/*
  Softshell: Dynamic Scheduling on GPUs.
  http://www.icg.tugraz.at/project/mvp

  Copyright (C) 2012 Institute for Computer Graphics and Vision,
                     Graz University of Technology

  Author(s):  Markus Steinberger - steinberger ( at ) icg.tugraz.at
              Bernhard Kainz - kainz ( at ) icg.tugraz.at
              Michael Kenzel - kenzel ( at ) icg.tugraz.at
              Stefan Hauswiesner - hauswiesner ( at ) icg.tugraz.at
              Bernhard Kerbl - kerbl ( at ) icg.tugraz.at
              Dieter Schmalstieg - schmalstieg ( at ) icg.tugraz.at

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/


/*
* file created by    Markus Steinberger / steinberger ( at ) icg.tugraz.at
*
* modifications by
*/

#ifndef SOFTSHELL_MEGAKERNEL_KERNELISSUER_INCLUDED
#define SOFTSHELL_MEGAKERNEL_KERNELISSUER_INCLUDED

#include "tools/types.h"
#include "tools/thread.h"
#include "tools/idmanager.h"
#include "tools/utils.h"
#include <vector>


namespace Softshell
{
  class KernelIssuer : public Thread
  {
  private:

    static const uint ConcurrentLaunches = 1;

    volatile bool run;
    volatile bool sleep;

    Signal startsignal;
    Signal signal;
    Mutex mutex;

    uint mydevice;
    dim blockSize;
    uint blocks;

    uint currentLaunchId;
    uint runningKernels;
    
    cudaEvent_t events[ConcurrentLaunches];
    uint ids[ConcurrentLaunches];
    uint running, issuing;

  protected:
    cudaStream_t stream;
    cudaStream_t stream_alt;

  private:
    bool issue()
    {
      if(runningKernels >= ConcurrentLaunches)
       return false;

      if(++currentLaunchId == 0)
        ++currentLaunchId;
      ids[issuing] = currentLaunchId;
      CUDA_CHECKED_CALL(cudaEventCreateWithFlags(&events[issuing], cudaEventDisableTiming | cudaEventBlockingSync));

      call(blocks, blockSize, ids[issuing]);
      CUDA_CHECKED_CALL(cudaEventRecord(events[issuing], stream));

      issuing = (issuing + 1) % ConcurrentLaunches;
      ++runningKernels;
      return true;
    }
    void finished()
    {

      CUDA_CHECKED_CALL(cudaEventDestroy(events[running]));
      events[running] = 0;
      running = (running + 1) % ConcurrentLaunches;
      --runningKernels;
    }
    void stopQueuedKernels()
    {
      while(runningKernels > 0)
      {
        terminate(ids[running]);
        if(events[running] != 0)
        {
          CUDA_CHECKED_CALL(cudaEventSynchronize(events[running]));
          CUDA_CHECKED_CALL(cudaEventDestroy(events[running]));
          events[running] = 0;
        }
        running = (running + 1) % ConcurrentLaunches;
        --runningKernels;
      }
    }
    int execute()
    {
      running = issuing = 0;
      runningKernels = 0;
      CUDA_CHECKED_CALL(cudaSetDevice(mydevice));
      CUDA_CHECKED_CALL(cudaStreamCreate(&stream));
      CUDA_CHECKED_CALL(cudaStreamCreate(&stream_alt));

      bool firsttime = true;
      while(run)
      {
        Guard g(mutex);
        if(firsttime) startsignal.signal();
        if(sleep)
        {
          if(!firsttime)
          {
            //send message to kill the running kernels
            g.unlock();
            stopQueuedKernels();
            informStopped();
            g.lock();
          }
          if(sleep)
          {
            g.unlock();
            signal.wait();
            g.lock();
          }
          if(!firsttime)
            informStarted();
        }
        if(firsttime)
        {
          informStarted();
          firsttime = false;
        }
        if(run && !sleep)
        {
          //issue one to X kernels, depending on the number of running kernels
          for(int i = runningKernels; i < ConcurrentLaunches; ++i)
            issue();
          g.unlock();
          //wait for the previous call to end
          cudaError_t err = cudaEventSynchronize(events[running]);
          if(err != cudaSuccess)
            failed(err);
          g.lock();
          finished();
        }
      }

      cudaStreamDestroy(stream);
      cudaStreamDestroy(stream_alt);
      //printf("thread execute finished\n");
      return 0;
    }
  protected:
    virtual void failed(cudaError_t &) = 0;
    virtual void informStopped() { }
    virtual void informStarted() { }
    virtual void terminate(uint launchId) = 0;
    virtual void call(uint blocks, const dim& blockSize, uint launchId) = 0;
  public:
    KernelIssuer(uint device_id, dim blockDim) :
        currentLaunchId(0), run(true), sleep(true), mydevice(device_id), blockSize(blockDim), blocks(60)
    {
      Thread::start();
      startsignal.wait();
    }
    virtual ~KernelIssuer() { }
    void setNumberOfBlocks(uint blocks) { KernelIssuer::blocks = blocks; }
    void setBlockSize(dim blockDim) { blockSize = blockDim; }
    void start()
    {
      Guard g(mutex);
      sleep = false;
      signal.signal();
    }
    void pause()
    {
      Guard g(mutex);
      sleep = true;
    }
    void end()
    {
      Guard g(mutex);
      run = false;
      sleep = false;
      signal.signal();
    }
    void join()
    {
      //printf("KernerIssuer: before thread join:\n");
      Thread::join();
      //printf("KernerIssuer: after thread join:\n");
    }
    std::vector<uint> getActiveLaunches()
    {
      Guard g(mutex);
      std::vector<uint> res(runningKernels,0);
      for(uint i = running, c = 0; c < runningKernels; ++c, i = (i + 1)%ConcurrentLaunches)
        res[c] = ids[i];
      return res;
    }
    void stopActiveLaunches()
    {
      Guard g(mutex);
      //send terminate message for all running kernels
      uint nowRunningKernes = runningKernels;
      uint runningId = running;
      while(nowRunningKernes > 0)
      {
        //printf("terminating: %d\n",ids[runningId]);
        terminate(ids[runningId]);
        runningId = (runningId + 1) % ConcurrentLaunches;
        --nowRunningKernes;
      }
    }
    bool isPaused() const { return sleep; }
    bool isRunning() const { return run; }
  };
}


#endif //SOFTSHELL_ISSUER_KERNELISSUER_INCLUDED
