/*
  Softshell: Dynamic Scheduling on GPUs.
  http://www.icg.tugraz.at/project/mvp

  Copyright (C) 2012 Institute for Computer Graphics and Vision,
                     Graz University of Technology

  Author(s):  Markus Steinberger - steinberger ( at ) icg.tugraz.at
              Bernhard Kainz - kainz ( at ) icg.tugraz.at
              Michael Kenzel - kenzel ( at ) icg.tugraz.at
              Stefan Hauswiesner - hauswiesner ( at ) icg.tugraz.at
              Bernhard Kerbl - kerbl ( at ) icg.tugraz.at
              Dieter Schmalstieg - schmalstieg ( at ) icg.tugraz.at

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/


/*
* file created by    Markus Steinberger / steinberger ( at ) icg.tugraz.at
*
* modifications by
*/

#ifndef SOFTSHELL_MEGAKERNEL_MEGAKERNEL_SIMPLE_CUH_INCLUDED
#define SOFTSHELL_MEGAKERNEL_MEGAKERNEL_SIMPLE_CUH_INCLUDED

#include "tools/utils.h"
#include "tools/types.h"
#include "tools/common.cuh"
#include "api/workitem.h"
#include "megakernel/controller.h"
#include "megakernel/launchstates.cuh"
#include "megakernel/execstate.cuh"
#include "timing/timesync.cuh"
#include "queue/dscheduler.cuh"
#include "queue/queue.cuh"
#include "communicator/hparams.h"
#include "distributor/proceduremanager.cuh"
#include "aggregation/localaggregation.cuh"
#include "megakernel/localprepull.cuh"
#include "memory/roundrobinalloc.cuh"
#include "megakernel/controller.cuh"
#include "priorities/eventPriorityEval.h"
#include <stdio.h>


namespace Softshell
{
  extern __constant__ float maxExecTime;
  extern __constant__ uint runningThreads;
  extern __device__ uint sleepTime;
  extern __constant__ uint defaultSharedMemory;
  extern __constant__ int stopOnNoWork;

 #define state \
  ((volatile BlockExecState*)(__sharedMemPointer(0)))
#define ExecStates \
  ((volatile ExecState*)__sharedMemPointer(sizeof(BlockExecState)))


  __device__ void d_worker_simple(uint launchId,  uint additionalSharedMemory)
  {
    volatile ExecState* es = ExecStates;
    {
      __execStateOffset() = __basicSharedRequirements() + sizeof(BlockExecState)/sizeof(uint);

      BlockExecState* bes = (BlockExecState*)(__sharedMemPointer(0));
      bes->launchId = launchId;
      bes->run = true;
      bes->starttime = getTimeCycles();
      bes->sharedOffset = __basicSharedRequirements()+(sizeof(BlockExecState)+sizeof(ExecState))/sizeof(uint);
      bes->sharedOffsetEnd = additionalSharedMemory/sizeof(uint);
      bes->sharedOffsetTop = (additionalSharedMemory-16)/sizeof(uint);
      
      es->syncpoint = 2;
      es->sharedOffset = bes->sharedOffset;
      es->threadOffset = 0;
    }
      
    __shared__ int didWork;
    didWork = 0;
    state->checkRunstate(threadIdx.x == 0);
    syncthreads(0);
    while(true)
    //for(int i = 0; i < 1000; ++i)
    {      
      if(threadIdx.x == 0)
      {
        if(d_queue.dequeue((Workpackage* volatile *)&es->wp, (DeviceEntryProcedureFunc volatile *) &es->proc, es->nThreads))
        {
          //printf("%d %d got wp: %llx %llx %d\n",blockIdx.x, threadIdx.x, es->wp, es->proc, es->nThreads);
          es->eventId = ((Workpackage*)es->wp)->getEventId();
          es->eventLaunchId = ((Workpackage*)es->wp)->getEventLaunchId();
          didWork = 1;
        }
        else if(didWork & stopOnNoWork)
        {
          //printf("%d %d end due to no work\n", launchId, blockIdx.x);
          d_launchStates->endLaunch(launchId);
          state->run = false;
        }

      }
      __syncthreads();

      if(es->proc != 0 && threadIdx.x < es->nThreads)
      {
        //if(threadIdx.x == 0)
        //  printf(" %d executing: %llx %llx %d\n", blockIdx.x, es->wp, es->proc, es->nThreads);
        DeviceEntryProcedureFunc(es->proc)((Workpackage*)es->wp);
      }
      __syncthreads();
      if(es->proc != 0 && threadIdx.x == 0)
      {
        es->proc = 0;
        //printf("%d notifying wp done\n", blockIdx.x); 
        d_eventManager.finishedWorkpackagesForEvent(es->eventId, es->eventLaunchId, 1);
      }

      state->checkRunstate(threadIdx.x == 0);
      __syncthreads();
      if(!state->run)
        return;
    }
    //printf("%d megakernel end due to runs\n",launchId);
  }

  __global__ void d_megakernel_simple(uint launchId, ControllerFunc controllerfunc, uint sortingThreads, uint additionalSharedMemory)
  {
    //initial check
    ___i_shared_all[0] = d_launchStates->checkRunState(launchId);
    syncthreads();
    if(!___i_shared_all[0])
      return;

    if(blockIdx.x + blockIdx.y + blockIdx.z == 0)
    {
      volatile BlockExecState* bstate = (BlockExecState*)(___i_shared_all + (additionalSharedMemory-sizeof(BlockExecState))/sizeof(uint));
      bstate->launchId = launchId;
      bstate->run = true;
      bstate->starttime = getTimeCycles();
      controllerfunc(bstate, sortingThreads);
    }
    else
     d_worker_simple(launchId, additionalSharedMemory);
  }


  template<class PriorityEval, bool SortQueue, bool AggregateWorkItems>
  __global__ void d_megakernel_simple_def_controller(uint launchId, uint sortingThreads, uint additionalSharedMemory)
  {
    //initial check
    ___i_shared_all[0] = d_launchStates->checkRunState(launchId);
    syncthreads();
    if(!___i_shared_all[0])
      return;

    if(blockIdx.x + blockIdx.y + blockIdx.z == 0)
    {
      volatile BlockExecState* bstate = (BlockExecState*)(___i_shared_all + (additionalSharedMemory-sizeof(BlockExecState))/sizeof(uint));
      bstate->launchId = launchId;
      bstate->run = true;
      bstate->starttime = getTimeCycles();
      Controller<PriorityEval, SortQueue, AggregateWorkItems>(bstate, sortingThreads);
    }
    else
     d_worker_simple(launchId, additionalSharedMemory);
  }

  
  __global__ void d_separate_worker_simple(uint launchId, uint additionalSharedMemory)
  {
    //initial check
    ___i_shared_all[0] = d_launchStates->checkRunState(launchId);
    syncthreads();
    if(!___i_shared_all[0])
      return;
    d_worker_simple(launchId, additionalSharedMemory);
  }

  template __global__ void d_megakernel_simple_def_controller<EventPriorityEval, true, true>(uint launchId, uint sortingThreads, uint additionalSharedMemory);
  template __global__ void d_megakernel_simple_def_controller<EventPriorityEval, true, false>(uint launchId, uint sortingThreads, uint additionalSharedMemory);
  template __global__ void d_megakernel_simple_def_controller<EventPriorityEval, false, true>(uint launchId, uint sortingThreads, uint additionalSharedMemory);
  template __global__ void d_megakernel_simple_def_controller<EventPriorityEval, false, false>(uint launchId, uint sortingThreads, uint additionalSharedMemory);
}


#endif //SOFTSHELL_MEGAKERNEL_MEGAKERNEL_CUH_INCLUDED
