/*
  Softshell: Dynamic Scheduling on GPUs.
  http://www.icg.tugraz.at/project/mvp

  Copyright (C) 2012 Institute for Computer Graphics and Vision,
                     Graz University of Technology

  Author(s):  Markus Steinberger - steinberger ( at ) icg.tugraz.at
              Bernhard Kainz - kainz ( at ) icg.tugraz.at
              Michael Kenzel - kenzel ( at ) icg.tugraz.at
              Stefan Hauswiesner - hauswiesner ( at ) icg.tugraz.at
              Bernhard Kerbl - kerbl ( at ) icg.tugraz.at
              Dieter Schmalstieg - schmalstieg ( at ) icg.tugraz.at

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/


/*
* file created by    Markus Steinberger / steinberger ( at ) icg.tugraz.at
*
* modifications by
*/

#include "megakernel/controller.h"
#include "megakernel/megakernelissuer.h"
#include "megakernel/megakernel.h"

#include "def_controller.cuh"
#include "controller.cuh"
#include "priorities/eventPriorityEval.h"

#include "communicator/hcommunicator.h"
#include "communicator/messageids.h"
#include "tools/config.h"

#include "memory/ScatterAlloc.cuh"

namespace Softshell
{
  MegaKernelIssuer::MegaKernelIssuer(Config* _config,HCommunicator* hcommunicator, uint device) :
    KernelIssuer(device, _config->registerConfig("MegakernelBlockDim", 512U)),
    runningBlockDim(_config->registerConfig("MegakernelBlockDim", 512U)),
    sortingThreads(_config->registerConfig("ControllerSortingThreads", 256U)),
    allocSpace(_config->registerConfig("MegakernelSharedMemory", 4096U)),
    maxConcurrentWorkpackages(_config->registerConfig("MaxConcurrentWorkpackages", 8U)),
    controllerInfo(_config->registerConfig("Controller",getDefaultController(_config))),
    useSimpleMegakernel(_config->registerConfig("SimpleMegakernel",0U)),
    separateController(_config->registerConfig("SeparateController",0U)),
    communicator(hcommunicator),
    config(_config)
  {
    cudaMemcpyToSymbol(runningThreads,&runningBlockDim, sizeof(uint));
    controllerInfo.func();
  }
  MegaKernelIssuer::MegaKernelIssuer(Config* _config, HCommunicator* hcommunicator, uint device, const ControllerInfo& controller) :
    runningBlockDim(_config->registerConfig("MegakernelBlockDim", 512U)),
    sortingThreads(_config->registerConfig("ControllerSortingThreads", 256U)),
    allocSpace(_config->registerConfig("MegakernelSharedMemory", 4096U)),
    maxConcurrentWorkpackages(_config->registerConfig("MaxConcurrentWorkpackages", 8U)),
    KernelIssuer(device, _config->registerConfig("MegakernelBlockDim", 512U)),
    controllerInfo(_config->registerConfig("Controller",controller)),
    useSimpleMegakernel(_config->registerConfig("SimpleMegakernel",0U)),
    separateController(_config->registerConfig("SeparateController",0U)),
    communicator(hcommunicator),
    config(_config)
  {
    cudaMemcpyToSymbol(runningThreads,&runningBlockDim, sizeof(uint));
    controllerInfo.func();
  }

  void MegaKernelIssuer::call(uint blocks, const dim& blockSize, uint launchId)
  {
    static size_t usedBlockSize = 0;
    if(usedBlockSize != runningBlockDim)
    {
      usedBlockSize = runningBlockDim;
      cudaMemcpyToSymbol(runningThreads,&runningBlockDim, sizeof(uint));
    }

    if(maxConcurrentWorkpackages < 1)
      maxConcurrentWorkpackages = 1;
    if(maxConcurrentWorkpackages > 12)
      maxConcurrentWorkpackages = 12;

    uint GroupSize = runningBlockDim/maxConcurrentWorkpackages;
    GroupSize = (GroupSize + 31)/32*32;

    if(!useSimpleMegakernel)
    {
      while(GroupSize*maxConcurrentWorkpackages > runningBlockDim)
        --maxConcurrentWorkpackages;
      runningBlockDim = GroupSize*maxConcurrentWorkpackages;
    }

    static ControllerInfo DefControllerTT = getDefaultController(true, true);
    static ControllerInfo DefControllerTF = getDefaultController(true, false);
    static ControllerInfo DefControllerFT = getDefaultController(false, true);
    static ControllerInfo DefControllerFF = getDefaultController(false, false);

    //call the kernel

    size_t sharedmemExec = 0;
    sharedmemExec += runningBlockDim/32 * sizeof(uint); //execstateoffsets
    sharedmemExec += 128; //sizeof(BlockExecState)
    sharedmemExec += allocSpace;
    if(!useSimpleMegakernel)
    {
      sharedmemExec += 32;//sizeof(LocalPrePull)
      sharedmemExec += 48*maxConcurrentWorkpackages; //sizeof(ExecState)
    }

    size_t sharedmem;

    if(!separateController)
    {
      size_t sharedmemController = controllerInfo.requiredSharedMemory();
      sharedmem = 16U+std::max(sharedmemExec, sharedmemController);
    }
    else
      sharedmem= 16U+sharedmemExec;
    sharedmem = (sharedmem + 15)/16*16;

    if(separateController)
    {
      controllerInfo.launchController(launchId, stream_alt);
      if(useSimpleMegakernel)
        d_separate_worker_simple<<<blocks,runningBlockDim,sharedmem,stream>>>(launchId, sharedmem);
      else
        d_separate_worker<<<blocks,runningBlockDim,sharedmem,stream>>>(launchId, sharedmem, maxConcurrentWorkpackages, GroupSize);
    }
    else if(controllerInfo == DefControllerTT)
    {
      if(useSimpleMegakernel)
        d_megakernel_simple_def_controller<EventPriorityEval,true,true><<<blocks,runningBlockDim,sharedmem,stream>>>(launchId, sortingThreads, sharedmem);
      else
        d_megakernel_def_controller<EventPriorityEval,true,true><<<blocks,runningBlockDim,sharedmem,stream>>>(launchId, sortingThreads, sharedmem, maxConcurrentWorkpackages, GroupSize);
    }
    else if(controllerInfo == DefControllerTF)
    {
      if(useSimpleMegakernel)
        d_megakernel_simple_def_controller<EventPriorityEval,true,false><<<blocks,runningBlockDim,sharedmem,stream>>>(launchId, sortingThreads, sharedmem);
      else
        d_megakernel_def_controller<EventPriorityEval,true,false><<<blocks,runningBlockDim,sharedmem,stream>>>(launchId, sortingThreads, sharedmem, maxConcurrentWorkpackages, GroupSize);
    }
    else if(controllerInfo == DefControllerFT)
    {
      if(useSimpleMegakernel)
        d_megakernel_simple_def_controller<EventPriorityEval,false,true><<<blocks,runningBlockDim,sharedmem,stream>>>(launchId, sortingThreads, sharedmem);
      else
        d_megakernel_def_controller<EventPriorityEval,false,true><<<blocks,runningBlockDim,sharedmem,stream>>>(launchId, sortingThreads, sharedmem, maxConcurrentWorkpackages, GroupSize);
    }
    else if(controllerInfo == DefControllerFF)
    {
      if(useSimpleMegakernel)
        d_megakernel_simple_def_controller<EventPriorityEval,false,false><<<blocks,runningBlockDim,sharedmem,stream>>>(launchId, sortingThreads, sharedmem);
      else
        d_megakernel_def_controller<EventPriorityEval,false,false><<<blocks,runningBlockDim,sharedmem,stream>>>(launchId, sortingThreads, sharedmem, maxConcurrentWorkpackages, GroupSize);
    }
    else
    {
      if(useSimpleMegakernel)
        d_megakernel_simple<<<blocks,runningBlockDim,sharedmem,stream>>>(launchId, controllerInfo.func(), sortingThreads, sharedmem);
      else
        d_megakernel<<<blocks,runningBlockDim,sharedmem,stream>>>(launchId, controllerInfo.func(), sortingThreads, sharedmem, maxConcurrentWorkpackages, GroupSize);
    }

  }
  void MegaKernelIssuer::terminate(uint launchId)
  {
    communicator->send(H2D_Terminate, sizeof(uint), &launchId);
  }
}
