/*
  Softshell: Dynamic Scheduling on GPUs.
  http://www.icg.tugraz.at/project/mvp

  Copyright (C) 2012 Institute for Computer Graphics and Vision,
                     Graz University of Technology

  Author(s):  Markus Steinberger - steinberger ( at ) icg.tugraz.at
              Bernhard Kainz - kainz ( at ) icg.tugraz.at
              Michael Kenzel - kenzel ( at ) icg.tugraz.at
              Stefan Hauswiesner - hauswiesner ( at ) icg.tugraz.at
              Bernhard Kerbl - kerbl ( at ) icg.tugraz.at
              Dieter Schmalstieg - schmalstieg ( at ) icg.tugraz.at

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/


/*
* file created by    Markus Steinberger / steinberger ( at ) icg.tugraz.at
*
* modifications by
*/

#include "memory/mappedMemory.h"

#include "tools/utils.h"
#include "tools/types.h"

#include <iostream>
#include <limits>

namespace Softshell
{ 
  MappedMemory::MemInfo::MemInfo(int _offset, uint _size, MemInfo* _prev, MemInfo* _next, bool _used) : offset(_offset), size(_size), prev(_prev), next(_next), used(_used) 
  {
  }

  int MappedMemory::MemInfo::alloc(uint _size)
  {
    _size += 16 - (_size%16);
    if(used)
    {
      //std::cout << "MemInfo: " << offset << " (" << size << ") already in use" << std::endl;
      return -1;
    }
    if(_size > size)
    {  
      //std::cout << "MemInfo: " << offset << " (" << size << ") too small" << std::endl;
      return -1;
    }
    if(_size + 16 < size)
    {
      uint restsize = size - _size;
      size = _size;
      used = true;
      MemInfo *nM = new MemInfo(offset+size, restsize, this, next);
      if(next) next->prev = nM;
      next = nM;
      //std::cout << "MemInfo: " << offset << " (" << size << ") resized and taken" << std::endl;
      return offset;
    }
    else
    {
      //std::cout << "MemInfo: " << offset << " (" << size << ") taken directly" << std::endl;
      used = true;
      return offset;
    }
  }
  void MappedMemory::MemInfo::free()
  {
    used = false;
    if(next && next->used == false)
    {
      MemInfo *d = next;
      size += next->size;
      next = next->next;
      if(next)
        next->prev = this;
      delete d;
    }
    if(prev && prev->used == false)
    {
      prev->size += size;
      if(next) next->prev = prev;
      prev->next = next;
      delete this;
    }
  }

  MappedMemory::MappedMemory(size_t _memorysize) : memorysize(_memorysize)
  {
    //std::cout << "allocating: " << CONTROLLED_MEMORY_SIZE << std::endl; 
    CUDA_CHECKED_CALL(cudaHostAlloc((void**)&h_memory, memorysize, cudaHostAllocMapped));
    CUDA_CHECKED_CALL(cudaHostGetDevicePointer((void**)&d_memory, h_memory, 0));
    begin = new MemInfo(0, _memorysize);
  }
  int MappedMemory::allocOffset(uint size)
  {
    Guard g(mutex);
    //std::cout << "MemInfo: trying to alloc " << size << std::endl;
    for(MemInfo* it = begin; it != 0; it = it->next)
    {
      int offset = it->alloc(size);
      if(offset != -1)
      {
        //std::cout << std::endl;
        return offset;
      }
    }
    std::cout << "WARNING OUT OF MAPPED MEMORY" << std::endl;
    return -1;
  }
  std::pair<void*,void*> MappedMemory::alloc(uint size)
  {
    int o = allocOffset(size);
    if(o == -1)
      return std::pair<void*,void*>((void*)0,(void*)(0));
    else
      return std::pair<void*,void*>((void*)(h_memory + o),(void*)(d_memory + o));
  }
  void MappedMemory::free(int p)
  {
    Guard g(mutex);
    for(MemInfo* it = begin; it != 0; it = it->next)
    {
      if(it->offset == p)
      {
        it->free();
        return;
      }
    }
    std::cout << "WARNING COULD NOT FIND MAPPED MEMORY TO BE FREED" << std::endl;
  }

  void MappedMemory::free(std::pair<void*,void*> ppair)
  {
    unsigned char *pc = reinterpret_cast<unsigned char*>(ppair.first);
    size_t d = pc - h_memory;
    free(d);
  }
  void MappedMemory::free(void* p)
  {
    unsigned char *pc = reinterpret_cast<unsigned char*>(p);
    size_t d;
    if(pc > h_memory && pc - h_memory < memorysize) d = pc - h_memory;
    else if(pc > d_memory && pc - d_memory < memorysize) d = pc - d_memory;
    else 
    {
      std::cout << "WARNING COULD NOT FIND MAPPED MEMORY TO BE FREED (POINTER IS NEITHER HOST NOR DEVICE POINTER)" << std::endl;
      return;
    }
    free(d);
  }
  void* MappedMemory::d_offsetToPointer(int offset) const
  {
    if(offset == -1) return 0;
    return reinterpret_cast<void*>(d_memory + offset);
  }
  void* MappedMemory::h_offsetToPointer(int offset) const
  {
    if(offset == -1) return 0;
    return reinterpret_cast<void*>(h_memory + offset);
  }
  void* MappedMemory::getDevicePointer(void* hostpointer) const
  {
    unsigned char *pc = reinterpret_cast<unsigned char*>(hostpointer);
    if(pc > h_memory && pc - h_memory < memorysize)
      return d_memory + (pc - h_memory);
    else 
      return 0;
  }
  void* MappedMemory::getHostPointer(void* devicepointer) const
  {
    unsigned char *pc = reinterpret_cast<unsigned char*>(devicepointer);
    if(pc > d_memory && pc - d_memory < memorysize)
      return h_memory + (pc - d_memory);
    else 
      return 0;
  }
}
