/*
  Softshell: Dynamic Scheduling on GPUs.
  http://www.icg.tugraz.at/project/mvp

  Copyright (C) 2012 Institute for Computer Graphics and Vision,
                     Graz University of Technology

  Author(s):  Markus Steinberger - steinberger ( at ) icg.tugraz.at
              Bernhard Kainz - kainz ( at ) icg.tugraz.at
              Michael Kenzel - kenzel ( at ) icg.tugraz.at
              Stefan Hauswiesner - hauswiesner ( at ) icg.tugraz.at
              Bernhard Kerbl - kerbl ( at ) icg.tugraz.at
              Dieter Schmalstieg - schmalstieg ( at ) icg.tugraz.at

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/


/*
* file created by    Markus Steinberger / steinberger ( at ) icg.tugraz.at
*
* modifications by
*/

#ifndef INTEGRATED_LINKED_LIST_CUH
#define INTEGRATED_LINKED_LIST_CUH

#include "asms.cuh"

namespace std_gpu
{


  template<class ListElement, bool Count = false>
  class _custom_linked_list
  {
    // single linked list, which is save to push back from any thread
    // visitor can go through list and remove unneeded items
  public:
    class Iterator
    {
      ListElement* element;
      ListElement* prev_element;
     public:
      __device__ Iterator(ListElement* el_prev, ListElement* el) :
        element(el), prev_element(el_prev) { }
      __device__ ListElement* operator* ()
      {
        return element;
      }
      __device__ ListElement* operator-> ()
      {
        return element;
      }
       __device__ const Iterator& operator ++()
      {
        prev_element = element;
        element = element->_next;
        return * this;
      }
       __device__ bool operator == (const Iterator& other) const
      {
        return element == other.element;
      }
        __device__ bool operator != (const Iterator& other) const
      {
        return element != other.element;
      }
    };
    class const_Iterator
    {
      const ListElement* element;
      const ListElement* prev_element;
    public:
      __device__ const_Iterator(ListElement* el_prev, ListElement* el) :
        element(el), prev_element(el_prev) { }
      __device__ const ListElement* operator* ()
      {
        return element;
      }
      __device__ const ListElement* operator ->()
      {
        return element;
      }
       __device__ const const_Iterator& operator ++()
      {
        prev_element = element;
        element = element->_next;
        return * this;
      }
       __device__ bool operator == (const const_Iterator& other) const
      {
        return element == other.element;
      }
        __device__ bool operator != (const const_Iterator& other) const
      {
        return element != other.element;
      }
    };

   protected:   
    ListElement* volatile _front;
    ListElement* volatile _back;
    volatile int _size;

    static __device__ ListElement* atomicSet(ListElement* volatile * address, ListElement* comp, ListElement* swap)
    {
      return (ListElement*) atomicCAS((PointerEquivalent*)(address), (PointerEquivalent)(comp), (PointerEquivalent)(swap));
    }
    static __device__ ListElement* atomicReplace(ListElement* volatile * address, ListElement* swap)
    {
      return (ListElement*) atomicExch((PointerEquivalent*)(address),(PointerEquivalent)(swap));
    }
  public:
    __device__ void init()
    {
      _front = _back = nullptr;
      _size = 0;
    }

    __device__ void push_back(ListElement* element)
    {
      ListElement* prev = atomicReplace(&_back, element);
      //ListElement* prev =  (ListElement*) atomicExch((PointerEquivalent*)(&_back),(PointerEquivalent)(element));
      if(prev != nullptr)
        atomicReplace(&prev->_next,element);
        //atomicExch((PointerEquivalent*)(&prev->_next),(PointerEquivalent)(element));
      else
        atomicReplace(&_front,element);
        //atomicExch((PointerEquivalent*)(&_front),(PointerEquivalent)(element));
      if(Count)
        atomicAdd((int*)&_size,1);
    }

    __device__ ListElement* pop_front()
    {
      ListElement* target;
      while(true)
      {
       target = _front;
       if(target == nullptr)
         return nullptr;
       ListElement* _next = target->_next;
       ListElement* old = atomicSet(&_front, target, _next);
       if(old == target)
       {
         if(_next == 0)
         {
           old = atomicSet(&_back, target, nullptr);
           if(old != target)
           {
             while(target->_next == nullptr);
             atomicReplace(&_front,target->_next);
             //atomicExch((PointerEquivalent*)(&_front),(PointerEquivalent)(target->_next));
           }
         }
         if(Count)
           atomicSub((int*)&_size,1);
         return target;
       }
      }
    }

    __device__ Iterator begin()
    {
      return Iterator(nullptr,_front);
    }
    __device__ Iterator end()
    {
      return Iterator(_back,nullptr);
    }
    __device__ const_Iterator begin() const
    {
      return const_Iterator(nullptr,_front);
    }
    __device__ const_Iterator end() const
    {
      return const_Iterator(_back,nullptr);
    }
     
    __device__ void erase(const Iterator& it)
    {
      ListElement* _next = it.element->_next;
      if(_next == nullptr)
        if(atomicSet(_back,it.element,it.prev_element) != it.element)
          do{_next = it.element->_next; } while(_next == nullptr);
      if(it.prev_element == nullptr)
        atomicSet(&_front,it.element,_next);
      else
        atomicReplace(&it.prev_element->_next,_next);
        //atomicExch((PointerEquivalent*)(&it.prev_element->_next),(PointerEquivalent)(_next));

      if(Count)
        atomicSub((int*)&_size,1);
    }
    __device__ void clear()
    {
      ListElement* it = _front;
      _front = nullptr;
      _back = nullptr;
      ListElement* temp = nullptr;
      for( ; it != nullptr; it = temp)
      {
        temp = it->_next;
        delete it;
      }
      _size = 0;
    }
    __device__ int size() const
    {
      if(Count)
        return _size;
      else
        return -1;
    }
  };

  template<class ListElement, bool Count = false, bool PassOwnership = false>
  class custom_linked_list : public _custom_linked_list<ListElement,Count>
  {
  public:
    __device__ custom_linked_list()
    {
      _front = _back = nullptr;
      _size = 0;
    }
    __device__ custom_linked_list(const custom_linked_list& other) : _custom_linked_list<ListElement,Count>(other)
    {
      if(!PassOwnership)
      {
        ListElement *f = other._front;
        _front = f;
        ListElement *last = nullptr;
        for( ; f != nullptr; f = f->_next)
        {
          ListElement * nelement = new ListElement(*f);
          if(last) last->_next = nelement;
          last = nelement;
        }
        _back = last;
      }
      else
      {
        const_cast<custom_linked_list*>(&other)->_front = nullptr;
        const_cast<custom_linked_list*>(&other)->_back = nullptr;
      }
    }
    __device__ ~custom_linked_list()
    {
      ListElement* it = _front;
      _front = nullptr;
      _back = nullptr;
      ListElement* temp = nullptr;
      for( ; it != nullptr; it = temp)
      {
        temp = it->_next;
        delete it;
      }
      _size = 0;
    }
  };

  template<class T>
  struct IntegratedListElement
  {
    friend class Iterator;
    IntegratedListElement* volatile _next;
    T data;
  public:
    __device__ IntegratedListElement() : _next(nullptr) { }
    __device__ IntegratedListElement(const T& d) : _next(nullptr), data(d)  { }
    __device__ T& operator-> ()
    {
      return data;
    }
    __device__ const T& operator-> () const
    {
      return data;
    }
    __device__ T& operator* ()
    {
      return data;
    }
    __device__ const T& operator* () const
    {
      return data;
    }
  };
  template<class T, bool Count = false>
  class _integrated_linked_list : public _custom_linked_list<IntegratedListElement<T>, Count >
  {
  public:
    typedef IntegratedListElement<T> ListElement;
    __device__ T* push_back(const T& data)
    {
      ListElement* element = new ListElement(data);
      ListElement* prev = atomicReplace(&_back, element);
      //ListElement* prev = (ListElement*)atomicExch((PointerEquivalent*)(&_back),(PointerEquivalent)(element));
      if(prev != nullptr)
        atomicReplace(&prev->_next,element);
        //atomicExch((PointerEquivalent*)(&prev->_next),(PointerEquivalent)(element));
      else
        atomicReplace(&_front,element);
        //atomicExch((PointerEquivalent*)(&_front),(PointerEquivalent)(element));
      if(Count)
        atomicAdd((int*)&_size,1);
      return &(*(*element));
    }
    __device__ void push_back(ListElement* element)
    {
      _custom_linked_list<IntegratedListElement<T>, Count >::push_back(element);
    }
  };

  template<class ListElement, bool Count = false, bool PassOwnership = false>
  class integrated_linked_list : public _integrated_linked_list<ListElement,Count>
  {
  public:
     __device__ integrated_linked_list()
    {
      _front = _back = nullptr;
      _size = 0;
    }
    __device__ integrated_linked_list(const custom_linked_list& other)  : _integrated_linked_list<ListElement,Count>(other)
    {
      if(!PassOwnership)
      {
        ListElement *f = other._front;
        _front = f;
        ListElement *last = nullptr;
        for( ; f != nullptr; f = f->_next)
        {
          ListElement * nelement = new ListElement(*f);
          if(last) last->_next = nelement;
          last = nelement;
        }
        _back = last;
      }
      else
      {
        const_cast<integrated_linked_list*>(&other)->_front = nullptr;
        const_cast<integrated_linked_list*>(&other)->_back = nullptr;
      }
    }
    __device__ ~integrated_linked_list()
    {
      ListElement* it = _front;
      _front = nullptr;
      _back = nullptr;
      ListElement* temp = nullptr;
      for( ; it != nullptr; it = temp)
      {
        temp = it->_next;
        delete it;
      }
      _size = 0;
    }
  };

}

#endif //MAINTAINED_LINKED_LIST_CUH