/*
  Softshell: Dynamic Scheduling on GPUs.
  http://www.icg.tugraz.at/project/mvp

  Copyright (C) 2012 Institute for Computer Graphics and Vision,
                     Graz University of Technology

  Author(s):  Markus Steinberger - steinberger ( at ) icg.tugraz.at
              Bernhard Kainz - kainz ( at ) icg.tugraz.at
              Michael Kenzel - kenzel ( at ) icg.tugraz.at
              Stefan Hauswiesner - hauswiesner ( at ) icg.tugraz.at
              Bernhard Kerbl - kerbl ( at ) icg.tugraz.at
              Dieter Schmalstieg - schmalstieg ( at ) icg.tugraz.at

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/


/*
* file created by    Markus Steinberger / steinberger ( at ) icg.tugraz.at
*
* modifications by
*/

#ifndef MAINTAINED_LINKED_LIST_CUH
#define MAINTAINED_LINKED_LIST_CUH

#include "pair.cuh"
#include "hash.cuh"
#include "asms.cuh"

namespace std_gpu
{



  template<class T>
  class _maintained_linked_list
  {
    // single linked list, which is save to push back from any thread
    // visitor can go through list and remove unneeded items

    struct ListElement
    {
      volatile T data;
      ListElement* volatile next;
      volatile int out;
      __device__ ListElement() { }
      __device__ ListElement(const T& d, ListElement* n = 0) : data(d), next(n), out(0) { }
      
    };

    uint _size;
    ListElement _front;
    ListElement* volatile _back;
    //ListElement* _toDelete;
    //ListElement* _lastVisitBack;

    static __device__ ListElement* atomicSet(ListElement* volatile * address, ListElement* comp, ListElement* swap)
    {
      PointerEquivalent old = atomicCAS((PointerEquivalent*)(address), (PointerEquivalent)(comp), (PointerEquivalent)(swap));
      return (ListElement*)(old);
    }
    static __device__ ListElement* atomicReplace(ListElement* volatile * address, ListElement* swap)
    {
      PointerEquivalent old = atomicExch((PointerEquivalent*)(address),(PointerEquivalent)(swap));
      return (ListElement*)(old);
    }
  public:
    __device__ void init()
    {
      _size = 0;
      _front.data = T(0);
      _front.next = 0;
      _back = &_front;
      //_lastVisitBack = 0;
      //_toDelete = 0;
    }

    __device__ volatile T& push_back(const T& data)
    {
      ListElement* element = new ListElement(data);
      if(element == 0)
      {
        printf("Error in Maintained Linked List: OUT OF MEMORY\nABORT\n");
        ::trap();
      }
      //ListElement* cback = _back;
      //ListElement* old = cback, *current;
      __threadfence();
      //do
      //{
      //  current = old;
      //  old = atomicSet(&current->next, 0, element);
      //} while(old != 0);
      //atomicSet(&_back, cback, element);
      //atomicSet(&_back, current, element);
      
      
      ListElement* prev = atomicReplace(&_back, element);
      atomicReplace(&prev->next,element);
      return element->data;
    }
    template<class Visitor>
    __device__ void visit(Visitor visitor, void*& commonLocal, uint linid = 0, uint threads = 1)
    {
      //ListElement* cback = 0;
      //ListElement*& element = (ListElement*&)(commonLocal);
      ListElement* element;
      ////remove elements from delete list
      //if(linid == 0 )
      //{
      //  while(_toDelete != 0 && _toDelete != _lastVisitBack)
      //  {
      //    ListElement* next = _toDelete->next;
      //    delete _toDelete;
      //    _toDelete = next;
      //  }
      //  cback = _back;
      //  _lastVisitBack = cback;
      //}

      
      ListElement* next, *last = &_front;
      element = _front.next;
      
      //ListElement* deleteBack = 0;
      if(threads > 1)
        __syncthreads();

      
      //int mycoco = 0;
      while(element != 0 /* && ++mycoco < 1000*/)
      {
        bool erase = true;
        if( element->out != 1 )
          erase = visitor.visit(element->data);
        if(threads > 1)
          __syncthreads();
        next = element->next;
        if(linid == 0)
        {
          if(erase)
          {
            //do not get rid of back...
            if(next == 0)
            {
              //printf("setting to zero (%llx->%llx->%llx)\n",last,element,next);
              element->out = 1;
              __threadfence();
            }
            else
            {
              //printf("deleting (%llx->%llx->%llx)\n",last,element,next);
              atomicReplace(&last->next,next);
              __threadfence();
              
              delete element;

              //last->next = next;
              //element->next = cback;
              //__threadfence();
              //if(deleteBack != 0)
              //{
              //  deleteBack->next = element;
              //  deleteBack = element;
              //}
              //else
              //  _toDelete = deleteBack = element;
            }
          }
          else
            last = element;
          //if(element == cback)
          //  next = 0;
          element = next;
        }
        if(threads > 1)
          __syncthreads();
      }
      //if(linid == 0 && mycoco >= 1000)
      //{
      //  printf("visit coco %d\n",mycoco);
      //  ::trap();
      //}
    }
  };

}

#endif //MAINTAINED_LINKED_LIST_CUH