/*
  Softshell: Dynamic Scheduling on GPUs.
  http://www.icg.tugraz.at/project/mvp

  Copyright (C) 2012 Institute for Computer Graphics and Vision,
                     Graz University of Technology

  Author(s):  Markus Steinberger - steinberger ( at ) icg.tugraz.at
              Bernhard Kainz - kainz ( at ) icg.tugraz.at
              Michael Kenzel - kenzel ( at ) icg.tugraz.at
              Stefan Hauswiesner - hauswiesner ( at ) icg.tugraz.at
              Bernhard Kerbl - kerbl ( at ) icg.tugraz.at
              Dieter Schmalstieg - schmalstieg ( at ) icg.tugraz.at

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/


/*
* file created by    Markus Steinberger / steinberger ( at ) icg.tugraz.at
*
* modifications by
*/

#include "timing/timesync.cuh"
#include "timing/timesync.h"
#include "communicator/dcommunicator.cuh"
#include "tools/utils.h"

#ifndef USE_CUDA_HEAP
#include "memory/UseScatterAlloc.cuh"
#endif

#include <iostream>

namespace Softshell
{

  __device__ clock64_t TIMESYNC_mp_cycle_offset[TIMESYNC_buffersize];
  __device__ double TIMESYNC_max_seconds_per_cycle;
  __device__ double TIMESYNC_seconds_per_cycle;
  __device__ clock64_t TIMESYNC_pulse;
  __device__ clock64_t TIMESYNC_realpulse;


  __device__ inline void TIMESYNC_atomic_min(clock64_t t, clock64_t* loc)
  {
    clock64_t old = *loc;
    while(old > t)
    {
      clock64_t told = atomicCAS(loc, old, t);
      if(told == old)
        break;
      else
        old = told;
    }
  }


  class TimeSyncer : public DMessageReceiver
  {
  public:
    __device__ TimeSyncer(DCommunicator* comm)
    {
      comm->registerMessageReceiver(this, H2D_TimePulse);
    }
    __device__  void receiveMessage(MessageHeader& h, volatile uint* data, uint linid, uint threads)
    {
      if(linid == 0)
      {
        clock64_t c = getTimeCycles();
        if(h.id == H2D_TimePulse)
        {
          //calc diff in time and update
          clock64_t ht = *reinterpret_cast<volatile clock64_t*>(data);

          double real_dt = (ht - TIMESYNC_realpulse);
          double dt = (c - TIMESYNC_pulse);
          double frac = real_dt/dt;
          //if we are too far off, there was another message blocking the queue
          if(frac > 1.0)
            return;

          double new_seconds_per_cycle = frac*TIMESYNC_max_seconds_per_cycle;

          //printf("updating time: \n\tdt %f realdt %f\n\tspc %f -> %f\n\tpulse %lld -> %lld \n\trealpulse %lld -> %lld\n",
          //  dt, real_dt,
          //  TIMESYNC_seconds_per_cycle*10000000000.0, new_seconds_per_cycle*10000000000.0,
          //  TIMESYNC_pulse, c,
          //  TIMESYNC_realpulse, ht);


          TIMESYNC_seconds_per_cycle = new_seconds_per_cycle;
          TIMESYNC_pulse = c;
          TIMESYNC_realpulse = ht;

        }
      }
    }
  };


  __global__ void TIMESYNC_write_timestamp(clock64_t *stamps)
  {
    clock64_t cyclecount = __clock64();
    uint myid = smid();
    TIMESYNC_atomic_min(cyclecount, &stamps[myid]);
    if(blockIdx.x == 0 && threadIdx.x == 0)
      new TimeSyncer(d_communicator);
  }

  //__global__ void tellTime()
  //{
  //  if(threadIdx.x == 0)
  //  {
  //    uint id = smid();
  //    double time = getTime();
  //    printf("%d: %f\n", id, time);
  //  }
  //}
  //__global__ void tellOffsetBuff(clock64_t* b)
  //{
  //  if(threadIdx.x == 0)
  //  {
  //    uint id = smid();
  //    printf("%d: %lld\n", id,  b[id]);
  //  }
  //}
  //  __global__ void tellOffset()
  //{
  //  if(threadIdx.x == 0)
  //  {
  //    uint id = smid();
  //    printf("%d: %lld\n", id,  TIMESYNC_mp_cycle_offset[id]);
  //  }
  //}

  void initTimeSync(int mpc, int grid, double clockrate)
  {
    clock64_t* d_temparray;
    CUDA_CHECKED_CALL(cudaMalloc(&d_temparray, sizeof(clock64_t)*TIMESYNC_buffersize));
    CUDA_CHECKED_CALL(cudaMemset(d_temparray, 0xFFFFFFFF, sizeof(clock64_t)*TIMESYNC_buffersize));

    double startime = TimeSync::time();
    TIMESYNC_write_timestamp<<<grid, 1>>>(d_temparray);
    CUDA_CHECKED_CALL(cudaDeviceSynchronize());
    cudaMemcpyToSymbol(TIMESYNC_mp_cycle_offset, d_temparray, sizeof(clock64_t)*TIMESYNC_buffersize, 0, cudaMemcpyDeviceToDevice);


    //tellOffsetBuff<<<5,512>>>(d_temparray);
    //CUDA_CHECKED_CALL(cudaDeviceSynchronize());

    //std::vector<clock64_t> cycles(TIMESYNC_buffersize);
    //cudaMemcpy(&cycles[0], d_temparray, sizeof(clock64_t)*TIMESYNC_buffersize, cudaMemcpyDeviceToHost);
    //for(uint i = 0; i < TIMESYNC_buffersize; ++i)
    //  std::cout << cycles[i] << std::endl;


    double sec_per_cycle = 1.0/clockrate;
    cudaMemcpyToSymbol(TIMESYNC_max_seconds_per_cycle, &sec_per_cycle, sizeof(double));
    cudaMemcpyToSymbol(TIMESYNC_seconds_per_cycle, &sec_per_cycle, sizeof(double));

    clock64_t realpulse = startime*clockrate;
    cudaMemcpyToSymbol(TIMESYNC_realpulse, &realpulse, sizeof(clock64_t));
    clock64_t null = 0;
    cudaMemcpyToSymbol(TIMESYNC_pulse, &null, sizeof(clock64_t));

    cudaFree(d_temparray);

    //CUDA_CHECKED_CALL(cudaDeviceSynchronize());
    //tellOffset<<<24,512>>>();
    //CUDA_CHECKED_CALL(cudaDeviceSynchronize());
    //tellTime<<<5,512>>>();
    //CUDA_CHECKED_CALL(cudaDeviceSynchronize());
  }


}
