/*
  Softshell: Dynamic Scheduling on GPUs.
  http://www.icg.tugraz.at/project/mvp

  Copyright (C) 2012 Institute for Computer Graphics and Vision,
                     Graz University of Technology

  Author(s):  Markus Steinberger - steinberger ( at ) icg.tugraz.at
              Bernhard Kainz - kainz ( at ) icg.tugraz.at
              Michael Kenzel - kenzel ( at ) icg.tugraz.at
              Stefan Hauswiesner - hauswiesner ( at ) icg.tugraz.at
              Bernhard Kerbl - kerbl ( at ) icg.tugraz.at
              Dieter Schmalstieg - schmalstieg ( at ) icg.tugraz.at

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/


/*
* file created by    Markus Steinberger / steinberger ( at ) icg.tugraz.at
*
* modifications by
*/

#ifndef SOFTSHELL_TOOLS_CONFIG_INCLUDED
#define SOFTSHELL_TOOLS_CONFIG_INCLUDED

#include <string>
#ifndef __GNUC__
#include <unordered_map>
#else
#include <map>
#endif
#include <exception>
#include <list>
#include <vector>
#include <sstream>
#include <iostream>
#include <stdexcept>
#include "tools/types.h"
#include "tools/thread.h"

#ifdef WIN32
#pragma warning( push )
#pragma warning( disable : 4244 4800 )
#endif

#define nullptr 0

namespace Softshell
{
  class Config
  {
  protected:
    template<class T>
    class EntryPoolMapping
    {
      static uint id;
    public:
      static uint get(uint& freeId)
      {
        if(id == 0)
          id = ++freeId;
        return id;
      }
    };

    class EntryCallbackBase
    {
    public:
      virtual EntryCallbackBase* clone() const = 0;
      virtual ~EntryCallbackBase() { }
    };
    template<class T>
    class EntryCallbackTypedIntermediate : public EntryCallbackBase
    {
    public:
      virtual void call(T& val, const std::string& name) const = 0;
    };
    template<class T, class CBArg = void>
    class EntryCallbackTyped : public EntryCallbackTypedIntermediate<T>
    {
    public:
      typedef void(*Callback)(const std::string& name, T& value, CBArg* arg);
      EntryCallbackTyped(Callback callback, CBArg* argument = 0) : cb(callback), arg(argument)
      { }
      void call(T& val, const std::string& name) const
      {
        cb(name, val, arg);
      }
      EntryCallbackBase* clone() const
      {
        return new EntryCallbackTyped(*this);
      }
    private:
      Callback cb;
      CBArg* arg;
    };
    class EntryCallback
    {
      EntryCallbackBase* callback;
    public:
      EntryCallback() : callback(0) { }
      template<class T, class CBarg>
      EntryCallback(T value, typename EntryCallbackTyped<T, CBarg>::Callback callback, CBarg* arg = 0) : callback(new EntryCallbackTyped<T, CBarg>(callback, arg)) { }
      ~EntryCallback() { if(callback) delete callback; }
      EntryCallback(const EntryCallback& other) : callback(other.callback->clone()) { }
      const EntryCallback& operator = (const EntryCallback& other)
      {
        callback = other.callback->clone();
        return *this;
      }

      template<class T>
      void call(T& val, const std::string& name)
      {
        static_cast<EntryCallbackTypedIntermediate<T>*>(callback)->call(val, name);
      }
    };

    class EntryBase
    {
    public:
      virtual EntryBase* cloneNoCallback() const = 0;
      virtual EntryBase* clone() const = 0;
      virtual ~EntryBase() { }
    };

    template<class T>
    class EntryTyped : public EntryBase
    {
    protected:

      T val;
      typedef std::list<EntryCallback> CallbackListType;
      CallbackListType callbacks;

      template<class TI, class CBarg>
      void addCallback(typename EntryCallbackTyped<T,CBarg>::Callback callback, CBarg* arg)
      {
        if(callback == 0)
          return;
        TI dummy = TI();
        callbacks.push_back(EntryCallback(dummy, callback, arg));
      }
    public:
      template<class CBarg>
      EntryTyped(T value, typename EntryCallbackTyped<T,CBarg>::Callback callback, CBarg* arg = 0) : val(value)
      {
        addCallback<T>(callback, arg);
      }
      template<class TAssign>
      void assign(TAssign newval, const std::string& name)
      {
        T nvval = newval;
        if(!(nvval == val))
        {
          val = nvval;
          for(CallbackListType::iterator it = callbacks.begin(); it != callbacks.end(); ++it)
            it->call(val, name);
        }
      }
      template<class CBarg>
      T& addListener(typename EntryCallbackTyped<T,CBarg>::Callback callback, CBarg* arg = 0)
      {
        addCallback<T>(callback, arg);
        return val;
      }
      const T& get() const
      {
        return val;
      }
      T& get()
      {
        return val;
      }
      EntryBase* cloneNoCallback() const
      {
        EntryTyped * newEntry = new EntryTyped(*this);
        newEntry->callbacks.clear();
        return newEntry;
      }
      EntryBase* clone() const
      {
        return new EntryTyped(*this);
      }
    };
  public:
    class Entry
    {

      EntryBase* entry;
      uint pool;
      bool own;

      template<class TTry, class T>
      bool tryAssign(const T& newval, const std::string& name)
      {
        uint trypoolid = EntryPoolMapping<TTry>::get(freeConfigMappings);
        if(trypoolid != pool)
          return false;
        //found it
        static_cast<EntryTyped<TTry>*>(entry)->assign(newval, name);
        return true;
      }
      template<class TTry>
      bool tryPipe(const std::string& newval, const std::string& name)
      {
        uint trypoolid = EntryPoolMapping<TTry>::get(freeConfigMappings);
        if(trypoolid != pool)
          return false;
        //found it
        std::stringstream sstr(newval);
        TTry val;
        sstr >> val;
        static_cast<EntryTyped<TTry>*>(entry)->assign(val, name);
        return true;
      }
      template<class T>
      bool tryAssignMatch(const T& newval, const std::string& name)
      {
        uint poolid = EntryPoolMapping<T>::get(freeConfigMappings);
        if(poolid != pool)
          return false;
        //same pool -> works
        static_cast<EntryTyped<T>*>(entry)->assign(newval, name);
        return true;
      }
      template<class T>
      void assignNumbers(const T& newval, const std::string& name)
      {
        if(tryAssignMatch(newval, name))
          return;
        else
        {
          //try standard types
          if(tryAssign<bool>(newval, name)) return;
          if(tryAssign<int>(newval, name)) return;
          if(tryAssign<unsigned int>(newval, name)) return;
          if(tryAssign<float>(newval, name)) return;
          if(tryAssign<double>(newval, name)) return;

          //a bit more exotic
          if(tryAssign<char>(newval, name)) return;
          if(tryAssign<unsigned char>(newval, name)) return;
          if(tryAssign<long>(newval, name)) return;
          if(tryAssign<unsigned long>(newval, name)) return;
          if(tryAssign<long long>(newval, name)) return;
          if(tryAssign<unsigned long long>(newval, name)) return;

          throw std::runtime_error(std::string("Config Error: different base types in assign for \"") + name + std::string("\""));
        }
      }

      template<class TTry, class T>
      bool tryGet(T& outval) const
      {
        uint trypoolid = EntryPoolMapping<TTry>::get(*(const_cast<uint*>(&freeConfigMappings)));
        if(trypoolid != pool)
          return false;
        //found it
        outval = static_cast<const EntryTyped<TTry>*>(entry)->get();
        return true;
      }
      template<class TTry>
      bool tryPipe(std::string& outval) const
      {
        uint trypoolid = EntryPoolMapping<TTry>::get(freeConfigMappings);
        if(trypoolid != pool)
          return false;
        //found it
        std::stringstream sstr;
        TTry val = static_cast<const EntryTyped<TTry>*>(entry)->get();
        sstr << val;
        outval = sstr.str();
        return true;
      }

      template<class T>
      T getNumbers(const std::string& name) const
      {
        uint poolid = EntryPoolMapping<T>::get(freeConfigMappings);
        if(poolid == pool) //same pool -> works
          return static_cast<const EntryTyped<T>*>(entry)->get();
        else
        {
          T val;
          //try standard types
          if(tryGet<bool>(val)) return val;
          if(tryGet<int>(val)) return val;
          if(tryGet<unsigned int>(val)) return val;
          if(tryGet<float>(val)) return val;
          if(tryGet<double>(val)) return val;

          //a bit more exotic
          if(tryGet<char>(val)) return val;
          if(tryGet<unsigned char>(val)) return val;
          if(tryGet<long>(val)) return val;
          if(tryGet<unsigned long>(val)) return val;
          if(tryGet<long long>(val)) return val;
          if(tryGet<unsigned long long>(val)) return val;

          //if custom types are used, they must match!
          throw std::runtime_error(std::string("Config Error: different base types in get for \"") + name + std::string("\""));
        }
      }
    public:
      static uint freeConfigMappings;

      Entry() : entry(0), pool(0), own(false) { }
      template<class T>
      Entry(T value) : entry(new EntryTyped<T>(value)), pool(EntryPoolMapping<T>::get(freeConfigMappings)), own(true) {  }
      template<class T, class CBarg>
      Entry(T value, typename EntryCallbackTyped<T, CBarg>::Callback callback, CBarg* arg = 0) : entry(new EntryTyped<T>(value, callback, arg)), pool(EntryPoolMapping<T>::get(freeConfigMappings)), own(true) { }
      ~Entry() { if(entry && own) delete entry; }
      Entry(const Entry& other) : entry(other.entry), pool(other.pool), own(other.own)
      {
        if(!own)
          throw std::runtime_error("Config Entry took ownership from not owning other");
        const_cast<Entry*>(&other)->own = 0;
      }
      const Entry& operator = (const Entry& other)
      {
        entry = other.entry;
        pool = other.pool;
        own = other.own;
        if(!own)
          throw std::runtime_error("Config Entry took ownership from not owning other");
        const_cast<Entry*>(&other)->own = 0;
        return * this;
      }
      static Entry deepCopy(const Entry& other)
      {
        Entry newEntry;
        newEntry.pool = other.pool;
        newEntry.own = true;
        newEntry.entry = other.entry->clone();
        return newEntry;
      }
      static Entry shallowCopy(const Entry& other)
      {
        Entry newEntry;
        newEntry.pool = other.pool;
        newEntry.own = true;
        newEntry.entry = other.entry->cloneNoCallback();
        return newEntry;
      }

      uint getPool() const
      {
        return pool;
      }
      template<class T>
      void assign(const T& newval, const std::string& name);

      template<class T>
      T get(const std::string name) const
      {
        uint poolid = EntryPoolMapping<T>::get(freeConfigMappings);
        if(poolid == pool) //same pool -> works
          return static_cast<const EntryTyped<T>*>(entry)->get();
        else
          //if custom types are used, they must match!
          throw std::runtime_error(std::string("Config Error: different base types in get for \"") + name + std::string("\""));
      }

      template<class T>
      T& getS()
      {
        return static_cast<EntryTyped<T>*>(entry)->get();
      }
      template<class T, class CBarg>
      T& addListenerS( typename EntryCallbackTyped<T, CBarg>::Callback callback, CBarg* arg = 0)
      {
        static_cast<EntryTyped<T>*>(entry)->addListener(callback, arg);
        return static_cast<EntryTyped<T>*>(entry)->get();
      }

    };

  private:
#ifndef __GNUC__
    typedef std::unordered_map<std::string, Entry> ConfigsType;

#else
    typedef std::map<std::string, Entry>  ConfigsType;
#endif
    ConfigsType configs;
    Mutex mutex;

  protected:
    virtual void newEntry(const std::string& name, const Entry& entry) {  }
    void insertEntry(const std::string& name, Entry entry)
    {
      Guard g(mutex);
      configs.insert(std::pair<std::string, Entry>(name, entry));
    }

  public:
    template<class T, class CBArg>
    T& registerConfig(const std::string& name, typename EntryCallbackTyped<T,CBArg>::Callback changeNotify, CBArg* cbArgument)
    {
      return registerConfig<T>(name, T(), changeNotify, cbArgument);
    }

    template<class T>
    T& registerConfig(const std::string& name, const T initialvalue = T(), typename EntryCallbackTyped<T,void>::Callback changeNotify = 0)
    {
      return registerConfig<T,void>(name, initialvalue, changeNotify, 0);
    }

    template<class T, class CBArg>
    T& registerConfig(const std::string& name, const T initialvalue, typename EntryCallbackTyped<T,CBArg>::Callback changeNotify, CBArg* cbArgument)
    {
      Guard g(mutex);
      //get typed pool
      uint poolid = EntryPoolMapping<T>::get(Entry::freeConfigMappings);
      //check if entry already exists

      ConfigsType::iterator found = configs.find(name);
      if(found != configs.end())
      {
        if(found->second.getPool() != poolid)
          throw std::runtime_error(std::string("Config Error: different base typed for same argument \"") + name + std::string("\""));
        return found->second.addListenerS<T,CBArg>(changeNotify, cbArgument);
      }

      //create a new one
      std::pair<ConfigsType::iterator, bool> inserted = configs.insert(std::make_pair(name,Entry(initialvalue, changeNotify, cbArgument)));
      newEntry(name, inserted.first->second);
      return inserted.first->second.getS<T>();
    }


    template<class T>
    void set(const std::string& name, T val)
    {
      Guard g(mutex);
      //check if entry exists
      ConfigsType::iterator found = configs.find(name);
      if(found == configs.end())
        throw std::runtime_error(std::string("Config name \"") + name + ("\" not found"));

      found->second.assign(val, name);
    }
    template<class T>
    T get(const std::string& name) const
    {
      ConstGuard g(mutex);
      ConfigsType::const_iterator found = configs.find(name);
      if(found == configs.end())
        throw std::runtime_error(std::string("Config name \"") + name + ("\" not found"));
      return found->second.get<T>(name);
    }
  };
  template<class T>
  uint Config::EntryPoolMapping<T>::id = 0;

  template<>
  inline void Config::Entry::assign<bool>(const bool& newval, const std::string& name)
  { assignNumbers(newval, name); }
  template<>
  inline void Config::Entry::assign<int>(const int& newval, const std::string& name)
  { assignNumbers(newval, name); }
  template<>
  inline void Config::Entry::assign<uint>(const uint& newval, const std::string& name)
  { assignNumbers(newval, name); }
  template<>
  inline void Config::Entry::assign<long>(const long& newval, const std::string& name)
  { assignNumbers(newval, name); }
  template<>
  inline void Config::Entry::assign<unsigned long>(const unsigned long& newval, const std::string& name)
  { assignNumbers(newval, name); }
  template<>
  inline void Config::Entry::assign<long long>(const long long& newval, const std::string& name)
  { assignNumbers(newval, name); }
  template<>
  inline void Config::Entry::assign<unsigned long long>(const unsigned long long& newval, const std::string& name)
  { assignNumbers(newval, name); }
  template<>
  inline void Config::Entry::assign<char>(const char& newval, const std::string& name)
  { assignNumbers(newval, name); }
  template<>
  inline void Config::Entry::assign<unsigned char>(const unsigned char &newval, const std::string& name)
  { assignNumbers(newval, name); }
  template<>
  inline void Config::Entry::assign<float>(const float &newval, const std::string& name)
  { assignNumbers(newval, name); }
  template<>
  inline void Config::Entry::assign<double>(const double &newval, const std::string& name)
  { assignNumbers(newval, name); }

  template<>
  inline void Config::Entry::assign<std::string>(std::string const & newval, const std::string& name)
  {
    if(tryAssign<std::string>(newval, name)) return;
    if(tryPipe<bool>(newval, name)) return;
    if(tryPipe<int>(newval, name)) return;
    if(tryPipe<unsigned int>(newval, name)) return;
    if(tryPipe<float>(newval, name)) return;
    if(tryPipe<double>(newval, name)) return;

    //a bit more exotic
    if(tryPipe<char>(newval, name)) return;
    if(tryPipe<unsigned char>(newval, name)) return;
    if(tryPipe<long>(newval, name)) return;
    if(tryPipe<unsigned long>(newval, name)) return;
    if(tryPipe<long long>(newval, name)) return;
    if(tryPipe<unsigned long long>(newval, name)) return;

    throw std::runtime_error(std::string("Config Error: different base types in assign for \"") + name + std::string("\""));
  }
  template<>
  inline void Config::Entry::assign<const char*>( const char* const &newval, const std::string& name)
  {
    if(tryAssignMatch(newval, name)) return;
    assign(std::string(newval), name);
  }
  template<>
  inline void Config::Entry::assign<char*>(char* const & newval, const std::string& name)
  {
    if(tryAssignMatch(newval, name)) return;
    if(tryAssign<const char*>(newval, name)) return;
    assign(std::string(newval), name);
  }
  template<class T>
  inline void Config::Entry::assign(const T& newval, const std::string& name)
  {
    uint poolid = EntryPoolMapping<T>::get(freeConfigMappings);
    if(poolid == pool) //same pool -> works
      static_cast<EntryTyped<T>*>(entry)->assign(newval, name);
    else
      //if custom types are used, they must match!
      throw std::runtime_error(std::string("Config Error: different base types in assign for \"") + name + std::string("\""));
  };


  template<>
  inline bool Config::Entry::get<bool>(const std::string name) const
  { return getNumbers<bool>(name); }
  template<>
  inline int Config::Entry::get<int>(const std::string name) const
  { return getNumbers<int>(name); }
  template<>
  inline unsigned int Config::Entry::get<unsigned int>(const std::string name) const
  { return getNumbers<unsigned int>(name); }
  template<>
  inline long Config::Entry::get<long>(const std::string name) const
  { return getNumbers<long>(name); }
  template<>
  inline unsigned long Config::Entry::get<unsigned long>(const std::string name) const
  { return getNumbers<unsigned long>(name); }
  template<>
  inline long long Config::Entry::get<long long>(const std::string name) const
  { return getNumbers<long long>(name); }
  template<>
  inline unsigned long long Config::Entry::get<unsigned long long>(const std::string name) const
  { return getNumbers<unsigned long long>(name); }
  template<>
  inline char Config::Entry::get<char>(const std::string name) const
  { return getNumbers<char>(name); }
  template<>
  inline unsigned char Config::Entry::get<unsigned char>(const std::string name) const
  { return getNumbers<unsigned char>(name); }
  template<>
  inline float Config::Entry::get<float>(const std::string name) const
  { return getNumbers<float>(name); }
  template<>
  inline double Config::Entry::get<double>(const std::string name) const
  { return getNumbers<double>(name); }
  template<>
  inline const char* Config::Entry::get<const char*>(const std::string name) const
  {
    const char* val;
    if(tryGet<char*>(val)) return val;
    if(tryGet<const char*>(val)) return val;
    throw std::runtime_error(std::string("Config Error: different base types in get for \"") + name + std::string("\""));
  }

  template<>
  inline std::string Config::Entry::get<std::string>(const std::string name) const
  {
    std::string val;
    if(tryGet<std::string>(val)) return val;
    if(tryGet<char*>(val)) return val;

    if(tryPipe<bool>(val)) return val;
    if(tryPipe<int>(val)) return val;
    if(tryPipe<unsigned int>(val)) return val;
    if(tryPipe<float>(val)) return val;
    if(tryPipe<double>(val)) return val;

    //a bit more exotic
    if(tryPipe<char>(val)) return val;
    if(tryPipe<unsigned char>(val)) return val;
    if(tryPipe<long>(val)) return val;
    if(tryPipe<unsigned long>(val)) return val;
    if(tryPipe<long long>(val)) return val;
    if(tryPipe<unsigned long long>(val)) return val;

    throw std::runtime_error(std::string("Config Error: different base types in get for \"") + name + std::string("\""));
  }

};

#ifdef WIN32
#pragma warning( pop )
#endif

#endif //SOFTSHELL_TOOLS_CONFIG_INCLUDED
