/*
  Softshell: Dynamic Scheduling on GPUs.
  http://www.icg.tugraz.at/project/mvp

  Copyright (C) 2012 Institute for Computer Graphics and Vision,
                     Graz University of Technology

  Author(s):  Markus Steinberger - steinberger ( at ) icg.tugraz.at
              Bernhard Kainz - kainz ( at ) icg.tugraz.at
              Michael Kenzel - kenzel ( at ) icg.tugraz.at
              Stefan Hauswiesner - hauswiesner ( at ) icg.tugraz.at
              Bernhard Kerbl - kerbl ( at ) icg.tugraz.at
              Dieter Schmalstieg - schmalstieg ( at ) icg.tugraz.at

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/


/*
* file created by    Markus Steinberger / steinberger ( at ) icg.tugraz.at
*
* modifications by
*/

#ifndef SOFTSHELL_TOOLS_CONFIGCOLLECTION_INCLUDED
#define SOFTSHELL_TOOLS_CONFIGCOLLECTION_INCLUDED


#include "tools/config.h"
#include <sstream>

namespace Softshell
{
  class ConfigCollection : public Config
  {
    static const char Seperator = '>';
    ConfigCollection* parent;
#ifndef __GNUC__
    typedef std::unordered_map<std::string, Config*> ChildrenAType;
    typedef std::unordered_map<std::string, ConfigCollection*> ChildrenBType;

#else
    typedef std::map<std::string, Config*> ChildrenAType;
    typedef std::map<std::string, ConfigCollection*> ChildrenBType;
#endif
    ChildrenAType children_a;
    ChildrenBType children_b;

    template<class T>
    void argumentOfChild(const std::string& name, T intialvalue)
    {
      registerConfig(name, intialvalue);
    }

 protected:

    void newEntry(const std::string& name, const Entry& entry)
    {
      //parent should mirror child entries
      if(parent)
        parent->insertEntry(name, Entry::shallowCopy(entry));
    }

  public:

    ConfigCollection() : parent(0) {  }
    ConfigCollection(ConfigCollection& _parent) : parent(&_parent)
    {
      static uint UnnamedId = 0;
      std::stringstream name;
      name << "Unnamed_" << UnnamedId++;
      _parent.registerChild(name.str(), this);
    }
    ConfigCollection(ConfigCollection& _parent, const std::string& name) : parent(&_parent)
    {
      _parent.registerChild(name, this);
    }

    void registerChild(const std::string& name, Config* child)
    {
      if(children_b.find(name) != children_b.end())
        return;
      children_a.insert(std::make_pair(name, child));
    }
    void registerChild(const std::string& name, ConfigCollection* child)
    {
      if(children_a.find(name) != children_a.end())
        return;
      children_b.insert(std::make_pair(name, child));
    }
    void unregisterChild(const std::string& name)
    {
      children_a.erase(name);
      children_b.erase(name);
    }

    template<class T>
    void set(const std::string& name, T val)
    {
      //is it an argument for me or for my children?
      size_t pos = name.find_first_of(Seperator);
      if(pos != std::string::npos)
      {
        std::string child = name.substr(0,pos);
        std::string name2 = name.substr(pos+1);
        ChildrenAType::const_iterator found_a = children_a.find(child);
        ChildrenBType::const_iterator found_b = children_b.find(child);
        if(found_a != children_a.end())
        {
          found_a->second->set(name2, val);
          return;
        }
        else if(found_b != children_b.end())
        {
          found_b->second->set(name2, val);
          return;
        }
        //does not match any children, so handle it as argument
      }

      //set for me and for all children
      Config::set(name, val);
      try
      {
        for(ChildrenAType::const_iterator it = children_a.begin(); it != children_a.end(); ++it)
          it->second->set(name, val);
        for(ChildrenBType::const_iterator it = children_b.begin(); it != children_b.end(); ++it)
          it->second->set(name, val);
      }
      catch(std::runtime_error)
      {
        // dont care if it is not found in children
      }

    }
    template<class T>
    T get(const std::string& name) const
    {
       //is it an argument for me or for my children?
      size_t pos = name.find_first_of(Seperator);
      if(pos != std::string::npos)
      {
        std::string child = name.substr(0,pos);
        std::string name2 = name.substr(pos+1);
        ChildrenAType::const_iterator found_a = children_a.find(child);
        ChildrenBType::const_iterator found_b = children_b.find(child);
        if(found_a != children_a.end())
          return found_a->second->get<T>(name2);
        else if(found_b != children_b.end())
          return found_b->second->get<T>(name2);
        //does not match any children, so handle it as argument
      }
      return Config::get<T>(name);
    }
  };
};

#endif //SOFTSHELL_TOOLS_CONFIG_INCLUDED
