/*
  Softshell: Dynamic Scheduling on GPUs.
  http://www.icg.tugraz.at/project/mvp

  Copyright (C) 2012 Institute for Computer Graphics and Vision,
                     Graz University of Technology

  Author(s):  Markus Steinberger - steinberger ( at ) icg.tugraz.at
              Bernhard Kainz - kainz ( at ) icg.tugraz.at
              Michael Kenzel - kenzel ( at ) icg.tugraz.at
              Stefan Hauswiesner - hauswiesner ( at ) icg.tugraz.at
              Bernhard Kerbl - kerbl ( at ) icg.tugraz.at
              Dieter Schmalstieg - schmalstieg ( at ) icg.tugraz.at

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/


/*
* file created by    Markus Steinberger / steinberger ( at ) icg.tugraz.at
*
* modifications by
*/

#ifndef SOFTSHELL_TOOLS_IDMANAGER_INCLUDED
#define SOFTSHELL_TOOLS_IDMANAGER_INCLUDED

#include "tools/thread.h"
#include "tools/types.h"
#include <utility>
#include <exception>
#include <string>


namespace Softshell
{
  template<class T, size_t RingbufferWidth = 4096>
  class IdManager
  {
    uint idCounter;
    Mutex mutex;
    typedef std::pair<uint, T> MappingT;
    MappingT mappings[RingbufferWidth];


  public:
    uint request(T requester) volatile
    {
      ConstGuard guard(mutex);
      if(++idCounter==0 || idCounter == 0xDEADBEEF)
        ++idCounter;
      size_t tries = 0;
      while(mappings[idCounter%RingbufferWidth].first != 0 && tries++ < RingbufferWidth)
        if(++idCounter==0 || idCounter == 0xDEADBEEF)
          ++idCounter;
      if(tries >= RingbufferWidth)
#if __GNUC__
        throw "IdManager out of Ids!";
#else
        throw std::exception("IdManager out of Ids!");
#endif
      mappings[idCounter%RingbufferWidth].first = idCounter;
      mappings[idCounter%RingbufferWidth].second = requester;

      return idCounter;
    }
    void remove(uint id) volatile
    {
      ConstGuard guard(mutex);
      mappings[id%RingbufferWidth].first = 0;
    }

    IdManager() : idCounter(0)
    {
      for(size_t i = 0; i < RingbufferWidth; ++i)
        mappings[i].first = 0;
    }
    T find(uint id) volatile
    {
      ConstGuard guard(mutex);
      if(mappings[id%RingbufferWidth].first == id)
        return mappings[id%RingbufferWidth].second;
      else
        return T(0);
    }
  };
};

#endif //SOFTSHELL_TOOLS_IDMANAGER_INCLUDED
