/*
  Softshell: Dynamic Scheduling on GPUs.
  http://www.icg.tugraz.at/project/mvp

  Copyright (C) 2012 Institute for Computer Graphics and Vision,
                     Graz University of Technology

  Author(s):  Markus Steinberger - steinberger ( at ) icg.tugraz.at
              Bernhard Kainz - kainz ( at ) icg.tugraz.at
              Michael Kenzel - kenzel ( at ) icg.tugraz.at
              Stefan Hauswiesner - hauswiesner ( at ) icg.tugraz.at
              Bernhard Kerbl - kerbl ( at ) icg.tugraz.at
              Dieter Schmalstieg - schmalstieg ( at ) icg.tugraz.at

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/



/*
* file created by    Markus Steinberger / steinberger ( at ) icg.tugraz.at
*
* modifications by
*/

#ifndef SOFTSHELL_TOOLS_THREAD_INCLUDED
#define SOFTSHELL_TOOLS_THREAD_INCLUDED



#if defined(_MSC_VER)
#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include<windows.h>
#define __cpu_thread_local __declspec(thread)
#else
#include<pthread.h>
#include <sys/time.h>
#include <unistd.h>
#include <errno.h>
#define __cpu_thread_local __thread
#endif

#include <cstdio>
namespace Softshell
{
  class Signal
  {
#ifdef WIN32
    HANDLE signal_;
#else
    pthread_cond_t cond_;
    pthread_mutex_t mutex_;
    bool signaled_;
#endif
    bool mine;

    void destroy()
    {
      if(mine)
      {
#ifdef WIN32
        CloseHandle(signal_);
#else
        pthread_mutex_destroy(&mutex_);
        pthread_cond_destroy(&cond_);
#endif
        mine = false;
      }
    }
  public:
    Signal(bool signaled = false) : mine(true)
    {
#ifdef WIN32
      signal_ = CreateEvent(NULL, FALSE, signaled?TRUE:FALSE, NULL);
#else
      signaled_ = signaled;
      pthread_cond_init(&cond_, NULL);
      pthread_mutex_init(&mutex_, NULL);
#endif
    }
    Signal(const Signal& other) : mine(true),
  #ifdef WIN32
      signal_(other.signal_)
#else
      cond_(other.cond_),
      mutex_(other.mutex_),
      signaled_(other.signaled_)
#endif
    {
      const_cast<Signal*>(&other)->mine = false;
    }
    const Signal& operator = (const Signal& other)
    {
      destroy();
      mine = true;
#ifdef WIN32
      signal_ = other.signal_;
#else
      cond_ = other.cond_;
      mutex_ = other.mutex_;
      signaled_ = other.signaled_;
#endif
      const_cast<Signal*>(&other)->mine = false;
      return *this;
    }
    void wait(size_t timeout = 0)
    {
      //printf("wait!!!! %llx %lx\n",&mutex_,pthread_self());
#ifdef WIN32
      if(timeout == 0)
        WaitForSingleObject( signal_, INFINITE);
      else
        WaitForSingleObject( signal_, static_cast<DWORD>(timeout));
#else
      pthread_mutex_lock( &mutex_);
      int errorCode = 0;
      while (!signaled_)
      {
        if (timeout == 0)
          errorCode = pthread_cond_wait(&cond_, &mutex_);
        else
        {
          timespec ts;
          timeval tp;

          gettimeofday(&tp, NULL);
          ts.tv_nsec = (tp.tv_usec + (timeout%1000)*1000) * 1000;
          ts.tv_sec  = tp.tv_sec +(timeout/1000) +ts.tv_nsec/1000000000;
          ts.tv_nsec %= 1000000000;

          errorCode = pthread_cond_timedwait(&cond_, &mutex_, &ts);
        }
        if (errorCode) {
          if (errorCode == ETIMEDOUT) {
            if (signaled_)
              errorCode = 0;
            break;
          }
          printf("Error in Signal::wait(): %d", errorCode);
        }
      }
      signaled_ = false;
      pthread_mutex_unlock(&mutex_);
#endif
    }
    void signal()
    {
      //printf("signal!!!! %llx %lx\n",&mutex_,pthread_self());
#ifdef WIN32
      SetEvent(signal_);
#else
      pthread_mutex_lock( &mutex_);
      signaled_ = true;
      pthread_cond_signal(&cond_);
      pthread_mutex_unlock(&mutex_);
#endif
    }
    ~Signal()
    {
      destroy();
    }
  };
  class Mutex
  {
#ifdef WIN32
    HANDLE mutex_;
#else
    pthread_mutex_t mutex_;
#endif

  public:
    Mutex()
    {
#ifdef WIN32
      mutex_ = CreateMutex(NULL, FALSE, NULL);
#else
      pthread_mutex_init(&mutex_, NULL);
#endif
    }
    Mutex(const Mutex& other)
    {
#ifdef WIN32
      mutex_ = CreateMutex(NULL, FALSE, NULL);
#else
      pthread_mutex_init(&mutex_, NULL);
#endif
    }
    const Mutex& operator = (const Mutex& other)
    {
      return *this;
    }
    void acquire()
    {
#ifdef WIN32
      if(mutex_)
        WaitForSingleObject( mutex_, INFINITE);
#else
      pthread_mutex_lock( &mutex_);
#endif
    }
    void release()
    {
#ifdef WIN32
      if(mutex_)
        ReleaseMutex( mutex_ );
#else
      pthread_mutex_unlock( &mutex_);
#endif
    }
    ~Mutex()
    {
#ifdef WIN32
      CloseHandle(mutex_);
#else
      pthread_mutex_destroy(&mutex_);
#endif
    }
  };

  class Guard
  {
    Mutex& _mutex;
    bool locked;
  public:
    Guard(Mutex& mutex) : _mutex(mutex)
    {
      _mutex.acquire();
      locked = true;
    }
    ~Guard()
    {
      if(locked)
        _mutex.release();
    }
    bool lock()
    {
      if(locked)
        return false;
      _mutex.acquire();
      return locked = true;
    }
    bool unlock()
    {
      if(!locked)
        return false;
      _mutex.release();
      locked = false;
      return true;
    }
  };

  class ConstGuard : public Guard
  {
  public:
    ConstGuard(volatile const Mutex& mutex) : Guard(*const_cast<Mutex*>(&mutex)) { }
  };


  class Thread
  {
  public:
    Thread()
    {
#ifdef WIN32
      isInit_ = false;
#endif
    };
    void start()
    {
#ifdef WIN32
      threadHandle_ = CreateThread(NULL,0,ThreadEntryPoint,static_cast<LPVOID>(this),0,&threadId_);
      isInit_ = true;
#else
      pthread_create(&threadId_, NULL, ThreadEntryPoint, static_cast<void*>(this));
#endif
    }
    virtual ~Thread()
    {
#ifdef WIN32
      if(isInit_)
        CloseHandle(threadHandle_);
#endif
    }
    static void sleep(int ms)
    {
#ifdef WIN32
      Sleep(ms);
#else
      usleep(ms * 1000);
#endif
    }
    bool kill()
    {
#ifdef WIN32
      return (TerminateThread(threadHandle_, 1) == 0);
#else
      return (pthread_cancel(threadId_) == 0);
#endif
    }
    int join()
    {
#ifdef WIN32
      int ret;
      WaitForSingleObject(threadHandle_, INFINITE);
      GetExitCodeThread(threadHandle_, reinterpret_cast<LPDWORD>(&ret));
      return ret;
#else
      void* ret;
      //printf("hallo\n");
      //int joinsuccessful =
      pthread_join(threadId_, &ret);

      //printf("join_successful: %d\n", joinsuccessful);
      return *reinterpret_cast<int*>(&ret);
#endif
    }

    static bool yield()
    {
#ifdef WIN32
      return SwitchToThread()?true:false;
#else
      return pthread_yield();
#endif
    }
  protected:
    virtual int execute() = 0;

  private:
#ifdef WIN32
    bool isInit_;
    DWORD threadId_;
    HANDLE threadHandle_;
    static DWORD WINAPI ThreadEntryPoint(LPVOID pthis)
    {
      Thread * pt = static_cast<Thread*>(pthis);
      return pt->execute();
    }
#else
    pthread_t threadId_;
    static void * ThreadEntryPoint(void * pthis)
    {
      Thread * pt = static_cast<Thread*>(pthis);
      return reinterpret_cast<void*>(pt->execute());
    }
#endif

  };

};
#endif //SOFTSHELL_TOOLS_THREAD_INCLUDED
