Set Implicit Arguments.
Require Export TLC.LibTactics TLC.LibCore TLC.LibString TLC.LibFset TLC.LibSet.
Generalizable Variables A B.
Require Export TLC.LibProd TLC.LibListZ.


(**************************************************************)
(** ** Notation to force “if” to be on booleans
       (and never on decidable equalities). *)

Notation "'if' b 'then' v1 'else' v2" :=
  (if (b : bool) then v1 else v2)
  (at level 200, right associativity) : type_scope.


(**************************************************************)
(** * Proving automatically instances *)

(** ** Comparable **)

Instance Comparable_Decidable : forall A (a1 a2 : A),
    Comparable A ->
    Decidable (a1 = a2).
  introv C. inverts* C.
Defined.

(** Builds an instance of [Comparable] from a non-recursive inductive. **)

(* I probably could have used [decide equality], but I didn’t find it
  easily usable in this context. *)
Ltac prove_decidable_eq :=
  let prove_injection Eq :=
    solve [ injection Eq; intros; substs~
          | intros; substs~; reflexivity ] in
  let rec t tr :=
    match goal with
    (** Trivial cases **)
    | _ =>
      applys decidable_make false; fold_bool; rew_refl; discriminate
    | _ =>
      applys decidable_make true; fold_bool; rew_refl; reflexivity
    (** Look for already defined typeclasses **)
    | _ =>
      typeclass || (apply Comparable_Decidable; typeclass)
    (** A little trivial case **)
    | _ =>
      match goal with
      |- Decidable (?f1 = ?f2) =>
        let D := fresh "D" in asserts D: (tr f1 <> tr f2);
        [ discriminate
        | applys decidable_make false; fold_bool; rew_refl;
          let AE := fresh "AE" in intro AE; rewrite AE in *; false* ]
      end
    (** General case **)
    | |- Decidable (?f1 ?x1 = ?f2 ?x2) =>
      let tf := type of f1 in
      let Decf := fresh "Dec_f" in let Decx := fresh "Dec_x" in
      let tr' := constr:(fun f : tf => tr (f x1)) in
      asserts Decx: (Decidable (x1 = x2));
      [ let T := type of x1 in try (typeclass || solve [ t (@id T) ])
      | asserts Decf: (Decidable (f1 = f2));
        [ t tr'
        | applys decidable_make (decide (f1 = f2 /\ x1 = x2));
          let I := fresh "I" in
          let I1 := fresh "I_f" in let I2 := fresh "I_x" in
          rewrite decide_spec; rewrite isTrue_eq_isTrue; iff I;
          [ lets (I1&I2): (rm I); try rewrite I1; rewrite~ I2
          | inverts I as I; splits~;
            let Eq := fresh "Eq" in
            asserts Eq: (tr (f1 x1) = tr (f2 x2));
            [ rewrite~ I | prove_injection Eq ]
          ] ] ]
    | _ =>
      (** Nothing intelligent to do, let us nevertheless simplify the
        task of the user to know what is the context. **)
      let TR := fresh "tr" in set (TR := tr)
    end in
  match goal with
  | |- Decidable (?x = _) =>
    let T := type of x in
    t (@id T)
  end.

Ltac prove_comparable_simple_inductive :=
  let x := fresh "x" in let y := fresh "y" in
  apply make_comparable; intros x y; destruct x; destruct y;
  prove_decidable_eq.


(**************************************************************)
(** Auxiliary function to perform case analysis on an option
    without requiring an explicit match. *)

Definition option_case {A B : Type} (d : B) (f : A -> B) (op : option A) : B :=
  match op with
  | None => d
  | Some x => f x
  end.

Lemma option_case_out : forall A B d (f : A -> B) op b,
  option_case d f op = b ->
  (op = None /\ b = d) \/ (exists a, op = Some a /\ b = f a).
Proof. introv E. destruct* op. Qed.

Lemma option_case_map : forall A B C d (g : A -> B) (f : B -> C) op,
  option_case (f d) (fun x => f (g x)) op = f (option_case d g op).
Proof. introv. destruct~ op. Qed.


(**************************************************************)
(** ** Char-related functions *)

(* Note that string is extracted to "char list" in Ocaml *)

(* Int_of_char is currently directly extracted in Caml as:
     "(fun c -> float_of_int (int_of_char c))".
   No property of this function is currently needed. *)

Parameter int_of_char : Ascii.ascii -> int.

(* Comparison on Ascii characters implemented as (=) on "char" in Caml *)

Global Instance ascii_comparable : Comparable Ascii.ascii.
  prove_comparable_simple_inductive. Defined.


(**************************************************************)
(** ** String-related functions *)

(** Binding for Coq stdlib concatentation function *)

Definition string_concat : string -> string -> string :=
  String.append.

(** Binding for Coq stdlib substring function *)

Definition string_sub s (n l : int) : string :=
  substring (abs n) (abs l) s.


(**************************************************************)
(** ** Operators specific *)

Global Instance op_unary_inhab : forall P : Type,
  Inhab (P -> P).
Proof. introv. apply prove_Inhab. introv. autos*. Defined.

Global Instance op_binary_inhab : forall P : Type,
  Inhab (P -> P -> P).
Proof. introv. apply prove_Inhab. introv. autos*. Defined.


(**************************************************************)
(** ** Extraction for comparison operators on integers *)

Global Instance comparison_comparable : Comparable comparison.
  prove_comparable_simple_inductive. Defined.

(* Directly extracted towards (<) in OCaml *)

Global Instance lt_int_decidable : forall i1 i2 : int, Decidable (i1 < i2).
  intros. applys decidable_make (decide (i1 ?= i2 = Datatypes.Lt)).
  rewrite lt_zarith. unfold Z.lt. rew_refl. reflexivity.
Defined.

(* Directly extracted towards (<=) in OCaml *)

Global Instance le_int_decidable : forall i1 i2 : int, Decidable (i1 <= i2).
  intros. applys decidable_make (decide (i1 ?= i2 <> Datatypes.Gt)).
  rewrite le_zarith. unfold Z.le, Z.lt. rew_refl. fequals.
Defined.

(* Directly extracted towards (>=) in OCaml *)

Global Instance ge_nat_decidable : forall n1 n2 : nat, Decidable (n1 >= n2).
  intros. applys sumbool_decidable. applys Compare_dec.ge_dec.
Defined.


(**************************************************************)
(* To be added to LibListZ *)

Lemma ZNth_to_Nth : forall A (n : nat) x (l : list A),
  LibListZ.ZNth n l x -> LibList.Nth n l x.
Proof. introv (H&_). rewrite~ abs_pos_nat in H. Qed.

Lemma Nth_to_ZNth : forall A (n : nat) x (l : list A),
  LibList.Nth n l x -> LibListZ.ZNth n l x.
Proof. introv H. splits*. rewrite~ abs_pos_nat. math. Qed.


(**************************************************************)
(** ** Generalization of Pickable to function that return options *)

(* Note: easier to use Pickable_option than Pickable directly. *)

Class Pickable_option (A : Type) (P : A -> Prop) := pickable_option_make {
  pick_option : option A;
  pick_option_correct : forall a, pick_option = Some a -> P a;
  pick_option_defined : (exists a, P a) -> (exists a, pick_option = Some a) }.

Implicit Arguments pick_option [A [Pickable_option]].
Extraction Inline pick_option.

Global Instance Pickable_option_Pickable :
    forall (A : Type) `{Inhab A} (P : A -> Prop),
    Pickable_option P -> Pickable P.
  introv I [[pi|] C D].
   applys pickable_make pi. introv _. apply~ C.
   applys pickable_make arbitrary. introv E. forwards (a&N): D E. inverts N.
Defined.

(** Application to LibHeap operation *)
Require Export LibHeap.

Global Instance binds_pickable_option : forall K V : Type,
    `{Comparable K} -> `{Inhab V} ->
    forall (h : heap K V) (v : K),
    Pickable_option (binds h v).
  introv CK IV; introv. applys pickable_option_make (read_option h v).
   apply read_option_binds.
   introv [a Ba]. forwards R: @binds_read_option Ba. exists~ a.
Defined.


(***********************************************************)
(***********************************************************)
(***********************************************************)
(** * Heap with a counter for allocating fresh locations *)

Module HeapGen (Export Heap : HeapSpec) : HeapSpec.
Generalizable Variable K V.

Definition heap K V := (nat * heap K V)%type.

Section HeapDefs.
(*Variables K V : Type.*)
Context `{Comparable K} `{Inhab V}.
Definition empty : heap K V := (0%nat, empty).
Definition dom (h : heap K V) := dom (snd h).
Definition binds (h : heap K V) := binds (snd h).
Definition to_list (h : heap K V) := to_list (snd h).
Definition read (h : heap K V) := read (snd h).
Definition write (h : heap K V) k v :=
  let (id, h0) := h in
  (S id, write (snd h) k v).
Definition rem (h : heap K V) k :=
  let (id, h0) := h in
  (S id, rem (snd h) k).
Definition indom (h : heap K V) := indom (snd h).
Definition read_option (h : heap K V) := read_option (snd h).
End HeapDefs.

Section HeapAxioms.
Context `{Comparable K} `{Inhab V}.
Implicit Types h : heap K V.

Lemma indom_equiv_binds : forall h k,
  indom h k = (exists v, binds h k v).
Proof. destruct h. eapply indom_equiv_binds. Qed.

Lemma dom_empty :
  dom (@empty K V) = \{}.
Proof. unfold empty. eapply dom_empty. Qed.

Lemma binds_equiv_read : forall h k,
  indom h k -> (forall v, (binds h k v) = (read h k = v)).
Proof. destruct h. eapply binds_equiv_read. Qed.

Lemma dom_write : forall h r v,
  dom (write h r v) = dom h \u \{r}.
Proof. destruct h. eapply dom_write. Qed.

Lemma binds_write_eq : forall h k v,
  binds (write h k v) k v.
Proof. destruct h. eapply binds_write_eq. Qed.

Lemma binds_write_neq : forall h k v k' v',
  binds h k v -> k <> k' ->
  binds (write h k' v') k v.
Proof. destruct h. eapply binds_write_neq. Qed.

Lemma binds_write_inv : forall h k v k' v',
  binds (write h k' v') k v ->
  (k = k' /\ v = v') \/ (k <> k' /\ binds h k v).
Proof. destruct h. eapply binds_write_inv. Qed.

Lemma binds_rem : forall h k k' v,
  binds h k v -> k <> k' -> binds (rem h k') k v.
Proof. destruct h. eapply binds_rem. Qed.

Lemma binds_rem_inv : forall h k v k',
  binds (rem h k') k v -> k <> k' /\ binds h k v.
Proof. destruct h. eapply binds_rem_inv. Qed.

Lemma not_indom_rem : forall h k,
  ~ indom (rem h k) k.
Proof. destruct h. eapply not_indom_rem. Qed.

Lemma binds_equiv_read_option : forall h k v,
  (binds h k v) = (read_option h k = Some v).
Proof. destruct h. eapply binds_equiv_read_option. Qed.

Lemma not_indom_equiv_read_option : forall h k,
  (~ indom h k) = (read_option h k = None).
Proof. destruct h. eapply not_indom_equiv_read_option. Qed.

Lemma read_option_def : forall h k,
  read_option h k = (If indom h k then Some (read h k) else None).
Proof. destruct h. eapply read_option_def. Qed.

Lemma indom_decidable : forall `{Comparable K} V (h:heap K V) k,
  Decidable (indom h k).
Proof. destruct h. eapply indom_decidable. Qed.

End HeapAxioms.

End HeapGen.


(**************************************************************)
(* Binding between LibLogic and Coq sdtlib *)

Lemma and_andb : forall b1 b2, and b1 b2 = andb b1 b2.
Proof. intros () (); reflexivity. Qed.


(**************************************************************)
(* Bindings between LibList and Coq sdtlib *)

Lemma mem_In : forall A `{Comparable A} l (a : A),
  LibList.mem a l = isTrue (List.In a l).
Proof.
  introv. induction l.
   rewrite mem_nil. fold_bool. rew_refl. introv Abs. inverts Abs.
   simpl. rew_refl. rewrite IHl. rewrite~ eqb_sym.
Qed.

Lemma In_mem : forall A `{Comparable A} l (a : A),
  List.In a l = istrue (LibList.mem a l).
Proof.
  introv. induction l; extens; iff I.
   inverts~ I.
   rewrite mem_nil in I. false*.
   simpl. rew_refl. rewrite <- IHl. inverts~ I.
   simpls. rew_refl in I. rewrite <- IHl in I. inverts~ I.
Qed.

Lemma filter_in : forall A `{Comparable A} (f : predb A) l a,
  List.In a (filter f l) <-> (List.In a l /\ f a).
Proof. introv CA. introv. do 2 erewrite In_mem. rewrite filter_mem_eq. rew_refl*. Qed.

Lemma map_in : forall A `{Comparable A} B `{Comparable B} (f : A -> B) l x,
  List.In x l ->
  List.In (f x) (map f l).
Proof. introv CA CB I. erewrite In_mem in *. apply* map_mem. Qed.

Lemma incl_nil : forall A (l : list A),
  List.incl nil l.
Proof. introv I. inverts I. Qed.

Lemma incl_nil_inv : forall A (l : list A),
  List.incl l nil -> l = nil.
Proof. introv I. destruct~ l. false I. left~. Qed.

Lemma incl_app_l : forall A `{Comparable A} (l1 l2 : list A),
  List.incl l1 (LibList.append l1 l2).
Proof. introv CA I. erewrite In_mem in *. rewrite mem_app. rew_refl*. Qed.

Lemma incl_app_r : forall A `{Comparable A} (l1 l2 : list A),
  List.incl l2 (LibList.append l1 l2).
Proof. introv CA I. erewrite In_mem in *. rewrite mem_app. rew_refl*. Qed.

Lemma incl_mem : forall A `{Comparable A} (l1 l2 : list A) x,
  List.incl l1 l2 ->
  LibList.mem x l1 ->
  LibList.mem x l2.
Proof. introv I M. erewrite <- In_mem in *. autos*. Qed.

Lemma mem_incl : forall A `{Comparable A} (l1 l2 : list A),
  (forall x,
    LibList.mem x l1 ->
    LibList.mem x l2) ->
  List.incl l1 l2.
Proof. introv I M. erewrite In_mem in *. autos*. Qed.

Lemma incl_Forall : forall A `{Comparable A} (l1 l2 : list A),
  List.incl l1 l2 <-> Forall (fun a => LibList.mem a l2) l1.
Proof.
  introv. induction l1.
   iff I.
    constructors.
    apply incl_nil.
   iff I.
    constructors.
     rewrite <- In_mem. apply I. left~.
     rewrite <- IHl1. eapply incl_tran.
      apply incl_appr with (m := a :: nil). apply List.incl_refl.
      apply I.
    forwards (M&F): Forall_inv (rm I). rewrite <- In_mem in M. apply* incl_cons.
Qed.

Global Instance incl_decidable : forall A (C : Comparable A) (l1 l2 : list A),
    Decidable (List.incl l1 l2).
  introv C; introv. applys Decidable_equiv (Forall (fun a => LibList.mem a l2) l1).
   symmetry. apply~ incl_Forall.
   typeclass.
Defined.

Lemma incl_inv_head : forall A `{Comparable A} (l1 l2 : list A) a,
  List.incl (a :: l1) l2 ->
  LibList.mem a l2.
Proof. introv I. rewrite <- In_mem. apply I. left~. Qed.

Lemma incl_inv_tail : forall A (l1 l2 : list A) a,
  List.incl (a :: l1) l2 ->
  List.incl l1 l2.
Proof. introv Il I. apply Il. right~. Qed.

Lemma incl_split : forall A `{Comparable A} (l1 l2 : list A),
  List.incl l1 l2 ->
  exists l2',
    length l2' <= length l1
    /\ List.incl l2' l2
    /\ List.incl l1 l2'.
Proof.
  introv CA I. induction l1.
   exists (@nil A). splits~.
    math.
    apply~ incl_nil.
   forwards M: incl_inv_head I. forwards T: incl_inv_tail (rm I).
    forwards (l2'&E&I1&I2): (rm IHl1) (rm T). tests M': (LibList.mem a l2').
     exists l2'. splits~.
      rew_list. math.
      apply~ incl_cons. erewrite~ In_mem.
     exists (a :: l2'). splits~.
      rew_list. math.
      apply~ incl_cons. erewrite~ In_mem.
      apply~ incl_cons.
       constructors~.
       apply~ incl_tl.
Qed.

Lemma not_incl : forall A `{Comparable A} (l1 l2 : list A),
  ~ List.incl l1 l2 ->
  exists x, LibList.mem x l1 /\ ~ LibList.mem x l2.
Proof.
  introv N. rewrite incl_Forall in N. rewrite Forall_iff_forall_mem in N.
  rew_logic in N. lets (x&Px): (rm N). clear N (* WTF *). exists x. splits*.
  tests T: (LibList.mem x l1); autos~. false Px. introv I. false* T.
Qed.

Definition double_incl A (l1 l2 : list A) :=
  List.incl l1 l2 /\ List.incl l2 l1.

Lemma double_incl_mem : forall A `{Comparable A} (l1 l2 : list A) a,
  double_incl l1 l2 ->
  LibList.mem a l1 = LibList.mem a l2.
Proof. introv (I1&I2). extens. do 2 rewrite <- In_mem. iff* I. Qed.

Lemma mem_double_incl : forall A `{Comparable A} (l1 l2 : list A),
  (forall a, LibList.mem a l1 = LibList.mem a l2) ->
  double_incl l1 l2.
Proof. introv E. splits; introv I; erewrite In_mem in *; rewrite <- E in *; autos~. Qed.

Lemma double_incl_refl : forall A (l : list A),
  double_incl l l.
Proof. introv. splits; apply List.incl_refl. Qed.

Lemma double_incl_trans : forall A (l1 l2 l3 : list A),
  double_incl l1 l2 ->
  double_incl l2 l3 ->
  double_incl l1 l3.
Proof. introv (I1&I2) (I3&I4). splits; apply* incl_tran. Qed.

Lemma double_incl_sym : forall A (l1 l2 : list A),
  double_incl l1 l2 ->
  double_incl l2 l1.
Proof. introv (I1&I2). splits*. Qed.

Lemma double_incl_rev : forall A `{Comparable A} (l : list A),
  double_incl l (rev l).
Proof.
  introv CA. introv. splits; introv I; erewrite In_mem in *.
   rewrite* rev_mem in I.
   rewrite* rev_mem.
Qed.

Lemma No_duplicates_incl_length : forall A `{Comparable A} (l l' : list A),
  No_duplicates l' ->
  List.incl l' l ->
  LibList.length l' <= LibList.length l.
Proof.
  introv C ND Inc. gen l. induction ND; introv Inc.
   rewrite LibList.length_nil. math.
   rewrite LibList.length_cons. sets l': (LibList.remove x l0). forwards: IHND l'.
    apply mem_incl with (H := _). introv I. unfolds l'. rewrite remove_mem.
     rew_refl. splits.
      applys incl_mem Inc. simpl. rew_refl*.
      intro. substs. rewrite <- Mem_mem in I. false*.
    asserts: (1 + LibList.length l' <= LibList.length l0).
     unfolds l'. unfold LibList.remove.
      rewrite <- filter_length with (l := l0) (p := fun y => decide (y <> x)).
      asserts Lem: (forall (a : int) (b c : nat), a <= c -> a + b <= (b + c)%nat).
       math.
      lets C': C. destruct C'. eapply Lem. clear - Inc C. destruct filter eqn: E.
       false. forwards I: mem_nil C x. rewrite <- E in I. fold_bool. rew_refl in I.
        false I. rewrite filter_mem_eq. rew_refl. splits~. eapply incl_mem.
         apply Inc.
         simpl. rew_refl*.
       rewrite LibList.length_cons. math.
     math.
Qed.

Lemma No_duplicates_double_incl_length : forall A `{Comparable A} (l l' : list A),
  No_duplicates l' ->
  double_incl l l' ->
  LibList.length l' <= LibList.length l.
Proof. introv ND (?&?). apply* No_duplicates_incl_length. Qed.


(**************************************************************)
(* Might be useful in LibOption *)

Global Instance option_none_decidable : forall A (o : option A),
    Decidable (o = None).
  introv. destruct o.
   applys Decidable_equiv False; [ iff Abs; inverts Abs | typeclass ].
   applys Decidable_equiv True; [ iff~ | typeclass ].
Defined.


(**************************************************************)
(* Might be useful in LibList *)

Fixpoint list_take_first A (P : A -> bool) (l : list A) : option A :=
  match l with
  | nil => None
  | a :: l =>
    ifb P a then Some a
    else list_take_first P l
  end.

Lemma list_take_first_prop : forall A `{Comparable A} P (l : list A) r,
  list_take_first P l = Some r ->
  LibList.mem r l /\ istrue (P r).
Proof.
  introv E. induction l; [false*|].
   unfolds in E. cases_if; fold_bool; rew_refl in *.
    inverts E. splits~. simpl. rew_refl. left~.
    forwards (I&HP): IHl E. splits~. simpl. rew_refl. right~.
Qed.

Lemma list_take_first_nil : forall A (P : A -> bool),
  list_take_first P nil = None.
Proof. reflexivity. Qed.

Lemma list_take_first_ex : forall A `{Comparable A} P (l : list A),
  (exists r, LibList.mem r l /\ istrue (P r)) ->
  exists r, list_take_first P l = Some r.
Proof.
  introv (e&He). induction l; [false*|].
   simpl in He. rew_refl in He. lets ([E|N]&Pe): (rm He).
    exists e. simpl. substs. cases_if~. fold_bool. rew_refl in *. false*.
    forwards~ (r&E): IHl. simpl. cases_if; eexists; autos*.
Qed.

Lemma fold_left_ex_ind : forall A `{Comparable A} B (P : A -> Prop) (Q : B -> Prop) (f : A -> B -> B) i l r,
  (forall a r, P a \/ Q r -> Q (f a r)) ->
  (exists a, LibList.mem a l /\ P a) ->
  fold_left f i l = r ->
  Q r.
Proof.
  introv R E F. asserts EQ: (Q i \/ exists a, LibList.mem a l /\ P a). right*.
  clear E. gen i. induction l; introv F EQ; lets [Qi|E]: (rm EQ).
   rewrite* <- F.
   lets (a&Ia&Pa): (rm E). false.
   simpl in F. applys IHl F. left. apply* R.
   lets (e&I&Pe): (rm E). simpl in I. rew_refl in I. inverts I as Ie.
    applys IHl F. left. apply* R.
    applys IHl F. right*.
Qed.


Fixpoint find_remove A {C : Comparable A} a (l : list A) :=
  match l with
  | nil => None
  | b :: l =>
    ifb a = b then Some l
    else LibOption.map (fun l => b :: l) (find_remove a l)
  end.

Lemma find_remove_none : forall A (C : Comparable A) (a : A) l,
  find_remove a l = None <->
  LibList.mem a l = false.
Proof.
  introv. induction l.
   iff~.
   simpls. cases_if; fold_bool; rew_refl in *.
    substs. iff I; false*. fold_bool. rew_refl in I. false* I.
    iff I; fold_bool; rew_refl in *; rew_logic in *.
     forwards E: option_map_none_inv I. rewrite IHl in E. rewrite~ E.
     rewrite~ (proj2 IHl). fold_bool. rew_refl*.
Qed.

Lemma find_remove_mem : forall A (C : Comparable A) (a : A) l l',
  find_remove a l = Some l' ->
  LibList.mem a l.
Proof.
  introv E. gen l'. induction l; introv E; tryfalse.
  simpls. cases_if; fold_bool; rew_refl~ in *.
  forwards (l0&Ef&El): map_inv E. right*.
Qed.

Lemma find_remove_eq : forall A (C : Comparable A) (a : A) l l',
  find_remove a l = Some l' ->
  exists l1 l2,
    l = l1 ++ a :: l2 /\ l' = l1 ++ l2.
Proof.
  introv E. gen l'. induction l; introv E; tryfalse.
  simpls. cases_if; fold_bool; rew_refl in *.
   substs. inverts E. exists~ (@nil A) l'.
   forwards (l0&Ef&El): map_inv E. forwards (l1&l2&E1&E2): IHl Ef.
    substs. exists~ (a0 :: l1) l2.
Qed.


Lemma remove_Nth_same : forall A `{Comparable A} l (a0 a : A) n1 n2,
  (forall n1 n2,
     Nth n1 l a ->
     Nth n2 l a ->
     n1 = n2) ->
  Nth n1 (LibList.remove a0 l) a ->
  Nth n2 (LibList.remove a0 l) a ->
  n1 = n2.
Proof.
  introv Nth_same N1 N2. gen n1 n2. induction l; introv N1 N2.
   inverts N1.
   unfold LibList.remove in *. rewrite filter_cons in *. case_if.
    inverts N1 as N1; inverts N2 as N2.
     autos*.
     eapply Nth_mem in N2. rewrite filter_mem_eq in N2.
      rew_refl in N2. lets (M&?): (rm N2). lets (n'&N): mem_Nth M.
      forwards*: Nth_same Nth_here Nth_next. false*.
     eapply Nth_mem in N1. rewrite filter_mem_eq in N1.
      rew_refl in N1. lets (M&?): (rm N1). lets (n'&N): mem_Nth M.
      forwards*: Nth_same Nth_here Nth_next. false*.
     fequals. apply~ IHl. introv N1' N2'. forwards: Nth_same.
      applys~ Nth_next N1'.
      applys~ Nth_next N2'.
      autos*.
    apply~ IHl. introv N1' N2'. forwards: Nth_same.
     applys Nth_next N1'.
     applys Nth_next N2'.
     autos*.
Qed.

Lemma removes_Nth_same : forall A `{Comparable A} l l0 (a : A) n1 n2,
  (forall n1 n2,
     Nth n1 l a ->
     Nth n2 l a ->
     n1 = n2) ->
  Nth n1 (LibList.removes l0 l) a ->
  Nth n2 (LibList.removes l0 l) a ->
  n1 = n2.
Proof.
  introv Nth_same N1 N2. gen n1 n2 l. induction l0; introv Nth_same N1 N2.
   applys~ Nth_same N1 N2.
   simpl in N1, N2. applys~ IHl0 N1 N2.
    introv N1' N2'. applys~ remove_Nth_same N1' N2'.
Qed.

Lemma remove_assoc_Nth_same : forall A `{Comparable A} B `{Comparable B} l (a0 a : A) (b1 b2 : B) n1 n2,
  (forall b1 b2 n1 n2,
     Nth n1 l (a, b1) ->
     Nth n2 l (a, b2) ->
     n1 = n2) ->
  Nth n1 (remove_assoc a0 l) (a, b1) ->
  Nth n2 (remove_assoc a0 l) (a, b2) ->
  n1 = n2.
Proof.
  introv CB Nth_same N1 N2. gen n1 n2. induction l; introv N1 N2.
   inverts N1.
   destruct a1 as [a' b']. rewrite remove_assoc_cons in *. case_if.
    apply~ IHl. introv N1' N2'. forwards: Nth_same.
     applys Nth_next N1'.
     applys Nth_next N2'.
     autos*.
    inverts N1 as N1; inverts N2 as N2.
     autos*.
     eapply Nth_mem in N2. eapply mem_mem_assoc in N2.
      rewrite~ mem_assoc_remove_assoc_neq in N2.
      lets (b&M): mem_assoc_exists_mem N2. lets (n'&N): mem_Nth M.
      forwards*: Nth_same Nth_here Nth_next. false*.
     eapply Nth_mem in N1. apply mem_mem_assoc in N1.
      rewrite~ mem_assoc_remove_assoc_neq in N1.
      lets (b&M): mem_assoc_exists_mem N1. lets (n'&N): mem_Nth M.
      forwards*: Nth_same Nth_here Nth_next. false*.
     fequals. apply~ IHl. introv N1' N2'. forwards: Nth_same.
      applys~ Nth_next N1'.
      applys~ Nth_next N2'.
      autos*.
Qed.

Lemma unique_assoc_assoc_mem : forall A B `{Comparable A} `{Comparable B} `{Inhab B} a (l : list (A * B)) b,
  (forall n1 n2 b1 b2,
    Nth n1 l (a, b1) ->
    Nth n2 l (a, b2) ->
    n1 = n2) ->
  LibList.mem (a, b) l ->
  LibList.assoc a l = b.
Proof.
  introv S M. induction l.
   false*.
   destruct a0. rewrite assoc_cons. case_if.
    substs. simpl in M. rew_refl in M. inverts M as M.
     inverts~ M.
     lets (n&N): mem_Nth M. false S.
      apply~ Nth_here.
      applys~ Nth_next N.
      false*.
    apply~ IHl.
     introv N1 N2. forwards L: S.
      applys~ Nth_next N1.
      applys~ Nth_next N2.
      inverts* L.
     simpl in M. rew_refl in M. inverts~ M; tryfalse.
Qed.


(**************************************************************)
(** ** Pickable instances to describe infinite sets. *)

Require Export TLC.LibList.

Definition int_list_greater : forall ns,
    { n : nat | Forall (fun n' : int => n' < n) ns }.
  induction ns as [|n ns].
   exists 0%nat. apply Forall_nil.
   lets (n'&IH): (rm IHns).
    destruct (decide (n < n')) eqn: T; fold_bool; rewrite decide_spec in T; rew_refl in T.
     exists n'. apply~ Forall_cons.
     exists (1 + Z.to_nat n)%nat. apply~ Forall_cons.
      simpl. rewrite <- Z2Nat.inj_succ; try math. rewrite Z2Nat.id; math.
      rewrite Forall_iff_forall_mem with (CA := _) in *. introv M. apply IH in M.
       simpl. rewrite <- Z2Nat.inj_succ; try math. rewrite Z2Nat.id; math.
Defined.

Lemma nat_infinite : forall ns,
  exists (n : nat), ~ mem n ns.
Proof.
  introv. lets (n&Fn): int_list_greater (map (fun n : nat => n : int) ns).
  exists n. introv A. rewrite Forall_iff_forall_mem with (CA := _) in Fn. forwards~: Fn.
   rewrite map_mem. eexists. splits*.
   math.
Qed.

Global Instance nat_infinite_pick : forall ns,
    Pickable (fun n : nat => ~ mem n ns).
  introv. lets (n&Fn): int_list_greater (map (fun n : nat => n : int) ns).
  applys pickable_make n. introv _.
  introv A. rewrite Forall_iff_forall_mem with (CA := _) in Fn. forwards~: Fn.
   rewrite map_mem. eexists. splits*.
   math.
Defined.


Lemma string_infinite : forall F,
  exists (f : string), ~ mem f F.
Proof.
  introv. forwards (n&Fn): int_list_greater (map (String.length : _ -> int) F).
  asserts (f&Lf): (exists f, String.length f = n).
    clear. induction n.
     exists~ ""%string.
     lets (f&IH): (rm IHn). exists ("f" ++ f)%string. simpls~.
  exists f. introv M. rewrite Forall_iff_forall_mem with (CA := _) in Fn. forwards: (rm Fn) n.
   rewrite* map_mem.
   math.
Qed.


(**************************************************************)
(** ** Partially Decidable **)

(** When we are not looking for completness, but that correctness is
  enough, we sometimes want to be able to decide something without
  having the guarantee to succeed every time, hence this type
  class. **)

Class PartiallyDecidable (P : Prop) : Type := PartiallyDecidable_make {
    try_to_decide : bool ;
    try_to_decide_spec : try_to_decide -> P
  }.

Implicit Arguments try_to_decide [[PartiallyDecidable]].

Global Instance Decidable_PartiallyDecidable : forall P,
    Decidable P ->
    PartiallyDecidable P.
  introv D. applys PartiallyDecidable_make (decide P). rew_refl~.
Defined.

(** This lemma is useful when we want to switch to the rest of the
  files without dealing yet with the extraction.  But it is important
  for it not to be an instance! **)
Definition PartiallyDecidable_false : forall P,
    PartiallyDecidable P.
  introv. applys PartiallyDecidable_make false. rew_refl*.
Defined.

Global Instance PartiallyDecidable_weaken : forall P Q : Prop,
    (P -> Q) ->
    PartiallyDecidable P ->
    PartiallyDecidable Q.
  introv I D. applys PartiallyDecidable_make (try_to_decide P).
  introv T. apply I. apply~ try_to_decide_spec.
Defined.


(**************************************************************)
(** ** Partially Pickable **)

(** Same than [Pickable], but the function is allowed to fail. This is
  very similar to [PartiallyDecidable]. **)

Class Partially_pickable (A : Type) (P : A -> Prop) := partially_pickable_make {
  try_to_pick : option A ;
  try_to_pick_correct : forall a, try_to_pick = Some a -> P a }.

Implicit Arguments try_to_pick [A [Partially_pickable]].
Extraction Inline try_to_pick.

Global Instance Pickable_option_Partially_pickable :
    forall (A : Type) (P : A -> Prop),
    Pickable_option P -> Partially_pickable P.
  introv [pi C D]. applys~ partially_pickable_make pi.
Defined.

Definition partially_pickable_false : forall (A : Type) (P : A -> Prop),
    Partially_pickable P.
  introv. applys partially_pickable_make (@None A). introv I. inverts I.
Defined.


(**************************************************************)
(** * LibHeap additionnal features *)

Lemma to_list_spec : forall K `{Comparable K} V `{Comparable V} (h : heap K V) k v,
  LibList.mem (k, v) (to_list h) <-> binds h k v.
Admitted.

Definition check_heap K V (P : K -> V -> bool) (h : heap K V) :=
  decide (Forall (fun (kv : K * V) =>
    istrue (let (k, v) := kv in P k v)) (to_list h)).

Lemma check_heap_spec : forall K `{Comparable K} V `{Comparable V} P (h : heap K V),
  check_heap P h <-> forall k v,
    binds h k v ->
    P k v.
Proof.
  introv CK CV. iff I.
   introv B. unfolds in I. rew_refl in *. eapply to_list_spec with (H := _) (H0 := _) in B.
    applys Forall_mem B. apply I.
   unfolds. rew_refl. apply Forall_iff_forall_mem with (CA := _ : Comparable (K * V)). intros (k&v) Ih.
    apply to_list_spec in Ih. autos*.
Qed.

Definition filter_heap K V (P : K -> V -> bool) (h : heap K V) :=
  filter (fun (kv : K * V) => let (k, v) := kv in P k v) (to_list h).

Lemma filter_heap_spec : forall K `{Comparable K} V `{Comparable V} P (h : heap K V) k v,
  LibList.mem (k, v) (filter_heap P h) <-> binds h k v /\ P k v.
Proof.
  introv. unfolds filter_heap. iff I.
  rewrite filter_mem_eq in I. rew_refl in I. lets (Ih&HP): (rm I). splits~. apply~ to_list_spec.
  rewrite filter_mem_eq. lets (B&HP): (rm I). rew_refl. splits~. apply~ to_list_spec.
Qed.

Global Instance binds_decidable : forall K V (CK : Comparable K) (CV : Comparable V) (IV : Inhab V)
     (h : heap K V) k v,
   Decidable (binds h k v).
  introv CK CV IV; introv. applys decidable_make (unsome_default false
    (option_map (fun v' => decide (v = v')) (read_option h k))).
  destruct (read_option h k) eqn:E; simpl; fold_bool; rew_refl.
  rewrite isTrue_eq_isTrue. iff I.
  substs. apply~ @read_option_binds.
  forwards B: @read_option_binds E. applys~ @binds_func I B.
  introv B. forwards E': @binds_read_option B. rewrite E in E'. false.
Defined.

Lemma read_option_write_eq : forall (K : Type) (HK : Comparable K) (V : Type)
    (HV : Inhab V) (h : heap K V) (k : K) (v : V),
  read_option (write h k v) k = Some v.
Proof.
  introv HV. introv. rewrite read_option_def. cases_if as I.
  rewrite~ read_write_eq.
  false I. apply indom_write_eq.
Qed.

Lemma read_option_write_neq : forall (K : Type) (HK : Comparable K) (V : Type)
    (HV : Inhab V) (h : heap K V) (k k' : K) (v : V), k <> k' ->
  read_option (write h k' v) k = read_option h k.
Proof.
  introv HV D. do 2 rewrite read_option_def. cases_if as I.
  rewrite~ read_write_neq; try applys~ @indom_write_inv I. cases_if as I'; trivial.
  false I'. applys~ @indom_write_inv I.
  cases_if as I'; trivial. false I. apply~ @indom_write.
Qed.


Section HeapCmp.

Variables K V : Type.
Hypothesis CK : Comparable K.
Hypothesis CV : Comparable V.
Hypothesis IV : Inhab V.

Definition heap_compare (h1 h2 : heap K V) :=
  let dom1 := LibList.map fst (to_list h1) in
  let dom2 := LibList.map fst (to_list h2) in
  let check k := decide (read_option h1 k = read_option h2 k) in
  decide (Forall check dom1 /\ Forall check dom2).

Axiom heap_compare_correct : forall h1 h2,
  heap_compare h1 h2 <-> (h1 '= h2). (* TODO *)

Lemma heap_comparable : Comparable (heap K V).
  apply make_comparable. intros h1 h2.
  applys decidable_make (heap_compare h1 h2).
  extens. apply heap_compare_correct.
Qed.

End HeapCmp.

