Require Export AbstractionsSignatures.
Require Export FuncFunctors.
Require Export NormalizeTest.

Module EnvNotRelational.
  Import FuncLattice.
  Section NumAbstraction.
    Variable A : Type.
    Variable L : AbLattice.t A.
    Variable AN : NumAbstraction.t L.
    Variable refine_max:nat.
    Variable P:program.

    Definition gamma (Env:array A) (env:env) : Prop :=
      forall x n,
        In x (vars P) ->
        env x = n -> GammaNum (get Env x) n.

    Lemma gamma_monotone : forall (E1 E2:array A),
      E1 ⊑♯ E2 -> gamma E1 ⊑ gamma E2.
    Proof.
      intros E1 E2 H1 ρ H2 x n H H'.
      apply (@Gamma.monotone _ _ _  _ GammaNum) with (get E1 x); auto.
    Qed.
    
    Lemma gamma_meet_morph : forall E1 E2:array A,
      (gamma E1 ⊓  gamma E2) ⊑ gamma (E1 ⊓♯ E2).
    Proof.
      intros; intros T1 T2 x n H1 H2.
      apply (@Gamma.monotone _ _ _  _ GammaNum) with
        ((get E1 x) ⊓♯ (get E2 x)).
      apply PosetDec.refl.
      apply EquivDec.sym.
      apply get_meet.
      apply (@Gamma.meet_morph _ _ _  _ GammaNum); auto.
      destruct T2; split; auto.
    Qed.

    Instance Gamma : Gamma.t (℘ env) (FuncLattice.array A).
    exists gamma.
    apply gamma_monotone.
    apply gamma_meet_morph.
  Defined.

  Lemma gamma_bottom_eq :forall E env x,
    gamma E env ->
    In x (vars P) ->
    (get E x) =♯ (⊥♯) ->
    False.
  Proof.
    intros.
    elim (NumAbstraction.bottom_empty (env x)).
    apply (@Gamma.monotone _ _ _  _ GammaNum) with (get E x); auto.
  Qed.

  Lemma test_bot : forall (E:array A) (l:list var),
    {exists x, In x l /\ (get E x) =♯ (⊥♯)}
    +{~ exists x, In x l /\ (get E x) =♯ (⊥♯)}.
  Proof.
    induction l.
    right; intros (x, (H1,H2)).
    inversion_clear H1.
    destruct (EquivDec.dec (get E a) (⊥♯)).
    left; exists a; auto with datatypes.
    destruct IHl; [left|right]; auto.
    destruct e as [x (H1,H2)].
    exists x; auto with datatypes.
    intros (x, (H1,H2)); elim n0.
    elim H1; intros.
    subst; elim n; auto.
    exists x; split; auto.
  Qed.

  Definition filter (E:array A) : array A :=
    if test_bot E (vars P) then (⊥♯) else E.

  Lemma filter_correct : forall E : array A,
    gamma E ⊑  gamma (filter E).
  Proof.
    unfold filter; repeat intro.
    destruct test_bot; auto.
    destruct e as [y (T1,T2)].
    elim (gamma_bottom_eq E x y); auto.
  Qed.
  Hint Resolve filter_correct.

  Fixpoint SemExpr (Env:array A) (e:expr) : A :=
    match e with
      Const n => NumAbstraction.const n
      | Unknown => NumAbstraction.top
      | Var x => get Env x
      | Numop o e1 e2 => NumAbstraction.sem_op o (SemExpr Env e1) (SemExpr Env e2)
    end.

  Lemma SemExpr_correct : forall Env e env n,
    gamma Env env ->
    sem_expr P env e n ->
    GammaNum (SemExpr Env e) n.
  Proof.
    induction e; simpl; intros.
    inversion_clear H0; apply NumAbstraction.const_correct.
    apply NumAbstraction.top_correct.
    inversion_clear H0; auto.
    inversion_clear H0.
    apply NumAbstraction.sem_op_correct with n1 n2; eauto.
  Qed.

  Definition assign (Env:array A) (x:var) (e:expr) : array A :=
    filter (subst Env x (SemExpr Env e)).

  Lemma assign_correct : forall Env x e,
    Collect.assign P x e (gamma Env) ⊑ gamma (assign Env x e).
  Proof.
    unfold assign; simpl; intros Env x e env H.
    apply filter_correct.
    intros x' n H1 H2.
    inversion_clear H.
    destruct H0 as [k [T1 [T2 T3]]].
    case (eq_word_dec x' x); intros.
    rewrite get_subst1; auto.
    subst.
    apply SemExpr_correct with x0; auto.
    inversion_clear T3; subst; auto.
    rewrite get_subst2; auto.
    inversion_clear T3.
    rewrite <- H2; rewrite <- H0; auto.
  Qed.

  Fixpoint subst_list_rec (N:A) (l:list var) (Env:array A) : array A :=
    match l with
      nil => Env
      | x::q => subst_list_rec N q (subst Env x N)
    end.

  Lemma subst_list_rec_old : forall N l Env x,
    get Env x = N ->
    get (subst_list_rec N l Env) x = N.
  Proof.
    induction l; simpl; intros; auto.
    rewrite IHl; auto.
    case (eq_word_dec x a); intros; unfold assign; simpl.
    rewrite get_subst1; auto.
    rewrite get_subst2; auto.
  Qed.

  Lemma subst_list_rec_correct : forall N l Env x,
    In x l ->
    get (subst_list_rec N l Env) x = N.
  Proof.
    induction l; simpl; intros. 
    inversion_clear H.
    destruct H; subst; unfold assign.
    rewrite subst_list_rec_old; auto.
    rewrite get_subst1; auto.
    auto.
  Qed.

  Definition top : array A :=
    subst_list_rec (NumAbstraction.top) (vars P) (⊥♯).

  Lemma init_env_correct : ⊤ ⊑ gamma top.
  Proof.
    unfold gamma, top; repeat intro.
    rewrite subst_list_rec_correct; auto.
    apply NumAbstraction.top_correct.
  Qed.

  Fixpoint BackSemExpr (Env:array A) (e:expr) (N:A) {struct e} : array A :=
    match e with
      Const n => if (EquivDec.dec ((NumAbstraction.const n) ⊓♯ N) (⊥♯)) then (⊥♯)
        else Env
      | Unknown => Env
      | Var x => filter (modify Env x (fun N' => N ⊓♯ N'))
      | Numop o e1 e2 =>
        let (N1,N2) := NumAbstraction.back_sem_op o N (SemExpr Env e1) (SemExpr Env e2) in
          filter ((BackSemExpr Env e1 N1) ⊓♯ (BackSemExpr Env e2 N2))
    end.

  Lemma BackSemExpr_correct : forall Env e N env n,
    sem_expr P env e n ->
    gamma Env env ->
    GammaNum N n ->
    gamma (BackSemExpr Env e N) env.
  Proof.
    Opaque MeetDec.op.
    induction e; simpl; intros; auto.
    case (@EquivDec.dec A); intros; auto.
    assert (GammaNum (⊥♯) n0).
    apply (@Gamma.monotone _ _ _  _ GammaNum) with ((NumAbstraction.const n) ⊓♯ N).
    apply PosetDec.refl; auto.
    apply (@Gamma.meet_morph _ _ _ _ GammaNum); auto.
    inversion_clear H.
    split; auto.
    apply NumAbstraction.const_correct; auto.
    elim (NumAbstraction.bottom_empty _ H2).
    apply filter_correct.
    intros x' n' H2 H3.
    inv H.
    case (eq_word_dec x' x); intros; subst.
    rewrite get_modify1; auto.
    apply (@Gamma.meet_morph _ _ _ _ GammaNum); split; auto.
    rewrite get_modify2; auto.
    inversion_clear H.
    generalize (NumAbstraction.back_sem_op_correct o N (SemExpr Env e1) (SemExpr Env e2) n1 n2 n).
    case NumAbstraction.back_sem_op; intros.
    apply filter_correct.
    elim H with a a0; auto; intros.
    apply gamma_meet_morph; split; auto.
    apply IHe1 with n1; auto.
    apply IHe2 with n2; auto.
    eapply SemExpr_correct; eauto.
    eapply SemExpr_correct; eauto.
  Qed.

  Definition back_assign (Env_before Env_after:array A) (x:var) (e:expr) : array A :=
    BackSemExpr Env_before e (get Env_after x).

  Lemma back_affect_correct : forall Env Env' ρ ρ' x n e,
    gamma Env ρ ->
    gamma Env' ρ' ->
    sem_expr P ρ e n ->
    Semantics.subst ρ x n ρ' -> 
      In x (vars P) ->
      gamma (back_assign Env Env' x e) ρ.
  Proof.
    intros.
    inversion_clear H2.
    unfold back_assign.
    eapply BackSemExpr_correct; eauto.
  Qed.

  Fixpoint back_test'' (t:test) (Env:array A) {struct t} : array A :=
    match t with
      | Numcomp c e1 e2 =>
        let (N1,N2) := NumAbstraction.back_test c (SemExpr Env e1) (SemExpr Env e2) in
          filter ((BackSemExpr Env e1 N1) ⊓♯ (BackSemExpr Env e2 N2))
      | And t1 t2 => filter ((back_test'' t1 Env) ⊓♯ (back_test'' t2 Env))
      | Or t1 t2 => (back_test'' t1 Env) ⊔♯ (back_test'' t2 Env)
      | Not _ => Env
    end.

  Lemma back_test''_correct : forall t Env env,
    gamma Env env ->
    sem_test P env t true ->
    gamma (back_test'' t Env) env.
  Proof.
    Opaque JoinDec.op.
    intros t; induction t; intros; simpl.
    generalize (NumAbstraction.back_test_correct c (SemExpr Env e1) (SemExpr Env e2)).
    case NumAbstraction.back_test; intros.
    apply filter_correct.
    inversion_clear H0.
    elim (H1 a a0 n1 n2); intros; auto.
    apply gamma_meet_morph; split.
    apply BackSemExpr_correct with n1; auto.
    apply BackSemExpr_correct with n2; auto.
    apply SemExpr_correct with env; auto.
    apply SemExpr_correct with env; auto.
    auto.
    apply filter_correct.
    elim (sem_test_and _ _ _ _ H0); intros.
    apply gamma_meet_morph; split; auto.
    elim (sem_test_or _ _ _ _ H0); intros.
    apply gamma_monotone with (back_test'' t1 Env); auto.
    apply gamma_monotone with (back_test'' t2 Env); auto.
  Qed.

  Fixpoint back_test' (t:test) (n:nat) (Env:array A) {struct n} : array A :=
    match n with
      | O => Env
      | Datatypes.S n => 
        let Env' := back_test'' t Env in
          if PosetDec.dec Env Env' then Env
            else back_test' t n Env' 
    end.

  Lemma back_test'_correct : forall t env n Env,
    sem_test P env t true ->
    gamma Env env ->
    gamma (back_test' t n Env) env.
  Proof.
    Opaque PosetDec.dec.
    induction n; simpl; auto; intros.
    destruct (@PosetDec.dec (array A)); auto.
    apply IHn; auto.
    apply back_test''_correct; auto.
  Qed.

  Definition assert (t:test) (Env:array A) : array A :=
    back_test' (normalize_test t) refine_max Env.

  Lemma assert_correct : forall t Env ,
    Collect.assert P t (gamma Env) ⊑ gamma (assert t Env).
  Proof.
    unfold assert; intros t E e H.
    destruct H.
    apply back_test'_correct; auto.
    apply normalize_test_correct; auto.
  Qed.

  End NumAbstraction.

  Instance Make {A} `{L:AbLattice.t A} (N:NumAbstraction.t L) (n:nat) (p:program) : 
    AbEnv.t (FuncLattice.Lattice L) p.
  Proof.
    refine (AbEnv.Make _ _ _ (Gamma A L N p) (assign _ _ _ _) _ (top _ _ _ _) _ (assert _ _ _ _ _) _).
    apply (assign_correct A L N p).
    apply (init_env_correct A L N p).
    apply (assert_correct A L N n p).
  Defined.

End EnvNotRelational.


(* 
*** Local Variables: ***
*** coq-prog-name: "coqtop" ***
*** coq-prog-args: ("-emacs-U" "-I" "lib/" "-I" ".") ***
*** End: ***
 *)
