Require Export ZArith.
Require Export List.
Require Export Bool.
Require Export Word.
(* SYNTAX *)

Ltac inv H := inversion H; try (subst; clear H).

Definition var  := Word.
Definition pp  := Word.

Inductive op  := Add | Sub | Mult.

Inductive expr  := 
   Const (n:Z)
 | Unknown
 | Var (x:var)
 | Numop (o:op) (e1 e2:expr).

Inductive comp  := Eq | Lt.

Inductive test  :=
 | Numcomp (c:comp) (e1 e2:expr)
 | Not (t:test)
 | And (t1 t2:test)
 | Or (t1 t2:test).

Inductive instr  :=
   Assign (p:pp) (x:var) (e:expr)
 | Skip (p:pp)
 | Assert (p:pp) (t:test)
 | If (p:pp) (t:test) (b1 b2:instr)
 | While (p:pp) (t:test) (b:instr)
 | Seq (i1:instr) (i2:instr).

Record program  := Prog {
  p_instr:instr;
  p_end: pp;
  vars: list var
}.

Fixpoint first (i:instr) : pp :=
  match i with
    | Assign p x e => p
    | Skip p => p
    | Assert p t => p
    | If p t i1 i2 => p
    | While p t i => p
    | Seq i1 i2 => first i1
  end.

Definition env  := var -> Z.

Inductive sem_op : op -> Z -> Z -> Z -> Prop :=
 | sem_add : forall n1 n2, sem_op Add n1 n2 (n1+n2)
 | sem_sub : forall n1 n2, sem_op Sub n1 n2 (n1-n2)
 | sem_mult : forall n1 n2, sem_op Mult n1 n2 (n1*n2).

Inductive sem_expr (p:program) (ρ:env) : expr -> Z -> Prop :=
   sem_const : forall n, sem_expr p ρ (Const n) n
 | sem_unknow : forall n, sem_expr p ρ Unknown n
 | sem_var : forall x, In x (vars p) -> sem_expr p ρ (Var x) (ρ x)
 | sem_numop : forall o e1 e2 n1 n2 n,
     sem_expr p ρ e1 n1 ->
     sem_expr p ρ e2 n2 ->
     sem_op o n1 n2 n ->
     sem_expr p ρ (Numop o e1 e2) n.

Open Scope Z_scope.

Inductive sem_comp : comp -> Z -> Z -> Prop :=
 | sem_eq : forall n1 n2, n1=n2 -> sem_comp Eq n1 n2
 | sem_le : forall n1 n2, n1<n2 -> sem_comp Lt n1 n2.

Inductive sem_test (p:program) (ρ:env) : test -> bool -> Prop :=
 | sem_numcomp_true : forall c e1 e2 n1 n2,
     sem_expr p ρ e1 n1 ->
     sem_expr p ρ e2 n2 ->
     sem_comp c n1 n2 ->
     sem_test p ρ (Numcomp c e1 e2) true
 | sem_numcomp_false : forall c e1 e2 n1 n2,
     sem_expr p ρ e1 n1 ->
     sem_expr p ρ e2 n2 ->
     ~ sem_comp c n1 n2 ->
     sem_test p ρ (Numcomp c e1 e2) false
 | sem_not : forall t b,
     sem_test p ρ t b ->
     sem_test p ρ (Not t) (negb b)
 | sem_and : forall t1 t2 b1 b2,
     sem_test p ρ t1 b1 ->
     sem_test p ρ t2 b2 ->
     sem_test p ρ (And t1 t2) (andb b1 b2)
 | sem_or : forall t1 t2 b1 b2,
     sem_test p ρ t1 b1 ->
     sem_test p ρ t2 b2 ->
     sem_test p ρ (Or t1 t2) (orb b1 b2).

Inductive subst : env -> var -> Z -> env -> Prop :=
  subst_def : forall (ρ ρ' : env) x n,
    ρ' x = n ->
    (forall y, x<>y -> ρ y = ρ' y) ->
    subst ρ x n ρ'.

Inductive Kind :=
   KAssign (x:var) (e:expr)
 | KSkip
 | KAssert (t:test)
 | KSeq1 (i:instr) (l:pp)
 | KSeq2 (k:Kind).

Inductive config :=
| Final (ρ:env)
| Inter (i:instr) (ρ:env).

Inductive sos (p:program) : Kind -> (instr * env) -> config -> Prop :=
 | sos_affect : forall l x e n ρ1 ρ2,
     sem_expr p ρ1 e n ->
     subst ρ1 x n ρ2 ->
     In x (vars p) ->
     sos p (KAssign x e) (Assign l x e,ρ1) (Final ρ2)
 | sos_skip : forall l ρ,
     sos p KSkip (Skip l,ρ) (Final ρ)
 | sos_assert_true : forall l t ρ,
     sem_test p ρ t true ->
     sos p (KAssert t) (Assert l t,ρ) (Final ρ)
 | sos_if_true : forall l t b1 b2 ρ,
     sem_test p ρ t true ->
     sos p (KAssert t) (If l t b1 b2,ρ) (Inter b1 ρ)
 | sos_if_false : forall l t b1 b2 ρ,
     sem_test p ρ t false ->
     sos p (KAssert (Not t)) (If l t b1 b2,ρ) (Inter b2 ρ)
 | sos_while_true : forall l t b ρ,
     sem_test p ρ t true ->
     sos p (KAssert t) (While l t b,ρ) (Inter (Seq b (While l t b)) ρ)
 | sos_while_false : forall l t b ρ,
     sem_test p ρ t false ->
     sos p (KAssert (Not t)) (While l t b,ρ) (Final ρ)
 | sos_seq1 : forall k i1 i2 ρ ρ',
     sos p k (i1,ρ) (Final ρ') ->
     sos p (KSeq1 i1 (first i2)) (Seq i1 i2,ρ) (Inter i2 ρ')
 | sos_seq2 : forall k i1 i1' i2 ρ ρ',
     sos p k (i1,ρ) (Inter i1' ρ') ->
     sos p (KSeq2 k) (Seq i1 i2,ρ) (Inter (Seq i1' i2) ρ').
  
Inductive sos_plus (p:program) : (instr * env) -> config -> Prop :=
| sos_plus0 : forall i ρ, sos_plus p (i,ρ) (Inter i ρ)
| sos_plus1 : forall k s1 s2, sos p k s1 s2 -> sos_plus p s1 s2
| sos_trans : forall k s1 i ρ s3, sos p k s1 (Inter i ρ) -> sos_plus p (i,ρ) s3 -> sos_plus p s1 s3.
Hint Constructors sos_plus.

Inductive reachable_sos (p:program) : pp*env -> Prop :=
| reachable_sos_intermediate : forall ρ0 i ρ,
  sos_plus p (p_instr p,ρ0) (Inter  i ρ) ->
  reachable_sos p (first i,ρ)
| reachable_sos_final : forall ρ0 ρ,
  sos_plus p (p_instr p,ρ0) (Final ρ) ->
  reachable_sos p (p_end p,ρ).

(* 
*** Local Variables: ***
*** coq-prog-name: "coqtop" ***
*** coq-prog-args: ("-emacs-U" "-I" "lib" "-I" ".") ***
*** End: ***
 *)


