Require Export TabTree.
Require Export FuncSignature.

Implicit Arguments leaf [A].
Implicit Arguments node [A].

Inductive height_inf (A:Type) : tree A -> nat -> Prop :=
 | inf_leaf : forall (p:nat), (height_inf A leaf p)
 | inf_node : forall (a:A) (tO tI:tree A) (p:nat),
      (height_inf A tO p) -> 
      (height_inf A tI p) -> 
      (height_inf A (node a tO tI) (S p)).
Implicit Arguments height_inf [A].

Record FuncTree (A:Type) (max:nat) :  Type := FuncTree_constr
  {tr : tree A;
   tr_bound : height_inf tr (S max)}.
Implicit Arguments FuncTree_constr [A].
Implicit Arguments tr [A].
Implicit Arguments tr_bound [A].

Definition leafP := inf_leaf.
Definition nodeP := inf_node.
Implicit Arguments leafP [A p].
Implicit Arguments nodeP [A a tO tI p].
Definition ft := FuncTree_constr.
Implicit Arguments ft [A max].


Section ArrayBinFuncFiniteSetLattice.

Variable At:Type.
Variable L:AbLattice.t At.

Definition apply_tree := apply_tree At (⊥♯).

Notation FuncTree := (FuncTree At).

Definition Feq (t1 t2 : tree At) : Prop :=
  forall p : positive, EquivDec.eq (apply_tree t1 p) (apply_tree t2 p).

Lemma apply_tree_bot :
 forall (max:nat) (t : tree At) (p : positive),
  (height_inf t (S max)) ->
  ~(inf_log p max) ->
  (apply_tree t p) = (⊥♯).
induction max; simpl; intros; auto.
inversion_clear H; simpl; auto.
inversion_clear H1; simpl.
inversion_clear H2; simpl.
destruct p; auto.
elim H0; constructor.
inversion_clear H; simpl; auto.
destruct p.
apply IHmax; auto.
red; intros; elim H0; constructor; auto.
apply IHmax; auto.
red; intros; elim H0; constructor; auto.
elim H0; constructor.
Qed.

Definition apply_functree (max:nat)
 (f : FuncTree max) (p : positive) :=
  apply_tree (tr _ f) p.

Definition Forder (f1 f2 : tree At) : Prop :=
  forall p : positive, PosetDec.order (apply_tree f1 p) (apply_tree f2 p).

Definition Fbottom :
  forall (max:nat), FuncTree max.
refine (fun max => FuncTree_constr _ leaf _).
intros.
constructor.
Defined.

Lemma apply_bottom :
 forall (max:nat) (p : positive),
 apply_functree _ (Fbottom max) p = (⊥♯).
unfold apply_functree, Fbottom in |- *; simpl in |- *; intros.
reflexivity.
Qed.
Definition Feq_bot_dec :
  forall t1 : tree At,
  {(forall p : positive, EquivDec.eq (apply_tree t1 p) (⊥♯))} +
  {~ (forall p : positive, EquivDec.eq (apply_tree t1 p) (⊥♯))}.
induction t1 as [| a t0 IHt0 t2 IHt2]; simpl in |- *.
left; simpl in |- *; auto.
case (PosetDec.dec a (⊥♯)); intros.
case IHt0; intros.
case IHt2; intros.
left; simple destruct p; simpl in |- *; auto.
right; red in |- *; intros H; elim n; intros p; generalize (H (xI p));
 simpl in |- *; auto.
right; red in |- *; intros H; elim n; intros p; generalize (H (xO p));
 simpl in |- *; auto.
right; red in |- *; intros H; elim n; generalize (H 1%positive);
 simpl in |- *; auto.
Defined.

Definition Feq_dec : forall t1 t2 : tree At, {Feq t1 t2} + {~ Feq t1 t2}.
induction t1 as [| a t0 IHt0 t2 IHt2]; intros T; case T; intros.
left; unfold Feq in |- *; auto.
case (EquivDec.dec a (⊥♯)); intros.
case (Feq_bot_dec t); intros.
case (Feq_bot_dec t0); intros.
left; unfold Feq in |- *; intros.
case p; simpl in |- *; auto.
right; unfold Feq in |- *; intros H; elim n; intros p;
 generalize (H (xI p)); simpl in |- *; auto.
simpl.
right; unfold Feq in |- *; intros H; elim n; intros p;
 generalize (H (xO p)); simpl in |- *; auto.
right; unfold Feq in |- *; intros H; elim n; generalize (H 1%positive); auto.
case (EquivDec.dec a (⊥♯)); intros.
case (Feq_bot_dec t0); intros.
case (Feq_bot_dec t2); intros.
left; unfold Feq in |- *; intros.
case p; simpl in |- *; auto.
right; unfold Feq in |- *; intros H; elim n; intros p;
 generalize (H (xI p)); simpl in |- *; auto.
right; unfold Feq in |- *; intros H; elim n; intros p;
 generalize (H (xO p)); simpl in |- *; auto.
right; unfold Feq in |- *; intros H; elim n; generalize (H 1%positive); auto.
case (EquivDec.dec a a0); intros.
case (IHt0 t); intros.
case (IHt2 t1); intros.
left; unfold Feq in |- *; intros.
case p; simpl in |- *; auto.
right; unfold Feq in |- *; intros H; elim n; intros p;
 generalize (H (xI p)); simpl in |- *; auto.
right; unfold Feq in |- *; intros H; elim n; intros p;
 generalize (H (xO p)); simpl in |- *; auto.
right; unfold Feq in |- *; intros H; elim n; generalize (H 1%positive); auto.
Defined.

Definition Forder_dec : forall t1 t2 : tree At, {Forder t1 t2} + {~ Forder t1 t2}.
induction t1 as [| a t0 IHt0 t2 IHt2]; intros T; case T; intros.
left; unfold Forder in |- *; auto.
left; unfold Forder in |- *; auto.
unfold apply_tree; simpl; auto.
case (PosetDec.dec a (⊥♯)); intros.
case (Feq_bot_dec t0); intros.
case (Feq_bot_dec t2); intros.
left; unfold Forder in |- *; intros.
case p; simpl in |- *; auto.
right; unfold Forder in |- *; intros H; elim n; intros p;
 generalize (H (xI p)); simpl in |- *; auto.
right; unfold Forder in |- *; intros H; elim n; intros p;
 generalize (H (xO p)); simpl in |- *; auto.
right; unfold Forder in |- *; intros H; elim n; generalize (H 1%positive); auto.
case (PosetDec.dec a a0); intros.
case (IHt0 t); intros.
case (IHt2 t1); intros.
left; unfold Forder in |- *; intros.
case p; simpl in |- *; auto.
right; unfold Forder in |- *; intros H; elim n; intros p;
 generalize (H (xI p)); simpl in |- *; auto.
right; unfold Forder in |- *; intros H; elim n; intros p;
 generalize (H (xO p)); simpl in |- *; auto.
right; unfold Forder in |- *; intros H; elim n; generalize (H 1%positive); auto.
Defined.

Inductive is_bottom : tree At -> Prop :=
  | is_bottom_leaf : is_bottom leaf
  | is_bottom_node :
      forall (a :At) (t1 t2 : tree At),
      EquivDec.eq a (⊥♯) ->
      is_bottom t1 -> is_bottom t2 -> is_bottom (node a t1 t2).

Inductive order' : tree At -> tree At -> Prop :=
  | order'_leaf : forall t : tree At, order' leaf t
  | order'_node_leaf :
      forall a (t1 t2 : tree At),
      EquivDec.eq a (⊥♯) ->
      is_bottom t1 -> is_bottom t2 -> order' (node a t1 t2) leaf
  | order'_node_node :
      forall (a b :At) (A1 A2 B1 B2 : tree At),
      PosetDec.order a b ->
      order' A1 B1 -> order' A2 B2 -> order' (node a A1 A2) (node b B1 B2).

Inductive eq' : tree At -> tree At -> Prop :=
  | eq'_leaf : eq' leaf leaf
  | eq'_leaf_node :
      forall (b :At) (t1 t2 : tree At),
      EquivDec.eq (⊥♯) b ->
      is_bottom t1 -> is_bottom t2 -> eq' leaf (node b t1 t2)
  | eq'_node_leaf :
      forall (b :At) (t1 t2 : tree At),
      EquivDec.eq (⊥♯) b ->
      is_bottom t1 -> is_bottom t2 -> eq' (node b t1 t2) leaf
  | eq'_node_node :
      forall (a b :At) (A1 A2 B1 B2 : tree At),
      EquivDec.eq a b ->
      eq' A1 B1 -> eq' A2 B2 -> eq' (node a A1 A2) (node b B1 B2).

Ltac CaseEq a := generalize (refl_equal a); pattern a at -1 in |- *; case a.

Lemma is_bottom_dec : forall t : tree At, {is_bottom t} + {~ is_bottom t}.
induction t.
left; constructor; auto.
case (PosetDec.dec a (⊥♯)); intros.
case IHt1; intros.
case IHt2; intros.
left; constructor; auto.
right; red in |- *; intros; elim n.
inversion_clear H; auto.
right; red in |- *; intros; elim n.
inversion_clear H; auto.
right; red in |- *; intros; elim n.
inversion_clear H; auto.
Qed.

Lemma eq'_dec : forall t1 t2 : tree At, {eq' t1 t2} + {~ eq' t1 t2}.
induction t1 as [| a t0 IHt0 t2 IHt2].
induction t2 as [| a t0 IHt0 t1 IHt1].
left; constructor; auto.
case (PosetDec.dec a (⊥♯)); intros.
case (is_bottom_dec t0); intros.
case (is_bottom_dec t1); intros.
left; constructor; auto.
right; red in |- *; intros; elim n.
inversion_clear H; auto.
right; red in |- *; intros; elim n.
inversion_clear H; auto.
right; red in |- *; intros; elim n.
inversion_clear H; auto.
intros t3; induction t3 as [| a0 t1 IHt1 t4 IHt4].
case (PosetDec.dec a (⊥♯)); intros.
case (is_bottom_dec t0); intros.
case (is_bottom_dec t2); intros.
left; constructor; auto.
right; red in |- *; intros; elim n.
inversion_clear H; auto.
right; red in |- *; intros; elim n.
inversion_clear H; auto.
right; red in |- *; intros; elim n.
inversion_clear H; auto.
case (EquivDec.dec a a0); intros.
case (IHt0 t1); intros.
case (IHt2 t4); intros.
left; constructor; auto.
right; red in |- *; intros; elim n.
inversion_clear H; auto.
right; red in |- *; intros; elim n.
inversion_clear H; auto.
right; red in |- *; intros; elim n.
inversion_clear H; auto.
Qed.

Lemma is_bottom2_eq' :
 forall t1 t2 : tree At, is_bottom t1 -> is_bottom t2 -> eq' t1 t2.
induction t1 as [| a t0 IHt0 t2 IHt2].
intros.
inversion_clear H; inversion_clear H0; constructor; eauto.
intros.
inversion_clear H; inversion_clear H0; constructor; eauto.
Qed.

Lemma is_bottom_eq' :
 forall t1 t2 : tree At, is_bottom t1 -> eq' t1 t2 -> is_bottom t2.
induction t1 as [| a t0 IHt0 t2 IHt2].
intros.
inversion_clear H.
inversion_clear H0; constructor; eauto.
intros.
inversion_clear H.
inversion_clear H0; constructor; eauto.
Qed.
Hint Resolve is_bottom_eq' is_bottom2_eq'.

Lemma eq'_sym : forall t1 t2 : tree At, eq' t1 t2 -> eq' t2 t1.
induction t1 as [| a t0 IHt0 t2 IHt2]; intros.
inversion_clear H; constructor; auto.
inversion_clear H; constructor; auto.
Qed.

Lemma eq'_trans : forall t1 t2 t3 : tree At, eq' t1 t2 -> eq' t2 t3 -> eq' t1 t3.
induction t1 as [| a t0 IHt0 t2 IHt2].
intros t2 t3 H; inversion_clear H; intros H'; inversion_clear H'; constructor;
 eauto.
intros t3 t4 H; inversion_clear H; intros H'; inversion_clear H'; constructor;
 eauto.
apply is_bottom_eq' with B1; auto.
apply eq'_sym; auto.
apply is_bottom_eq' with B2; auto.
apply eq'_sym; auto.
Qed.

Lemma order'_is_bottom : forall t1 t2 : tree At, is_bottom t1 -> order' t1 t2.
induction t1 as [| a t0 IHt0 t2 IHt2]; intros.
constructor.
inversion_clear H; case t1; constructor; auto.
apply PosetDec.trans with (⊥♯); auto.
Qed.

Lemma order'_is_bottom2 :
 forall t1 t2 : tree At, is_bottom t2 -> order' t1 t2 -> is_bottom t1.
induction t1 as [| a t0 IHt0 t2 IHt2]; intros.
generalize H0; clear H0; inversion_clear H.
intros H; inversion_clear H; constructor; eauto.
intros H; inversion_clear H; constructor; eauto.
generalize H0; clear H0; inversion_clear H.
intros H; inversion_clear H; constructor; eauto.
intros H; inversion_clear H; constructor; eauto.
Qed.

Lemma order'_antisym :
 forall t1 t2 : tree At, order' t1 t2 -> order' t2 t1 -> eq' t1 t2.
intros t1 t2 H.
induction H
 as [T| a t1 t2 H H0 H1| a b A1 A2 B1 B2 H H1 IHorder'1 H2 IHorder'2];
 intros H4.
inversion H4; constructor; auto.
constructor; auto.
inversion_clear H4; constructor; auto.
Qed.

Lemma order'_trans :
 forall t1 t2 t3 : tree At, order' t1 t2 -> order' t2 t3 -> order' t1 t3.
induction t1 as [| a t0 IHt0 t2 IHt2].
intros t2 t3 H; inversion_clear H; intros H'; inversion_clear H'; constructor;
 eauto.
intros t3 t4 H; inversion_clear H.
intros; destruct t4 as [| a0 T t1]; constructor; try assumption.
apply PosetDec.trans with (⊥♯); auto.
apply order'_is_bottom; auto.
apply order'_is_bottom; auto.
intros H'.
inversion_clear H'.
constructor.
apply PosetDec.antisym; auto.
apply PosetDec.trans with b; auto.
apply order'_is_bottom2 with B1; auto.
apply order'_is_bottom2 with B2; auto.
constructor.
apply PosetDec.trans with b; auto.
apply IHt0 with B1; auto.
apply IHt2 with B2; auto.
Qed.

Lemma is_bottom_apply :
 forall t : tree At,
 is_bottom t ->
 forall p : positive, EquivDec.eq (⊥♯) (apply_tree t p).
intros T H; induction H as [| a t1 t2 H H1 IHis_bottom1 H2 IHis_bottom2].
simple destruct p; auto.
simple destruct p; auto.
Qed.

Lemma apply_is_bottom :
 forall t : tree At,
 (forall p : positive, EquivDec.eq (⊥♯) (apply_tree t p)) ->
 is_bottom t.
induction t; intros; constructor.
generalize (H 1%positive); auto.
apply IHt1; intros.
generalize (H (xO p)); auto.
apply IHt2; intros.
generalize (H (xI p)); auto.
Qed.

Lemma eq'_to_Feq : forall t1 t2 : tree At, eq' t1 t2 -> Feq t1 t2.
induction t1 as [| a t0 IHt0 t2 IHt2]; intros.
inversion_clear H.
unfold Feq in |- *.
simple destruct p; auto.
unfold Feq in |- *.
simple destruct p; simpl in |- *; auto.
apply is_bottom_apply; auto.
apply is_bottom_apply; auto.
unfold Feq in |- *.
inversion_clear H.
simple destruct p; simpl in |- *; auto.
intros; apply EquivDec.sym; apply is_bottom_apply; auto.
intros; apply EquivDec.sym; apply is_bottom_apply; auto.
simple destruct p; simpl in |- *; auto.
apply IHt2; auto.
apply IHt0; auto.
Qed.

Lemma Feq_to_eq' : forall t1 t2 : tree At, Feq t1 t2 -> eq' t1 t2.
induction t1 as [| a t0 IHt0 t2 IHt2]; intros.
apply is_bottom2_eq'.
constructor.
apply apply_is_bottom.
intros; generalize (H p).
destruct p; simpl; auto.
destruct t1.
apply is_bottom2_eq'.
apply apply_is_bottom.
intros; generalize (H p).
destruct p; simpl; auto.
constructor.
constructor.
generalize (H xH); simpl; auto.
apply IHt0; intros p; generalize (H (xO p)); simpl; auto.
apply IHt2; intros p; generalize (H (xI p)); simpl; auto.
Qed.

Lemma Forder_to_order' : forall t1 t2 : tree At, Forder t1 t2 -> order' t1 t2.
unfold Forder in |- *; induction t1 as [| a t0 IHt0 t2 IHt2].
simple destruct t2; intros; constructor.
simple destruct t1; intros.
constructor.
generalize (H 1%positive); auto.
apply apply_is_bottom; intros.
generalize (H (xO p)); auto.
apply apply_is_bottom; intros.
generalize (H (xI p)); auto.
constructor.
generalize (H 1%positive); auto.
apply IHt0; intros.
generalize (H (xO p)); auto.
apply IHt2; intros.
generalize (H (xI p)); auto.
Qed.

Lemma is_bottom_Forder_leaf : forall t, is_bottom t -> Forder t leaf.
intros t0; induction t0; simpl; intros; intros p.
destruct p; simpl; auto.
inversion_clear H.
destruct p; simpl; auto.
generalize (IHt0_2 H2 p); destruct p; auto.
generalize (IHt0_1 H1 p); destruct p; auto.
Qed.

Lemma order'_to_Forder : forall t1 t2 : tree At, order' t1 t2 -> Forder t1 t2.
induction t1 as [| a t0 IHt0 t2 IHt2]; intros.
unfold Forder in |- *.
intros; destruct p; simpl; auto.
inversion_clear H.
apply is_bottom_Forder_leaf; constructor; auto.
intros p; destruct p; simpl; auto.
apply IHt2; auto.
apply IHt0; auto.
Qed.

  Definition t : Type := FuncTree WordSize.

  Definition get : t -> Word -> At := fun f w =>
    apply_functree WordSize f (proj1_sig w).

  Definition eq : t -> t -> Prop := fun f1 f2 =>
    forall a1 a2 : Word, a1 = a2 -> EquivDec.eq (get f1 a1) (get f2 a2).

  Lemma eq_refl : forall x : t, eq x x.
  Proof.
    intros f a1 a2 H; unfold get.
    assert (proj1_sig a1 = proj1_sig a2).
    inversion_clear H; auto.
    rewrite H0; auto.
  Qed.

  Lemma Feq_to_eq : forall x y : t, Feq (tr _ x) (tr _ y) -> eq x y.
  Proof.
    unfold Feq, eq, get, apply_functree; intros.
    assert (proj1_sig a1 = proj1_sig a2).
    inversion_clear H0; auto.
    rewrite H1; apply H.
  Qed.

  Lemma eq_to_Feq : forall x y : t, eq x y -> Feq (tr _ x) (tr _ y).
  Proof.
    unfold Feq, eq, get, apply_functree; intros.
    case (inf_log_dec p WordSize); intros.
    apply (H (exP p (inf_log_inf_log_bool _ _ i)) (exP p (inf_log_inf_log_bool _ _ i))); simpl ;auto.
    rewrite (apply_tree_bot WordSize); auto.
    rewrite (apply_tree_bot WordSize); auto.
    destruct y; auto.
    destruct x; auto.
  Qed.

  Lemma eq_dec : forall x y : t, {eq x y}+{~ eq x y}.
  Proof. 
    intros x y; case (eq'_dec (tr _ x) (tr _ y)); intros; [left | right].
    apply Feq_to_eq; apply eq'_to_Feq; auto.
    red; intro; elim n.
    apply Feq_to_eq'; apply eq_to_Feq; auto.
  Qed.

  Definition order : t -> t -> Prop := fun f1 f2 =>
    forall a1 a2 : Word, a1 = a2 -> PosetDec.order (get f1 a1) (get f2 a2).

  Lemma Forder_to_order : forall x y : t, Forder (tr _ x) (tr _ y) -> order x y.
  Proof.
    unfold Forder, order, get, apply_functree; intros.
    assert (proj1_sig a1 = proj1_sig a2).
    inversion_clear H0; auto.
    rewrite H1; apply H.
  Qed.

  Lemma order_to_Forder : forall x y : t, order x y -> Forder (tr _ x) (tr _ y).
  Proof.
    unfold Forder, order, get, apply_functree; intros.
    case (inf_log_dec p WordSize); intros.
    apply (H (exP p (inf_log_inf_log_bool _ _ i)) (exP p (inf_log_inf_log_bool _ _ i))); simpl ;auto.
    rewrite (apply_tree_bot WordSize); auto.
    destruct y; auto.
    destruct x; auto.
  Qed.

  Lemma order_dec : forall x y : t, {order x y}+{~ order x y}.
  Proof. 
    intros x y; case (Forder_dec (tr _ x) (tr _ y)); intros; [left | right].
    apply Forder_to_order; auto.
    red; intro; elim n.
    apply order_to_Forder; auto.
  Qed.

  Section map.

  Variable f : At -> At.

  Variable f_bottom : EquivDec.eq (⊥♯) (f (⊥♯)).

  Fixpoint map_tree (t : tree At) : tree At :=
    match t with
    | leaf  => leaf
    | node a tO tI =>
        node (f a) (map_tree tO) (map_tree tI)
  end.

  Lemma apply_map_tree :
   forall (t : tree At) (p : positive),
     EquivDec.eq (apply_tree (map_tree t) p)
          (f (apply_tree t p)).
  Proof.
    induction t0; simpl; intros; auto.
     case p; simpl in |- *; auto.
  Qed.

  Lemma map_tree_height_inf :
   forall (t : tree At) (max:nat),
   height_inf t max ->
   height_inf (map_tree t) max.
  Proof.
    induction t0; simpl; intros; auto.
    inversion_clear H.
    constructor; auto.
  Qed.

  End map.

  Definition map : (At -> At) -> t -> t.
    intros f m;
    refine (FuncTree_constr _ (map_tree f (tr WordSize m)) _).
    apply map_tree_height_inf.
    case m; auto.
  Defined.
 
  Lemma get_map : forall (f : At -> At),
    (EquivDec.eq (⊥♯) (f (⊥♯))) ->
    forall (t1 : t) (a : Word),
      EquivDec.eq (get (map f t1) a) (f (get t1 a)).
  Proof.
    unfold t, map, get, apply_functree; intros; simpl.
    apply apply_map_tree; auto.
  Qed.

  Section f.

  Variable f : At -> At -> At.

  Variable f_bottom1 : forall x, EquivDec.eq x (f x (⊥♯)).
  Variable f_bottom2 : forall x, EquivDec.eq x (f (⊥♯) x).

  Fixpoint map2_tree (t1 t2 : tree At) {struct t2} : tree At :=
    match t1, t2 with
    | leaf, _ => t2
    | _, leaf => t1
    | node a1 tO1 tI1, node a2 tO2 tI2 =>
        node (f a1 a2) (map2_tree tO1 tO2) (map2_tree tI1 tI2)
  end.

  Lemma apply_map2_tree :
   forall (t1 t2 : tree At) (p : positive),
     EquivDec.eq (apply_tree (map2_tree t1 t2) p)
          (f (apply_tree t1 p) (apply_tree t2 p)).
  Proof.
    induction t1 as [| a t0 IHt0 t2 IHt2]; intros t'; case t'; simpl in |- *;
     intros. 
     auto.
     case p; simpl in |- *; auto.
     case p; simpl in |- *; auto.
     case p; simpl in |- *; auto.
  Qed.

  Lemma map2_tree_height_inf :
   forall (t1 t2 : tree At) (max:nat),
   height_inf t1 max ->
   height_inf t2 max ->
   height_inf (map2_tree t1 t2) max.
  Proof.
    induction t1 as [|a t11 t12]; destruct t2 as [|b t21 t22]; simpl; intros; auto.
    inversion_clear H in H0.
    inversion_clear H0.
    constructor; auto.
  Qed.

  Definition map2_max : forall (max:nat),
   FuncTree max -> FuncTree max -> FuncTree max.
  Proof.
    intros max f1 f2;
    refine (FuncTree_constr _ (map2_tree (tr _ f1) (tr _ f2)) _).
    apply map2_tree_height_inf.
    case f1; auto.
    case f2; auto.
  Defined.
 
  End f.

  Definition map2 : (At -> At -> At) -> t -> t -> t := fun f =>
    map2_max f WordSize.

  Lemma get_map2 : forall (f : At -> At -> At),
    (forall x : At, EquivDec.eq x (f (⊥♯) x)) ->
    (forall x : At, EquivDec.eq x (f x (⊥♯))) ->
    forall (t1 t2 : t) (a : Word), EquivDec.eq (get (map2 f t1 t2) a) (f (get t1 a) (get t2 a)).
  Proof.
    unfold t, map2, map2_max, get, apply_functree; intros f H1 H2; simpl.
    intros.
    apply apply_map2_tree; auto.
  Qed.

  Section f'.

  Variable f : At -> At -> At.

  Variable f_bottom1 : forall x, EquivDec.eq (⊥♯) (f x (⊥♯)).
  Variable f_bottom2 : forall x, EquivDec.eq (⊥♯) (f (⊥♯) x).

  Fixpoint map2_tree' (t1 t2 : tree At) {struct t2} : tree At :=
    match t1, t2 with
    | leaf, _ => leaf
    | _, leaf => leaf
    | node a1 tO1 tI1, node a2 tO2 tI2 =>
        node (f a1 a2) (map2_tree' tO1 tO2) (map2_tree' tI1 tI2)
  end.

  Lemma apply_map2_tree' :
   forall (t1 t2 : tree At) (p : positive),
     EquivDec.eq (apply_tree (map2_tree' t1 t2) p)
          (f (apply_tree t1 p) (apply_tree t2 p)).
  Proof.
    induction t1 as [| a t0 IHt0 t2 IHt2]; intros t'; case t'; simpl in |- *;
     intros. 
     auto.
     case p; simpl in |- *; auto.
     case p; simpl in |- *; auto.
     case p; simpl in |- *; auto.
  Qed.

  Lemma map2_tree'_height_inf :
   forall (t1 t2 : tree At) (max:nat),
   height_inf t1 max ->
   height_inf t2 max ->
   height_inf (map2_tree' t1 t2) max.
  Proof.
    induction t1 as [|a t11 t12]; destruct t2 as [|b t21 t22]; simpl; intros; auto.
    inversion_clear H in H0.
    inversion_clear H0.
    constructor; auto.
  Qed.

  Definition map2'_max : forall (max:nat),
   FuncTree max -> FuncTree max -> FuncTree max.
  Proof.
    intros max f1 f2;
    refine (FuncTree_constr _ (map2_tree' (tr _ f1) (tr _ f2)) _).
    apply map2_tree'_height_inf.
    case f1; auto.
    case f2; auto.
  Defined.
 
  End f'.

  Definition map2' : (At -> At -> At) -> t -> t -> t := fun f =>
    map2'_max f WordSize.

  Lemma get_map2' : forall (f : At -> At -> At),
    (forall x : At, EquivDec.eq (⊥♯) (f (⊥♯) x)) ->
    (forall x : At, EquivDec.eq (⊥♯) (f x (⊥♯))) ->
    forall (t1 t2 : t) (a : Word), EquivDec.eq (get (map2' f t1 t2) a) (f (get t1 a) (get t2 a)).
  Proof.
    unfold t, map2', map2'_max, get, apply_functree; intros f H1 H2; simpl.
    intros.
    apply apply_map2_tree'; auto.
  Qed.

  Section f''.

  Variable f : At -> At -> At.

  Variable f_bottom : EquivDec.eq (⊥♯) (f (⊥♯) (⊥♯)).

  Fixpoint map2_tree'' (t1 t2 : tree At) {struct t2} : tree At :=
    match t1, t2 with
    | leaf, leaf => leaf
    | leaf, _ => map_tree (f (⊥♯)) t2
    | _, leaf => map_tree (fun a => f a (⊥♯)) t1
    | node a1 tO1 tI1, node a2 tO2 tI2 =>
        node (f a1 a2) (map2_tree'' tO1 tO2) (map2_tree'' tI1 tI2)
  end.

  Lemma apply_map2_tree'' :
   forall (t1 t2 : tree At) (p : positive),
     EquivDec.eq (apply_tree (map2_tree'' t1 t2) p)
          (f (apply_tree t1 p) (apply_tree t2 p)).
  Proof.
    induction t1 as [| a t0 IHt0 t2 IHt2]; intros t'; case t'; simpl in |- *;
     intros. 
     auto.
     case p; simpl in |- *; auto; intros; apply apply_map_tree; auto.
     case p; simpl in |- *; auto; intros.
       apply apply_map_tree with (f:=(fun a0 : At => f a0 (⊥♯))); auto. 
       apply apply_map_tree with (f:=(fun a0 : At => f a0 (⊥♯))); auto. 
     case p; simpl in |- *; auto.
  Qed.

  Lemma map2_tree''_height_inf :
   forall (t1 t2 : tree At) (max:nat),
   height_inf t1 max ->
   height_inf t2 max ->
   height_inf (map2_tree'' t1 t2) max.
  Proof.
    induction t1 as [|a t11 t12]; destruct t2 as [|b t21 t22]; simpl; intros; auto.
    inversion_clear H0; constructor.
    apply map_tree_height_inf; auto.
    apply map_tree_height_inf; auto.
    inversion_clear H; constructor.
    apply map_tree_height_inf; auto.
    apply map_tree_height_inf; auto.
    inversion_clear H in H0.
    inversion_clear H0.
    constructor; auto.
  Qed.

  Definition map2''_max : forall (max:nat),
   FuncTree max -> FuncTree max -> FuncTree max.
  Proof.
    intros max f1 f2;
    refine (FuncTree_constr _ (map2_tree'' (tr _ f1) (tr _ f2)) _).
    apply map2_tree''_height_inf.
    case f1; auto.
    case f2; auto.
  Defined.
 
  End f''.

  Definition map2'' : (At -> At -> At) -> t -> t -> t := fun f =>
    map2''_max f WordSize.

  Lemma get_map2'' : forall (f : At -> At -> At),
    (EquivDec.eq (⊥♯) (f (⊥♯) (⊥♯))) ->
    forall (t1 t2 : t) (a : Word), EquivDec.eq (get (map2'' f t1 t2) a) (f (get t1 a) (get t2 a)).
  Proof.
    unfold t, map2'', map2''_max, get, apply_functree; intros f H1 H2; simpl.
    intros.
    apply apply_map2_tree''; auto.
  Qed.

Lemma height_inf_modify : 
 forall (t:tree At) (p:positive) (n:nat) (v:At->At),
 (height_inf t (S n)) -> (inf_log p n) ->
 (height_inf (modify_tree _ (⊥♯) t p v) (S n)).
induction t0; simpl; intros.
generalize p H0; clear H0 H p.
induction n; simpl; intros.
inversion_clear H0; simpl.
repeat constructor.
destruct p; simpl; constructor ;auto.
constructor.
apply IHn.
inversion_clear H0; assumption.
apply IHn.
inversion_clear H0; assumption.
constructor.
constructor.
constructor.
inversion_clear H.
destruct p; constructor; auto.
inversion_clear H0 in H1 H2.
apply IHt0_2; auto.
inversion_clear H0 in H1 H2.
apply IHt0_1; auto.
Qed.


  Definition modify : t -> Word -> (At -> At) -> t.
  Proof.
    intros F x f.
    refine (FuncTree_constr _ (modify_tree _ (⊥♯) (tr _ F) (proj1_sig x) f) _).
    apply height_inf_modify.
    destruct F; auto.
    destruct x; auto.
  Defined.

Lemma get_modify1 : forall F x p f,
   x = p ->
  get (modify F p f) x = f (get F x).
Proof.
  destruct F; unfold modify, get, apply_functree in |- *; simpl; intros.
  destruct x; destruct p; simpl in *; subst.
  inversion_clear H.
  unfold apply_tree; rewrite apply_modify_tree1; auto.
Qed.

Lemma get_modify2 : forall F x p f,
  ~ x = p ->
  get (modify F p f) x = (get F x).
Proof.
  destruct F; unfold modify, get, apply_functree in |- *; simpl; intros.
  destruct x; destruct p; simpl in *; subst.
  unfold apply_tree; rewrite apply_modify_tree2; auto.
  intro; subst; elim H.
  replace e with e0; auto.
  apply bool_proof_irrelevance.
Qed.

  Definition bottom : t := (Fbottom WordSize).

  Lemma bottom_eq_bottom : forall a, EquivDec.eq (get bottom a) (⊥♯). 
  Proof.
    unfold bottom, Fbottom, get, apply_functree; simpl; auto.
  Qed.

  Section mapi.

  Section mapi_positive.

  Variable f : positive -> At -> At.

  Variable f_bottom : forall p, EquivDec.eq (⊥♯) (f p (⊥♯)).

  Fixpoint mapi_tree_rec (t1 : tree At) (p:positive) {struct t1} : tree At :=
    match t1 with
    |  leaf => leaf
    | node a tO tI =>
        node (f (inv_pos p xH) a)
                (mapi_tree_rec tO (xO p))
                (mapi_tree_rec tI (xI p))
  end.

  Lemma apply_mapi_tree_rec :
   forall (t1 : tree At) (p q : positive),
     EquivDec.eq (apply_tree (mapi_tree_rec t1 q) p)
          (f (inv_pos q p) (apply_tree t1 p)).
  Proof.
    induction t1 as [| a t0 IHt0 t2 IHt2]; simpl in |- *;
     intros. 
     auto.
     case p; simpl in |- *; auto.
     intros p0; generalize (IHt2 p0 (xI q)); simpl; auto.
     intros p0; generalize (IHt0 p0 (xO q)); simpl; auto.
  Qed.

  Lemma mapi_tree_rec_height_inf :
   forall (t1 : tree At) (max:nat) q,
   height_inf t1 max ->
   height_inf (mapi_tree_rec t1 q) max.
  Proof.
    induction t1 as [|a t11 t12]; simpl; intros; auto.
    inversion_clear H.
    constructor; auto.
  Qed.

  Definition mapi_tree (t1 : tree At) : tree At :=
    mapi_tree_rec t1 xH.

  Lemma apply_mapi_tree :
   forall (t1 : tree At) (p : positive),
     EquivDec.eq (apply_tree (mapi_tree t1) p)
          (f p (apply_tree t1 p)).
  Proof.
    unfold mapi_tree; intros.
    assert (p=inv_pos 1 p).
    reflexivity.
    pattern p at 2; rewrite H.
    apply apply_mapi_tree_rec; auto.
  Qed.

  Lemma mapi_tree_height_inf :
   forall (t1 : tree At) (max:nat),
   height_inf t1 max ->
   height_inf (mapi_tree t1) max.
  Proof.
    unfold mapi_tree; intros; apply mapi_tree_rec_height_inf; auto.
  Qed.

  End mapi_positive.

  
  Variable f : Word -> At -> At.

  Variable f_eq1 : forall p1 p2, p1 = p2 -> f p1 = f p2.
  Variable f_bottom : forall p, EquivDec.eq (⊥♯) (f p (⊥♯)).

  Definition mapi :  t -> t.
  Proof.
    intros f1.
    set (fp:=fun p => 
           match inf_log_dec p WordSize with
             left h => f (exP p (inf_log_inf_log_bool _ _ h))
           | right _ => fun x => x
           end).
    refine (FuncTree_constr _ (mapi_tree fp (tr _ f1)) _).
    apply mapi_tree_height_inf.
    case f1; auto.
  Defined.

  Lemma get_mapi :
   forall (t1 : t) (p : Word),
     EquivDec.eq (get (mapi t1) p) (f p (get t1 p)).
  Proof.
    unfold mapi, get, apply_functree; intros; simpl.
    set (fp:=fun p => 
       match inf_log_dec p WordSize with
         left h => f (exP p (inf_log_inf_log_bool _ _ h))
       | right _ => fun x => x
       end).
    assert (forall (p : positive), EquivDec.eq (⊥♯) (fp p (⊥♯))).
    intros; unfold fp; case inf_log_dec; intros; auto.
    apply EquivDec.trans with (1:=(apply_mapi_tree fp H (tr WordSize t1) (proj1_sig p))).
    replace (fp (proj1_sig p)) with (f p); auto.
    unfold fp; case inf_log_dec; intros.
    apply f_eq1; destruct p; simpl. 
    replace e with (inf_log_inf_log_bool _ _ i); auto.
    apply bool_proof_irrelevance.
    elim n; destruct p; simpl; auto.
  Qed.

  End mapi.

  Section f2.

  Section f2_rec.

  Variable f : positive -> At -> At -> At.

  Variable f_bottom1 : forall x p, EquivDec.eq x (f p x (⊥♯)).
  Variable f_bottom2 : forall x p, EquivDec.eq x (f p (⊥♯) x).

  Fixpoint map2i_tree_rec (t1 t2 : tree At) (p:positive) {struct t2} : tree At :=
    match t1, t2 with
    | leaf, _ => t2
    | _, leaf => t1
    | node a1 tO1 tI1, node a2 tO2 tI2 =>
        node (f (inv_pos p xH) a1 a2)
                (map2i_tree_rec tO1 tO2 (xO p))
                (map2i_tree_rec tI1 tI2 (xI p))
  end.

  Lemma apply_map2i_tree_rec :
   forall (t1 t2 : tree At) (p q : positive),
     EquivDec.eq (apply_tree (map2i_tree_rec t1 t2 q) p)
          (f (inv_pos q p) (apply_tree t1 p) (apply_tree t2 p)).
  Proof.
    induction t1 as [| a t0 IHt0 t2 IHt2]; intros t'; case t'; simpl in |- *;
     intros. 
     auto.
     case p; simpl in |- *; auto.
     case p; simpl in |- *; auto.
     case p; simpl in |- *; auto.
     intros p0; generalize (IHt2 t3 p0 (xI q)); simpl; auto.
     intros p0; generalize (IHt0 t1 p0 (xO q)); simpl; auto.
  Qed.

  Lemma map2i_tree_rec_height_inf :
   forall (t1 t2 : tree At) (max:nat) q,
   height_inf t1 max ->
   height_inf t2 max ->
   height_inf (map2i_tree_rec t1 t2 q) max.
  Proof.
    induction t1 as [|a t11 t12]; destruct t2 as [|b t21 t22]; simpl; intros; auto.
    inversion_clear H in H0.
    inversion_clear H0.
    constructor; auto.
  Qed.

  Definition map2i_tree (t1 t2 : tree At) : tree At :=
    map2i_tree_rec t1 t2 xH.

  Lemma apply_map2i_tree :
   forall (t1 t2 : tree At) (p : positive),
     EquivDec.eq (apply_tree (map2i_tree t1 t2) p)
          (f p (apply_tree t1 p) (apply_tree t2 p)).
  Proof.
    unfold map2i_tree; intros.
    assert (p=inv_pos 1 p).
    reflexivity.
    pattern p at 2; rewrite H.
    apply apply_map2i_tree_rec; auto.
  Qed.

  Lemma map2i_tree_height_inf :
   forall (t1 t2 : tree At) (max:nat),
   height_inf t1 max ->
   height_inf t2 max ->
   height_inf (map2i_tree t1 t2) max.
  Proof.
    unfold map2i_tree; intros; apply map2i_tree_rec_height_inf; auto.
  Qed.

  End f2_rec.


  Variable f : Word -> At -> At -> At.

  Variable f_eq1 : forall p1 p2, p1 = p2 -> f p1 = f p2.
  Variable f_bottom2 : forall x p, EquivDec.eq x (f p (⊥♯) x).
  Variable f_bottom1 : forall x p, EquivDec.eq x (f p x (⊥♯)).

  Definition map2i : t -> t -> t.
  Proof.
    intros f1 f2.
    set (fp:=fun p => 
           match inf_log_dec p WordSize with
             left h => f (exP p (inf_log_inf_log_bool _ _ h))
           | right _ => JoinDec.op
           end).
    refine (FuncTree_constr _ (map2i_tree fp (tr _ f1) (tr _ f2)) _).
    apply map2i_tree_height_inf.
    case f1; auto.
    case f2; auto.
  Defined.

  Lemma get_map2i :
   forall (t1 t2 : t) (p : Word),
     EquivDec.eq (get (map2i t1 t2) p)
          (f p (get t1 p) (get t2 p)).
  Proof.
    unfold map2i, get, apply_functree; intros; simpl.
    set (fp:=fun p => 
       match inf_log_dec p WordSize with
         left h => f (exP p (inf_log_inf_log_bool _ _ h))
       | right _ => JoinDec.op
       end).
    assert (forall (x : At) (p : positive), EquivDec.eq x (fp p x (⊥♯))).
    intros; unfold fp; case inf_log_dec; intros; auto.
    assert (forall (x : At) (p : positive), EquivDec.eq x (fp p (⊥♯) x)).
    intros; unfold fp; case inf_log_dec; intros; auto.
    apply EquivDec.trans with (1:=(apply_map2i_tree fp H H0 (tr WordSize t1) (tr WordSize t2) (proj1_sig p))).
    replace (fp (proj1_sig p)) with (f p); auto.
    unfold fp; case inf_log_dec; intros.
    apply f_eq1; destruct p; simpl. 
    replace e with (inf_log_inf_log_bool _ _ i); auto.
    apply bool_proof_irrelevance.
    elim n; destruct p; simpl; auto.
  Qed.
 
  End f2.


  Section map2i'.

  Section map2i_rec.

  Variable f : positive -> At -> At -> At.

  Variable f_bottom : forall p, EquivDec.eq (⊥♯) (f p (⊥♯) (⊥♯)).

  Fixpoint map2i'_tree_rec (t1 t2 : tree At) (p:positive) {struct t2} : tree At :=
    match t1, t2 with
    | leaf, _ => mapi_tree_rec (fun p a => f p (⊥♯) a) t2 p
    | _, leaf => mapi_tree_rec (fun p a => f p a (⊥♯)) t1 p
    | node a1 tO1 tI1, node a2 tO2 tI2 =>
        node (f (inv_pos p xH) a1 a2)
                (map2i'_tree_rec tO1 tO2 (xO p))
                (map2i'_tree_rec tI1 tI2 (xI p))
  end.

  Lemma apply_map2i'_tree_rec :
   forall (t1 t2 : tree At) (p q : positive),
     EquivDec.eq (apply_tree (map2i'_tree_rec t1 t2 q) p)
          (f (inv_pos q p) (apply_tree t1 p) (apply_tree t2 p)).
  Proof.
    induction t1 as [| a t0 IHt0 t2 IHt2]; intros t'; case t'; 
    simpl; intros. 
    auto.
    case p; simpl in |- *; auto; intros.
    apply (apply_mapi_tree_rec (fun (p1 : positive) (a : At) => f p1 (⊥♯) a)
                  f_bottom t1  p0 (xI q)).
    apply (apply_mapi_tree_rec (fun (p1 : positive) (a : At) => f p1 (⊥♯) a)
                  f_bottom t0  p0 (xO q)).
    case p; simpl in |- *; auto; intros.
    apply (apply_mapi_tree_rec (fun (p1 : positive) (a : At) => f p1 a (⊥♯))
                  f_bottom t2  p0 (xI q)).
    apply (apply_mapi_tree_rec (fun (p1 : positive) (a : At) => f p1 a (⊥♯))
                  f_bottom t0  p0 (xO q)).
    case p; simpl in |- *; auto.
    intros p0; generalize (IHt2 t3 p0 (xI q)); simpl; auto.
    intros p0; generalize (IHt0 t1 p0 (xO q)); simpl; auto.
  Qed.

  Lemma map2i'_tree_rec_height_inf :
   forall (t1 t2 : tree At) (max:nat) q,
   height_inf t1 max ->
   height_inf t2 max ->
   height_inf (map2i'_tree_rec t1 t2 q) max.
  Proof.
    induction t1 as [|a t11 t12]; simpl.
    destruct t2 as [|b t21 t22]; simpl; intros; auto.
    generalize (mapi_tree_rec_height_inf (fun (p : positive) (a : At) => f p (⊥♯) a) (node b t21 t22) _ q H0).
    simpl.
    intro W; apply W.
    destruct t2 as [|b t21 t22]; simpl; intros; auto.
    generalize (mapi_tree_rec_height_inf (fun (p : positive) (a : At) => f p a (⊥♯)) (node a t11 t1_1) _ q H);
      simpl; intros W; apply W.
    inversion_clear H in H0.
    inversion_clear H0.
    constructor; auto.
  Qed.

  Definition map2i'_tree (t1 t2 : tree At) : tree At :=
    map2i'_tree_rec t1 t2 xH.

  Lemma apply_map2i'_tree :
   forall (t1 t2 : tree At) (p : positive),
     EquivDec.eq (apply_tree (map2i'_tree t1 t2) p)
          (f p (apply_tree t1 p) (apply_tree t2 p)).
  Proof.
    unfold map2i'_tree; intros.
    assert (p=inv_pos 1 p).
    reflexivity.
    pattern p at 2; rewrite H.
    apply apply_map2i'_tree_rec; auto.
  Qed.

  Lemma map2i'_tree_height_inf :
   forall (t1 t2 : tree At) (max:nat),
   height_inf t1 max ->
   height_inf t2 max ->
   height_inf (map2i'_tree t1 t2) max.
  Proof.
    unfold map2i'_tree; intros; apply map2i'_tree_rec_height_inf; auto.
  Qed.

  End map2i_rec.

  Variable f : Word -> At -> At -> At.

  Variable f_eq1 : forall p1 p2, p1 = p2 -> f p1 = f p2.
  Variable f_bottom : forall p, EquivDec.eq (⊥♯) (f p (⊥♯) (⊥♯)).

  Definition map2i' : t -> t -> t.
  Proof.
    intros f1 f2.
    set (fp:=fun p => 
           match inf_log_dec p WordSize with
             left h => f (exP p (inf_log_inf_log_bool _ _ h))
           | right _ => JoinDec.op
           end).
    refine (FuncTree_constr _ (map2i'_tree fp (tr _ f1) (tr _ f2)) _).
    apply map2i'_tree_height_inf.
    case f1; auto.
    case f2; auto.
  Defined.

  Lemma get_map2i' :
   forall (t1 t2 : t) (p : Word),
     EquivDec.eq (get (map2i' t1 t2) p)
          (f p (get t1 p) (get t2 p)).
  Proof.
    unfold map2i', get, apply_functree; intros; simpl.
    set (fp:=fun p => 
       match inf_log_dec p WordSize with
         left h => f (exP p (inf_log_inf_log_bool _ _ h))
       | right _ => JoinDec.op
       end).
    assert (forall (p : positive), EquivDec.eq (⊥♯) (fp p (⊥♯) (⊥♯))).
    intros; unfold fp; case inf_log_dec; intros; auto.
    apply EquivDec.trans with (1:=(apply_map2i'_tree fp H (tr WordSize t1) (tr WordSize t2) (proj1_sig p))).
    replace (fp (proj1_sig p)) with (f p); auto.
    unfold fp; case inf_log_dec; intros.
    apply f_eq1; destruct p; simpl. 
    replace e with (inf_log_inf_log_bool _ _ i); auto.
    apply bool_proof_irrelevance.
    elim n; destruct p; simpl; auto.
  Qed.
 
  End map2i'.

End ArrayBinFuncFiniteSetLattice.
