Require Export ZArith.
Require Export lattice_def.
Require Export Word.

Open Scope Z_scope.

Module FiniteSet.
Class t (t : Type) : Type := Make {
  eq :> EquivDec.t t;
  cardinal : Z;
  cardinal_positive : cardinal > 0;
  inject : t -> Z;
  nat2t : Z -> t;
  inject_bounded : forall x : t, 0 <= (inject x) < cardinal;
    inject_nat2t : forall n : Z, 
     0 <= n < cardinal -> inject (nat2t n) = n;
     inject_injective : forall x y : t, inject x = inject y -> x =♯ y;
       inject_comp_eq : forall x y : t, x =♯ y -> inject x = inject y
}.
Hint Resolve @cardinal_positive.
End FiniteSet.

Instance WordEquivDec : EquivDec.t Word.
refine (EquivDec.Make _ (@eq Word) _ _ _ _ ).
apply @refl_equal.
apply @sym_eq.
apply @trans_eq.
apply eq_word_dec.
Defined.

Let t := Word.

  Definition inject : t -> Z := fun w =>
    let (p,_) := w in (Zpred (Zpos p)).

  Definition nat2t : Z -> t := fun z =>
    match z with
       Z0 => Word_1
     | Zpos p => match (inf_log_dec (Psucc p) WordSize) with
                    left H => exP (Psucc p) (inf_log_inf_log_bool _ _ H)
                  | right _ => Word_1
                 end
     | Zneg p => Word_1
    end.

  Fixpoint height_pos (p:positive) : nat :=
    match p with
      xH => O
    | xI p => S (height_pos p)
    | xO p => S (height_pos p)
    end.

  Fixpoint pow2P (n:nat) : positive :=
    match n with
      O => xH
    | S p => xI (pow2P p)
    end.

  Lemma Zpos_xO : forall p, Zpos (xO p) = 2 * (Zpos p).
  Proof.
    auto with zarith.
  Qed.

  Lemma Zpos_xI : forall p, Zpos (xI p) = 2 * (Zpos p) + 1.
  Proof.
    auto with zarith.
  Qed.

  Lemma pow2P_bigger_height : forall p,
    (Zpos p) <= (Zpos (pow2P (height_pos p))).
  Proof.
    induction p; simpl; auto.
    rewrite Zpos_xI; rewrite Zpos_xO.
    auto with zarith.
    auto with zarith.
  Qed.

  Lemma height_word : forall w : Word, (height_pos w <= WordSize)%nat.
  Proof.
    destruct w; simpl.
    generalize dependent WordSize.
    intros n H.
    generalize (inf_log_bool_inf_log _ _ H).
    generalize n; clear n H.
    induction x; simpl; intros.
    inversion_clear H; auto with arith.
    inversion_clear H; auto with arith.
    auto with arith.
  Qed.

  Lemma pow2P_monotone : forall n1 n2, 
    (n1 <= n2)%nat ->(Zpos (pow2P n1)) <= (Zpos (pow2P n2)).
  Proof.
    induction 1; simpl.
    auto with zarith.
    rewrite Zpos_xI.
    generalize (Zle_0_pos (pow2P m)); intros.
    omega.
  Qed.

  Lemma word_bounded : forall w : Word,
    (Zpos w) <= (Zpos (pow2P WordSize)).
  Proof. 
    intros w; apply Zle_trans with (Zpos (pow2P (height_pos w))).
    apply pow2P_bigger_height.
    apply pow2P_monotone.
    apply height_word.
  Qed.

  Definition cardinal : Z := (Zpos (pow2P WordSize)).

  Lemma cardinal_positive : cardinal > 0.
  Proof.
    unfold cardinal; generalize (Zgt_pos_0 (pow2P WordSize)).
    omega.
  Qed.

  Lemma inject_bounded : forall x : t, 0 <= (inject x) < cardinal.
  Proof.
    destruct x; unfold inject, cardinal in *.
    split.
    destruct x; simpl; auto with zarith.
    cut (Zpos x <= Zpos (pow2P WordSize)); intros.
    unfold Zpred.
    omega.
    generalize (word_bounded (exP x e)); auto.
  Qed.


  Lemma le_pow2P_inf_log : forall p n,
    Zpos p <= Zpos (pow2P n) -> inf_log p n.
  Proof.
    induction p; intros.
    rewrite Zpos_xI in H.
    destruct n.
    simpl in H.
    rewrite Zpos_xI in H.
    generalize (Zgt_pos_0 p); intros.
    absurd (Zpos p <= 0); omega.
    simpl in H.
    do 2 rewrite Zpos_xI in H.
    constructor; apply IHp; omega.
    destruct n.
    simpl in H; rewrite Zpos_xO in H.
    generalize (Zgt_pos_0 p); intros.
    absurd (Zpos p <= 0); omega.
    simpl in H.
    rewrite Zpos_xI in H; rewrite Zpos_xO in H.
    constructor; apply IHp; omega.
    constructor.
  Qed.

  Lemma inject_nat2t : forall n : Z, 
     0 <= n < cardinal -> inject (nat2t n) = n.      
  Proof.
    intros; unfold inject, nat2t.
    destruct n.
    unfold Word_1; auto.
    case inf_log_dec; unfold exP; intros.
    rewrite Zpos_succ_morphism.
    rewrite Zpred_succ; auto.
    elim n; clear n.
    apply le_pow2P_inf_log.
    rewrite Zpos_succ_morphism. 
    unfold cardinal in H.
    unfold Zsucc; omega.
    generalize (Zlt_neg_0 p); intros.
    absurd (Zneg p >= 0); omega.
  Qed.

  Lemma inject_injective : forall x y : t, inject x = inject y -> x =♯ y.
  Proof.
    intros (x,Hx) (y,hy); unfold inject; intros; simpl.
    assert (Zpos x = Zpos y).
    rewrite (Zsucc_pred (Zpos x)).
    rewrite (Zsucc_pred (Zpos y)).
    rewrite H; auto.
    inversion H0; subst.
    replace hy with Hx; auto.
    apply bool_proof_irrelevance.
  Qed.

  Lemma inject_comp_eq : forall x y : t, x =♯ y -> inject x = inject y.
  Proof.
    intros (x,Hx) (y,hy); unfold inject; intros.
    inversion H.
    subst; auto.
  Qed.

Instance WordFiniteSet : FiniteSet.t Word.
refine (FiniteSet.Make _ _ 
  cardinal
  cardinal_positive
  inject
  nat2t
  inject_bounded
  inject_nat2t
  inject_injective
  inject_comp_eq).
Defined.


(*
Module Func_FiniteSet_Lattice.
  
  Class t0 (A:Set) (B:Type) (t:Type) : Type := Make0 {
    FS :> FiniteSet.t A;
    L :> AbLattice.t B;
   get : t -> A -> B
  }.

  Section t0.
    Variable A : Set.
    Variable B t : Type.
    Variable T : t0 A B t.

  Definition eq : t -> t -> Prop := fun f1 f2 : t =>
    forall a1 a2 : A, a1 =♯ a2 -> @EquivDec.eq B _ (get f1 a1) (get f2 a2).

  Definition order : t -> t -> Prop := fun f1 f2 : t =>
    forall a1 a2 : A, a1 =♯ a2 -> (get f1 a1) ⊑♯ (get f2 a2).

  End t0.

  Class t (A:Set) (B:Type) (t:Type) : Type := Make {
    T0 :> t0 A B t; 
    eq_refl : forall x : t, eq x x;
    eq_dec : forall x y : t, {eq x y}+{~ eq x y}}.
  order_dec : forall x y : t, {order x y}+{~ order x y}.
 map2 : (B -> B -> B) -> t -> t -> t.
 get_map2 : forall (f : B -> B -> B),
    (forall x : B, x =♯ (f (⊥♯) x)) ->
    (forall x : B, x =♯ (f x (⊥♯))) ->
    forall (t1 t2 : t) (a : A), (get (map2 f t1 t2) a) =♯ (f (get t1 a) (get t2 a)).

  Parameter map2' : (B -> B -> B) -> t -> t -> t.
  Parameter get_map2' : forall (f : B -> B -> B),
    (forall x : B, (⊥♯) =♯ (f (⊥♯) x)) ->
    (forall x : B, (⊥♯) =♯ (f x (⊥♯))) ->
    forall (t1 t2 : t) (a : A), (get (map2' f t1 t2) a) =♯ (f (get t1 a) (get t2 a)).

  Parameter map2'' : (B -> B -> B) -> t -> t -> t.
  Parameter get_map2'' : forall (f : B -> B -> B),
    (⊥♯) =♯ (f (⊥♯) (⊥♯)) ->
    forall (t1 t2 : t) (a : A), (get (map2'' f t1 t2) a) =♯ (f (get t1 a) (get t2 a)).

  Parameter map2i : (A -> B -> B -> B) -> t -> t -> t.
  Parameter get_map2i : forall (f : A -> B -> B -> B),
    (forall a1 a2, a1 =♯ a2 -> f a1 = f a2) ->
    (forall x a, x =♯ (f a (⊥♯) x)) ->
    (forall x a, x =♯ (f a x (⊥♯))) ->
    forall (t1 t2 : t) (a : A),
       (get (map2i f t1 t2) a) =♯ (f a (get t1 a) (get t2 a)).

  Parameter map2i' : (A -> B -> B -> B) -> t -> t -> t.
  Parameter get_map2i' : forall (f : A -> B -> B -> B),
    (forall a1 a2, a1 =♯ a2 -> f a1 = f a2) ->
    (forall a, (⊥♯) =♯ (f a (⊥♯) (⊥♯))) ->
    forall (t1 t2 : t) (a : A),
       (get (map2i' f t1 t2) a) =♯ (f a (get t1 a) (get t2 a)).

  Parameter modify : t -> A -> (B -> B) -> t.
  Parameter get_modify1 : forall f a2 a1 fv,
    a2 =♯ a1 ->
    get (modify f a1 fv) a2 = fv (get f a2).
  Parameter get_modify2 : forall f a2 a1 fv,
    ~ a2 =♯ a1 ->
    get (modify f a1 fv) a2 = get f a2.

  Parameter bottom : t.
  Parameter bottom_eq_bottom : forall a, @EquivDec.eq B _ (get bottom a) (⊥♯). 

End Func_FiniteSet_Lattice.
*)
