Require Export lattice_def.
Require Export CompleteLattice.
Require Import ZArith.
Require Export z_extra_props.


Inductive Z' : Set :=
 | z : Z -> Z'
 | infty_pos : Z'
 | infty_neg : Z'.
Inductive order_Z' : Z' -> Z' -> Prop :=
 | order_z_z : forall a b : Z, (Zlt a b) -> (order_Z' (z a) (z b))
 | order_infty: (order_Z' infty_neg infty_pos)
 | order_infty_pos : forall x: Z, (order_Z' (z x) infty_pos)
 | order_infty_neg : forall x: Z, (order_Z' infty_neg (z x)).

Notation "a <' b" := (order_Z' a b) (at level 50).
Notation "a <=' b" := (a <' b \/ a=b) (at level 50).


Lemma infty_neg_min : forall a : Z', infty_neg <=' a.
destruct a; intros; auto || left; constructor; auto.
Qed.

Lemma infty_pos_max : forall a : Z', a <=' infty_pos.
destruct a; intros; auto || left; constructor; auto.
Qed.

Lemma order_Z'_antisym : forall a b: Z', a<'b -> ~b<'a.
intros a b H; inversion_clear H; red; intros H; inversion_clear H.
apply (Zlt_asym _ _ H0 H1).
Qed.

Lemma order_Z'_strict : forall a b: Z', a<'b -> ~a=b.
intros a b H; inversion_clear H; red; intros H.
injection H; intros; subst; omega.
discriminate H.
discriminate H.
discriminate H.
Qed.

Lemma order_Z'_trans : forall a b c: Z', a<'b -> b<'c -> a<'c.
intros a b c H; inversion_clear H; intros H'; inversion_clear H'; constructor.
auto with zarith.
Qed.

Lemma order_Z'_trans1 : forall a b c: Z', a<'b -> b<='c -> a<='c.
intros a b c H; inversion_clear H; 
 (intros [H'|H']; [inversion_clear H';left;constructor|rewrite <- H'; left; constructor; auto]).
auto with zarith.
Qed. 

Lemma order_Z'_trans2 : forall a b c: Z', 
 a<'b -> b<='c -> a<'c.
intros a b c H H1; destruct H1.
apply order_Z'_trans with b; auto.
subst; auto.
Qed. 

Lemma order_Z'_trans3 : forall a b c: Z', 
 b<='c -> c<'a -> b<'a.
intros a b c H H1; destruct H.
apply order_Z'_trans with c; auto.
subst; auto.
Qed. 

Lemma order_Z'_disj : forall b c: Z', 
 b<='c -> c<'b -> False.
intros.
assert (b <' b).
apply order_Z'_trans3 with c; auto.
apply (order_Z'_strict _ _ H1); auto.
Qed. 

Lemma not_order_Z' : forall a b: Z', ~ a<'b -> b<='a.
destruct a; intros.
destruct b; intros.
case (Z_dec' z1 z0); intros.
case s; intros.
left; constructor; auto.
elim H; constructor; auto.
rewrite e; auto.
elim H; constructor; auto.
left; constructor; auto.
destruct b; intros; auto.
left; constructor; auto.
left; constructor; auto.
destruct b; intros; auto.
elim H; constructor; auto.
elim H; constructor; auto.
Qed.

Lemma order_Z'_dec : forall a b : Z', {a<'b}+{~a<'b}.
destruct a; intros.
destruct b; intros.
case (Z_lt_dec z0 z1); intros.
left; constructor; auto.
right; intros H; inversion H; elim n; auto.
left; constructor; auto.
right; intros H; inversion H.
right; intros H; inversion H.
destruct b; intros.
left; constructor; auto.
left; constructor; auto.
right; intros H; inversion H.
Qed.

Inductive interval : Set := 
 | ITV : forall a b : Z', a <=' b -> interval.

Definition min : interval -> Z' :=
 fun i => match i with
  | ITV a _ _ => a
 end.

Definition max : interval -> Z' :=
 fun i => match i with
  | ITV _ a _ => a
 end.

Lemma min_order_max : forall i:interval, (min i)<='(max i).
destruct i; simpl; auto.
Qed.

Definition eq_interval : interval -> interval -> Prop :=
 fun i1 i2 => (min i1)=(min i2) /\ (max i1)=(max i2).

Lemma eq_interval_refl : forall i : interval, eq_interval i i.
unfold eq_interval; auto.
Qed.

Lemma eq_interval_sym : forall i1 i2 : interval, 
 eq_interval i1 i2 -> eq_interval i2 i1.
unfold eq_interval; intuition.
Qed.

Lemma eq_interval_trans : forall i1 i2 i3 : interval, 
 eq_interval i1 i2 -> eq_interval i2 i3 -> eq_interval i1 i3.
unfold eq_interval; intuition.
rewrite <- H; auto.
rewrite <- H3; auto.
Qed.

Definition order_interval : interval -> interval -> Prop :=
 fun i1 i2 => 
   ((min i2)<='(min i1)) /\ ((max i1)<='(max i2)).

Lemma order_interval_refl : forall i1 i2 : interval, 
  eq_interval i1 i2 -> order_interval i1 i2.
unfold eq_interval,order_interval; intuition.
Qed.

Lemma order_interval_antisym : forall i1 i2 : interval, 
 order_interval i1 i2 -> order_interval i2 i1 -> eq_interval i1 i2.
generalize order_Z'_antisym; intros Hantisym.
generalize order_Z'_strict; intros Hstrict.
unfold eq_interval,order_interval; intuition; solve [apply False_ind; eauto].
Qed.

Ltac Rew := match goal with
 | id : _ = _ |- _ => try rewrite id; clear id
end.

Lemma order_interval_trans : forall i1 i2 i3 : interval, 
 order_interval i1 i2 -> order_interval i2 i3 -> order_interval i1 i3.
generalize order_Z'_trans1; intros H1.
unfold order_interval; intuition eauto; try (repeat Rew; auto; fail).
right; rewrite H2; auto.
right; rewrite H2; auto.
right; rewrite H2; auto.
right; rewrite H2; auto.
Qed.

Definition join_interval : interval -> interval -> interval.
refine (fun i1 i2 => 
 let min1 := (min i1) in
 let min2 := (min i2) in
 let max1 := (max i1) in
 let max2 := (max i2) in 
 let min' := (match (order_Z'_dec min1 min2) with
      (left _) => min1
    | (right _) => min2
   end) in 
 let max' := (match (order_Z'_dec max1 max2) with
      left _ => max2
    | right _ => max1
   end) in
 (ITV min' max' _)).
unfold min',max'.
cut (min1<'max1 \/ min1=max1); intros.
cut (min2<'max2 \/ min2=max2); intros.
case (order_Z'_dec min1 min2); case (order_Z'_dec max1 max2); intros; auto.
apply order_Z'_trans1 with min2; auto.
elim (not_order_Z' _ _ n0); intros.
apply order_Z'_trans1 with min1; auto.
rewrite H1; auto.
unfold min2,max2 in * |- *; apply min_order_max.
unfold min1,max1 in * |- *; apply min_order_max.
Defined.

Lemma join_interval_bound1 : forall i1 i2 : interval, 
  order_interval i1 (join_interval i1 i2).
intros i1 i2.
unfold order_interval,join_interval.
case (order_Z'_dec (min i1) (min i2)); intros;
case (order_Z'_dec (max i1) (max i2)); simpl; intros; split; auto.
apply not_order_Z'; auto.
apply not_order_Z'; auto.
Qed.

Lemma join_interval_bound2 : forall i1 i2 : interval, 
  order_interval i2 (join_interval i1 i2).
intros i1 i2.
unfold order_interval,join_interval.
case (order_Z'_dec (min i1) (min i2)); intros;
case (order_Z'_dec (max i1) (max i2)); simpl; intros; split; auto.
apply not_order_Z'; auto.
apply not_order_Z'; auto.
Qed.

Lemma join_interval_least_bound : forall i1 i2 i: interval, 
  order_interval i1 i ->
  order_interval i2 i ->
 order_interval (join_interval i1 i2) i.
intros i1 i2 i.
unfold order_interval,join_interval.
case (order_Z'_dec (min i1) (min i2)); intros H (H1,H2) (H3,H4);
case (order_Z'_dec (max i1) (max i2)); simpl; intros; split; auto.
Qed.

Definition meet_interval : interval -> interval -> option interval.
refine (fun i1 i2 => 
 let min1 := (min i1) in
 let min2 := (min i2) in
 let max1 := (max i1) in
 let max2 := (max i2) in 
 let min' := (match (order_Z'_dec min1 min2) with
      (left _) => min2
    | (right _) => min1
   end) in 
 let max' := (match (order_Z'_dec max1 max2) with
      left _ => max1
    | right _ => max2
   end) in
 match (order_Z'_dec max' min') with
   left _ => None
 | right _ => Some (ITV min' max' _)
 end).
generalize dependent n.
unfold min',max'.
cut (min1<'max1 \/ min1=max1); [intro|idtac].
cut (min2<'max2 \/ min2=max2); [intro|idtac].
case (order_Z'_dec min1 min2); case (order_Z'_dec max1 max2); intros; auto.
apply not_order_Z'; auto.
apply not_order_Z'; auto.
unfold min2,max2 in * |- *; apply min_order_max.
unfold min1,max1 in * |- *; apply min_order_max.
Defined.

Inductive eqbot  : option interval  -> option interval -> Prop :=
  eqbot1 : eqbot None None
| eqbot2 : forall x1 x2, eq_interval x1 x2 -> eqbot (Some x1) (Some x2).

Inductive orderbot  : option interval -> option interval -> Prop :=
  orderbot1 : forall x, orderbot None x
| orderbot2 : forall x1 x2, order_interval x1 x2 ->
   orderbot (Some x1) (Some x2).

Lemma meet_interval_bound1 : forall i1 i2 : interval, 
  orderbot (meet_interval i1 i2) (Some i1).
intros i1 i2.
unfold order_interval,meet_interval.
case (order_Z'_dec (min i1) (min i2)); intros;
case (order_Z'_dec (max i1) (max i2)); simpl; intros;
case order_Z'_dec; auto; intros; try constructor; split; simpl; auto.
apply not_order_Z'; auto.
apply not_order_Z'; auto.
Qed.

Lemma meet_interval_bound2 : forall i1 i2 : interval, 
  orderbot (meet_interval i1 i2) (Some i2).
intros i1 i2.
unfold order_interval,meet_interval.
case (order_Z'_dec (min i1) (min i2)); intros;
case (order_Z'_dec (max i1) (max i2)); simpl; intros; 
case order_Z'_dec; auto; intros; try constructor; split; simpl; auto.
apply not_order_Z'; auto.
apply not_order_Z'; auto.
Qed.

Lemma meet_interval_least_bound : forall i1 i2 i: interval, 
  order_interval i i1 ->
  order_interval i i2 ->
  orderbot (Some i) (meet_interval i1 i2).
intros i1 i2 i.
unfold order_interval,meet_interval.
case (order_Z'_dec (min i1) (min i2)); intros H (H1,H2) (H3,H4);
case (order_Z'_dec (max i1) (max i2)); simpl; intros;
case order_Z'_dec; auto; intros; try (constructor; split; simpl; auto);
apply False_ind;
apply (order_Z'_disj (min i) (max i));
try (apply min_order_max).
apply order_Z'_trans2 with (min i2); auto.
apply order_Z'_trans3 with (max i1); auto.
apply order_Z'_trans2 with (min i2); auto.
apply order_Z'_trans3 with (max i2); auto.
apply order_Z'_trans2 with (min i1); auto.
apply order_Z'_trans3 with (max i1); auto.
apply order_Z'_trans2 with (min i1); auto.
apply order_Z'_trans3 with (max i2); auto.
Qed.

Definition widen_interval : interval -> interval -> interval.
refine (fun i1 i2 => 
 let min1 := (min i1) in
 let min2 := (min i2) in
 let max1 := (max i1) in
 let max2 := (max i2) in 
 let min' := (match (order_Z'_dec min2 min1) with
      (left _) => infty_neg
    | (right _) => min1
   end) in 
 let max' := (match (order_Z'_dec max1 max2) with
      left _ => infty_pos
    | right _ => max1
   end) in
 (ITV min' max' _)).
unfold min',max'.
case (order_Z'_dec min2 min1); case (order_Z'_dec max1 max2); intros; auto.
left; constructor.
case max1; intros; (left;constructor; auto) || auto.
case min1; intros; (left;constructor; auto) || auto.
unfold min1,max1 in * |- *; apply min_order_max.
Defined.

Lemma order_Z'_eq_trans : forall x y t, x <=' y -> y <=' t -> x <=' t.
Proof.
  destruct 1.
  intros; apply order_Z'_trans1 with y; auto.
  subst; auto.
Qed.

Lemma not_order_Z'' : forall a b: Z', ~ a<'b -> b<='a.
Proof.
  intros; apply not_order_Z'; auto.
Qed.

Lemma order_interval_dec : forall i1 i2, {order_interval i1 i2}+{~order_interval i1 i2}.
destruct i1; destruct i2.
case (order_Z'_dec a a0); intros.
right; intros H; inversion_clear H; simpl in *.
destruct H0.
apply (order_Z'_antisym _ _ H o1).
elim (order_Z'_strict _ _ o1); auto.
case (order_Z'_dec b0 b); intros.
right; intros H; inversion_clear H; simpl in *.
destruct H1.
apply (order_Z'_antisym _ _ H o1).
elim (order_Z'_strict _ _ o1); auto.
left; constructor; simpl; intros.
apply not_order_Z'; auto.
apply not_order_Z'; auto.
Qed.

Definition narrow_interval : interval -> interval -> option interval :=
  fun i1 i2 => 
  let min := (match (min i1) with
                infty_neg => min i2
              | _ => min i1
              end) in
  let max :=  (match (max i1) with
                 infty_pos => max i2
               | _ => max i1
               end) in
   match order_Z'_dec max min with
    | left _ => None
    | right h => Some (ITV min max (not_order_Z' _ _ h))
   end.

Definition top_interval : interval.
refine 
 (ITV infty_neg infty_pos _).
left; constructor.
Defined.

Lemma top_interval_is_top : forall i : interval, order_interval i top_interval.
unfold top_interval,order_interval; intros; split; auto.
apply infty_neg_min.
apply infty_pos_max.
Qed.

Lemma widen_interval_eq1 : forall i1 i2 i3,
 eq_interval i1 i2 ->
 eq_interval (widen_interval i1 i3) (widen_interval i2 i3).
intros i1 i2 i3.
unfold eq_interval,widen_interval.
split.
case (order_Z'_dec (min i1) (min i2)); intro;
case (order_Z'_dec (min i3) (min i1)); intro;
case (order_Z'_dec (min i3) (min i2)); simpl; intuition.
elim (order_Z'_strict _ _ o H0).
elim (order_Z'_strict _ _ o H0).
elim n0; rewrite <- H0; auto.
elim n0; rewrite H0; auto.
case (order_Z'_dec (max i1) (max i2)); intro;
case (order_Z'_dec (max i1) (max i3)); intro;
case (order_Z'_dec (max i2) (max i3)); simpl; intuition.
elim (order_Z'_strict _ _ o H1).
elim (order_Z'_strict _ _ o H1).
elim n0; rewrite <- H1; auto.
elim n0; rewrite H1; auto.
Qed.

Lemma widen_interval_eq2 : forall i1 i2 i3,
 eq_interval i1 i2 ->
 eq_interval (widen_interval i3 i1) (widen_interval i3 i2).
intros i1 i2 i3.
unfold eq_interval,widen_interval.
split.
case (order_Z'_dec (min i1) (min i2)); intro;
case (order_Z'_dec (min i1) (min i3)); intro;
case (order_Z'_dec (min i2) (min i3)); simpl; intuition.
elim (order_Z'_strict _ _ o H0).
elim (order_Z'_strict _ _ o H0).
elim n0; rewrite <- H0; auto.
elim n0; rewrite H0; auto.
case (order_Z'_dec (max i1) (max i2)); intro;
case (order_Z'_dec (max i3) (max i1)); intro;
case (order_Z'_dec (max i3) (max i2)); simpl; intuition.
elim (order_Z'_strict _ _ o H1).
elim (order_Z'_strict _ _ o H1).
elim n0; rewrite <- H1; auto.
elim n0; rewrite H1; auto.
Qed.

Lemma widen_interval_bound1 : forall i1 i2 : interval,
  order_interval i1 (widen_interval i1 i2).
intros i1 i2.
unfold order_interval,widen_interval.
case (order_Z'_dec (min i2) (min i1)); intros;
case (order_Z'_dec (max i1) (max i2)); simpl; intros; split; auto;
apply infty_neg_min || apply infty_pos_max.
Qed.

Lemma widen_interval_bound2 : forall i1 i2 : interval,
  order_interval i2 (widen_interval i1 i2).
intros i1 i2.
unfold order_interval,widen_interval.
case (order_Z'_dec (min i2) (min i1)); intros;
case (order_Z'_dec (max i1) (max i2)); simpl; intros; split; auto;
apply infty_neg_min || apply infty_pos_max || apply not_order_Z'; auto.
Qed.

Hint Resolve infty_pos_max infty_neg_min.

Lemma narrow_interval_bound1 : forall i1 i2 : interval,
  orderbot (narrow_interval i1 i2) (Some i1).
intros i1 i2.
unfold order_interval,narrow_interval; simpl.
case_eq (max i1); intros;
case_eq (min i1); intros;
destruct order_Z'_dec;
constructor; simpl; split; auto;
try (rewrite H0; auto; fail);
try (rewrite H; auto; fail).
Qed.

Lemma narrow_interval_eq_def : forall i1 i2 i,
  narrow_interval i1 i2 = Some i ->
  eq_interval i i1 \/ 
  (eq_interval i i2 /\ min i1 = infty_neg /\ max i1 = infty_pos) \/
  (min i = min i1 /\ max i = max i2 /\ max i1 = infty_pos) \/
  (min i = min i2 /\ max i = max i1 /\ min i1 = infty_neg).
Proof.
  unfold narrow_interval; intros i1 i2 i.
  unfold eq_interval.
  destruct (max i1); destruct (min i1); 
  destruct order_Z'_dec;
  intros HH; inversion_clear HH; 
  unfold eq_interval; simpl; intuition.
Qed.

Lemma narrow_interval_eq_def' : forall i1 i2,
  narrow_interval i1 i2 = None ->
  (max i2 <' min i1 /\ max i1 = infty_pos) \/
  (max i1 <' min i2 /\ min i1 = infty_neg).
Proof.
  unfold narrow_interval; intros i1 i2.
  generalize (min_order_max i1); intro.
  destruct (max i1); destruct (min i1); 
  destruct order_Z'_dec;
  intros HH; inversion_clear HH;
  try (elim (order_Z'_disj _ _ H);assumption); auto.
  elim (order_Z'_disj _ _ (min_order_max i2)); auto.
Qed.

Lemma eqbot_orderbot_trans1 : forall x y z,
  eqbot x y -> orderbot y z -> orderbot x z.
Proof.
  intros.
  inversion_clear H in H0; inversion_clear H0; constructor.
  apply order_interval_trans with x2; auto.
  apply order_interval_refl; auto.
Qed.

Lemma eqbot_orderbot_trans2 : forall x y z,
  orderbot x y -> eqbot y z -> orderbot x z.
Proof.
  intros.
  inversion_clear H in H0; inversion_clear H0; constructor.
  apply order_interval_trans with x2; auto.
  apply order_interval_refl; auto.
Qed.

Hint Resolve eqbot1 eqbot2 eq_interval_sym
  meet_interval_bound2 meet_interval_bound1
  min_order_max.
Lemma narrow_interval_bound2 : forall i1 i2 : interval,
  orderbot (meet_interval i1 i2) (narrow_interval i1 i2).
intros i1 i2.
case_eq (narrow_interval i1 i2); intros.
destruct (narrow_interval_eq_def _ _ _ H) as [H1 | [(H1,H2) | [(H1,(H2,H3)) | (H1,(H2,H3))]]].
apply eqbot_orderbot_trans2 with (Some i1); auto.
apply eqbot_orderbot_trans2 with (Some i2); auto.
unfold meet_interval.
destruct (order_Z'_dec (min i1) (min i2));
destruct (order_Z'_dec (max i1) (max i2));
case order_Z'_dec; intros; simpl;
try rewrite H; 
try (constructor; split; simpl; auto);
try rewrite H1; try rewrite H2; auto.
unfold meet_interval.
destruct (order_Z'_dec (min i1) (min i2));
destruct (order_Z'_dec (max i1) (max i2));
case order_Z'_dec; intros; simpl;
try rewrite H;
constructor; split; simpl; auto;
try rewrite H1; try rewrite H2; try rewrite H3; auto.
apply  not_order_Z'; auto.
destruct (min i2); try (right;auto;fail).
elim n; rewrite H3; constructor.
elim n; rewrite H3; constructor.
destruct (min i2); try (right;auto;fail).
elim n; rewrite H3; constructor.
elim n; rewrite H3; constructor.
apply  not_order_Z'; auto.
destruct (narrow_interval_eq_def' _ _ H) as [(H0,H1)|(H0,H1)].
unfold meet_interval.
destruct (order_Z'_dec (min i1) (min i2));
destruct (order_Z'_dec (max i1) (max i2));
case order_Z'_dec; intros; simpl;
try constructor.
elim (order_Z'_antisym _ _ H0).
apply order_Z'_trans2 with (min i2); auto.
elim n0; apply order_Z'_trans with (1:=H0); auto.
elim n0; apply order_Z'_trans with (2:=H0); auto.
elim n1; auto.
unfold meet_interval.
destruct (order_Z'_dec (min i1) (min i2));
destruct (order_Z'_dec (max i1) (max i2));
case order_Z'_dec; intros; simpl;
try constructor.
elim n; auto.
elim n; apply order_Z'_trans2 with (min i2); auto.
elim n; apply order_Z'_trans3 with (max i1); auto.
elim n0; apply order_Z'_trans2 with (min i2); auto.
Qed.

Definition left_interval : Z -> interval.
refine 
 (fun b => ITV infty_neg (z b) _).
left; constructor.
Defined.

Definition right_interval : Z -> interval.
refine 
 (fun a => ITV (z a) infty_pos _).
left; constructor.
Defined.

Definition z_interval : forall a b :Z, (Zle a b) -> interval.
refine 
 (fun a b H => ITV (z a) (z b) _).
elim (Zle_lt_or_eq _ _ H); intros.
left; constructor; auto.
right; subst; auto.
Defined.

Definition infty_neg_interval : interval.
refine (ITV infty_neg infty_neg _).
auto.
Defined.

Definition infty_pos_interval : interval.
refine (ITV infty_pos infty_pos _).
auto.
Defined.

Lemma six_kind_of_interval : forall i : interval,
    (eq_interval i top_interval)
 \/ (eq_interval i infty_neg_interval)
 \/ (eq_interval i infty_pos_interval)
 \/ (exists b, (eq_interval i (left_interval b)))
 \/ (exists a, (eq_interval i (right_interval a)))
 \/ (exists a, exists b, exists H:Zle a b, (eq_interval i (z_interval a b H))).
destruct i.
destruct a; destruct b.
right; right; right; right; right.
exists z0; exists z1.
cut (z0 <= z1)%Z; intros.
exists H.
unfold eq_interval, z_interval; simpl; auto.
decompose [or] o.
inversion H; intros; auto with zarith.
inversion H; intros; auto with zarith.
right; right; right; right; left.
exists z0; unfold eq_interval, right_interval; simpl; auto.
elim o; intros H; inversion H.
elim o; intros H; inversion H.
right; right; left; unfold eq_interval,infty_pos_interval; simpl; auto.
elim o; intros H; inversion H.
right; right; right; left; unfold eq_interval,left_interval; simpl; eauto.
left; unfold eq_interval,top_interval; simpl; auto.
right; left; unfold eq_interval,infty_neg_interval; simpl; auto.
Qed.

Definition acc_interval i := forall int : interval,
    Acc
      (fun p1 p2 =>
       eq_interval (snd p1) (widen_interval (snd p2) (fst p1)) /\ 
         ~ eq_interval (snd p2) (snd p1)) 
      (int,i).

Lemma Acc_eq_acc : forall (A:Set) (Ord Eq:A->A->Prop),
 (forall x y z, Ord x y -> Eq y z -> Ord x z) ->
 forall x y, Acc Ord x -> Eq y x -> Acc Ord y.
intros.
inversion_clear H0.
constructor; intros.
apply H2.
apply H with y; auto.
Qed.

Lemma order_eq_order_interval : forall i1 i2 i3,
 order_interval i1 i2 -> eq_interval i2 i3 -> order_interval i1 i3.
intros; apply order_interval_trans with i2; auto.
apply order_interval_refl; auto.
Qed.

Lemma acc_interval_eq : forall i1 i2,
 acc_interval i1 -> eq_interval i1 i2 -> acc_interval i2.
unfold acc_interval; intros.
apply Acc_eq_acc with
 (fun p1 p2 : interval*interval => eq_interval (fst p1) (fst p2) /\
               eq_interval (snd p1) (snd p2))
 (int,i1); auto.
intros; intuition.
apply eq_interval_trans with (1:=H3).
apply widen_interval_eq1; auto.
elim H4; apply eq_interval_trans with (snd z0); auto.
split; simpl.
apply eq_interval_refl.
apply eq_interval_sym; auto.
Qed.

Lemma acc_interval_property_top : acc_interval top_interval.
unfold acc_interval; constructor.
unfold order_interval,eq_interval; simpl.
destruct y; simpl.
case (order_Z'_dec (min i) infty_neg); intro;
case (order_Z'_dec infty_pos (max i)); intro.
inversion_clear o.
inversion_clear o.
inversion_clear o.
clear n n0; intros ((H2,H3),H4).
elim H4; rewrite H2; rewrite H3; auto.
Qed.

Lemma acc_interval_property_left : forall b : Z,
 acc_interval (left_interval b).
constructor.
unfold left_interval,order_interval,eq_interval; simpl.
destruct y; simpl.
case (order_Z'_dec (min i) infty_neg); intro;
case (order_Z'_dec (z b) (max i)); intro.
inversion_clear o.
inversion_clear o.
intros ((H1,H1'),H2).
apply acc_interval_eq with top_interval.
apply acc_interval_property_top.
split; simpl; auto.
intros ((H1,H3),H2).
elim H2; split; auto.
Qed.

Lemma acc_interval_property_right : forall b : Z,
 acc_interval (right_interval b).
constructor.
unfold right_interval,order_interval,eq_interval; simpl.
destruct y; simpl.
case (order_Z'_dec infty_pos (max i)); intro;
case (order_Z'_dec (min i) (z b)); intro;
intros ((H2,H3),H4).
inversion_clear o.
inversion_clear o.
apply acc_interval_eq with top_interval.
apply acc_interval_property_top.
unfold eq_interval; auto.
elim H4; auto.
Qed.

Lemma acc_interval_property_infty_neg : acc_interval infty_neg_interval.
constructor.
unfold infty_neg_interval,order_interval,eq_interval; simpl.
destruct y; simpl.
case (order_Z'_dec (min i) infty_neg); intro;
case (order_Z'_dec infty_neg (max i)); intro;
intros ((H2,H3),H4).
inversion_clear o.
inversion_clear o.
apply acc_interval_eq with top_interval.
apply acc_interval_property_top.
unfold eq_interval; auto.
elim H4; auto.
Qed.

Lemma acc_interval_property_infty_pos : acc_interval infty_pos_interval.
constructor.
unfold infty_pos_interval,order_interval,eq_interval; simpl.
destruct y; simpl.
case (order_Z'_dec (min i) infty_pos); intro;
case (order_Z'_dec infty_pos (max i)); intro;
intros ((H2,H3),H4).
inversion_clear o0.
apply acc_interval_eq with top_interval.
apply acc_interval_property_top.
unfold eq_interval; auto.
inversion_clear o.
elim H4; auto.
Qed.

Lemma acc_interval_property_z : forall a b H,
 acc_interval (z_interval a b H).
constructor.
unfold z_interval,order_interval,eq_interval; simpl.
destruct y; simpl.
case (order_Z'_dec (min i) (z a)); intro;
case (order_Z'_dec (z b) (max i)); intro;
intros ((H2,H3),H4).
apply acc_interval_eq with top_interval.
apply acc_interval_property_top.
unfold eq_interval; auto.
apply acc_interval_eq with (left_interval b).
apply acc_interval_property_left.
unfold eq_interval; auto.
apply acc_interval_eq with (right_interval a).
apply acc_interval_property_right.
unfold eq_interval; auto.
elim H4; auto.
Qed.

Lemma acc_interval_property : forall i:interval, acc_interval i.
intros i; elim (six_kind_of_interval i); intuition.
apply acc_interval_eq with top_interval.
apply acc_interval_property_top.
apply eq_interval_sym; auto.
apply acc_interval_eq with infty_neg_interval.
apply acc_interval_property_infty_neg.
apply eq_interval_sym; auto.
apply acc_interval_eq with infty_pos_interval.
apply acc_interval_property_infty_pos.
apply eq_interval_sym; auto.
elim H0; intros.
apply acc_interval_eq with (left_interval x).
apply acc_interval_property_left.
apply eq_interval_sym; auto.
elim H; intros.
apply acc_interval_eq with (right_interval x).
apply acc_interval_property_right.
apply eq_interval_sym; auto.
decompose [ex] H.
elim H0; intros.
apply acc_interval_eq with (z_interval x x0 x1).
apply acc_interval_property_z.
apply eq_interval_sym; auto.
Qed. 

Definition acc_interval' i := forall int : interval,
    Acc
      (fun p1 p2 =>
       eqbot (Some (snd p1)) (narrow_interval (snd p2) (fst p1)) /\ 
         ~ eq_interval (snd p2) (snd p1)) 
      (int,i).

Lemma case_interval: forall i,
 ( min i = infty_neg /\ max i = infty_pos ) \/
 ( min i = infty_neg /\ max i <' infty_pos ) \/
 ( infty_neg <' min i /\ max i = infty_pos ) \/
 ( infty_neg <' min i /\ max i <' infty_pos ).
Proof.
  destruct i; simpl.
  destruct a; destruct b; simpl; auto;
  try (do 3 right; split; constructor) ||
      (do 2 right; left; split; constructor) ||
      (right; left; split; constructor).
Qed.

Definition sym_interval : nat -> interval.
intros n.
refine 
 (ITV (z (- (Z_of_nat n))) (z (Z_of_nat n)) _).
destruct n.
right; constructor; reflexivity.
left; constructor.
assert (0<Z_of_nat (S n))%Z.
simpl; constructor.
omega.
Defined.


Lemma meet_interval_eq1_order : forall i1 i2 j,
 eq_interval i1 i2 ->
 orderbot (meet_interval i1 j) (meet_interval i2 j).
Proof.
  intros.
  destruct H.
  generalize (meet_interval_bound1 i1 j).
  generalize (meet_interval_bound2 i1 j).
  generalize (meet_interval_least_bound i2 j).
  case_eq (meet_interval i2 j);
  case_eq (meet_interval i1 j); intros; try (constructor;fail).
  apply H3; auto.
  inversion_clear H5.
  destruct i1; destruct i2; simpl in *; subst; auto.
  inversion H4; auto.
  apply H3; auto.
  inversion_clear H5.
  destruct i1; destruct i2; simpl in *; subst; auto.
  inversion_clear H4; auto.
Qed.

Lemma orderbot_antisym : forall x y,
  orderbot x y -> orderbot y x -> eqbot x y.
Proof.
  intros.
  inversion_clear H in H0; inversion_clear H0; constructor.
  apply order_interval_antisym; auto.
Qed.

Lemma meet_interval_eq1 : forall i1 i2 j,
 eq_interval i1 i2 ->
 eqbot (meet_interval i1 j) (meet_interval i2 j).
Proof.
  intros; apply orderbot_antisym.
  apply meet_interval_eq1_order; auto.
  apply meet_interval_eq1_order; auto.
Qed.

Hint Resolve eq_interval_refl.

Lemma eqbot_refl : forall x, eqbot x x.
Proof.
  destruct x; constructor; auto.
Qed.

Hint Resolve eqbot_refl.

Lemma narrow_interval_eq1 : forall i1 i2 j,
  eq_interval i1 i2 ->
  eqbot (narrow_interval i1 j) (narrow_interval i2 j).
Proof.
  intros.
  destruct H.
  unfold narrow_interval.
  rewrite H; rewrite H0.
  destruct (max i2); destruct (min i2); auto.
Qed.

Lemma narrow_interval_eq2 : forall i1 i2 j,
  eq_interval i1 i2 ->
  eqbot (narrow_interval j i1) (narrow_interval j i2).
Proof.
  intros.
  destruct H.
  unfold narrow_interval.
  rewrite H; rewrite H0.
  destruct (max i2); destruct (min i2); auto.
Qed.

Lemma eqbot_trans : forall x y z,
  eqbot x y -> eqbot y z -> eqbot x z.
Proof.
  intros.
  inversion_clear H in H0; inversion_clear H0; constructor.
  apply eq_interval_trans with x2; auto.
Qed.

Lemma eqbot_sym : forall x y, eqbot x y -> eqbot y x.
Proof.
  intros.
  inversion_clear H; constructor; auto.
Qed.

Lemma acc_interval'_eq : forall i1 i2,
 acc_interval' i1 -> eq_interval i1 i2 -> acc_interval' i2.
unfold acc_interval'; intros.
apply Acc_eq_acc with
 (fun p1 p2 : interval*interval => eq_interval (fst p1) (fst p2) /\
               eq_interval (snd p1) (snd p2))
 (int,i1); auto.
intros; intuition.
apply eqbot_trans with (narrow_interval (snd y) (fst x)); auto.
apply  narrow_interval_eq1; auto.
elim H4; apply eq_interval_trans with (snd z0); auto.
Qed.

Lemma order_Z'_eq_antisym : forall x y,
 x<='y -> y<='x -> x=y.
Proof.
  intros.
  destruct H; auto.
  elim (order_Z'_disj _ _ H0); auto.
Qed.

Inductive bound_length_interval (i:interval )(b:Z) : Prop :=
 bound_length_interval_def : forall z1 z2,
  min i = z z1 -> max i = z z2 -> ((z2-z1)<=b)%Z ->
  bound_length_interval i b.

Lemma inj_order_Z'_eq : forall x y, (z x)<='(z y) -> (x<=y)%Z.
Proof.
  intros.
  inversion_clear H.
  inversion_clear H0; omega.
  inversion_clear H0; omega.
Qed.

Lemma acc_interval'_property1 : forall i:interval, 
 infty_neg <' min i /\ max i <' infty_pos ->
 acc_interval' i.
Proof.
  intros i2 (H1,H2) i1.
  constructor; destruct y; simpl.
  case_eq (narrow_interval i2 i).
  intros i3 H; generalize (narrow_interval_eq_def _ _ _ H); intuition.
  inversion_clear H4; elim H5; apply eq_interval_trans with i3; auto.
  rewrite H7 in H2; inversion_clear H2.
  rewrite H7 in H2; inversion_clear H2.
  rewrite H7 in H1; inversion_clear H1.
  intros H; generalize (narrow_interval_eq_def' _ _ H); intuition.
  rewrite H6 in H2; inversion_clear H2.
  rewrite H6 in H1; inversion_clear H1.
Qed.

Lemma acc_interval'_property2 : forall i:interval, 
 infty_neg <' min i /\ max i = infty_pos ->
 acc_interval' i.
Proof.
  intros i2 (H1,H2) i1.
  constructor; intros (j1,j2); simpl.
  unfold narrow_interval.
  unfold eq_interval.
  rewrite H2; destruct (min i2); 
  case order_Z'_dec; simpl; intros h (H,H3);
  inversion_clear H; destruct H0; simpl in *.
  apply acc_interval'_property1.
  rewrite H; split.
  constructor.
  destruct (max j2); try constructor.
  elim H3; split; auto.
  generalize (min_order_max j2); rewrite H; intros.
  inversion_clear H4.
  inversion_clear H5.
  elim H3; split; auto.
  inversion_clear H1.
Qed.

Lemma acc_interval'_property3 : forall i:interval, 
 infty_neg = min i /\ max i <' infty_pos ->
 acc_interval' i.
Proof.
  intros i2 (H1,H2) i1.
  constructor; intros (j1,j2); simpl.
  unfold narrow_interval.
  unfold eq_interval.
  rewrite <- H1; destruct (max i2); 
  case order_Z'_dec; simpl; intros h (H,H3);
  inversion_clear H; destruct H0; simpl in *.
  apply acc_interval'_property1.
  rewrite H0; split.
  destruct (min j2); try constructor.
  elim H3; split; auto.
  constructor.
  inversion_clear H2.
  generalize (min_order_max j2); rewrite H0; intros.
  inversion_clear H4.
  inversion_clear H5.
  elim H3; split; auto.
Qed.

Lemma eq_Z'_dec : forall x y : Z', {x=y}+{~x=y}.
Proof.
  destruct x; destruct y; try (left;constructor;reflexivity) ||
  (right; intros H; inversion_clear H;fail).
  destruct (Z_eq_dec z0 z1); intros.
  subst; left; constructor.
  right; intro; elim n.
  inversion_clear H; auto.
Qed.

Lemma acc_interval'_property4 : forall i:interval, 
 min i = infty_neg /\ max i = infty_pos ->
 acc_interval' i.
Proof.
  intros i2 (H1,H2) i1.
  constructor; intros (j1,j2); simpl.
  unfold narrow_interval.
  unfold eq_interval.
  rewrite H1; rewrite H2.
  case order_Z'_dec; simpl; intros h (H3,H4).
  elim (order_Z'_disj _ _ (min_order_max j1)); auto.
  inversion_clear H3.
  destruct H; simpl in *.
  destruct (eq_Z'_dec infty_neg (min j2)).
  apply acc_interval'_property3.
  split; auto.
  destruct (max j2); try constructor.
  elim H4; auto.
  destruct (eq_Z'_dec infty_pos (max j2)).
  apply acc_interval'_property2.
  split; auto.
  destruct (min j2); try constructor.
  elim H4; auto.
  apply acc_interval'_property1; split.
  destruct (min j2); try constructor; intuition.
  destruct (max j2); try constructor; intuition.
Qed.
  
Lemma acc_interval'_property : forall i:interval, acc_interval' i.
Proof.
  intros i1 i2.
  destruct (case_interval i1) as [H7|[H7|[H7|H7]]]; destruct H7.
  apply acc_interval'_property4; split; auto.
  apply acc_interval'_property3; split; auto.
  apply acc_interval'_property2; split; auto.
  apply acc_interval'_property1; split; auto.
Qed.

Lemma eq_interval_dec : forall i1 i2, {eq_interval i1 i2}+{~eq_interval i1 i2}.
destruct i1; destruct i2.
case (eq_Z'_dec a a0); intros.
case (eq_Z'_dec b b0); intros.
left; constructor; simpl; auto.
right; intros H; inversion_clear H; elim n; auto.
right; intros H; inversion_clear H; elim n; auto.
Qed.

Instance IntervalEquiv : EquivDec.t (option interval).
exists eqbot.

destruct x; constructor.
apply eq_interval_refl.

intros.
inversion_clear H; constructor.
apply eq_interval_sym; auto.

intros.
inversion_clear H in H0; inversion_clear H0; constructor.
apply eq_interval_trans with x2; auto.

destruct x; destruct y; try (right; intros H; inversion_clear H; fail).
destruct (eq_interval_dec i i0).
left; constructor; auto.
right; intros H; inversion_clear H; elim n; auto.
left; constructor.
Defined.

Instance IntervalPoset : PosetDec.t (option interval).
refine (PosetDec.Make _ _ orderbot _ _ _ _).

intros [x|] y H; inversion_clear H; constructor.
apply order_interval_refl; auto.

intros.
inversion_clear H in H0; inversion_clear H0; constructor.
apply order_interval_antisym; auto.

intros.
inversion_clear H in H0; inversion_clear H0; constructor.
apply order_interval_trans with x2; auto.

destruct x; destruct y; try (right; intros H; inversion_clear H; fail).
destruct (order_interval_dec i i0).
left; constructor; auto.
right; intros H; inversion_clear H; elim n; auto.
left; constructor.
left; constructor.
Defined.

Instance IntervalJoin : JoinDec.t (option interval).
refine (JoinDec.Make _ _
  (fun x y =>
    match (x,y) with
    | (None,_) => y
    | (_,None) => x
    | (Some x,Some y) => Some (join_interval x y)
    end) _ _ _).

destruct x; destruct y; constructor.
apply join_interval_bound1.
apply order_interval_refl; apply eq_interval_refl.

destruct x; destruct y; constructor.
apply join_interval_bound2.
apply order_interval_refl; apply eq_interval_refl.

intros.
inversion_clear H in H0; inversion_clear H0; constructor; auto.
apply join_interval_least_bound; auto.
Defined.

Instance IntervalMeet : MeetDec.t (option interval).
refine (MeetDec.Make _ _
  (fun x y =>
    match (x,y) with
    | (None,_) => None
    | (_,None) => None
    | (Some x,Some y) => meet_interval x y
    end) _ _ _).

intros x y; destruct x; destruct y; try constructor.
apply meet_interval_bound1.

intros x y; destruct x; destruct y; try constructor.
apply meet_interval_bound2.    

intros.
inversion_clear H in H0; inversion_clear H0; try constructor; auto.
apply meet_interval_least_bound; auto.
Defined.

Instance IntervalBot : BotDec.t (option interval).
refine (BotDec.Make _ _ None _).
intros; constructor.
Defined.

Instance IntervalTop : TopDec.t (option interval).
refine (TopDec.Make _ _ (Some top_interval) _).
intros.
destruct x; constructor.
split; unfold top_interval; simpl; auto.
Defined.

Definition t := option interval.

  Definition wid : t -> t -> t := fun x y =>
    match (x,y) with
    | (None,_) => y
    | (_,None) => x
    | (Some x,Some y) => Some (widen_interval x y)
    end.

Definition narr: t -> t -> t := fun x y =>
  match x,y with
    Some x,Some y => narrow_interval x y
  | _,_ => None
  end.
 
  Definition widen : t -> t -> t := fun x y =>
    match (x,y) with
    | (None,_) => y
    | (_,None) => x
    | (Some x,Some y) => Some (widen_interval x y)
    end.
 
Instance IntervalWiden : Widen.t (option interval).
refine (Widen.Make _ _ _ _ widen _ _ _ _ _ _ _).

destruct x; destruct y; unfold widen; constructor.
apply widen_interval_bound1.
apply order_interval_refl; apply eq_interval_refl.

destruct x; destruct y; unfold widen; constructor.
apply widen_interval_bound2.
apply order_interval_refl; apply eq_interval_refl.

intros x y l H; inversion_clear H; destruct l; unfold widen; constructor; auto.
apply widen_interval_eq1; auto.

intros x y l H; inversion_clear H; destruct l; unfold widen; constructor; auto.
apply widen_interval_eq2; auto.

unfold widen; simpl.
destruct x; constructor; auto.

unfold widen; simpl.
apply eqbot_refl.

  constructor; destruct y.
  unfold Widen.rel; simpl.
  destruct t0. 
  destruct t1.
  intros _.
  assert (forall p,
      Acc
     (fun x0 y : t * t =>
      (snd x0) =♯ (widen (snd y) (fst x0)) /\ ~  (snd y) =♯ (snd x0))
     (Some (fst p), Some (snd p))).
  intros p.
  assert ( Acc
      (fun p1 p2 =>
       eq_interval (snd p1) (widen_interval (snd p2) (fst p1)) /\ 
         ~ eq_interval (snd p2) (snd p1)) p).
  destruct p; apply acc_interval_property.
  induction H as [p _ Hind].
  destruct p; simpl; constructor.
  destruct y; simpl.
  intros (H1,H2).
  destruct t0.
  destruct t1.
  apply Hind with (y:=(i3,i4)).
  simpl in H1; inversion_clear H1.
  split; simpl; auto.
  simpl in H1; inversion_clear H1.
  unfold widen in H1; elim H2.
  inversion_clear H1; constructor; auto.
  apply (H (i,i0)).
  simpl; intros (H,H1).
  destruct x.
  simpl in H; inversion_clear H.
  elim H1; constructor.
  intros (H1,H2); simpl in *.
  elim H2.
  replace x with (widen x None).
  apply eqbot_sym; auto.
  destruct x; unfold widen; auto.
Defined.

Definition narrow: t -> t -> t := fun x y =>
  match x,y with
    Some x,Some y => narrow_interval x y
  | _,_ => None
  end.

Instance IntervalNarrow : Narrow.t (option interval).
refine (Narrow.Make _ _ _ _ _ _ narrow _ _ _ _ _).

destruct x; destruct y; simpl; try constructor.
apply narrow_interval_bound1.

destruct x; destruct y; simpl; try constructor.
apply narrow_interval_bound2.

destruct x; destruct y; destruct z0; simpl; intros;
  try constructor.
inversion_clear H; apply narrow_interval_eq1; auto.
inversion_clear H.
inversion_clear H.

destruct x; destruct y; destruct z0; simpl; intros;
  try constructor.
inversion_clear H; apply narrow_interval_eq2; auto.
inversion_clear H.
inversion_clear H.

constructor; destruct y.
unfold Narrow.rel; simpl.
destruct t1.
destruct t0.
intros _.
assert (forall p, Acc
     (fun x0 y : t * t =>
      (snd x0) =♯ (narrow (snd y) (fst x0)) /\ ~ (snd y) =♯ (snd x0))
     (Some (fst p), Some (snd p))).
intros p.
assert ( Acc
      (fun p1 p2 =>
       eqbot (Some (snd p1)) (narrow_interval (snd p2) (fst p1)) /\ 
         ~ eq_interval (snd p2) (snd p1)) p).
  destruct p; apply acc_interval'_property.
  induction H as [p _ Hind].
  destruct p; constructor.
  destruct y; simpl; intros (H1,H2).
  destruct t1.
  destruct t0.
  apply (Hind (i4,i3)).
  simpl; split; auto.
  inversion_clear H1.
  constructor; destruct y; simpl; intros (H3,H4).
  elim H4; apply eqbot_sym; auto.
  apply (H (i0,i)).
  unfold narrow; destruct x; intros (H,_); inversion_clear H.
  constructor; destruct y; simpl; intros (H3,H4).
  elim H4; apply eqbot_sym; auto.
Defined.

Instance IntervalLattice : AbLattice.t (option interval).

Inductive ZleI (x:Z) : Z' -> Prop :=
  ZleI_def0 : ZleI x infty_pos
| ZleI_def1 : forall y, x<=y -> ZleI x (z y).
Inductive ZgeI (x:Z) : Z' -> Prop :=
  ZgeI_def0 : ZgeI x infty_neg
| ZgeI_def1 : forall y, y<=x -> ZgeI x (z y).

Inductive gamma_ : t -> Z -> Prop :=
  gamma_def0 : forall i x, 
    ZgeI x (min i) -> ZleI x (max i) -> gamma_ (Some i) x.
Definition gamma := gamma_.

Lemma ZleI_trans1 : forall x y z,
  ZleI x y -> y <' z -> ZleI x z.
Proof.
  intros.
  inversion_clear H0 in H; inversion_clear H; try constructor.
  omega.
Qed.

Lemma ZleI_trans2 : forall x y z,
  ZleI x y -> y <=' z -> ZleI x z.
Proof.
  destruct 2.
  apply ZleI_trans1 with y; auto.
  subst; auto.
Qed.

Lemma ZgeI_trans1 : forall x y z,
  ZgeI x z -> y <' z -> ZgeI x y.
Proof.
  intros.
  inversion_clear H0 in H; inversion_clear H; try constructor.
  omega.
Qed.

Lemma ZgeI_trans2 : forall x y z,
  ZgeI x z -> y <=' z -> ZgeI x y.
Proof.
  destruct 2.
  apply ZgeI_trans1 with z0; auto.
  subst; auto.
Qed.

Lemma gamma_monotone : forall N1 N2 n,
  N1 ⊑♯ N2 ->
  gamma N1 n ->
  gamma N2 n.
Proof.
  intros.
  inversion_clear H in H0; inversion_clear H0; constructor.
  apply ZgeI_trans2 with (min x1); inversion_clear H1; auto.
  apply ZleI_trans2 with (max x1); inversion_clear H1; auto.
Qed.

Lemma not_ZgeI_ZleI : forall x y, ZgeI x y -> ZleI x y -> y = z x.
Proof.
  destruct 1; intros H'; inversion_clear H'.
  replace x with y; auto.
  omega.
Qed.

Hint Resolve not_ZgeI_ZleI ZgeI_trans1 ZgeI_trans2 
 ZleI_trans1 ZleI_trans2.

Lemma gamma_meet_morph : forall N1 N2 n,
  gamma N1 n -> gamma N2 n -> gamma (N1 ⊓♯ N2) n.
Proof.
  intros N1 N2 n H1; inversion_clear H1; intros H2; inversion_clear H2.

  simpl.
  unfold meet_interval.
  case (order_Z'_dec (max i) (max i0));
  case (order_Z'_dec (min i) (min i0)); 
  case order_Z'_dec; intros;
  try (constructor; simpl; auto; fail).
  assert (min i0 = z n); [eauto|idtac].
  assert (max i = z n); [eauto|idtac].
  elim (order_Z'_strict _ _ o); rewrite H2; auto.
  assert (min i = z n); [eauto|idtac].
  assert (max i = z n); [eauto|idtac].
  elim (order_Z'_strict _ _ o); rewrite H2; auto.
  assert (min i0 = z n); [eauto|idtac].
  assert (max i0 = z n); [eauto|idtac].
  elim (order_Z'_strict _ _ o); rewrite H2; auto.
  assert (min i = z n); [eauto|idtac].
  assert (max i0 = z n); [eauto|idtac].
  elim (order_Z'_strict _ _ o); rewrite H2; auto.
Qed.

Definition sub1' (x:Z') : Z' :=
  match x with
    infty_neg => infty_neg
  | infty_pos => infty_pos
  | z a => z (a-1)
  end.

Definition add1' (x:Z') : Z' :=
  match x with
    infty_neg => infty_neg
  | infty_pos => infty_pos
  | z a => z (a+1)
  end.

Definition min' (x y : Z') : Z' :=
  match x with
    infty_neg => x
  | infty_pos => y
  | z a => match y with
             infty_neg => infty_neg
           | infty_pos => x
           | z b => z (Zmin a b) 
           end
  end.

Lemma order_Z'_eq_refl : forall x, x <=' x.
Proof.
  intros; right; auto.
Qed.
Hint Resolve order_Z'_eq_refl.

Lemma order_Z'_eq_Zle : forall x y,
 x <= y -> (z x) <=' (z y).
Proof.
  intros.
  destruct (Zle_lt_or_eq _ _ H).
  left; constructor; auto.
  right; subst; auto.
Qed.
Hint Resolve order_Z'_eq_Zle.

Lemma le_min'_l : forall x y, (min' x y) <=' x.
Proof.
  destruct x; destruct y; simpl; auto; try (repeat constructor; fail).
Qed.

Lemma le_min'_r : forall x y, (min' x y) <=' y.
Proof.
  destruct x; destruct y; simpl; auto; try (repeat constructor; fail).
Qed.

Lemma case_min' : forall (P:Z'->Prop) x y, 
  (x <=' y -> P x) ->
  (y <=' x -> P y) -> P (min' x y).
Proof.
  destruct x; destruct y; simpl; intros; auto;
  (apply H0; repeat constructor) ||
  (apply H; repeat constructor) || 
  (apply case_Zmin; auto).
Qed.

Lemma order_Z'_eq_infty_neg : forall x, infty_neg <=' x.
Proof.
  destruct x; (constructor 1; constructor) || (constructor 2; auto).
Qed.
Lemma order_Z'_eq_infty_pos : forall x, x <=' infty_pos.
Proof.
  destruct x; (constructor 1; constructor) || (constructor 2; auto).
Qed.
Hint Resolve order_Z'_eq_infty_pos order_Z'_eq_infty_neg.

Definition max' (x y : Z') : Z' :=
  match x with
    infty_neg => y
  | infty_pos => x
  | z a => match y with
             infty_neg => z a
           | infty_pos => infty_pos
           | z b => z (Zmax a b) 
           end
  end.

Lemma le_max'_l : forall x y, x <=' (max' x y).
Proof.
  destruct x; destruct y; simpl; auto; try (repeat constructor; fail).
Qed.

Lemma le_max'_r : forall x y, y <=' (max' x y).
Proof.
  destruct x; destruct y; simpl; auto; try (repeat constructor; fail).
Qed.

Lemma case_max' : forall (P:Z'->Prop) x y, 
  (y <=' x -> P x) ->
  (x <=' y -> P y) -> P (max' x y).
Proof.
  destruct x; destruct y; simpl; intros; auto;
  (apply H0; repeat constructor) ||
  (apply H; repeat constructor) || 
  (apply case_Zmax; auto).
Qed.

Lemma order_Z'_eq_min' : forall a b d,
  a <' d -> a <=' b -> a <=' (min' b (sub1' d)).
Proof.
  do 4 intro; apply case_min'; intros; auto.
  destruct d; simpl in *; auto.
  inversion_clear H; auto.
  apply order_Z'_eq_Zle; omega.
Qed.

Lemma order_Z'_eq_max' : forall a c d,
  a <' d -> c <=' d -> (max' (add1' a) c) <=' d.
Proof.
  do 4 intro; apply case_max'; intros; auto.
  destruct a; simpl in *; auto.
  inversion_clear H; auto.
  apply order_Z'_eq_Zle; omega.
Qed.

Definition BackTestEq (N1 N2:t) : (t*t) :=
  (N1 ⊓♯ N2,N1 ⊓♯ N2).

Definition BackTestLt (N1 N2:t) : (t*t) :=
  match N1,N2 with
  | Some (ITV a b h1),Some (ITV c d h2) =>
      (N1 ⊓♯ (Some (ITV infty_neg (sub1' d) (order_Z'_eq_infty_neg _))),
       N2 ⊓♯(Some (ITV (add1' a) infty_pos (order_Z'_eq_infty_pos _))))
  | _,_ => (None,None)
  end.

Lemma Zmin_l : forall x y, x<=y -> Zmin x y = x.
Proof.
  intros; apply case_Zmin; auto with zarith.
Qed.

Lemma Zmin_r : forall x y, y<=x -> Zmin x y = y.
Proof.
  intros; apply case_Zmin; auto with zarith.
Qed.


Lemma min'_eq : forall b d,
  min' b d = (if order_Z'_dec b d then b else d).
Proof.
  destruct b; destruct d; simpl; case order_Z'_dec; 
  intros; 
  try solve[inversion_clear o|elim n;constructor]; auto.
  inversion_clear o.
  rewrite Zmin_l; try omega; auto.
  assert (z1<=z0).
  case (Z_le_dec z1 z0); auto.
  intro; elim n; constructor; omega.
  rewrite Zmin_r; auto.
Qed.

Lemma BackTestEq_correct : forall N1 N2 N1' N2' n1 n2,
  n1 = n2 ->
  gamma N1 n1 -> gamma N2 n2 ->
  BackTestEq N1 N2 = (N1',N2') ->
  gamma N1' n1 /\ gamma N2' n2.
Proof.
  unfold BackTestEq; intros N1 N2 N1' N2' n1 n2 H.
  intros; subst.
  match goal with
    [ _: (?X,?Y) = (?X1,?Y1) |- _] =>
    replace X1 with X;
    replace Y1 with Y; try congruence
  end.
  split; apply gamma_meet_morph; auto.
Qed.

Lemma BackTestLt_correct : forall N1 N2 N1' N2' n1 n2,
  n1 < n2 ->
  gamma N1 n1 -> gamma N2 n2 ->
  BackTestLt N1 N2 = (N1',N2') ->
  gamma N1' n1 /\ gamma N2' n2.
Proof.
  unfold BackTestLt; intros N1 N2 N1' N2' n1 n2 H.
  destruct N1; destruct N2; intros;
  injection H2; intros; clear H2; subst; auto;
  try solve [inversion_clear H1;inversion_clear H0].
  generalize H3; clear H3; destruct i; destruct i0.
  intros H3; injection H3; intros; subst; clear H3.
  inversion_clear H0; inversion_clear H1; simpl in *.
  split.
  apply (gamma_meet_morph (Some (ITV a b o))
    (Some (ITV infty_neg (sub1' b0) (order_Z'_eq_infty_neg (sub1' b0)))));
  auto; repeat (constructor;simpl); auto.
  inversion_clear H4; simpl; constructor; omega.
  apply (gamma_meet_morph (Some (ITV a0 b0 o0))
        (Some (ITV (add1' a) infty_pos (order_Z'_eq_infty_pos (add1' a)))));
  auto; repeat (constructor;simpl); auto.
  inversion_clear H2; simpl; constructor; omega.
Qed.

Definition minus_Z' (x:Z') : Z' :=
 match x with
 | z y => z (-y)
 | infty_pos => infty_neg
 | infty_neg => infty_pos
 end.

Lemma minus_Z'_inv_lt : forall x y,
 x <' y -> (minus_Z' y) <' (minus_Z' x).
Proof.
  destruct 1; simpl; try constructor.
  omega.
Qed.

Lemma minus_Z'_inv_le : forall x y,
 x <=' y  -> (minus_Z' y) <=' (minus_Z' x).
Proof.
  destruct 1.
  left; apply minus_Z'_inv_lt; auto.
  subst; auto.
Qed.

Definition Minus (N:t) : t :=
  match N with
    None => None
  | Some (ITV x y H) => Some (ITV (minus_Z' y) (minus_Z' x) (minus_Z'_inv_le _ _ H))
  end.

Lemma Minus_correct : forall N n,
  gamma N n -> gamma (Minus N) (-n).
Proof.
  destruct 1.
  destruct i; simpl in *; constructor; simpl.
  inversion_clear H0; simpl; constructor; omega.
  inversion_clear H; simpl; constructor; omega.
Qed.

Definition add' (x y:Z') : Z' :=
  match (x,y) with
   | (infty_pos,infty_neg) => infty_pos
   | (infty_pos,_) => infty_pos
   | (infty_neg,infty_pos) => infty_pos
   | (infty_neg,_) => infty_neg
   | (_,infty_neg) => infty_neg
   | (_,infty_pos) => infty_pos
   | (z a,z b) => (z (a+b))
  end.

Inductive optionprop (A:Set) (P:A->Prop) : option A -> Prop :=
 optionprop_def : forall a, P a -> optionprop A P (Some a).
Implicit Arguments optionprop [A].
Inductive optionprop2 (A:Set) (P:A->A->Prop) : option A -> option A -> Prop :=
 optionprop2_def : forall a1 a2, P a1 a2 -> optionprop2 A P (Some a1) (Some a2).
Implicit Arguments optionprop2 [A].

Lemma add'_monotone_lt : forall x y t u, 
  x <' t -> y <' u -> (add' x y) <' (add' t u).
Proof.
  destruct 1; destruct 1; unfold add'; repeat constructor.
  omega.
Qed.

Lemma add'_monotone : forall x y t u, 
  x <=' t-> y <=' u -> (add' x y) <=' (add' t u).
Proof.
  destruct 1; destruct 1; subst.
  destruct H; destruct H0; unfold add'; repeat constructor.
  omega.
  destruct H; destruct u; unfold add'; auto; try constructor;
  try (repeat constructor; intuition;fail); auto.
  destruct H0; destruct t0; unfold add';  auto; try constructor;
  try (repeat constructor; intuition;fail); auto.
  auto.
Qed.

Lemma add'_monotone1 : forall x y t u, 
  ZleI x t -> ZleI y u -> ZleI (x+y) (add' t u).
Proof.
  destruct 1; destruct 1; unfold add'; repeat constructor.
  omega.
Qed.

Lemma add'_monotone2 : forall x y t u, 
  ZgeI x t -> ZgeI y u -> ZgeI (x+y) (add' t u).
Proof.
  destruct 1; destruct 1; unfold add'; repeat constructor.
  omega.
Qed.

Definition sub' (x y:Z') : Z' :=
  match (x,y) with
   | (infty_pos,infty_pos) => infty_pos
   | (infty_pos,_) => infty_pos
   | (infty_neg,infty_neg) => infty_neg
   | (infty_neg,_) => infty_neg
   | (_,infty_neg) => infty_pos
   | (_,infty_pos) => infty_neg
   | (z a,z b) => (z (a-b))
  end.

Lemma sub'_monotone : forall x y t u, 
  x <=' t-> y <=' u -> (sub' x u) <=' (sub' t y).
Proof.
  destruct 1; destruct 1; subst.
  destruct H; destruct H0; unfold sub'; repeat constructor.
  omega.
  destruct H; destruct u; unfold sub'; auto; try constructor;
  try (repeat constructor; intuition;fail); auto.
  destruct H0; destruct t0; unfold sub'; auto; try constructor;
  try (repeat constructor; intuition;fail); auto.
  auto.
Qed.

Lemma sub'_monotone1 : forall x y t u, 
  ZleI x t -> ZgeI y u -> ZleI (x-y) (sub' t u).
Proof.
  destruct 1; destruct 1; unfold sub'; repeat constructor.
  omega.
Qed.

Lemma sub'_monotone2 : forall x y t u, 
  ZgeI x t -> ZleI y u -> ZgeI (x-y) (sub' t u).
Proof.
  destruct 1; destruct 1; unfold sub'; repeat constructor.
  omega.
Qed.

Inductive Zcomp0 (x:Z) : Set :=
 | Zcomp0_lt : x<0 -> Zcomp0 x
 | Zcomp0_0 : x=0 -> Zcomp0 x
 | Zcomp0_gt : 0<x -> Zcomp0 x.
Lemma Zcomp0_dec : forall z, Zcomp0 z.
Proof.
  destruct z0.
  constructor 2; auto.
  constructor 3; constructor.
  constructor 1; constructor.
Qed.

Inductive Case_Z' : Z' -> Set :=
   Case_Z'_infty_neg : Case_Z' infty_neg
 | Case_Z'_z_neg : forall x, x<0 -> Case_Z' (z x)
 | Case_Z'_z_0 : Case_Z' (z 0)
 | Case_Z'_z_pos : forall x, x>0 -> Case_Z' (z x)
 | Case_Z'_infty_pos : Case_Z' infty_pos.
Lemma Case_Z'_dec : forall z, Case_Z' z.
Proof.
  destruct z0.
  destruct z0.
  constructor 3.
  constructor 4; constructor.
  constructor 2; constructor.
  constructor.
  constructor.
Qed.

Definition mult' (x y:Z') : Z' :=
  match (x,y) with
   | (infty_pos,infty_pos) => infty_pos
   | (infty_pos,infty_neg) => infty_neg
   | (infty_pos,z b) => 
       match Zcomp0_dec b with
         Zcomp0_lt _ => infty_neg
       | Zcomp0_0 _ => z 0
       | Zcomp0_gt _ => infty_pos
       end
   | (infty_neg,infty_pos) => infty_neg
   | (infty_neg,infty_neg) => infty_pos
   | (infty_neg,z b) => 
       match Zcomp0_dec b with
         Zcomp0_lt _ => infty_pos
       | Zcomp0_0 _ => z 0
       | Zcomp0_gt _ => infty_neg
       end
   | (z a,infty_pos) => 
       match Zcomp0_dec a with
         Zcomp0_lt _ => infty_neg
       | Zcomp0_0 _ => z 0
       | Zcomp0_gt _ => infty_pos
       end
   | (z a,infty_neg) => 
       match Zcomp0_dec a with
         Zcomp0_lt _ => infty_pos
       | Zcomp0_0 _ => z 0
       | Zcomp0_gt _ => infty_neg
       end
   | (z a,z b) => z (a*b)
  end.

Definition mult_min' (a b c d:Z') :=
  (min' (min' (mult' a c) (mult' b d))
        (min' (mult' b c) (mult' a d))).
Definition mult_max' (a b c d:Z') :=
  (max' (max' (mult' a c) (mult' b d))
        (max' (mult' b c) (mult' a d))).

Hint Resolve order_Z'_eq_trans.

Lemma mult'_monotone : forall a b c d, 
  (mult_min' a b c d) <=' (mult_max' a b c d).
Proof.
  unfold mult_max', mult_min'; intros.
  repeat (apply case_min');
  repeat (apply case_max'); eauto.
Qed.

Ltac prod0 x y := 
  cut (x*y=0); [intros HHH; destruct (Zmult_integral _ _ HHH)|omega].

Lemma min'_n_n : forall x, min' x x = x.
Proof.
  intros; apply case_min'; auto.
Qed.

Lemma Zmin_Zmin_1 : forall x y,
  Zmin x (Zmin y x) = Zmin y x.
Proof.
  intros; repeat (apply case_Zmin); auto with zarith.
Qed.

Lemma Zmin_Zmin_2 : forall x y,
  Zmin x (Zmin x y) = Zmin x y.
Proof.
  intros; repeat (apply case_Zmin); auto with zarith.
Qed.


Lemma prod_le_0 : forall x y, 0 <= x*y -> 
 (x<=0/\y<=0) \/ (x>=0/\y>=0).
Proof.
  intros.
  destruct (Zpos_or_not x); destruct (Zpos_or_not y); intuition;
  try (assert (x=0); [omega|subst]);
  try (assert (y=0); [omega|subst]).
  assert (x*y<=0).
  auto with zarith sign.
  prod0 x y; subst; omega.
  assert (x*y<=0).
  auto with zarith sign.
  prod0 x y; subst; omega.
Qed.

Lemma prod_ge_0 : forall x y, x*y <= 0 -> 
 (x<=0/\y>=0) \/ (x>=0/\y<=0).
Proof.
  intros.
  destruct (Zpos_or_not x); destruct (Zpos_or_not y); intuition;
  try (assert (x=0); [omega|subst]);
  try (assert (y=0); [omega|subst]).
  assert (0<=x*y).
  auto with zarith sign.
  prod0 x y; subst; omega.
  assert (0<=x*y).
  auto with zarith sign.
  prod0 x y; subst; omega.
Qed.

Ltac prodLe0 := 
  match goal with
   [id:0 <= (?x*?y)|- _] => 
        destruct (prod_le_0 x y id) as [HH|HH];
        destruct HH; clear id
 | [id:(?x*?y) <= 0|- _] => 
        destruct (prod_ge_0 x y id) as [HH|HH];
        destruct HH; clear id
  end.

Lemma mult_min'_monotone : forall a b c d x y,
  ZgeI x a -> ZleI x b -> 
  ZgeI y c -> ZleI y d -> ZgeI (x*y) (mult_min' a b c d).
Proof.
  unfold mult_min'; intros.
  destruct a as [a| | ]; inversion_clear H;
  destruct b as [b| | ]; inversion_clear H0;
  destruct c as [c| | ]; inversion_clear H1;
  destruct d as [d| | ]; inversion_clear H2; unfold mult';
  try (unfold min'; constructor;
       destruct (Mult_interval_correct_min_max a b c d x y); intuition)
  ||
  (repeat (case Zcomp0_dec; intros); subst; 
  try (apply False_ind; omega) || constructor; 
  try (assert (x=0); [omega|subst]);
  try (assert (y=0); [omega|subst]);
  repeat rewrite Zmult_0_l;
  repeat rewrite Zmult_0_r;
  repeat rewrite min'_n_n; unfold min'; constructor;
  try rewrite Zmin_Zmin_1;
  try rewrite Zmin_Zmin_2; 
  auto).
  apply case_Zmin; intros; try omega.
  repeat  prodLe0; try (apply False_ind; omega).
  destruct (Zpos_or_not y).
  apply Zle_trans with 0; auto with sign zarith.
  apply Zmult_ineq1; apply Zmult_bounds1; split; try omega.
  prodLe0; try (apply False_ind; omega).
  apply sign_rule2; omega.
  destruct (Zpos_or_not c).
  rewrite Zmin_l; auto with sign zarith.
  rewrite Zmin_r; auto with sign zarith.
  destruct (Zpos_or_not y).
  apply Zle_trans with 0; auto with sign zarith.
  apply Zmult_ineq1.
  apply Zmult_bounds1; split; try omega.
  destruct (Zpos_or_not d).
  rewrite Zmin_r; auto with sign zarith.
  destruct (Zpos_or_not y).
  apply Zmult_ineq1; auto with sign zarith.
  apply Zle_trans with 0; auto with sign zarith.
  rewrite Zmin_l; auto with sign zarith.
  destruct (Zpos_or_not d).
  rewrite Zmin_r; auto with sign zarith.
  destruct (Zpos_or_not y).
  apply Zmult_ineq2; auto with sign zarith.
  apply Zle_trans with 0; auto with sign zarith.
  rewrite Zmin_l; auto with sign zarith.
  destruct (Zpos_or_not a).
  rewrite Zmin_l; auto with sign zarith.
  rewrite Zmin_r; auto with sign zarith.
  destruct (Zpos_or_not x).
  apply Zle_trans with 0; auto with sign zarith.
  apply Zmult_ineq2; auto with sign zarith.
  destruct (Zpos_or_not a).
  rewrite Zmin_l; auto with sign zarith.
  rewrite Zmin_r; auto with sign zarith.
  destruct (Zpos_or_not x).
  apply Zle_trans with 0; auto with sign zarith.
  auto with sign zarith.
  auto with sign zarith.
  auto with sign zarith.
  auto with sign zarith.
  auto with sign zarith.
  destruct (Zpos_or_not b).
  rewrite Zmin_r; auto with sign zarith.
  destruct (Zpos_or_not x).
  auto with sign zarith.
  apply Zle_trans with 0; auto with sign zarith.
  rewrite Zmin_l; auto with sign zarith.
  destruct (Zpos_or_not b).
  rewrite Zmin_l; auto with sign zarith.
  destruct (Zpos_or_not x).
  auto with sign zarith.
  apply Zle_trans with 0; auto with sign zarith.
  rewrite Zmin_r; auto with sign zarith.
  auto with sign zarith.
  auto with sign zarith.
  auto with sign zarith.
  auto with sign zarith.
Qed.

Lemma order_Z'_eq_geI : forall x y, ZgeI x y -> y <=' (z x).
Proof.
  destruct 1; auto.
Qed.

Lemma geI_order_Z'_eq : forall x y, y <=' (z x) -> ZgeI x y.
Proof.
  destruct y; destruct 1; try constructor.
  inversion_clear H; omega.
  injection H; intros; omega.
  inversion_clear H.
  inversion_clear H.
Qed.

Lemma order_Z'_eq_leI : forall x y, ZleI x y -> (z x) <=' y.
Proof.
  destruct 1; auto.
Qed.

Lemma leI_order_Z'_eq : forall x y, (z x) <=' y -> ZleI x y.
Proof.
  destruct y; destruct 1; try constructor.
  inversion_clear H; try omega.
  injection H; intros; omega.
  inversion_clear H.
  inversion_clear H.
Qed.

Hint Constructors ZgeI ZleI.
Hint Resolve geI_order_Z'_eq leI_order_Z'_eq.

Lemma order_Z'_eq_infty_pos' : forall x,
 infty_pos <=' x -> x = infty_pos.
Proof.
  destruct 1; auto.
  inversion_clear H; auto.
Qed.
Lemma order_Z'_eq_infty_neg' : forall x,
 x <=' infty_neg -> x = infty_neg.
Proof.
  destruct 1; auto.
  inversion_clear H; auto.
Qed.

Ltac simp_order := 
  match goal with
    [id : infty_neg <=' _ |- _] => clear id
  | [id : infty_pos <=' _ |- _] => generalize (order_Z'_eq_infty_pos' _ id); clear id; intros id; subst
  | [id : _ <=' infty_pos |- _] => clear id
  | [id : _ <=' infty_neg |- _] => generalize (order_Z'_eq_infty_neg' _ id); clear id; intros id; subst
  end.

Lemma min'_l : forall x y, x<='y -> min' x y = x.
Proof.
  intros; apply case_min'; intros; auto.
  inversion_clear H in H0; inversion_clear H0; auto.
  inversion_clear H in H1; inversion_clear H1.
  apply False_ind; omega.
Qed.
Lemma min'_r : forall x y, y<='x -> min' x y = y.
Proof.
  intros; apply case_min'; intros; auto.
  inversion_clear H in H0; inversion_clear H0; auto.
  inversion_clear H in H1; inversion_clear H1.
  apply False_ind; omega.
Qed.

Lemma order_Z'_eq_prop : forall x y, z x <=' z y -> x <= y.
Proof.
  destruct 1; inversion_clear H; omega.
Qed.

Lemma mult'_x_pos1 : forall a x,
  a <=' z x -> x < 0 -> mult' a infty_pos = infty_neg.
Proof.
  unfold mult'; intros.
  destruct a; auto.
  generalize (order_Z'_eq_prop _ _ H); clear H; intros.
  case Zcomp0_dec; intros; auto; apply False_ind; omega.
  simp_order.
  discriminate H.
Qed.

Lemma le_min'1 : forall a b c d, mult_min' a b c d <=' mult' a c.
Proof.
  unfold mult_min'; intros.
  apply order_Z'_eq_trans with (min' (mult' a c) (mult' b d));
  apply le_min'_l.
Qed.
Lemma le_min'2 : forall a b c d, mult_min' a b c d <=' mult' a d.
Proof.
  unfold mult_min'; intros.
  apply order_Z'_eq_trans with (min' (mult' b c) (mult' a d));
  apply le_min'_l || apply le_min'_r.
Qed.
Lemma le_min'3 : forall a b c d, mult_min' a b c d <=' mult' b c.
Proof.
  unfold mult_min'; intros.
  apply order_Z'_eq_trans with (min' (mult' b c) (mult' a d));
  apply le_min'_l || apply le_min'_r.
Qed.
Lemma le_min'4 : forall a b c d, mult_min' a b c d <=' mult' b d.
Proof.
  unfold mult_min'; intros.
  apply order_Z'_eq_trans with (min' (mult' a c) (mult' b d));
  apply le_min'_l || apply le_min'_r.
Qed.
Hint Resolve le_min'1 le_min'2 le_min'3 le_min'4.

Lemma mult'_pos1 : forall a b y, z 0 <=' y -> a <=' b -> mult' a y <=' mult' b y.
Proof.
  destruct 1.
  inversion_clear H; unfold mult'; intros.
  destruct a; destruct b; destruct H; inversion_clear H;
  try (destruct Zcomp0_dec; try (apply False_ind; omega); auto).
  repeat constructor; apply Zmult_gt_0_lt_compat_r; omega.
  auto.
  destruct a; destruct b; destruct H; inversion_clear H; 
  repeat (case Zcomp0_dec; intro); auto;
  try (apply False_ind; omega).
  subst.
  intros H; unfold mult'; destruct a; destruct b; destruct H; inversion_clear H;
  repeat rewrite Zmult_0_l;
  repeat rewrite Zmult_0_r; 
  try (destruct Zcomp0_dec; try (apply False_ind; omega); auto);
  auto.
Qed.
Lemma Zlt_neg : forall x y, -x < -y -> y < x.
Proof.
  intros; omega.
Qed.

Lemma mult'_neg1 : forall a b y, y <=' z 0 -> a <=' b -> mult' b y <=' mult' a y.
Proof.
  destruct 1.
  inversion_clear H; unfold mult'; intros.
  destruct a; destruct b; destruct H; inversion_clear H;
  try (destruct Zcomp0_dec; try (apply False_ind; omega); auto).
  repeat constructor; apply Zlt_neg.
  replace (-(z0*a0)) with (z0*(-a0)); [idtac|ring].
  replace (-(z1*a0)) with (z1*(-a0)); [idtac|ring].
  apply Zmult_gt_0_lt_compat_r; omega.
  auto.
  destruct a; destruct b; destruct H; inversion_clear H; 
  repeat (case Zcomp0_dec; intro); auto;
  try (apply False_ind; omega).
  subst.
  intros H; unfold mult'; destruct a; destruct b; destruct H; inversion_clear H;
  repeat rewrite Zmult_0_l;
  repeat rewrite Zmult_0_r; 
  try (destruct Zcomp0_dec; try (apply False_ind; omega); auto);
  auto.
Qed.
Hint Resolve mult'_neg1 mult'_pos1.

Lemma mult'_l_0 : forall x, mult' x (z 0) = z 0.
Proof.
  unfold mult'; intros.
  destruct x;
  repeat rewrite Zmult_0_r; 
  repeat (case Zcomp0_dec; intro); 
  try (apply False_ind; omega); auto.
Qed.
Lemma mult'_r_0 : forall x, mult' (z 0) x = z 0.
Proof.
  unfold mult'; intros.
  destruct x;
  repeat rewrite Zmult_0_l; 
  repeat (case Zcomp0_dec; intro); 
  try (apply False_ind; omega); auto.
Qed.

Lemma mult_min'_monotone'1 : forall a b c d a',
  a' <=' a -> a <=' b -> c <=' d ->
  (mult_min' a' b c d) <=' (mult_min' a b c d).
Proof.
  intros; unfold mult_min' at 2 4; 
  repeat apply case_min'; intros; auto.
  destruct c as [v| | ].
  destruct (Zcomp0_dec v) as [H5|H5|H5].
  apply order_Z'_eq_trans with (mult' b (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' b (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' a' (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' a' infty_pos); auto with zarith.
  apply order_Z'_eq_trans with (mult' b infty_neg); auto with zarith.
  destruct c as [v| | ].
  destruct (Zcomp0_dec v) as [H5|H5|H5].
  apply order_Z'_eq_trans with (mult' b (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' b (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' a' (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' a' infty_pos); auto with zarith.
  apply order_Z'_eq_trans with (mult' b infty_neg); auto with zarith.
  destruct d as [v| | ].
  destruct (Zcomp0_dec v) as [H5|H5|H5].
  apply order_Z'_eq_trans with (mult' b (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' b (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' a' (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' a' infty_pos); auto with zarith.
  apply order_Z'_eq_trans with (mult' b infty_neg); auto with zarith.
  destruct d as [v| | ].
  destruct (Zcomp0_dec v) as [H5|H5|H5].
  apply order_Z'_eq_trans with (mult' b (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' b (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' a' (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' a' infty_pos); auto with zarith.
  apply order_Z'_eq_trans with (mult' b infty_neg); auto with zarith.
Qed.

Lemma mult_min'_monotone'2 : forall a b c d b',
  a <=' b -> b <=' b' -> c <=' d ->
  (mult_min' a b' c d) <=' (mult_min' a b c d).
Proof.
  intros; unfold mult_min' at 2 4; 
  repeat apply case_min'; intros; auto.
  destruct d as [v| | ].
  destruct (Zcomp0_dec v) as [H5|H5|H5].
  apply order_Z'_eq_trans with (mult' b' (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' b' (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' a (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' a infty_pos); auto with zarith.
  apply order_Z'_eq_trans with (mult' b' infty_neg); auto with zarith.
  destruct d as [v| | ].
  destruct (Zcomp0_dec v) as [H5|H5|H5].
  apply order_Z'_eq_trans with (mult' b' (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' b' (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' a (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' a infty_pos); auto with zarith.
  apply order_Z'_eq_trans with (mult' b' infty_neg); auto with zarith.
  destruct c as [v| | ].
  destruct (Zcomp0_dec v) as [H5|H5|H5].
  apply order_Z'_eq_trans with (mult' b' (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' b' (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' a (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' a infty_pos); auto with zarith.
  apply order_Z'_eq_trans with (mult' b' infty_neg); auto with zarith.
  destruct c as [v| | ].
  destruct (Zcomp0_dec v) as [H5|H5|H5].
  apply order_Z'_eq_trans with (mult' b' (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' b' (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' a (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' a infty_pos); auto with zarith.
  apply order_Z'_eq_trans with (mult' b' infty_neg); auto with zarith.
Qed.

Lemma min'_comm : forall x y, min' x y = min' y x.
Proof.
  unfold min'; intros.
  destruct x ;destruct y; auto.
  repeat (apply case_Zmin); intros; auto.
Qed.

Lemma mult'_comm : forall x y, mult' x y = mult' y x.
Proof.
  unfold mult'; intros; destruct x; destruct y;
  try (case Zcomp0_dec; intros); auto.
  rewrite Zmult_comm; auto.
Qed.

Lemma mult_min'_comm : forall a b c d, mult_min' a b c d = mult_min' c d a b.
Proof.
  intros; unfold mult_min'.
  repeat rewrite (mult'_comm c).
  repeat rewrite (mult'_comm d).
  rewrite (min'_comm (mult' a d)); auto.  
Qed.

Lemma mult_min'_monotone' : forall a b c d a' b' c' d',
  a' <=' a -> a <=' b -> b <=' b' -> 
  c' <=' c -> c <=' d -> d <=' d' ->
  (mult_min' a' b' c' d') <=' (mult_min' a b c d).
Proof.
  generalize order_Z'_eq_trans; intros.
  intros.
  apply order_Z'_eq_trans with (mult_min' a b' c' d').
  apply mult_min'_monotone'1; eauto.
  apply order_Z'_eq_trans with (mult_min' a b c' d').
  apply mult_min'_monotone'2; eauto.
  repeat rewrite (mult_min'_comm a).
  apply order_Z'_eq_trans with (mult_min' c d' a b).
  apply mult_min'_monotone'1; eauto.
  apply mult_min'_monotone'2; eauto.
Qed.

Lemma le_max'1 : forall a b c d, mult' a c <=' mult_max' a b c d.
Proof.
  unfold mult_max'; intros.
  apply order_Z'_eq_trans with (max' (mult' a c) (mult' b d));
  apply le_max'_l.
Qed.
Lemma le_max'2 : forall a b c d, mult' a d <=' mult_max' a b c d.
Proof.
  unfold mult_max'; intros.
  apply order_Z'_eq_trans with (max' (mult' b c) (mult' a d));
  apply le_max'_l || apply le_max'_r.
Qed.
Lemma le_max'3 : forall a b c d, mult' b c <=' mult_max' a b c d.
Proof.
  unfold mult_max'; intros.
  apply order_Z'_eq_trans with (max' (mult' b c) (mult' a d));
  apply le_max'_l || apply le_max'_r.
Qed.
Lemma le_max'4 : forall a b c d, mult' b d <=' mult_max' a b c d.
Proof.
  unfold mult_max'; intros.
  apply order_Z'_eq_trans with (max' (mult' a c) (mult' b d));
  apply le_max'_l || apply le_max'_r.
Qed.
Hint Resolve le_max'1 le_max'2 le_max'3 le_max'4.

Lemma mult_max'_monotone'1 : forall a b c d a',
  a' <=' a -> a <=' b -> c <=' d ->
  (mult_max' a b c d) <=' (mult_max' a' b c d).
Proof.
  intros; unfold mult_max' at 1 3; 
  repeat apply case_max'; intros; auto.
  destruct c as [v| | ].
  destruct (Zcomp0_dec v) as [H5|H5|H5].
  apply order_Z'_eq_trans with (mult' a' (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' b (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' b (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' b infty_pos); auto with zarith.
  apply order_Z'_eq_trans with (mult' a' infty_neg); auto with zarith.
  destruct c as [v| | ].
  destruct (Zcomp0_dec v) as [H5|H5|H5].
  apply order_Z'_eq_trans with (mult' a' (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' b (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' b (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' b infty_pos); auto with zarith.
  apply order_Z'_eq_trans with (mult' a' infty_neg); auto with zarith.
  destruct d as [v| | ].
  destruct (Zcomp0_dec v) as [H5|H5|H5].
  apply order_Z'_eq_trans with (mult' a' (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' b (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' b (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' b infty_pos); auto with zarith.
  apply order_Z'_eq_trans with (mult' a' infty_neg); auto with zarith.
  destruct d as [v| | ].
  destruct (Zcomp0_dec v) as [H5|H5|H5].
  apply order_Z'_eq_trans with (mult' a' (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' b (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' b (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' b infty_pos); auto with zarith.
  apply order_Z'_eq_trans with (mult' a' infty_neg); auto with zarith.
Qed.

Lemma mult_max'_monotone'2 : forall a b c d b',
  a <=' b -> b <=' b' -> c <=' d ->
  (mult_max' a b c d) <=' (mult_max' a b' c d).
Proof.
  intros; unfold mult_max' at 1 3; 
  repeat apply case_max'; intros; auto.
  destruct d as [v| | ].
  destruct (Zcomp0_dec v) as [H5|H5|H5].
  apply order_Z'_eq_trans with (mult' a (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' b' (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' b' (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' b' infty_pos); auto with zarith.
  apply order_Z'_eq_trans with (mult' a infty_neg); auto with zarith.
  destruct d as [v| | ].
  destruct (Zcomp0_dec v) as [H5|H5|H5].
  apply order_Z'_eq_trans with (mult' a (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' b' (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' b' (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' b' infty_pos); auto with zarith.
  apply order_Z'_eq_trans with (mult' a infty_neg); auto with zarith.
  destruct c as [v| | ].
  destruct (Zcomp0_dec v) as [H5|H5|H5].
  apply order_Z'_eq_trans with (mult' a (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' b' (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' b' (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' b' infty_pos); auto with zarith.
  apply order_Z'_eq_trans with (mult' a infty_neg); auto with zarith.
  destruct c as [v| | ].
  destruct (Zcomp0_dec v) as [H5|H5|H5].
  apply order_Z'_eq_trans with (mult' a (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' b' (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' b' (z v)); auto with zarith.
  apply order_Z'_eq_trans with (mult' b' infty_pos); auto with zarith.
  apply order_Z'_eq_trans with (mult' a infty_neg); auto with zarith.
Qed.

Lemma max'_comm : forall x y, max' x y = max' y x.
Proof.
  unfold max'; intros.
  destruct x ;destruct y; auto.
  repeat (apply case_Zmax); intros; auto.
Qed.

Lemma mult_max'_comm : forall a b c d, mult_max' a b c d = mult_max' c d a b.
Proof.
  intros; unfold mult_max'.
  repeat rewrite (mult'_comm c).
  repeat rewrite (mult'_comm d).
  rewrite (max'_comm (mult' a d)); auto.  
Qed.

Lemma mult_max'_monotone' : forall a b c d a' b' c' d',
  a' <=' a -> a <=' b -> b <=' b' -> 
  c' <=' c -> c <=' d -> d <=' d' ->
  (mult_max' a b c d) <=' (mult_max' a' b' c' d').
Proof.
  generalize order_Z'_eq_trans; intros.
  intros.
  apply order_Z'_eq_trans with (mult_max' a' b c d).
  apply mult_max'_monotone'1; eauto.
  apply order_Z'_eq_trans with (mult_max' a' b' c d).
  apply mult_max'_monotone'2; eauto.
  repeat rewrite (mult_max'_comm a').
  apply order_Z'_eq_trans with (mult_max' c' d a' b').
  apply mult_max'_monotone'1; eauto.
  apply mult_max'_monotone'2; eauto.
Qed.

Lemma max'_n_n : forall x, max' x x = x.
Proof.
  intros; apply case_max'; auto.
Qed.

Lemma Zmax_Zmax_1 : forall x y,
  Zmax (Zmax y x) x = Zmax y x.
Proof.
  intros; repeat (apply case_Zmax); auto with zarith.
Qed.

Lemma Zmax_Zmax_2 : forall x y,
  Zmax (Zmax x y) x = Zmax x y.
Proof.
  intros; repeat (apply case_Zmax); auto with zarith.
Qed.

Lemma Zmax_l : forall x y, y <= x -> Zmax x y = x.
Proof.
  intros; apply case_Zmax; auto with zarith.
Qed.

Lemma Zmax_r : forall x y, x <= y -> Zmax x y = y.
Proof.
  intros; apply case_Zmax; auto with zarith.
Qed.



Lemma mult_max'_monotone : forall a b c d x y,
  ZgeI x a -> ZleI x b -> 
  ZgeI y c -> ZleI y d -> ZleI (x*y) (mult_max' a b c d).
Proof.
  unfold mult_max'; intros.
  destruct a as [a| | ]; inversion_clear H;
  destruct b as [b| | ]; inversion_clear H0;
  destruct c as [c| | ]; inversion_clear H1;
  destruct d as [d| | ]; inversion_clear H2; unfold mult';
  try (unfold max'; constructor;
       destruct (Mult_interval_correct_min_max a b c d x y); intuition)
  ||
  (repeat (case Zcomp0_dec; intros); subst; 
  try (apply False_ind; omega) || constructor; 
  try (assert (x=0); [omega|subst]);
  try (assert (y=0); [omega|subst]);
  repeat rewrite Zmult_0_l;
  repeat rewrite Zmult_0_r;
  repeat rewrite max'_n_n; unfold max'; constructor;
  try rewrite Zmax_Zmax_1;
  try rewrite Zmax_Zmax_2; 
  try (match goal with
         [|- ?x * ?y <= 0] => auto with sign zarith 
       | [|- _ * _ <= _ * _] => auto with sign zarith end);
  auto).
  destruct (Zpos_or_not c).
  rewrite Zmax_r; auto with sign zarith.
  rewrite Zmax_l; auto with sign zarith.
  destruct (Zpos_or_not y).
  apply Zle_trans with 0; auto with sign zarith.
  auto with sign zarith.
  destruct (Zpos_or_not y).
  apply Zle_trans with 0; auto with sign zarith.
  rewrite Zmax_l; auto with sign zarith.
  destruct (Zpos_or_not y).
  rewrite Zmax_r; auto with sign zarith.
  apply Zle_trans with 0; auto with sign zarith.
  destruct (Zpos_or_not y).
  rewrite Zmax_l; auto with sign zarith.
  destruct (Zpos_or_not d).
  rewrite Zmax_l; auto with sign zarith.
  rewrite Zmax_r; auto with sign zarith.
  destruct (Zpos_or_not a).
  rewrite Zmax_r; auto with sign zarith.
  rewrite Zmax_l; auto with sign zarith.
  destruct (Zpos_or_not x).
  apply Zle_trans with 0; auto with sign zarith.
  auto with sign zarith.
  destruct (Zpos_or_not a).
  rewrite Zmax_r; auto with sign zarith.
  rewrite Zmax_l; auto with sign zarith.
  destruct (Zpos_or_not x).
  apply Zle_trans with 0; auto with sign zarith.
  auto with sign zarith.
  destruct (Zpos_or_not b).
  rewrite Zmax_r; auto with sign zarith.
  destruct (Zpos_or_not x).
  auto with sign zarith.
  apply Zle_trans with 0; auto with sign zarith.
  rewrite Zmax_l; auto with sign zarith.
  destruct (Zpos_or_not b).
  rewrite Zmax_l; auto with sign zarith.
  destruct (Zpos_or_not x).
  auto with sign zarith.
  apply Zle_trans with 0; auto with sign zarith.
  rewrite Zmax_r; auto with sign zarith.
Qed.

Definition sem_op_add (N1 N2:t) : t := 
  match N1,N2 with
    Some i1,Some i2 => 
     Some (ITV (add' (min i1) (min i2))
               (add' (max i1) (max i2))
               (add'_monotone _ _ _ _ (min_order_max i1) (min_order_max i2)))
  | _,_ => None
  end.

Definition sem_op_sub (N1 N2:t) : t := 
  match N1,N2 with
  | Some i1,Some i2 => 
     Some (ITV (sub' (min i1) (max i2))
               (sub' (max i1) (min i2))
               (sub'_monotone _ _ _ _ (min_order_max i1) (min_order_max i2)))
  | _,_ => None
  end.

Definition sem_op_mult (N1 N2:t) : t := 
  match N1,N2 with
  | Some i1,Some i2 => 
     Some (ITV (mult_min' (min i1) (max i1) (min i2) (max i2))
               (mult_max' (min i1) (max i1) (min i2) (max i2))
               (mult'_monotone _ _ _ _))
  | _,_ => None
  end.

Lemma sem_op_add_correct : forall N1 N2 n1 n2 n,
  gamma N1 n1 -> gamma N2 n2 -> 
  n = n1 + n2 ->
  gamma (sem_op_add N1 N2) n.
Proof.
  intros N1 N2 n1 n2 n H; inversion_clear H.
  intros H2; inversion_clear H2.
  intros H4; inversion_clear H4; 
  unfold sem_op_add; constructor; simpl.
  apply add'_monotone2; auto.
  apply add'_monotone1; auto.
Qed.

Lemma sem_op_sub_correct : forall N1 N2 n1 n2 n,
  gamma N1 n1 -> gamma N2 n2 -> 
  n = n1 - n2 ->
  gamma (sem_op_sub N1 N2) n.
Proof.
  intros N1 N2 n1 n2 n H; inversion_clear H.
  intros H2; inversion_clear H2.
  intros H4; inversion_clear H4; 
  unfold sem_op_sub; constructor; simpl.
  apply sub'_monotone2; auto.
  apply sub'_monotone1; auto.
Qed.

Lemma sem_op_mult_correct : forall N1 N2 n1 n2 n,
  gamma N1 n1 -> gamma N2 n2 -> 
  n = n1 * n2 ->
  gamma (sem_op_mult N1 N2) n.
Proof.
  intros N1 N2 n1 n2 n H; inversion_clear H.
  intros H2; inversion_clear H2.
  intros H4; inversion_clear H4; 
  unfold sem_op_mult; constructor; simpl.
  apply mult_min'_monotone; auto.
  apply mult_max'_monotone; auto.
Qed.

Lemma backsem_add1 : forall (x y:Z) (a f:Z'),
 ZgeI (x+y) a -> ZleI y f ->
 ZgeI x (sub' a f).
Proof.
  intros.
  inversion_clear H0 in H; unfold sub'.
  inversion_clear H; constructor.
  inversion_clear H; constructor.
  omega.
Qed.

Lemma backsem_add2 : forall (x y:Z) (b e:Z'),
 ZleI (x+y) b -> ZgeI y e -> 
 ZleI x (sub' b e).
Proof.
  intros.
  inversion_clear H0 in H; unfold sub'.
  inversion_clear H; constructor.
  inversion_clear H; constructor.
  omega.
Qed.

Lemma not_ZgeI_ZleI_2 : forall a b x,
  ZgeI x b ->
  ZleI x a ->
  ~ a <' b.
Proof.
  red; intros.
  inversion_clear H in H0 H1; inversion_clear H1 in H0; inversion_clear H0.
  omega.
Qed.

Definition backsem_add_int (a b c d e f:Z') : t :=
  let max := min' d (sub' b e) in
  let min := max' c (sub' a f) in
    match order_Z'_dec max min with
      left _ => None
    | right h => Some (ITV min max (not_order_Z' _ _ h))
   end.


Definition backsem_add (N N1 N2:t) : t * t :=
      match N,N1,N2 with
        None,_,_ => (None,None)
      | _,None,_ => (None,None)
      | _,_,None => (None,None)
      | Some (ITV a b h1),Some (ITV c d h2),Some (ITV e f h3) => 
          (backsem_add_int a b c d e f,backsem_add_int a b e f c d)
      end.

Lemma backsem_add_int_correct : forall (x y:Z) (a b c d e f:Z'),
 ZgeI (x+y) a -> ZleI (x+y) b ->
 ZgeI x c -> ZleI x d ->
 ZgeI y e -> ZleI y f ->
 gamma (backsem_add_int a b c d e f) x.
Proof.
  intros; unfold backsem_add_int.
  assert (ZgeI x (max' c (sub' a f))).
  apply case_max'; intro; auto.
  apply backsem_add1 with y; auto.
  assert (ZleI x (min' d (sub' b e))).
  apply case_min'; intro; auto.
  apply backsem_add2 with y; auto.
  destruct order_Z'_dec.
  elim (not_ZgeI_ZleI_2 _ _ _ H5 H6); auto.
  constructor; simpl; auto.
Qed.

Lemma backsem_add_correct : forall N N1 N2 n1 n2 n N1' N2',
  gamma N1 n1 -> gamma N2 n2 ->
  n = n1 + n2 -> gamma N n ->
  backsem_add N N1 N2 = (N1',N2') ->
  gamma N1' n1 /\ gamma N2' n2.
Proof.
  unfold backsem_add; intros until 4.
  inversion_clear H2; inversion_clear H; inversion_clear H0.
  unfold backsem_add; intros.
  destruct i; destruct i0; destruct i1; simpl in *.
  injection H0; intros; subst; clear H0; auto.
  split.  
  apply backsem_add_int_correct with n2; auto.
  apply backsem_add_int_correct with n1; auto.
  rewrite Zplus_comm; auto.
  rewrite Zplus_comm; auto.
Qed.

Lemma backsem_sub1 : forall (x y:Z) (a e:Z'),
 ZgeI (x-y) a -> ZgeI y e ->
 ZgeI x (add' a e).
Proof.
  intros.
  inversion_clear H0 in H; unfold add'.
  inversion_clear H; constructor.
  inversion_clear H; constructor.
  omega.
Qed.

Lemma backsem_sub2 : forall (x y:Z) (b f:Z'),
 ZleI (x-y) b -> ZleI y f -> 
 ZleI x (add' b f).
Proof.
  intros.
  inversion_clear H0 in H; unfold add'.
  inversion_clear H; constructor.
  inversion_clear H; constructor.
  omega.
Qed.

Definition backsem_sub_int (a b c d e f:Z') : t :=
  let max := min' d (add' b f) in
  let min := max' c (add' a e) in
    match order_Z'_dec max min with
      left _ => None
    | right h => Some (ITV min max (not_order_Z' _ _ h))
   end.

Lemma backsem_sub3 : forall (x y:Z) (a e:Z'),
 ZgeI (x-y) a -> ZleI x e ->
 ZleI y (sub' e a).
Proof.
  intros.
  inversion_clear H0 in H; unfold sub'.
  inversion_clear H; constructor.
  inversion_clear H; constructor.
  omega.
Qed.

Lemma backsem_sub4 : forall (x y:Z) (b f:Z'),
 ZleI (x-y) b -> ZgeI x f -> 
 ZgeI y (sub' f b).
Proof.
  intros.
  inversion_clear H0 in H; unfold sub'.
  inversion_clear H; constructor.
  inversion_clear H; constructor.
  omega.
Qed.

Definition backsem_sub_int' (a b c d e f:Z') : t :=
  let max := min' f (sub' d a) in
  let min := max' e (sub' c b) in
    match order_Z'_dec max min with
      left _ => None
    | right h => Some (ITV min max (not_order_Z' _ _ h))
   end.

Definition backsem_sub (N N1 N2:t) : t * t :=
      match N,N1,N2 with
        None,_,_ => (None,None)
      | _,None,_ => (None,None)
      | _,_,None => (None,None)
      | Some (ITV a b h1),Some (ITV c d h2),Some (ITV e f h3) => 
          (backsem_sub_int a b c d e f,
           backsem_sub_int' a b  c d e f)
      end.

Lemma backsem_sub_int_correct : forall (x y:Z) (a b c d e f:Z'),
 ZgeI (x-y) a -> ZleI (x-y) b ->
 ZgeI x c -> ZleI x d ->
 ZgeI y e -> ZleI y f ->
 gamma (backsem_sub_int a b c d e f) x.
Proof.
  intros; unfold backsem_sub_int.
  assert (ZgeI x (max' c (add' a e))).
  apply case_max'; intro; auto.
  apply backsem_sub1 with y; auto.
  assert (ZleI x (min' d (add' b f))).
  apply case_min'; intro; auto.
  apply backsem_sub2 with y; auto.
  destruct order_Z'_dec.
  elim (not_ZgeI_ZleI_2 _ _ _ H5 H6); auto.
  constructor; simpl; auto.
Qed.

Lemma backsem_sub_int'_correct : forall (x y:Z) (a b c d e f:Z'),
 ZgeI (x-y) a -> ZleI (x-y) b ->
 ZgeI x c -> ZleI x d ->
 ZgeI y e -> ZleI y f ->
 gamma (backsem_sub_int' a b c d e f) y.
Proof.
  intros; unfold backsem_sub_int'.
  assert (ZgeI y (max' e (sub' c b))).
  apply case_max'; intro; auto.
  apply backsem_sub4 with x; auto.
  assert (ZleI y (min' f (sub' d a))).
  apply case_min'; intro; auto.
  apply backsem_sub3 with x; auto.
  destruct order_Z'_dec.
  elim (not_ZgeI_ZleI_2 _ _ _ H5 H6); auto.
  constructor; simpl; auto.
Qed.

Lemma backsem_sub_correct : forall N N1 N2 n1 n2 n N1' N2',
  gamma N1 n1 -> gamma N2 n2 ->
  n = n1 - n2 -> gamma N n ->
  backsem_sub N N1 N2 = (N1',N2') ->
  gamma N1' n1 /\ gamma N2' n2.
Proof.
  unfold backsem_sub; intros until 4.
  inversion_clear H2; inversion_clear H; inversion_clear H0.
  unfold backsem_sub; intros.
  destruct i; destruct i0; destruct i1; simpl in *.
  injection H0; intros; subst; clear H0; auto.
  split.  
  apply backsem_sub_int_correct with n2; auto.
  apply backsem_sub_int'_correct with n1; auto.
Qed.

Definition backsem_mult (N N1 N2:t) : t * t :=
      match N,N1,N2 with
        None,_,_ => (None,None)
      | _,None,_ => (None,None)
      | _,_,None => (None,None)
      | Some (ITV a b h1),Some (ITV c d h2),Some (ITV e f h3) =>  
          (Some (ITV c d h2),Some (ITV e f h3))
      end.

Lemma backsem_mult_correct : forall N N1 N2 n1 n2 n N1' N2',
  gamma N1 n1 -> gamma N2 n2 ->
  n = n1 * n2 -> gamma N n ->
  backsem_mult N N1 N2 = (N1',N2') ->
  gamma N1' n1 /\ gamma N2' n2.
Proof.
  unfold backsem_mult; intros until 4.
  inversion_clear H2; inversion_clear H; inversion_clear H0.
  destruct i; destruct i0; destruct i1; simpl in *.
  intros H0; injection H0; intros; subst; clear H0; auto.
  split; constructor; simpl; auto.  
Qed.

Definition const (n:Z) : t := 
  Some (ITV (z n) (z n) (order_Z'_eq_refl _)).

Lemma const_correct : forall n, gamma (const n) n.
Proof.
  unfold const; intros; constructor; simpl; constructor; omega.
Qed.

Lemma const_complete : forall n N,
  gamma N n -> (const n) ⊑♯ N.
Proof.
  unfold const; intros.
  inversion_clear H; simpl; constructor.
  destruct i; simpl in *.
  inversion_clear H0 in o; inversion_clear H1 in o; constructor; 
  simpl; try (repeat constructor; fail) ||
             (apply order_Z'_eq_Zle; auto; fail).
Qed.

Lemma top_correct : forall n, gamma (⊤♯) n.
Proof.
  simpl; constructor; simpl; constructor.
Qed.

Lemma bottom_empty : forall n, ~ gamma (⊥♯) n.
Proof.
  red; intros.
  inversion_clear H.
Qed.

Module Interval.

  Definition t := option interval.

  Instance Lattice : AbLattice.t t.

  Instance Gamma : Gamma.t (℘ Z) t.
  exists gamma.
  repeat intro; eapply gamma_monotone with N1; auto.
  repeat intro.
  destruct H; apply gamma_meet_morph; auto.
  Defined.

  Definition backtest_eq : t -> t -> t*t := BackTestEq.
  Lemma backtest_eq_correct : ∀N1 N2 N1' N2' : t, ∀n1 n2 : Z, 
    n1 = n2 ->
    gamma N1 n1 -> gamma N2 n2 ->
    backtest_eq N1 N2 = (N1', N2') ->
    gamma N1' n1 ∧ gamma N2' n2.
    Proof BackTestEq_correct.
    
  Definition backtest_lt : t -> t -> t*t := BackTestLt.
  Lemma backtest_lt_correct : ∀N1 N2 N1' N2' : t, ∀n1 n2 : Z, 
    n1 < n2 ->
    gamma N1 n1 -> gamma N2 n2 ->
    backtest_lt N1 N2 = (N1', N2') ->
    gamma N1' n1 ∧ gamma N2' n2.
  Proof BackTestLt_correct.

  Definition sem_add := sem_op_add.
  Definition sem_sub := sem_op_sub.
  Definition sem_mult := sem_op_mult.
  Definition sem_add_correct := sem_op_add_correct.
  Definition sem_sub_correct := sem_op_sub_correct.
  Definition sem_mult_correct := sem_op_mult_correct.

  Definition backsem_add := backsem_add.
  Definition backsem_add_correct := backsem_add_correct.

  Definition backsem_sub := backsem_sub.
  Definition backsem_sub_correct := backsem_sub_correct.

  Definition backsem_mult := backsem_mult.
  Definition backsem_mult_correct := backsem_mult_correct.


  Definition const : Z -> t := const.
  Definition const_correct : forall n, gamma (const n) n := const_correct.
  Definition const_complete : forall n N,
    gamma N n -> (const n) ⊑♯ N:= const_complete.

  Definition top : t := (⊤♯).
  Definition top_correct : forall n, gamma top n := top_correct.

  Definition bottom_empty : forall n, ~ gamma (⊥♯) n := bottom_empty.

  Definition check_positive (N : t) : bool :=
    match N with
      | None => true 
      | Some (ITV a b _) =>
        if order_Z'_dec (z (-1)) a then true else false
    end.

  Lemma check_positive_correct : forall N n,
    check_positive N = true ->
    gamma N n -> n>=0.
  Proof.
    destruct N; simpl; intros.
    destruct i.
    destruct order_Z'_dec; try discriminate.
    inversion H0; subst.
    inversion o0; subst; inversion H2; subst.
    omega.
    inversion H0.
  Qed.

  Definition check_in_range (N : t) (z1 z2:Z) : bool :=
    match N with
      | None => true 
      | Some (ITV a b _) =>
        if order_Z'_dec (z z1) a then 
          if order_Z'_dec b (z z2) then true
            else false
          else false
    end.

  Ltac inv H := inversion H; try (subst; clear H).

  Lemma check_in_range_correct : forall N z1 z2 n,
    check_in_range N z1 z2 = true ->
    gamma N n -> z1 < n < z2.
  Proof.
    destruct N; simpl; intros.
    destruct i.
    repeat destruct order_Z'_dec; try discriminate.
    inv H0; simpl in *.
    inv H2; inv o0.
    inv H3; inv o1.
    omega.
    inversion H0.
  Qed.

  Definition is_constant (N : t) : option Z :=
    match N with
      | None => None
      | Some (ITV (z a) (z b) _) =>
        if Z_eq_dec a b then Some a else None
      | _ => None
    end.

  Lemma check_is_constant_correct : forall N z n,
    is_constant N = Some z ->
    gamma N n -> n = z.
  Proof.
    destruct N; simpl; intros.
    destruct i.
    inv H0.
    simpl in *.
    inv H2; inv H3; try discriminate.
    destruct Z_eq_dec; try discriminate.
    subst; inv H; omega.
    discriminate.
  Qed.

End Interval.
