Require Import CompleteLattice.
Require Import lattice_def.

Lemma meet_top : forall (t:Type) (P: PosetDec.t t) (M:MeetDec.t t) (T:@TopDec.t t P) (x : t), (⊤♯) ⊓♯ x =♯ x.
Proof.
  intros.
  apply PosetDec.antisym.
  auto.
  apply MeetDec.greatest_lower_bound.
    apply TopDec.prop.
  auto.
Qed.

Section Lift.
Instance LiftEquivDec {A} (E: EquivDec.t A) : EquivDec.t (option A).
apply EquivDec.Make with (fun x y =>
match x, y with
| Some x, Some y => EquivDec.eq x y
| None, None => True
| _, _ => False
end);
induction x;
trivial;
induction y;
intuition.
induction z; intuition; eauto.
apply EquivDec.dec.
Defined.

Instance LiftPosetDec {A} (P:PosetDec.t A) : PosetDec.t (option A).
apply (PosetDec.Make _ (LiftEquivDec _)) with (fun x y =>
match x, y with
| Some x, Some y => PosetDec.order x y
| None, _ => True
| _, None => False
end);
induction x; intuition; induction y; intuition.
simpl;
apply PosetDec.antisym; assumption.
induction z. eauto. assumption.
apply PosetDec.dec.
Defined.

Instance LiftJoinDec {A} `(J:JoinDec.t A) : JoinDec.t (option A).
apply JoinDec.Make with (fun x y =>
match x, y with
| Some x, Some y => Some (JoinDec.op x y)
| Some x, None => Some x
| None, Some y => Some y
| None, None => None
end);
induction x; intuition;
induction y; intuition;
simpl; trivial.
induction z; intuition.
Defined.

Instance LiftMeetDec {A} `(J:PosetDec.t A) (op: A -> A -> option A)
  (bound1 : ∀x y z, op x y = Some z -> z ⊑♯ x)
  (bound2 : ∀x y z, op x y = Some z -> z ⊑♯ y)
  (greatest_lower_bound : ∀x y z t, op x y = Some t -> z ⊑♯ x → z ⊑♯ y → z ⊑♯ t)
  (glb' : ∀x y z, op x y = None → z ⊑♯ x → z ⊑♯ y → False)
  : MeetDec.t (option A).
apply MeetDec.Make with (fun x y =>
match x, y with
| None, _
| _, None => None
| Some x, Some y => op x y
end);
intros [x|] [y|]; try (intros [z|]); simpl; intuition;
case_eq (op x y); intuition.
apply bound1 with y; assumption.
apply bound2 with x; assumption.
apply greatest_lower_bound with x y; assumption.
apply glb' with x y z; assumption.
Defined.

Instance LiftTopDec A `{T:TopDec.t A} : TopDec.t (option A).
apply TopDec.Make with (Some TopDec.elem).
intros [x|]; simpl.
apply TopDec.prop.
trivial.
Defined.

Instance LiftBotDec A `{PosetDec.t A} : BotDec.t (option A).
apply BotDec.Make with None.
intros [x|]; simpl; trivial.
Defined.
End Lift.

Instance ProductEquivDec {A B} (Ea : EquivDec.t A) (Eb : EquivDec.t B) : EquivDec.t (A * B).
refine (EquivDec.Make (A * B) (fun x y => @EquivDec.eq _ Ea (fst x) (fst y) /\ @EquivDec.eq _ Eb (snd x) (snd y)) _ _ _ _);
firstorder; eauto.
destruct (EquivDec.dec (fst x) (fst y));
destruct (EquivDec.dec (snd x) (snd y));
intuition.
Defined.

Instance ProductPosetDec {A B} (Pa: PosetDec.t A) (Pb: PosetDec.t B) : PosetDec.t (A * B).
refine (PosetDec.Make (A*B) (ProductEquivDec _ _) (fun x y => @PosetDec.order _ Pa (fst x) (fst y) /\ @PosetDec.order _ Pb (snd x) (snd y)) _ _ _ _); firstorder;
eauto.
destruct (PosetDec.dec (fst x) (fst y));
destruct (PosetDec.dec (snd x) (snd y));
intuition.
Defined.

Instance ProductJoinDec {A B} `(Ja: JoinDec.t A) `(Jb: JoinDec.t B) : JoinDec.t (A*B).
apply JoinDec.Make with (fun x y =>
(JoinDec.op (fst x) (fst y),
JoinDec.op (snd x) (snd y)));
firstorder; simpl;
auto.
Defined.

Instance ProductMeetDec {A B} `(Ma: MeetDec.t A) `(Mb: MeetDec.t B) : MeetDec.t (A*B).
apply MeetDec.Make with (fun x y =>
(MeetDec.op (fst x) (fst y),
MeetDec.op (snd x) (snd y)));
firstorder; simpl;
auto.
Defined.

Instance ProductBotDec {A B} `(Ba: BotDec.t A) `(Bb: BotDec.t B) : BotDec.t (A*B).
apply BotDec.Make with (BotDec.elem, BotDec.elem).
firstorder; simpl; auto.
Defined.

Instance ProductTopDec {A B} `(Ta: TopDec.t A) `(Tb: TopDec.t B) : TopDec.t (A*B).
apply TopDec.Make with (TopDec.elem, TopDec.elem).
intros (a&b); split; apply TopDec.prop.
Defined.

Definition ProductWiden {A B} (wa: A->A->A) (wb:B->B->B) :=
  fun (x y : A*B) =>
    (wa (fst x) (fst y), wb (snd x) (snd y)).

Instance ProductLatticeDec {A B} `(L: LatticeDec.t A) `(M: LatticeDec.t B) : LatticeDec.t (A * B).
apply LatticeDec.Make with (ProductPosetDec _ _);
(apply ProductWiden; apply LatticeDec.widen) || intuition.
Defined.

Instance ProductGamma {A B a b γ γ'} `(L: Gamma.t a A γ) `(R: Gamma.t b B γ') :
  Gamma.t (a * b) (A * B) (λ x, (γ (fst x), γ' (snd x))).
Proof.
  destruct L. destruct R.
  split.
intros [na nb] [ma mb] [Ha Hb]; simpl in *; split; auto.
intros [na nb] [ma mb]; simpl; split; auto.
Defined.

(** Flat lattice *)
Module FlatLatticeDec.
Inductive TB {A:Type} :=
| TB_Top : TB
| TB_Bot : TB
| TB_Elem: forall a : A, TB
.

Definition tb_eq {A} (a_eq:A->A->Prop) := fun x y : TB =>
match x, y with
| TB_Top, TB_Top => True
| TB_Bot, TB_Bot => True
| TB_Elem x', TB_Elem y' => a_eq x' y'
| _, _ => False
end.

Instance tEquiv {A} `(Ea: EquivDec.t A) : EquivDec.t (@TB A).
apply EquivDec.Make with (tb_eq EquivDec.eq).
induction x; intuition. apply EquivDec.refl.
induction x; induction y; intuition. apply EquivDec.sym; intuition.
induction x; induction y; induction z; simpl in *; intuition.
eapply EquivDec.trans ; eassumption.
induction x; induction y; intuition.
apply EquivDec.dec.
Defined.

Definition tb_le {A} (a_eq:A->A->Prop) := fun x y : TB =>
match x, y with
| TB_Bot, _ => True
| _, TB_Top => True
| TB_Elem x', TB_Elem y' => a_eq x' y'
| _, _ => False
end.

Instance tPoset {A} `(Ea: EquivDec.t A) : PosetDec.t (@TB A).
apply (PosetDec.Make _ (tEquiv Ea) (tb_le EquivDec.eq)).
induction x; induction y; simpl; intuition.
induction x; induction y; intuition.
induction x; induction y; induction z; simpl; intuition.
eapply EquivDec.trans ; eassumption.
induction x; induction y; simpl; intuition.
apply EquivDec.dec.
Defined.

Definition tb_join {A} (a_eqb:A->A->bool) := fun x y : @TB A =>
match x, y with
| TB_Bot, _ => y
| _, TB_Bot => x
| TB_Top, _
| _, TB_Top => TB_Top
| TB_Elem x', TB_Elem y' => if a_eqb x' y' then x else TB_Top
end.

Instance tJoin {A} `(Ea: EquivDec.t A) : JoinDec.t (@TB A).
apply JoinDec.Make with (tb_join (fun x y => if EquivDec.dec x y then true else false)).
induction x as [| |x]; induction y as [| |y]; simpl; intuition.
destruct (EquivDec.dec x y); trivial.
induction x as [| |x]; induction y as [| |y]; simpl; intuition.
destruct (EquivDec.dec x y); intuition.
induction x as [| |x]; induction y as [| |y]; induction z as [| |z]; simpl; intuition.
destruct (EquivDec.dec x y); trivial.
destruct (EquivDec.dec x y). trivial.
elim n; apply EquivDec.trans with z; intuition.
Defined.

Definition tb_meet {A} (a_eqb:A->A->bool) := fun x y : TB =>
match x, y with
| TB_Bot, _
| _, TB_Bot => TB_Bot
| TB_Top, _ => y
| _, TB_Top => x
| TB_Elem x', TB_Elem y' => if a_eqb x' y' then x else TB_Bot
end.

Instance tMeet {A} `(Ea: EquivDec.t A) : MeetDec.t (@TB A).
apply MeetDec.Make with (tb_meet (fun x y => if EquivDec.dec x y then true else false)).
induction x as [| |x]; induction y as [| |y]; simpl; intuition.
destruct (EquivDec.dec x y); simpl; intuition.
induction x as [| |x]; induction y as [| |y]; simpl; intuition.
destruct (EquivDec.dec x y); simpl; intuition.
induction x as [| |x]; induction y as [| |y]; induction z as [| |z]; simpl; intuition.
destruct (EquivDec.dec x y). trivial.
elim n; apply EquivDec.trans with z; intuition.
Defined.

Instance t {A} `(Ea: EquivDec.t A) : LatticeDec.t (@TB A).
apply (LatticeDec.Make _ (tPoset _) (tJoin _) (tMeet _)).
apply BotDec.Make with TB_Bot.
induction x; simpl; intuition.
exact JoinDec.op.
Defined.
End FlatLatticeDec.

Section TopFlat.
Instance TopFlatPoset A `{EquivDec.t A} : PosetDec.t (option A).
set (X := LiftEquivDec _).
apply PosetDec.Make with (eq:=X) (order:= fun x y =>
match x, y with
| _, None => True
| Some x, Some y => EquivDec.eq x y
| None, _ => False
end);
induction x as [x|];
induction y as [y|];
try (induction z as [z|]);
intuition.
eauto.
apply EquivDec.dec.
Defined.

Instance TopFlatJoin A `{EquivDec.t A} : JoinDec.t (option A).
apply JoinDec.Make with (fun x y =>
match x, y with
| None, _
| _, None => None
| Some x', Some y' => if EquivDec.dec x' y' then x else None
end);
intros [x|] [y|]; intuition;
try (case (EquivDec.dec x y); intuition);
simpl; trivial.
unfold PosetDec.order in *.
unfold TopFlatPoset in *.
induction z as [z|]; intuition.
apply n. eauto.
Defined.

End TopFlat.

Require Import Containers.Maps.

Section MapLatDec.

Variables A B : Type.
Hypothesis Oa : OrderedType A.
Hypothesis Pb : PosetDec.t B.
Hypothesis Tb : @TopDec.t B Pb.

Definition find_default k (M : Map [ A, B] ) :=
match (M) [k] with
| Some x => x
| None => TopDec.elem
end.

Definition map_eq (f g : Map [A, B]) : Prop :=
  forall k, EquivDec.eq (find_default k f) (find_default k g).

Definition map_le (f g : Map [A, B]) : Prop :=
  forall k, PosetDec.order (find_default k f) (find_default k g).

Remark map_empty_top : forall f, map_le f [].
intros f k. unfold find_default.
assert (([]) [k] = @None B) as K;[auto|rewrite K].
destruct ((f) [k]); intuition.
apply TopDec.prop.
Qed.

Definition map_leb_one (f : Map [A, B]) := fun k y =>
  if PosetDec.dec (find_default k f) y then true else false.

Definition map_leb (f g : Map [ A, B]) := MF.for_all (map_leb_one f) g.

Lemma map_leb_proper : forall f, Proper (_eq ==> eq ==> eq) (map_leb_one f).
Proof.
intros f x y H_eq z w Heq; subst.
unfold map_leb_one, find_default.
case_eq ((f)[x]); intros;
 (rewrite (MF.find_o f) with y x;[|rewrite H_eq;intuition];
 rewrite H);
reflexivity.
Qed.

Lemma map_leb_iff_map_le (f g : Map [A,B]) :
  map_leb f g = true <-> map_le f g.
Proof.
destruct (MF.for_all_iff (map_leb_proper f) g) as [K1 K2].
unfold map_leb, map_le.
split; intros H.
intros k.
unfold find_default.
case_eq ((g)[k]).
intros b Hb.
generalize (K1 H k b (find_2 Hb)).
unfold map_leb_one.
unfold find_default.
case_eq ((f)[k]).
intros c Hc.
destruct (PosetDec.dec c b). trivial.
intros K; absurd (false = true); intuition.
destruct (PosetDec.dec TopDec.elem b). trivial.
intros _ K; absurd (false = true); intuition.
intros; apply TopDec.prop.
apply K2.
intros k e He.
unfold map_leb_one.
generalize (H k).
unfold find_default.
rewrite (find_1 He).
case_eq ((f)[k]).
intros b Hb K.
destruct (PosetDec.dec b e). trivial. contradiction.
intros.
destruct (PosetDec.dec TopDec.elem e). trivial. contradiction.
Qed.

Lemma map_le_dec : forall f g, {map_le f g} + {~ map_le f g}.
Proof.
intros.
pose proof (map_leb_iff_map_le f g) as [K1 K2].
case_eq (map_leb f g); intros H;[left|right].
apply (K1 H).
intros K.
absurd (false = true).
discriminate.
rewrite <- H.
rewrite (K2 K).
reflexivity.
Qed.

Lemma map_eq_dec : forall f g, {map_eq f g} + {~ map_eq f g}.
Proof.
intros f g.
destruct (map_le_dec f g);[|right].
destruct (map_le_dec g f);[left|right];
firstorder.
firstorder.
Defined.

End MapLatDec.

Implicit Arguments find_default [A B Oa Pb Tb].

Instance MapEquivDec A {B} `{OrderedType A} `{Pb: PosetDec.t B} `(Tb: @TopDec.t B Pb) : EquivDec.t Map [A, B].
apply EquivDec.Make with (map_eq A B _ _ _);
try firstorder.
intros x y z H1 H2 k.
apply (EquivDec.trans _ _ _ (H1 k) (H2 k)).
apply map_eq_dec.
Defined.

(*
Instance MapLiftEquivDec A {B} `{OrderedType A} `{Pb: PosetDec.t B} `(Tb: @TopDec.t B Pb) : EquivDec.t Map [A, option B].
apply MapEquivDec with (LiftPosetDec _).
exact (LiftTopDec _).
Defined.
*)

Instance MapPosetDec A {B} `{Oa:OrderedType A} `{Pb: PosetDec.t B} `(Tb: @TopDec.t B Pb) : PosetDec.t Map [A, B].
apply (PosetDec.Make _ (MapEquivDec A Tb)) with (map_le A B _ _ _).
intros x y H k. apply (PosetDec.refl _ _ (H k)).
intros x y H1 H2 k. apply (PosetDec.antisym _ _ (H1 k) (H2 k)).
intros x y z H1 H2 k. apply (PosetDec.trans _ _ _ (H1 k) (H2 k)).
apply map_le_dec.
Defined.

(*
Instance MapLiftPosetDec A {B} `{OrderedType A} `{Pb: PosetDec.t B} `(Tb: @TopDec.t B Pb) : PosetDec.t Map [A, option B].
apply MapPosetDec with (LiftPosetDec _).
exact (LiftTopDec _).
Defined.
*)

Definition MapJoin A {B} `{Oa:OrderedType A} `{Jb: JoinDec.t B} `(Tb: TopDec.t B):
  Map [ A , B ] -> Map [ A , B ] -> Map [ A , B ] := fun f g =>
fold
(fun k y => add k (JoinDec.op (find_default k f) y))
g
[]
.

Instance  MapJoinDec A {B} `{Oa:OrderedType A} `{Jb: JoinDec.t B} `(Tb: @TopDec.t B _) : JoinDec.t Map [A, B].
apply JoinDec.Make with (MapJoin A Tb);
unfold MapJoin.
intros f g. apply MF.fold_rec.
intros; apply (map_empty_top A B Oa _ Tb).
intros k e a m' m'' Hg Hm' Hadd Hf k'.
unfold find_default.
rewrite MF.add_o.
case (eq_dec k k'); intros U.
rewrite (MF.find_o _ U).
case ((f)[k']);auto.
case_eq ((a)[k']);[|intros; apply TopDec.prop].
intros c Hc.
case_eq ((f)[k']);[intros b Hb|intros Hb]; generalize (Hf k');
unfold find_default; rewrite Hc; rewrite Hb; trivial.
intros f g. apply MF.fold_rec.
intros; apply (map_empty_top A B Oa _ Tb).
intros k e a m' m'' Hg Hm' Hadd Hf k'.
unfold find_default.
rewrite MF.add_o.
rewrite (Hadd k').
case (eq_dec k k'); intros U.
rewrite (MF.add_eq_o _ _ U).
rewrite (MF.find_o _ U).
case ((f)[k']);[|intros; eapply PosetDec.trans;[apply TopDec.prop|]];auto.
rewrite (MF.add_neq_o _ _ U).
apply (Hf k').
(* least upper bound *)
intros f g h Hf Hg.
generalize dependent h; apply MF.fold_rec.
intros m Hm h _ K k.
generalize (K k).
unfold find_default.
case_eq ((h)[k]);[|intros; apply TopDec.prop].
case (@MF.In_dec _ _ _ _ _ m k).
 intros [u U]; elim (Hm k u U).
intros U.
pose proof (MF.not_find_in_iff m k) as [U1 U2].
rewrite (U1 U).
case (([])[k]);[|trivial].
intros b c Hc Hb.
apply PosetDec.trans with TopDec.elem.
apply TopDec.prop.
exact Hb.
intuition.
assert (m' ⊑♯ h[k<-⊤♯]) as Hh.
intros k'.
generalize (Hf k');
generalize (Hg k').
unfold find_default.
rewrite H2.
case (eq_dec k k').
  intros Heq.
  repeat (rewrite MF.add_eq_o;[|assumption]).
  intros; apply TopDec.prop.
intros Hneq.
repeat (rewrite MF.add_neq_o;[|assumption]). intuition.
assert (f ⊑♯ h[k<-⊤♯]) as Hf'.
  intros k'; generalize (Hf k').
  unfold find_default.
  case (eq_dec k k').
    intros;
    repeat (rewrite MF.add_eq_o;[|assumption]).
    apply TopDec.prop.
  intros Hneq.
  rewrite MF.add_neq_o;[|assumption].
  intuition.
intros k'.
generalize (Hf k');
generalize (Hg k');
generalize (H3 _ Hf' Hh k').
unfold find_default.
rewrite H2.
case (eq_dec k k'). 
  intros Heq;
  repeat (rewrite MF.add_eq_o;[|assumption]).
  repeat (rewrite <- (MF.find_o _ Heq)).
  auto.
intros Hneq.
repeat (rewrite MF.add_neq_o;[|assumption]).
case_eq ((h)[k']);[|intros; apply TopDec.prop].
intros b Hb1.
case_eq ((a)[k']); trivial.
Defined.

Definition MapWiden A {B} `{Oa:OrderedType A} `(Tb: TopDec.t B)
  (widen: B->B->B) :
  Map [ A , B ] -> Map [ A , B ] -> Map [ A , B ] := fun f g =>
fold
(fun k y => add k (widen (find_default k f) y))
g
[]
.

(*
Instance MapLiftJoinDec A B `{Oa:OrderedType A} `{Jb: JoinDec.t B} `{Tb: @TopDec.t B _} : JoinDec.t Map [A, option B].
apply MapJoinDec.
apply LiftJoinDec.
assumption.
Defined.
*)

(*
Definition MapMeet A B `{Oa: OrderedType A} `{Tb:TopDec.t B} (mb: B->B->option B) :
  Map [ A , B ] -> Map [ A , B ] -> Map [ A , option B ] := fun f g =>
  fold
  (fun k x => add k (mb x (find_default A B Oa _ Tb k g)))
  f
  (map (@Some _) g)
.

Definition MapLiftMeet A B `{Oa: OrderedType A}
  `(Mb:MeetDec.t (option B)) `(Tb:@TopDec.t (option B) _) :
  Map [ A , option B ] -> Map [ A , option B ] -> Map [ A , option B ] := fun f g =>
  fold
  (fun k x => add k (MeetDec.op x (find_default A _ Oa _ Tb k g)))
  f
  g
.
*)

Definition MapMeet0 A {B} `{Oa:OrderedType A} `{Mb: MeetDec.t B} `(Tb: @TopDec.t B _):
  Map [ A , B ] -> Map [ A , B ] -> Map [ A , B ] := fun f g =>
fold
(fun k x => add k (MeetDec.op x (find_default k g)))
f
g
.

Lemma map_meet_spec : forall A B (Oa:OrderedType A) `(Mb: MeetDec.t B) (Tb: @TopDec.t B _),
  forall f g : Map [A, B], 
  forall k : A,
    find_default k (MapMeet0 A Tb f g) =♯
    (find_default k f ⊓♯ find_default k g).
Proof.
  intros A B Oa H Mb Tb f g k.
  unfold MapMeet0.
  apply MF.fold_rec.
  unfold find_default at 2.
  intros m H1.
  case_eq ((m)[k]).
    intros b Hb; elim (H1 _ _ (find_2 Hb)).
  intros _.
  apply PosetDec.antisym;[|auto].
  apply MeetDec.greatest_lower_bound.
    apply TopDec.prop.
  auto.
  intros k0 e a m' m'' H0 H1 H2 H3.
  unfold find_default.
  rewrite (H2 k).
  case (eq_dec k0 k).
    intros Hkeq.
    repeat (rewrite (MF.add_eq_o);[|trivial]).
    rewrite (MF.find_o _ Hkeq).
    auto.
  intros Hkneq.
  repeat (rewrite (MF.add_neq_o);[|trivial]).
  apply H3.
Qed.

Lemma map_meet_1 : forall A B (Oa:OrderedType A) `(Mb:MeetDec.t B) `(Tb:@TopDec.t B _),
 ∀x y : Map  [A, B], MapMeet0 A Tb x y ⊑♯ x.
Proof.
  intros A B Oa H Mb Tb x y k.
  eapply PosetDec.trans.
    apply PosetDec.refl.
    apply map_meet_spec.
  auto.
Qed.

Lemma map_meet_2 : forall A B (Oa:OrderedType A) `(Mb:MeetDec.t B) `(Tb:@TopDec.t B _),
 ∀x y : Map  [A, B], MapMeet0 A Tb x y ⊑♯ y.
Proof.
  intros A B Oa H Mb Tb x y k.
  eapply PosetDec.trans.
    apply PosetDec.refl.
    apply map_meet_spec.
  auto.
Qed.

Lemma map_meet_3 : forall A B (Oa:OrderedType A) `(Mb:MeetDec.t B) `(Tb:@TopDec.t B _),
 ∀x y z : Map  [A, B], z ⊑♯ x → z ⊑♯ y → z ⊑♯ MapMeet0 A Tb x y.
Proof.
  intros A B Oa H Mb Tb x y z H0 H1 k.
  eapply PosetDec.trans;
    [|apply PosetDec.refl; apply EquivDec.sym; apply map_meet_spec].
  auto.
Qed.

Instance MapMeetDec A {B} `{Oa:OrderedType A} `{Mb: MeetDec.t B} `(Tb: @TopDec.t B _) : MeetDec.t Map [A, B].
apply MeetDec.Make with (MapMeet0 A Tb).
apply map_meet_1.
apply map_meet_2.
apply map_meet_3.
Defined.

Instance MapTopDec A {B} `{OrderedType A} `{PosetDec.t B} `(@TopDec.t B _) : TopDec.t Map [A,B].
apply TopDec.Make with [].
simpl; unfold map_le, find_default.
intros f k.
assert (([])[k] = @None B) as K.
auto.
rewrite K.
apply TopDec.prop.
Defined.


(* Ne se peut : bottom ?
Instance MapLatticeDec A {B} `{OrderedType A} `{Lb:LatticeDec.t B} `(Tb:@TopDec.t B _) : LatticeDec.t Map [A,B].
apply (LatticeDec.Make _ (MapPosetDec A Tb)).
apply (MapJoinDec A Tb).
apply (MapMeetDec A Tb).
*)

Definition map_gamma b A {B} `{Oa:OrderedType A} `(Tb: TopDec.t B) (op : B → b) :
  Map [A, B] -> (A -> b) := fun M k => op (find_default k M).

Lemma map_gamma_monotone : forall b A B (Oa:OrderedType A) op
  `(Gb: Gamma.t b B op) (Tb:@TopDec.t B _),
  ∀N1 N2 : Map [A , B],
  N1 ⊑♯ N2 →
      map_gamma b A Tb op N1 ⊑ map_gamma b A Tb op N2.
Proof.
  intros b A B Oa op H H0 Gb Tb N1 N2 H1 q.
  generalize (H1 q).
  unfold map_gamma, find_default.
  destruct ((N2)[q]);
    destruct ((N1)[q]);
      apply Gamma.monotone.
Qed.

Lemma map_gamma_meet_morph : forall b A B (Oa:OrderedType A) op
  `(Gb: Gamma.t b B op) (Tb:@TopDec.t B _),
  ∀N1 N2 : Map [A , B],
 (map_gamma b A Tb op N1 ⊓ map_gamma b A Tb op N2) ⊑
     map_gamma b A Tb op (N1 ⊓♯ N2).
Proof.
  intros b A B Oa op H H0 Gb Tb N1 N2 q.
  unfold MeetDec.op, MapMeetDec.
  unfold map_gamma.
  apply Poset.trans with (op (find_default q N1 ⊓♯ find_default q N2)).
  unfold find_default; simpl.
  destruct ((N2)[q]);
    destruct ((N1)[q]);
      apply Gamma.meet_morph.
  apply Gamma.monotone.
  apply PosetDec.refl.
  apply EquivDec.sym.
  apply map_meet_spec.
Qed.

(*
Instance MapGamma b A {B} `{Oa:OrderedType A} `{Gb: Gamma.t b B}
  `{BotDec.t Map [A,B]} {Tb:TopDec.t B} : Gamma.t (A -> b) (Map [A, B]).
apply Gamma.Make with (map_gamma b A Tb).
intros f g Hfg k; unfold map_gamma.
apply Gamma.monotone.
unfold find_default.
case_eq (g[k]);[|intros; apply TopDec.prop].
intros 
intros; apply (map_gamma_monotone _ _ _ Oa Gb Tb N1 N2). auto.
apply map_gamma_meet_morph.
Defined.
*)
(*
Require Import FaithLattice.

Instance FaithOfAbLattice {A} (L:AbLattice.t A) : AbLattice'.t A.
constructor; constructor.
exact (fun x y => if EquivDec.dec x y then true else false).
exact JoinDec.op.
exact BotDec.elem.
exact Widen.widen.
Defined.
*)

Section MapWithBot.

Variables A B : Type.
Hypothesis Oa : OrderedType A.
Hypothesis Pb : PosetDec.t B.
Hypothesis Bb : @BotDec.t B Pb.

Definition find_def_to_bot k (M : Map [ A, B] ) :=
match (M) [k] with
| Some x => x
| None => BotDec.elem
end.

Definition bot_map_eq (f g : Map [A, B]) : Prop :=
  forall k, EquivDec.eq (find_def_to_bot k f) (find_def_to_bot k g).

Definition bot_map_le (f g : Map [A, B]) : Prop :=
  forall k, PosetDec.order (find_def_to_bot k f) (find_def_to_bot k g).

Remark map_empty_bot : forall f, bot_map_le [] f.
intros f k. unfold find_def_to_bot.
assert (([]) [k] = @None B) as K;[auto|rewrite K].
destruct ((f) [k]); intuition.
Qed.

Remark map_empty_find_bot1 : forall k, find_def_to_bot k [] = ⊥♯. auto. Qed.
Remark map_empty_find_bot2 : forall m k (Hm:Empty m), find_def_to_bot k m = ⊥♯.
Proof. intros m k Hm. unfold find_def_to_bot.
case_eq ((m)[k]).
  intros b Hb. elim (Hm _ _ (find_2 Hb)).
reflexivity.
Qed.

Definition bot_map_geb_one (f : Map [A, B]) := fun k y =>
  if PosetDec.dec y (find_def_to_bot k f) then true else false.

Definition bot_map_leb (f g : Map [ A, B]) := MF.for_all (bot_map_geb_one g) f.

Lemma bot_map_leb_proper : forall f, Proper (_eq ==> eq ==> eq) (bot_map_geb_one f).
Proof.
intros f x y H_eq z w Heq; subst.
unfold bot_map_geb_one, find_def_to_bot.
case_eq ((f)[x]); intros;
 (rewrite (MF.find_o f) with y x;[|rewrite H_eq;intuition];
 rewrite H);
reflexivity.
Qed.

Lemma bot_map_leb_iff_map_le (f g : Map [A,B]) :
  bot_map_leb f g = true <-> bot_map_le f g.
Proof.
destruct (MF.for_all_iff (bot_map_leb_proper g) f) as [K1 K2].
unfold bot_map_leb, bot_map_le.
split; intros H.
intros k.
unfold find_def_to_bot.
case_eq ((f)[k]).
intros b Hb.
generalize (K1 H k b (find_2 Hb)).
unfold bot_map_geb_one.
unfold find_def_to_bot.
case_eq ((g)[k]).
intros c Hc.
destruct (PosetDec.dec b c). trivial.
intros K; absurd (false = true); intuition.
destruct (PosetDec.dec b BotDec.elem). trivial.
intros _ K; absurd (false = true); intuition.
intros; apply BotDec.prop.
apply K2.
intros k e He.
unfold bot_map_geb_one.
generalize (H k).
unfold find_def_to_bot.
rewrite (find_1 He).
case_eq ((g)[k]).
intros b Hb K.
destruct (PosetDec.dec e b). trivial. contradiction.
intros.
destruct (PosetDec.dec e BotDec.elem). trivial. contradiction.
Qed.

Lemma bot_map_le_dec : forall f g, {bot_map_le f g} + {~ bot_map_le f g}.
Proof.
intros.
pose proof (bot_map_leb_iff_map_le f g) as [K1 K2].
case_eq (bot_map_leb f g); intros H;[left|right].
apply (K1 H).
intros K.
absurd (false = true).
discriminate.
rewrite <- H.
rewrite (K2 K).
reflexivity.
Defined.

Lemma bot_map_eq_dec : forall f g, {bot_map_eq f g} + {~ bot_map_eq f g}.
Proof.
intros f g.
destruct (bot_map_le_dec f g);[|right].
destruct (bot_map_le_dec g f);[left|right];
firstorder.
firstorder.
Defined.

Instance BotMapEquivDec : EquivDec.t Map [ A, B].
apply EquivDec.Make with bot_map_eq;
try firstorder.
intros x y z H1 H2 k.
apply (EquivDec.trans _ _ _ (H1 k) (H2 k)).
apply bot_map_eq_dec.
Defined.

Instance BotMapPosetDec : PosetDec.t Map [A, B].
apply (PosetDec.Make _ BotMapEquivDec) with bot_map_le.
intros x y H k. apply (PosetDec.refl _ _ (H k)).
intros x y H1 H2 k. apply (PosetDec.antisym _ _ (H1 k) (H2 k)).
intros x y z H1 H2 k. apply (PosetDec.trans _ _ _ (H1 k) (H2 k)).
apply bot_map_le_dec.
Defined.

Definition BotMapJoin `{Jb: @JoinDec.t B Pb}: Map [ A , B ] -> Map [ A , B ] -> Map [ A , B ] := fun f g =>
fold
(fun k y => add k (JoinDec.op (find_def_to_bot k f) y))
g
f
.

Instance BotMapJoinDec `{Jb: @JoinDec.t B Pb} : JoinDec.t Map [A, B].
apply JoinDec.Make with BotMapJoin;
unfold BotMapJoin.
intros f g. apply MF.fold_rec.
intros; auto.
intros k e a m' m'' Hg Hm' Hadd Hf k'.
unfold find_def_to_bot.
rewrite MF.add_o.
case (eq_dec k k'); intros U.
rewrite (MF.find_o _ U).
case ((a)[k']);auto.
case_eq ((f)[k']);[|auto].
intros c Hc.
generalize (Hf k');
unfold find_def_to_bot; rewrite Hc; trivial.
intros f g. apply MF.fold_rec.
intros m Hm k. unfold find_def_to_bot.
case_eq ((m)[k]).
intros b Hb. elim (Hm _ _ (find_2 Hb)). auto.
intros k e a m' m'' Hg Hm' Hadd Hf k'.
unfold find_def_to_bot.
rewrite MF.add_o.
rewrite (Hadd k').
case (eq_dec k k'); intros U.
rewrite (MF.add_eq_o _ _ U).
rewrite (MF.find_o _ U).
case ((f)[k']); auto.
rewrite (MF.add_neq_o _ _ U).
apply (Hf k').
(* least upper bound *)
intros f g h Hf Hg.
generalize dependent h; apply MF.fold_rec.
intros m Hm h K _ k. exact (K k).
intros k e a m' m'' H H0 H1 H2 h Hf Hg.
assert (m' ⊑♯ h[k<-find_def_to_bot k f ⊔♯ e]) as Hh.
intros k'.
generalize (Hf k');
generalize (Hg k').
unfold find_def_to_bot.
rewrite H1.
case (eq_dec k k').
  intros Heq.
  repeat (rewrite MF.add_eq_o;[|assumption]).
  pose proof (MF.not_find_in_iff m' k) as [K1 _]; rewrite <- Heq; rewrite (K1 H0).
  auto.
intros Hneq.
repeat (rewrite MF.add_neq_o;[|assumption]). intuition.
assert (f ⊑♯ h[k<-find_def_to_bot k f ⊔♯ e]) as Hf'.
  intros k'; generalize (Hf k').
  unfold find_def_to_bot.
  case (eq_dec k k').
    intros Heq.
    repeat (rewrite MF.add_eq_o;[|assumption]).
    rewrite <- Heq. auto.
  intros Hneq.
  rewrite MF.add_neq_o;[|assumption].
  intuition.
intros k'.
generalize (Hf k');
generalize (Hg k');
generalize (H2 _ Hf' Hh k').
unfold find_def_to_bot.
rewrite H1.
case (eq_dec k k'). 
  intros Heq;
  repeat (rewrite MF.add_eq_o;[|assumption]).
  repeat (rewrite <- (MF.find_o _ Heq)).
  auto.
intros Hneq.
repeat (rewrite MF.add_neq_o;[|assumption]).
auto.
Defined.

Definition BotMapWiden (widen: B->B->B) :
  Map [ A , B ] -> Map [ A , B ] -> Map [ A , B ] := fun f g =>
fold
(fun k y => add k (widen (find_def_to_bot k f) y))
g
f
.

Definition BotMapMeet `{Mb: @MeetDec.t B Pb} :
  Map [ A , B ] -> Map [ A , B ] -> Map [ A , B ] := fun f g =>
fold
(fun k x => add k (MeetDec.op x (find_def_to_bot k g)))
f
[]
.

Lemma bot_map_meet_spec : forall `(Mb: @MeetDec.t B Pb),
  forall f g : Map [A, B], 
  forall k : A,
    find_def_to_bot k (BotMapMeet f g) =♯
    (find_def_to_bot k f ⊓♯ find_def_to_bot k g).
Proof.
  intros Mb f g k.
  unfold BotMapMeet.
  apply MF.fold_rec.
  intros m H.
  rewrite map_empty_find_bot1;
  rewrite map_empty_find_bot2.
  apply meet_bottom1.
  assumption.
  intros k0 e a m' m'' H0 H1 H2 H3.
  unfold find_def_to_bot.
  rewrite (H2 k).
  case (eq_dec k0 k).
    intros Hkeq.
    repeat (rewrite (MF.add_eq_o);[|trivial]).
    rewrite (MF.find_o _ Hkeq).
    auto.
  intros Hkneq.
  repeat (rewrite (MF.add_neq_o);[|trivial]).
  apply H3.
Qed.

Instance BotMapMeetDec `{@MeetDec.t B Pb} : MeetDec.t Map [A, B].
apply MeetDec.Make with BotMapMeet;
try (
  intros x y k;
  eapply PosetDec.trans;
    [apply PosetDec.refl;
      apply bot_map_meet_spec
    |auto]).
intros x y z Hx Hy k.
  eapply PosetDec.trans;
    [|apply PosetDec.refl; apply EquivDec.sym; apply bot_map_meet_spec].
  auto.
Defined.

Instance BotMapBotDec : BotDec.t Map [A,B].
apply BotDec.Make with [].
intros f k; rewrite map_empty_find_bot1.
auto.
Defined.

End MapWithBot.

