Require Export List.
Require Export Utf8.

Module EquivDec.
  Class t (A:Type) : Type := Make  {
    eq : A -> A -> Prop;
    refl : ∀ x, eq x x;
    sym : ∀ x y, eq x y -> eq y x;
    trans : ∀ x y z, eq x y -> eq y z -> eq x z;
    dec : ∀ x y, {eq x y}+{~ eq x y}
  }.
End EquivDec.
Notation "x =♯ y" := (EquivDec.eq x y) (at level 40).

Module PosetDec.
  Class t A : Type := Make  {
    eq :> EquivDec.t A;
    order : A -> A -> Prop;
    refl : ∀ x y,  x=♯y -> order x y;
    antisym : ∀ x y, order x y -> order y x -> x=♯y;
    trans : ∀ x y z, order x y -> order y z -> order x z;
    dec : ∀ x y, {order x y}+{~ order x y}
  }.
End PosetDec.
Notation "x ⊑♯ y" := (PosetDec.order x y) (at level 40).

Module JoinDec.
  Class t A `{PosetDec.t A} : Type := Make  {
    op : A -> A -> A;
    bound1 : ∀x y:A, x ⊑♯ op x y;
    bound2 : ∀x y:A, y ⊑♯ op x y;
    least_upper_bound : ∀x y z:A, x ⊑♯ z -> y ⊑♯ z -> (op x y) ⊑♯ z
  }.
End JoinDec.
Notation "x ⊔♯ y" := (JoinDec.op x y) (at level 40).

Module MeetDec.
  Class t A `{PosetDec.t A} : Type := Make  {
    op : A -> A -> A;
    bound1 : ∀x y:A, op x y ⊑♯ x;
    bound2 : ∀x y:A, op x y ⊑♯ y;
    greatest_lower_bound : ∀x y z:A, z ⊑♯ x -> z ⊑♯ y -> z ⊑♯ (op x y)
  }.
End MeetDec.
Notation "x ⊓♯ y" := (MeetDec.op x y) (at level 40).

Module BotDec.
  Class t A `{PosetDec.t A} : Type := Make  {
    elem : A;
    prop : forall x : A, elem ⊑♯ x
  }.
End BotDec.
Notation "⊥♯" := (BotDec.elem) (at level 40).

Module TopDec.
  Class t A `{PosetDec.t A} : Type := Make  {
    elem : A;
    prop : forall x : A, x ⊑♯ elem
  }.
End TopDec.
Notation "⊤♯" := (TopDec.elem) (at level 40).

Module LatticeDec.
  Class t (A:Type) : Type := Make
  { porder :> PosetDec.t A; 
    join :> JoinDec.t A;
    meet :> MeetDec.t A;
    bot :> BotDec.t A;
    widen : A -> A -> A
  }.
End LatticeDec.
Notation "x ∇♯ y" := (LatticeDec.widen x y) (at level 40).

Module Widen.
  Definition rel {A:Type} `{P:PosetDec.t A} `{BotDec.t A} (widen:A->A->A) : A*A -> A*A -> Prop := 
    fun x y =>
    (snd x) =♯ (widen (snd y) (fst x)) /\ 
    ~ (snd y) =♯ (snd x).
  Class t A `{P:PosetDec.t A} `{BotDec.t A} : Type := Make {
    widen : A -> A -> A;
    widen_bound1 : forall x y : A, x ⊑♯ (widen x y);
    widen_bound2 : forall x y : A, y ⊑♯ (widen x y);
    widen_eq1 : forall x y z: A, 
      x =♯ y -> (widen x z) =♯ (widen y z);
    widen_eq2 : forall x y z: A, 
      x =♯ y -> (widen z x) =♯ (widen z y);
    widen_bottom1 : forall x : A, x =♯ (widen x (⊥♯));
    widen_bottom2 : forall x : A, x =♯ (widen (⊥♯) x);
    widen_acc_property : forall x:A, Acc (rel widen) (x,x)
  }.
End Widen.

Module Narrow.
  Definition rel {A:Type} `{P:PosetDec.t A} `{BotDec.t A} `{MeetDec.t A} (narrow:A->A->A) : A*A -> A*A -> Prop := 
    fun x y =>
    (snd x) =♯ (narrow (snd y) (fst x)) /\ 
    ~ (snd y) =♯ (snd x).
  Class t A `{P:PosetDec.t A} `{BotDec.t A} `{MeetDec.t A} : Type := Make {
    narrow : A -> A -> A;
    narrow_bound1 : forall x y, (narrow x y) ⊑♯ x;
    narrow_bound2 : forall x y : A, (x ⊓♯ y) ⊑♯ (narrow x y);
    narrow_eq1 : forall x y z: A, 
      x =♯ y -> (narrow x z) =♯ (narrow y z);
    narrow_eq2 : forall x y z: A, 
      x =♯ y -> (narrow z x) =♯ (narrow z y);
    narrow_acc_property : forall x:A, Acc (rel narrow) (x,x)
  }.
End Narrow.

Module AbLattice.
  Class t (A:Type) : Type := Make
  { porder :> PosetDec.t A; 
    join :> JoinDec.t A;
    meet :> MeetDec.t A;
    bot :> BotDec.t A;
    widen :> Widen.t A;
    narrow :> Narrow.t A
  }.
End AbLattice.
Hint Resolve @EquivDec.refl @EquivDec.sym @EquivDec.trans 
             @PosetDec.refl @PosetDec.antisym @PosetDec.trans
             @JoinDec.bound1 @JoinDec.bound2 @JoinDec.least_upper_bound
             @MeetDec.bound1 @MeetDec.bound2 @MeetDec.greatest_lower_bound
             @BotDec.prop.

Instance AbLatticeDec {A} `(L:AbLattice.t A) : LatticeDec.t A.
apply LatticeDec.Make with (AbLattice.porder).
apply AbLattice.join.
apply AbLattice.meet.
apply AbLattice.bot.
exact Widen.widen.
Defined.

Section lattice_prop.

Require Import Inclusion.
Require Import Inverse_Image.

Variable t : Type.
Variable P : PosetDec.t t.
Variable B : BotDec.t t.
Variable J : JoinDec.t t.
Variable M : MeetDec.t t.
Variable N : Narrow.t t.

Lemma meet_monotone : forall x1 x2 y1 y2,
  x1 ⊑♯ x2 -> y1 ⊑♯ y2 -> (x1 ⊓♯ y1)  ⊑♯ (x2 ⊓♯ y2).
Proof.
  eauto.
Qed.

Lemma join_monotone : forall x1 x2 y1 y2,
  x1 ⊑♯ x2 -> y1 ⊑♯ y2 -> (x1 ⊔♯ y1) ⊑♯ (x2 ⊔♯ y2).
Proof.
  simpl.
  eauto.
Qed.

Lemma order_eq_order :
 forall x y z : t, x ⊑♯ y -> y =♯ z -> x ⊑♯ z.
Proof.
  eauto.
Qed.

Lemma eq_order_order :
 forall x y z : t, x =♯ y -> y ⊑♯ z -> x ⊑♯ z.
Proof.
  intros x y z H H1; apply PosetDec.trans with y; auto.
Qed.

Lemma meet_sym : forall x y :t, (x ⊓♯ y) =♯ (y ⊓♯ x).
  auto.
Qed.

Lemma join_sym : forall x y :t, (x ⊔♯ y) =♯ (y ⊔♯ x).
  auto.
Qed.

Lemma meet_assoc : forall x y z, (x ⊓♯ (y ⊓♯ z)) =♯ ((x ⊓♯ y) ⊓♯ z).
Proof.
  intros; apply PosetDec.antisym; apply MeetDec.greatest_lower_bound.
  apply MeetDec.greatest_lower_bound; auto.
  apply PosetDec.trans with (y ⊓♯ z); auto. 
  apply PosetDec.trans with (y ⊓♯ z); auto. 
  apply PosetDec.trans with (x ⊓♯ y); auto. 
  apply MeetDec.greatest_lower_bound; auto.
  apply PosetDec.trans with (x ⊓♯ y); auto. 
Qed.

Lemma join_assoc : forall x y z, (x ⊔♯ (y ⊔♯ z)) =♯ ((x ⊔♯ y) ⊔♯ z).
Proof.
  intros; apply PosetDec.antisym; apply JoinDec.least_upper_bound.
  apply PosetDec.trans with (JoinDec.op x y); auto.
  apply JoinDec.least_upper_bound; auto.
  apply PosetDec.trans with (JoinDec.op x y); auto.
  apply JoinDec.least_upper_bound; auto.
  apply PosetDec.trans with (JoinDec.op y z); auto. 
  apply PosetDec.trans with (JoinDec.op y z); auto. 
Qed.

Lemma meet_eq1 :
  forall x y z, x =♯ y -> (x ⊓♯ z) =♯ (y ⊓♯ z).
Proof.
  intros; apply PosetDec.antisym.
  apply MeetDec.greatest_lower_bound; auto.
  apply order_eq_order with x; auto.
  apply MeetDec.greatest_lower_bound; auto.
  apply order_eq_order with y; auto.
Qed.

Lemma join_eq1 :
  forall x y z, x =♯ y -> (x ⊔♯ z) =♯ (y ⊔♯ z).
Proof.
  intros; apply PosetDec.antisym.
  apply JoinDec.least_upper_bound; auto.
  apply eq_order_order with y; auto.
  apply JoinDec.least_upper_bound; auto.
  apply eq_order_order with x; auto.
Qed.

Lemma meet_eq2 :
  forall x y z, y =♯ z -> (x ⊓♯ z) =♯ (x ⊓♯ y).
Proof.
  intros; apply PosetDec.antisym.
  apply MeetDec.greatest_lower_bound; auto.
  apply order_eq_order with z; auto.
  apply MeetDec.greatest_lower_bound; auto.
  apply order_eq_order with y; auto.
Qed.

Lemma join_eq2 :
  forall x y z, y =♯ z -> (x ⊔♯ z) =♯ (x ⊔♯ y).
Proof.
  intros; apply PosetDec.antisym.
  apply JoinDec.least_upper_bound; auto.
  apply eq_order_order with y; auto.
  apply JoinDec.least_upper_bound; auto.
  apply eq_order_order with z; auto.
Qed.

Lemma meet_bottom1 : forall x : t, ⊥♯ =♯ (⊥♯ ⊓♯ x).
Proof.
  intros; apply PosetDec.antisym; auto.
Qed.

Lemma join_bottom1 : forall x : t, x =♯ (⊥♯ ⊔♯ x).
Proof.
  intros; apply PosetDec.antisym; auto.
Qed.

Lemma meet_bottom2 : forall x : t, ⊥♯ =♯ (x ⊓♯ ⊥♯).
Proof.
  intros; apply PosetDec.antisym; auto.
Qed.

Lemma join_bottom2 : forall x : t, x =♯ (x ⊔♯ ⊥♯).
Proof.
  intros; apply PosetDec.antisym; auto.
Qed.


  Lemma ascending_chain_condition_to_widening_operator :
    well_founded (fun x y => ~ y =♯ x /\ y ⊑♯ x) -> 
    Widen.t t.
  Proof.
    intros wf.
    exists JoinDec.op; intros; auto.
    apply join_eq1; auto.
    apply join_eq2; auto.
    set (R:=(fun x y : t => ~ y =♯ x /\ PosetDec.order y x)) in *.
    set (mes := fun p:t*t => snd p).
    apply Inclusion.Acc_incl with
      (fun p1 p2 => R (mes p1) (mes p2)).
    intros (x1,y1) (x2,y2); simpl; unfold R; intros.
    destruct H; simpl in *; intuition eauto.
    apply Acc_inverse_image with (f:=mes); auto.
  Defined.

  Lemma descending_chain_condition_to_narrowing_operator :
    well_founded (fun x y => ~ y =♯ x /\ PosetDec.order x y) -> 
    Narrow.t t.
  Proof.
    intros wf.
    exists MeetDec.op; intros; auto.
    apply meet_eq1; auto.
    apply meet_eq2; auto.
    set (R:=(fun x y : t => ~ y =♯ x /\ PosetDec.order x y)) in *.
    set (mes := fun p:t*t => snd p).
    apply Acc_incl with
      (fun p1 p2 => R (mes p1) (mes p2)).
    intros (x1,y1) (x2,y2); simpl; unfold R; intros.
    destruct H; simpl in *; split; auto.
    apply PosetDec.trans with (y2 ⊓♯ x1); auto.
    apply Acc_inverse_image with (f:=mes); auto.
  Qed.

  Lemma narrow_x_x: forall x:t,
    (Narrow.narrow x x) =♯ x.
  Proof.
    intros; apply PosetDec.antisym.
    apply Narrow.narrow_bound1.
    apply PosetDec.trans with (x ⊓♯ x); auto.
    apply Narrow.narrow_bound2.
  Qed.

End lattice_prop.


