Require Export lattice_def.
Require Export CompleteLattice.
Require Export ZArith.

Ltac CaseEq a := generalize (refl_equal a); pattern a at -1 in |- *; case a.

Module Sign.

  Inductive t : Set := bot | tops | pos | neg | zero | pos0 | neg0 | not0.

  Definition eq : t -> t -> Prop := fun x y => x=y.
  Lemma eq_refl : forall x : t, eq x x.
  Proof.
    destruct x; constructor.
  Qed.

  Lemma eq_sym : forall x y : t, eq x y -> eq y x.
  Proof.
    destruct 1; constructor.
  Qed.

  Lemma eq_trans : forall x y z : t, eq x y -> eq y z -> eq x z.
  Proof.
    destruct 1; destruct 1; constructor.
  Qed.

  Lemma eq_dec : forall x y : t, {eq x y}+{~ eq x y}.
  Proof.
    destruct x; destruct y; try (left;constructor) || (right;intros H;inversion H).
  Qed.

  Instance Equiv : EquivDec.t t.
  exists eq.
  apply eq_refl.
  apply eq_sym.
  apply eq_trans.
  apply eq_dec.
  Defined.

  Inductive order : t -> t -> Prop :=
    ordertops : forall x, order x tops
  | orderbot : forall x, order bot x
  | orderzero : order zero zero
  | orderpos : order pos pos
  | orderneg : order neg neg
  | orderpos0 : order pos0 pos0
  | orderpos0' : order pos pos0
  | orderpos0'' : order zero pos0
  | orderneg0 : order neg0 neg0
  | orderneg0' : order neg neg0
  | orderneg0'' : order zero neg0
  | ordernot0 : order not0 not0
  | ordernot0' : order pos not0
  | ordernot0'' : order neg not0.

  Lemma order_refl : forall x y : t, x =♯ y -> order x y.
  Proof.
    destruct 1; destruct x; try constructor.
  Qed.

  Lemma order_antisym : forall x y : t, order x y -> order y x -> eq x y.
  Proof.
    destruct 1; intros H; inversion_clear H; constructor.
  Qed.

  Lemma order_trans : forall x y z : t, order x y -> order y z -> order x z.
  Proof.
    destruct 1; intros H; inversion_clear H; constructor.
  Qed.

  Lemma order_dec : forall x y : t, {order x y}+{~ order x y}.
  Proof.
    destruct x; destruct y; try (left;constructor) || (right;intros H;inversion H).
  Qed.

  Instance Poset : PosetDec.t t.
  refine (PosetDec.Make _ _ order _ _ _ _).
  apply order_refl.
  apply order_antisym.
  apply order_trans.
  apply order_dec.
  Defined.

  Definition join  : t -> t -> t := fun x y =>
    match x,y with
    | bot,_ => y
    | _,bot => x
    | pos,pos => pos
    | neg,neg => neg
    | zero,zero => zero
    | pos,zero => pos0
    | zero,pos => pos0
    | pos0,pos0 => pos0
    | pos0,pos => pos0
    | pos0,zero => pos0
    | pos,pos0 => pos0
    | zero,pos0 => pos0
    | neg,zero => neg0
    | zero,neg => neg0
    | neg0,neg0 => neg0
    | neg0,neg => neg0
    | neg0,zero => neg0
    | neg,neg0 => neg0
    | zero,neg0 => neg0
    | pos,neg => not0
    | neg,pos => not0
    | not0,not0 => not0
    | not0,pos => not0
    | not0,neg => not0
    | neg,not0 => not0
    | pos,not0 => not0
    | _,_ => tops
    end.

  Lemma join_bound1 : forall x y : t, order x (join x y).
  Proof.
    destruct x; destruct y; constructor.
  Qed.

  Lemma join_bound2 : forall x y : t, order y (join x y).
  Proof.
    destruct x; destruct y; constructor.
  Qed.

  Lemma join_least_upper_bound : forall x y z : t, order x z -> order y z -> order (join x y) z.
  Proof.
    intros x y z H; inversion_clear H; intros H1; inversion_clear H1; try constructor.
  Qed.

  Instance Join : JoinDec.t t.
  refine (JoinDec.Make _ _ join _ _ _).
  apply join_bound1.
  apply join_bound2.
  apply join_least_upper_bound.
  Defined.

  Definition meet  : t -> t -> t := fun x y =>
    match x,y with
    | tops,_ => y
    | _,tops => x
    | pos,pos => pos
    | neg,neg => neg
    | zero,zero => zero
    | pos0,pos0 => pos0
    | pos0,pos => pos
    | pos0,zero => zero
    | pos,pos0 => pos
    | zero,pos0 => zero
    | neg0,neg0 => neg0
    | neg,neg0 => neg
    | neg0,neg => neg
    | zero,neg0 => zero
    | neg0,zero => zero
    | pos0,neg0 => zero
    | neg0,pos0 => zero
    | not0, pos0 => pos
    | not0, neg0 => neg
    | pos0, not0 => pos
    | neg0, not0 => neg
    | pos, not0 => pos
    | neg, not0 => neg
    | not0, pos => pos
    | not0, neg => neg
    | not0, not0 => not0
    | _,_ => bot
    end.

  Lemma meet_bound1 : forall x y : t, order (meet x y) x.
  Proof.
    destruct x; destruct y; constructor.
  Qed.

  Lemma meet_bound2 : forall x y : t, order (meet x y) y.
  Proof.
    destruct x; destruct y; constructor.
  Qed.

  Lemma meet_greatest_lower_bound : forall x y z : t, order z x -> order z y -> order z (meet x y).
  Proof.
    intros x y z H; inversion_clear H; intros H1; inversion_clear H1; try constructor.
  Qed.

  Instance Meet : MeetDec.t t.
  refine (MeetDec.Make _ _ meet _ _ _).
  apply meet_bound1.
  apply meet_bound2.
  apply meet_greatest_lower_bound.
  Defined.

  Definition bottom : t := bot.

  Lemma bottom_is_bottom : forall x : t, order bottom x.
  Proof.
    constructor.
  Qed.

  Instance Bot : BotDec.t t.
  exists bottom.
  apply bottom_is_bottom.
  Defined.

  Definition top : t := tops.

  Lemma top_is_top : forall x, order x top.
  Proof.
    constructor.
  Qed.

  Instance Top : TopDec.t t.
  exists top.
  apply top_is_top.
  Defined.

  Lemma acc :
   well_founded (fun x y => ~ eq y x /\ order y x).
  Proof.
   intros x.
   destruct x;
   repeat (
     constructor;
     intros y (H1,H2);
     destruct y; inversion_clear H2 in H1; try (elim H1; constructor; fail); clear H1).
  Qed.

  Lemma dcc :
   well_founded (fun x y => ~ eq y x /\ order x y).
  Proof.
   intros x.
   destruct x;
   repeat (
     constructor;
     intros y (H1,H2);
     destruct y; inversion_clear H2 in H1; try (elim H1; constructor; fail); clear H1).
  Qed.

  Require lattice_def.

  Instance Widen : Widen.t t.
  Proof.
    apply ascending_chain_condition_to_widening_operator; auto.
    apply Join.
    apply acc.
  Defined.

  Instance Narrow : Narrow.t t.
  Proof.
    apply descending_chain_condition_to_narrowing_operator.
    apply dcc.
  Defined.

  (* Instance Lattice : AbLattice.t t. *)

Open Scope Z_scope.

Inductive gamma : t -> Z -> Prop :=
 | gamma_top : forall n, gamma top n
 | gamma_zero : forall n, n=0 -> gamma zero n
 | gamma_pos : forall n, n>0 -> gamma pos n
 | gamma_neg : forall n, n<0 -> gamma neg n
 | gamma_pos0 : forall n, n>=0 -> gamma pos0 n
 | gamma_neg0 : forall n, n<=0 -> gamma neg0 n
 | gamma_not0 : forall n, n<>0 -> gamma not0 n.

Lemma gamma_monotone : forall N1 N2 n,
  order N1 N2 ->
  gamma N1 n ->
  gamma N2 n.
Proof.
  intros.
  inversion_clear H in H0; inversion_clear H0; constructor; auto; omega.
Qed.

Lemma gamma_meet_morph : forall N1 N2 n,
  gamma N1 n -> gamma N2 n -> gamma (meet N1 N2) n.
Proof.
  intros N1 N2 n H1; inversion_clear H1; simpl; try constructor;
  intros H2; inversion_clear H2; simpl; try constructor; 
  try omega || (apply False_ind;omega).
Qed.

Instance LatticeDec : LatticeDec.t t.
eapply LatticeDec.Make; auto with typeclass_instances.
Defined.

Instance Gamma : Gamma.t (℘ Z) t gamma.
apply Gamma.Make.
repeat intro; eapply gamma_monotone with N1; auto.
repeat intro.
destruct H; apply gamma_meet_morph; auto.
Defined.

Definition backtest_eq (N1 N2:t) : (t*t) :=
  (meet N1 N2,meet N1 N2).

Definition backtest_lt (N1 N2:t) : (t*t) :=
  match N1,N2 with
  | bot,_ => (bot,bot)
  | _,bot => (bot,bot)
  | pos0,pos => (pos0,pos)
  | pos0,pos0 => (pos0,pos)
  | pos0,tops => (pos0,pos)
  | pos0,not0 => (pos0,pos)
  | pos0,_ => (bot,bot)

  | pos,zero => (bot,bot)
  | pos,neg => (bot,bot)
  | pos,neg0 => (bot,bot)
  | pos,tops => (pos,pos)
  | pos,not0 => (pos,pos)
  | pos,pos0 => (pos,pos)

  | zero,zero => (bot,bot)
  | zero,neg => (bot,bot)
  | zero,neg0 => (bot,bot)
  | zero,tops => (zero,pos)
  | zero,pos0 => (zero,pos)
  | zero,not0 => (zero,pos)

  | neg0,neg => (neg,neg)
  | neg0,neg0 => (neg,neg0)
  | neg0,zero => (neg,zero)

  | not0,zero => (neg,zero)
  | not0,neg => (neg,neg)
  | not0,neg0 => (neg,neg0)

  | tops,zero => (neg,zero)
  | tops,neg => (neg,neg)
  | tops,neg0 => (neg,neg0)
  | _,_ => (N1,N2)
  end.

Lemma backtest_eq_correct : forall N1 N2 N1' N2' n1 n2,
  n1 = n2 ->
  gamma N1 n1 -> gamma N2 n2 ->
  backtest_eq N1 N2 = (N1',N2') ->
  gamma N1' n1 /\ gamma N2' n2.
Proof.
  intros N1 N2 N1' N2' n1 n2 H.
  inversion_clear H; intros H1; inversion_clear H1;
  intros H2; inversion_clear H2; simpl;
  intros H3; injection H3; clear H3; intros; subst; 
  split; try constructor; try (auto with zarith || omega || apply False_ind; omega).
Qed.

Lemma backtest_lt_correct : forall N1 N2 N1' N2' n1 n2,
  n1 < n2 ->
  gamma N1 n1 -> gamma N2 n2 ->
  backtest_lt N1 N2 = (N1',N2') ->
  gamma N1' n1 /\ gamma N2' n2.
Proof.
  intros N1 N2 N1' N2' n1 n2 H.
  intros H1; inversion_clear H1;
  intros H2; inversion_clear H2; simpl;
  intros H3; injection H3; clear H3; intros; subst;
  split; try constructor; try (auto with zarith || omega || apply False_ind; omega).
Qed.


Definition sem_add (N1 N2:t) : t := 
  match N1,N2 with
    bot,_ => bot
  | _,bot => bot
  | pos,pos => pos
  | neg,neg => neg
  | zero,_ => N2
  | pos0,pos0 => pos0
  | pos0,pos => pos
  | pos,pos0 => pos
  | neg0,neg0 => neg0
  | neg,neg0 => neg
  | neg0,neg => neg
  | _,zero => N1
  | _,_ => top
  end.

Definition sem_sub (N1 N2:t) : t := 
  match N1,N2 with
    bot,_ => bot
  | _,bot => bot
  | pos,neg => pos
  | pos,neg0 => pos
  | pos0,neg => pos
  | pos0,neg0 => pos0
  | neg,pos => neg
  | neg,pos0 => neg
  | neg0,pos => neg
  | neg0,pos0 => neg0
  | zero,pos => neg
  | zero,neg => pos
  | zero,pos0 => neg0
  | zero,neg0 => pos0
  | _,zero => N1
  | _,_ => top
  end.

Definition sem_mult (N1 N2:t) : t := 
  match N1,N2 with
    bot,_ => bot
  | _,bot => bot
  | pos,pos0 => pos0
  | pos0,pos => pos0
  | pos0,pos0 => pos0
  | pos,pos => pos
  | neg0,neg0 => pos0
  | neg0,neg => pos0
  | neg,neg0 => pos0
  | neg,neg => pos
  | pos,neg => neg
  | neg,pos => neg
  | neg,not0 => not0
  | pos,not0 => not0
  | not0,neg => not0
  | not0,pos => not0
  | zero,_ => zero
  | _,zero => zero
  | _,_ => top
  end.

Lemma sem_add_correct : forall N1 N2 n1 n2 n,
  gamma N1 n1 -> gamma N2 n2 -> 
   n = n1 + n2 ->
  gamma (sem_add N1 N2) n.
Proof.
  intros N1 N2 n1 n2 n H; inversion_clear H;
  intros H1; inversion_clear H1;
  intros H2; inversion_clear H2; 
  constructor; try (omega || subst; omega).
Qed.

Lemma sem_sub_correct : forall N1 N2 n1 n2 n,
  gamma N1 n1 -> gamma N2 n2 -> 
   n = n1 - n2 ->
  gamma (sem_sub N1 N2) n.
Proof.
  intros N1 N2 n1 n2 n H; inversion_clear H;
  intros H1; inversion_clear H1;
  intros H2; inversion_clear H2; 
  constructor; try (omega || subst; omega).
Qed.

Lemma sem_mult_correct : forall N1 N2 n1 n2 n,
  gamma N1 n1 -> gamma N2 n2 -> 
   n = n1 * n2 ->
  gamma (sem_mult N1 N2) n.
Proof.
  intros N1 N2 n1 n2 n H; inversion_clear H;
  intros H1; inversion_clear H1;
  intros H2; inversion_clear H2;
  constructor; try (omega || subst; omega).
  auto with zarith.
  replace 0 with (n1*0).
  apply Zmult_gt_0_lt_compat_l; auto.
  ring.
  generalize (Zmult_le_0_compat n1 n2); omega.
  intro; elim H.
  elim (Zmult_integral _ _ H1); omega.
  auto with zarith.
  replace (n1*n2) with (- ((-n1)*(n2))).
  generalize (Zmult_gt_0_compat (-n1) (n2)); omega.
  ring.
  replace (n1*n2) with (((-n1)*(-n2))).
  generalize (Zmult_gt_0_compat (-n1) (-n2)); omega.
  ring.
  replace (n1*n2) with (((-n1)*(-n2))).
  generalize (Zmult_le_0_compat (-n1) (-n2)); omega.
  ring.
  intro; elim H.
  elim (Zmult_integral _ _ H1); omega.
  generalize (Zmult_le_0_compat (n1) (n2)); omega.
  generalize (Zmult_le_0_compat (n1) (n2)); omega.
  replace (n1*n2) with (((-n1)*(-n2))).
  generalize (Zmult_le_0_compat (-n1) (-n2)); omega.
  ring.
  replace (n1*n2) with (((-n1)*(-n2))).
  generalize (Zmult_le_0_compat (-n1) (-n2)); omega.
  ring.
  intro; elim H.
  elim (Zmult_integral _ _ H1); omega.
  intro; elim H.
  elim (Zmult_integral _ _ H1); omega.
Qed.

Definition backsem_add (N N1 N2:t) : t*t := 
  match N,N1,N2 with
    bot,_,_ => (bot,bot)
  | _,bot,_ => (bot,bot)
  | _,_,bot => (bot,bot)

  | pos,not0,neg => (pos,neg)
  | pos,not0,zero => (pos,zero)
  | pos,zero,not0 => (zero,pos)
  | pos,neg,not0 => (neg,pos)
  | pos,neg,neg => (bot,bot)
  | pos,neg,zero => (bot,bot)
  | pos,neg,neg0 => (bot,bot)
  | pos,zero,neg => (bot,bot)
  | pos,zero,zero => (bot,bot)
  | pos,zero,neg0 => (bot,bot)
  | pos,neg0,neg => (bot,bot)
  | pos,neg0,zero => (bot,bot)
  | pos,neg0,neg0 => (bot,bot)
  | pos,pos0,neg => (pos,neg)
  | pos,tops,neg => (pos,neg)
  | pos,neg,_ => (neg,pos)
  | pos,neg0,_ => (neg0,pos)
  | pos,pos0,zero => (pos,zero)
  | pos,tops,zero => (pos,zero)
  | pos,zero,pos0 => (zero,pos)
  | pos,zero,tops => (zero,pos)

  | neg,not0,zero => (neg,zero)
  | neg,zero,not0 => (zero,neg)
  | neg,pos,not0 => (pos,neg)
  | neg,not0,pos => (neg,pos)
  | neg,pos,pos => (bot,bot)
  | neg,pos,zero => (bot,bot)
  | neg,pos,pos0 => (bot,bot)
  | neg,pos0,pos0 => (bot,bot)
  | neg,pos0,pos => (bot,bot)
  | neg,pos0,zero => (bot,bot)
  | neg,zero,pos0 => (bot,bot)
  | neg,zero,pos => (bot,bot)
  | neg,zero,zero => (bot,bot)
  | neg,tops,pos => (neg,pos)
  | neg,neg0,pos => (neg,pos)
  | neg,pos,tops => (pos,neg)
  | neg,pos,_ => (pos,neg)
  | neg,pos0,_ => (pos0,neg)
  | neg,neg0,zero => (neg,zero)
  | neg,tops,zero => (neg,zero)
  | neg,zero,neg0 => (zero,neg)
  | neg,zero,tops => (zero,neg)

  | zero,neg,neg => (bot,bot)
  | zero,neg,zero => (bot,bot)
  | zero,neg,neg0 => (bot,bot)
  | zero,pos,pos => (bot,bot)
  | zero,pos,zero => (bot,bot)
  | zero,pos,pos0 => (bot,bot)
  | zero,zero,not0 => (bot,bot)
  | zero,not0,zero => (bot,bot)
  | zero,zero,pos => (bot,bot)
  | zero,zero,neg => (bot,bot)
  | zero,pos0,pos => (bot,bot)
  | zero,neg0,neg => (bot,bot)
  | zero,tops,neg => (pos,neg)
  | zero,_,zero => (zero,zero)
  | zero,tops,pos => (neg,pos)
  | zero,tops,pos0 => (neg0,pos0)
  | zero,tops,neg0 => (pos0,neg0)
  | zero,neg,tops => (neg,pos)
  | zero,neg,pos0 => (neg,pos)
  | zero,neg0,tops => (neg0,pos0)
  | zero,neg0,pos => (neg,pos)
  | zero,neg0,neg0 => (zero,zero)
  | zero,zero,_ => (zero,zero)
  | zero,pos,tops => (pos,neg)
  | zero,pos,neg0 => (pos,neg)
  | zero,pos0,pos0 => (zero,zero)
  | zero,pos0,tops => (pos0,neg0)
  | zero,pos0,neg => (pos,neg)
  | zero,not0,neg => (pos,neg)
  | zero,not0,neg0 => (pos,neg)
  | zero,not0,pos => (neg,pos)
  | zero,not0,pos0 => (neg,pos)
  | zero,neg,not0 => (neg,pos)
  | zero,neg0,not0 => (neg,pos)
  | zero,pos,not0 => (pos,neg)
  | zero,pos0,not0 => (pos,neg)

  | _,_,_ => (N1,N2)
  end.


Definition minus (N:t) : t :=
  match N with
    bot => bot
  | tops => top
  | pos => neg
  | neg => pos
  | pos0 => neg0
  | neg0 => pos0
  | zero => zero
  | not0 => not0
  end.

Lemma minus_correct : forall N n,
  gamma N n -> gamma (minus N) (-n).
Proof.
  destruct 1; simpl; constructor; omega.
Qed.

Definition backsem_sub (N N1 N2:t) : t*t := 
  let (x,y) := backsem_add N N1 (minus N2) in 
   (x,minus y).

Lemma sign_pos_pos: forall x y,
  x > 0 -> y > 0 -> x*y > 0.
Proof.
  intros; replace 0 with (x*0); auto with zarith.
  apply Zmult_gt_compat_l; auto .
Qed.

Lemma sign_pos_zero: forall x y,
  x > 0 -> y = 0 -> x*y = 0.
Proof.
  intros; subst; ring.
Qed.

Lemma sign_pos_neg: forall x y,
  x > 0 -> y < 0 -> x*y < 0.
Proof.
  intros; replace 0 with (x*0); auto with zarith.
  apply Zmult_lt_compat_l; auto with zarith.
Qed.

Lemma sign_zer_pos: forall x y,
  x = 0 -> y > 0 -> x*y = 0.
Proof.
  intros; subst; ring.
Qed.

Lemma sign_zero_zero: forall x y,
  x = 0 -> y = 0 -> x*y = 0.
Proof.
  intros; subst; ring.
Qed.

Lemma sign_zero_neg: forall x y,
  x = 0 -> y < 0 -> x*y = 0.
Proof.
  intros; subst; ring.
Qed.

Lemma sign_neg_pos: forall x y,
  x < 0 -> y > 0 -> x*y < 0.
Proof.
  intros; replace 0 with (0*y); auto with zarith.
  apply Zmult_lt_compat_r; auto with zarith.
Qed.

Lemma sign_neg_zero: forall x y,
  x < 0 -> y = 0 -> x*y = 0.
Proof.
  intros; subst; ring.
Qed.

Lemma sign_neg_neg: forall x y,
  x < 0 -> y < 0 -> x*y > 0.
Proof.
  intros; replace (x*y) with (-x*-y).
  apply sign_pos_pos; auto with zarith.
  ring.
Qed.

Lemma backsem_add_correct : forall N N1 N2 n1 n2 n N1' N2',
  gamma N1 n1 -> gamma N2 n2 ->
  n = n1 + n2 -> gamma N n ->
  backsem_add N N1 N2 = (N1',N2') ->
  gamma N1' n1 /\ gamma N2' n2.
Proof.
  intros N N1 N2 n1 n2 n N1' N2' H H1.
  generalize H; generalize H1.
  inversion_clear H1; inversion_clear H; intros HH1 HH2;
  intros H3; inversion_clear H3; 
  intros H4; inversion_clear H4; simpl;
  intros H5; injection H5; intros; subst;
  split; try assumption || constructor; try (apply False_ind; omega) || omega.
Qed.

Lemma backsem_sub_correct : forall N N1 N2 n1 n2 n N1' N2',
  gamma N1 n1 -> gamma N2 n2 ->
  n = n1 - n2 -> gamma N n ->
  backsem_sub N N1 N2 = (N1',N2') ->
  gamma N1' n1 /\ gamma N2' n2.
Proof.
  intros N N1 N2 n1 n2 n N1' N2' H H1.
  generalize H; generalize H1.
  inversion_clear H1; inversion_clear H; intros HH1 HH2;
  intros H3; inversion_clear H3; 
  intros H4; inversion_clear H4; simpl;
  intros H5; injection H5; intros; subst;
  split; try assumption || constructor; try (apply False_ind; omega) || omega.
Qed.

Definition backsem_mult (N N1 N2:t) : t*t := 
  match N,N1,N2 with
    bot,_,_ => (bot,bot)
  | _,bot,_ => (bot,bot)
  | _,_,bot => (bot,bot)

  | pos,not0,neg => (neg,neg)
  | pos,not0,neg0 => (neg,neg)
  | pos,not0,pos => (pos,pos)
  | pos,not0,pos0 => (pos,pos)
  | pos,neg,not0 => (neg,neg)
  | pos,neg0,not0 => (neg,neg)
  | pos,pos,not0 => (pos,pos)
  | pos,pos0,not0 => (pos,pos)
  | pos,pos,neg => (bot,bot)
  | pos,pos,neg0 => (bot,bot)
  | pos,pos0,neg => (bot,bot)
  | pos,pos0,neg0 => (bot,bot)
  | pos,neg,pos => (bot,bot)
  | pos,neg0,pos => (bot,bot)
  | pos,neg,pos0 => (bot,bot)
  | pos,neg0,pos0 => (bot,bot)
  | pos,_,zero => (bot,bot)
  | pos,zero,_ => (bot,bot)
  | pos,tops,neg => (neg,neg)
  | pos,neg0,neg => (neg,neg)
  | pos,tops,pos => (pos,pos)
  | pos,pos0,pos => (pos,pos)
  | pos,neg,tops => (neg,neg)
  | pos,neg,neg0 => (neg,neg)
  | pos,pos,tops => (pos,pos)
  | pos,pos,pos0 => (pos,pos)
  | pos,pos0,pos0 => (pos,pos)
  | pos,neg0,neg0 => (neg,neg)
  | pos,pos0,tops => (pos,pos)
  | pos,tops,pos0 => (pos,pos)
  | pos,tops,neg0 => (neg,neg)
  | pos,neg0,tops => (neg,neg)

  | neg,not0,pos => (neg,pos)
  | neg,not0,pos0 => (neg,pos)
  | neg,not0,neg => (pos,neg)
  | neg,not0,neg0 => (pos,neg)
  | neg,pos,not0 => (pos,neg)
  | neg,pos0,not0 => (pos,neg)
  | neg,neg,not0 => (neg,pos)
  | neg,neg0,not0 => (neg,pos)
  | neg,pos,pos => (bot,bot)
  | neg,pos,pos0 => (bot,bot)
  | neg,pos0,pos => (bot,bot)
  | neg,pos0,pos0 => (bot,bot)
  | neg,neg,neg => (bot,bot)
  | neg,neg0,neg => (bot,bot)
  | neg,neg,neg0 => (bot,bot)
  | neg,neg0,neg0 => (bot,bot)
  | neg,_,zero => (bot,bot)
  | neg,zero,_ => (bot,bot)
  | neg,tops,pos => (neg,pos)
  | neg,neg0,pos => (neg,pos)
  | neg,tops,neg => (pos,neg)
  | neg,pos0,neg => (pos,neg)
  | neg,neg,tops => (neg,pos)
  | neg,neg,pos0 => (neg,pos)
  | neg,neg0,pos0 => (neg,pos)
  | neg,pos,tops => (pos,neg)
  | neg,pos,neg0 => (pos,neg)
  | neg,pos0,neg0 => (pos,neg)
  | neg,pos0,tops => (pos,neg)
  | neg,tops,pos0 => (neg,pos)
  | neg,neg0,tops => (neg,pos)
  | neg,tops,neg0 => (pos,neg)

  | zero,pos0,not0 => (zero,not0)
  | zero,neg0,not0 => (zero,not0)
  | zero,not0,pos0 => (not0,zero)
  | zero,not0,neg0 => (not0,zero)
  | zero,not0,pos => (bot,bot)
  | zero,not0,neg => (bot,bot)
  | zero,not0,not0 => (bot,bot)
  | zero,pos,not0 => (bot,bot)
  | zero,neg,not0 => (bot,bot)
  | zero,pos,pos => (bot,bot)
  | zero,pos,neg => (bot,bot)
  | zero,neg,pos => (bot,bot)
  | zero,neg,neg => (bot,bot)
  | zero,tops,pos => (zero,pos)
  | zero,pos0,pos => (zero,pos)
  | zero,neg0,pos => (zero,pos)
  | zero,tops,neg => (zero,neg)
  | zero,pos0,neg => (zero,neg)
  | zero,neg0,neg => (zero,neg)
  | zero,pos,tops => (pos,zero)
  | zero,pos,pos0 => (pos,zero)
  | zero,pos,neg0 => (pos,zero)
  | zero,neg,tops => (neg,zero)
  | zero,neg,pos0 => (neg,zero)
  | zero,neg,neg0 => (neg,zero)

  | not0,zero,_ => (bot,bot)
  | not0,_,zero => (bot,bot)
  | not0,pos0,pos0 => (pos,pos)
  | not0,neg0,neg0 => (neg,neg)
  | not0,neg0,pos0 => (neg,pos)
  | not0,pos0,neg0 => (pos,neg)
  | not0,pos0,_ => (pos,N2)
  | not0,_,pos0 => (N1,pos)
  | not0,neg0,_ => (neg,N2)
  | not0,_,neg0 => (N1,neg)

  | _,_,_ => (N1,N2)
  end.

Lemma split_ge0 : forall n, n>=0 -> n>0 \/ n=0.
Proof.
  intros; omega.
Qed.

Lemma split_le0 : forall n, n<=0 -> n<0 \/ n=0.
Proof.
  intros; omega.
Qed.

Lemma backsem_mult_correct : forall N N1 N2 n1 n2 n N1' N2',
  gamma N1 n1 -> gamma N2 n2 ->
  n = n1 * n2 -> gamma N n ->
  backsem_mult N N1 N2 = (N1',N2') ->
  gamma N1' n1 /\ gamma N2' n2.
Proof.
  intros N N1 N2 n1 n2 n N1' N2' H H1.
  generalize H; generalize H1.
   inversion_clear H1; inversion_clear H; intros HH1 HH2;
  intros H3; inversion_clear H3; 
  intros H4; inversion_clear H4; simpl;
  intros H5; injection H5; intros; subst; 
  (split; 
  [try assumption; (constructor||apply False_ind); try omega;
   try (destruct (Ztrichotomy n1 0) as [Hn2 | [ Hn2 | Hn2 ]]; (omega || (apply False_ind; try omega))) |
   try assumption; (constructor||apply False_ind); try omega;
   try (destruct (Ztrichotomy n2 0) as [Hn2 | [ Hn2 | Hn2 ]]; (omega || (apply False_ind; try omega)))];
   repeat (match goal with
   | id : _ >= 0 |- _ => destruct (split_ge0 _ id); clear id
   | id : _ <= 0 |- _ => destruct (split_le0 _ id); clear id
   end); 
   subst; try omega;
   try match goal with
   | id1 : n1*n2=0 |- _ => elim (Zmult_integral _ _ id1); omega
   | id1 : n1 >0, id2 : n2 <0 |- _ => generalize (sign_pos_neg n1 n2 id1 id2); intros; omega
   | id1 : n1 <0, id2 : n2 <0 |- _ => generalize (sign_neg_neg n1 n2 id1 id2); intros; omega
   | id1 : n1 <0, id2 : n2 >0 |- _ => generalize (sign_neg_pos n1 n2 id1 id2); intros; omega
   | id1 : n1 >0, id2 : n2 >0 |- _ => generalize (sign_pos_pos n1 n2 id1 id2); intros; omega
  end).
Qed.

Definition const (n:Z) : t := 
  if (Z_eq_dec 0 n) then zero else
  (if (Z_le_dec 0 n) then pos else neg).

Lemma const_correct : forall n, gamma (const n) n.
Proof.
  unfold const; intros.
  case Z_eq_dec; intros.
  constructor; auto.
  case Z_le_dec; intros; constructor; omega.
Qed.


Lemma bottom_empty : forall n, ~ gamma bottom n.
Proof.
  red; intros.
  inversion_clear H.
Qed.

Definition check_positive (N:t) :=
  match N with
    bot => true
  | tops => false
  | pos => true
  | neg => false
  | pos0 => true
  | neg0 => false
  | zero => true
  | not0 => false
  end.

Lemma check_positive_correct : forall N n,
  check_positive N = true ->
  gamma N n -> n>=0.
Proof.
  destruct N; simpl; intros; try discriminate;
  inversion_clear H0; omega.
Qed.

Definition check_in_range (N:t) (a b:Z) :=
  match N with
    bot => true
  | tops => false
  | pos => false
  | neg => false
  | pos0 => false
  | neg0 => false
  | zero => if Z_lt_dec a 0 then if Z_lt_dec 0 b then true else false else false
  | not0 => false
  end.

Lemma check_in_range_correct : forall N z1 z2 n,
  check_in_range N z1 z2 = true ->
  gamma N n -> z1 < n < z2.
Proof.
  destruct N; simpl; intros; try discriminate; inversion_clear H0.
  repeat destruct Z_lt_dec; omega || discriminate.
Qed.

Definition is_constant (N:t) :=
  match N with
    bot => Some 0
  | tops => None
  | pos => None
  | neg => None
  | pos0 => None
  | neg0 => None
  | zero => Some 0
  | not0 => None
  end.

Lemma check_is_constant_correct : forall N z n,
  is_constant N = Some z ->
  gamma N n -> n = z.
Proof.
  destruct N; simpl; intros; try discriminate; inversion_clear H0.
  congruence.
Qed.

End Sign.
