
(** Lattices contructed by combining other lattices **)

Set Implicit Arguments.
Require Export Lift.


Notation "'ifd' b 'then' v1 'else' v2" :=
  (match b with
   | left _ => v1
   | right _ => v2
   end)
  (at level 200, right associativity) : type_scope.

Notation " [ ] " := nil : list_scope.
Notation " [ x ] " := (cons x nil) : list_scope.
Notation " [ x ; .. ; y ] " := (cons x .. (cons y nil) ..) : list_scope.


Module SumLattice.

Section SumLattice.

Variables T1 T2 : Type.
Hypothesis L1 : LatticeDec.t T1.
Hypothesis L2 : LatticeDec.t T2.

Definition t0 := (T1 + T2)%type.
Definition t := option t0.

Inductive eqt0 : t0 → t0 → Prop :=
  | eqt_left : forall t1 t2,
    t1 =♯ t2 →
    eqt0 (inl t1) (inl t2)
  | eqt_right : forall t1 t2,
    t1 =♯ t2 →
    eqt0 (inr t1) (inr t2)
  | eqt_bot_rl : forall t1 t2,
    t1 =♯ (⊥♯) →
    t2 =♯ (⊥♯) →
    eqt0 (inr t1) (inl t2)
  | eqt_bot_lr : forall t1 t2,
    t1 =♯ (⊥♯) →
    t2 =♯ (⊥♯) →
    eqt0 (inl t1) (inr t2)
  .

Global Instance Equiv0 : EquivDec.t t0.
  apply EquivDec.Make with eqt0; try solve [
     repeat intros [?|?]; repeat (let H := fresh in intro H; inverts H); constructors* ].
  intros [x|x] [y|y];
    ((destruct (EquivDec.dec x y); [left; constructors*
                       |right; let H := fresh in intro H; inverts* H])
    || (destruct (EquivDec.dec x (⊥♯)); destruct (EquivDec.dec y (⊥♯));
         ((left; solve [ constructors~ ]) || (right; let H := fresh in intro H; inverts* H)))).
Defined.

Inductive order : t0 → t0 → Prop :=
  | order_left : forall t1 t2,
    t1 ⊑♯ t2 →
    order (inl t1) (inl t2)
  | order_right : forall t1 t2,
    t1 ⊑♯ t2 →
    order (inr t1) (inr t2)
  | order_bot : forall t1 t2,
    t1 =♯ inl (⊥♯) →
    order t1 t2
  .

Global Instance Poset0 : PosetDec.t t0.
  apply PosetDec.Make with
    (eq := Equiv0) (order := order).
  repeat intros [?|?]; repeat (let H := fresh in intro H; inverts H);
    (solve [ constructors~ ] || apply order_bot; constructors*).
  introv O1 O2. repeat inverts O1 as O1; repeat inverts O2 as O2; try solve [ constructors* ].
  introv O1 O2. repeat inverts O1 as O1; try solve [ apply order_bot; constructors~ ];
    repeat inverts O2 as O2; try solve [ constructors~; applys @PosetDec.trans O1 O2 ];
    apply order_bot; constructors*.
  intros [x|x] [y|y];
    (destruct (EquivDec.dec x (⊥♯)); [left; apply order_bot; constructors*|]);
    ((destruct (PosetDec.dec x y); [left; constructors~
                                   |right; intro O; repeat (inverts O as O); autos~])
     || (right; intro O; repeat inverts O as O; autos~)).
Defined.

Definition make1 t1 : t := Some (inl t1).
Definition make2 t2 : t := Some (inr t2).

Global Instance Poset : PosetDec.t t.
  apply CoLattice.PosetDec. apply LiftPosetDec. apply CoLattice.PosetDec. apply Poset0.
Defined.

Global Instance Top : TopDec.t t.
  apply TopDec.Make with None; simpl; autos*.
Defined.

Global Instance Bot : BotDec.t t.
  apply BotDec.Make with (Some (inl (⊥♯))); intros [[?|?]|]; simpl; constructors*.
Defined.

Global Instance Join : JoinDec.t t.
  apply JoinDec.Make with (fun x y =>
    match x, y return t with
    | Some (inl x), Some (inl y) => Some (inl (x ⊔♯ y))
    | Some (inr x), Some (inr y) => Some (inr (x ⊔♯ y))
    | Some (inl x), Some (inr y) =>
      ifd EquivDec.dec x (⊥♯) then Some (inr y)
      else ifd EquivDec.dec y (⊥♯) then Some (inl x)
      else None
    | Some (inr x), Some (inl y) =>
      ifd EquivDec.dec x (⊥♯) then Some (inl y)
      else ifd EquivDec.dec y (⊥♯) then Some (inr x)
      else None
    | _, _ => None
    end); repeat intros [[?|?]|];
    repeat (let H := fresh in intro H; repeat inverts H as H);
    simpls; repeat cases_if;
    try solve [ constructors; apply PosetDec.trans with (⊥♯); autos~ ];
    try solve [ repeat constructors~ ];
    constructors*.
Defined.

Global Instance Meet : MeetDec.t t.
  apply MeetDec.Make with (fun x y =>
    match x, y return t with
    | Some (inl x), Some (inl y) => Some (inl (x ⊓♯ y))
    | Some (inr x), Some (inr y) => Some (inr (x ⊓♯ y))
    | None, _ => y
    | _, None => x
    | _, _ => ⊥♯
    end); repeat intros [[?|?]|];
    repeat (let H := fresh in intro H; repeat inverts H as H); simpls;
    try solve [ constructors; apply PosetDec.trans with (⊥♯); autos~ ];
    try solve [ repeat constructors~ ].
Defined.

Definition widen x y : t :=
  match x, y with
  | Some (inl x), Some (inl y) => Some (inl (x ∇♯ y))
  | Some (inr x), Some (inr y) => Some (inr (x ∇♯ y))
  | _, _ => y
  end.

Global Instance Lattice : LatticeDec.t t.
  apply LatticeDec.Make with Poset;
    [| | |exact widen]; typeclass.
Defined.

Variables C1 C2 : Type.
Variable gamma1 : T1 → C1 → Prop.
Variable gamma2 : T2 → C2 → Prop.
Hypothesis G1 : Gamma.t (℘ C1) T1 gamma1.
Hypothesis G2 : Gamma.t (℘ C2) T2 gamma2.

Inductive gamma : t → ℘ (C1 + C2) :=
  | gamma_left : forall c t,
    gamma1 t c →
    gamma (Some (inl t)) (inl c)
  | gamma_right : forall c t,
    gamma2 t c →
    gamma (Some (inr t)) (inr c)
  | gamma_top : forall c,
    gamma None c
  (* These are strange cases, but I didn't see where we actually assumes they can't happen. *)
  | gamma_left_bot : forall c t,
    gamma1 (⊥♯) c →
    gamma t (inl c)
  | gamma_right_bot : forall c t,
    gamma2 (⊥♯) c →
    gamma t (inr c)
  .

Global Instance Gamma : Gamma.t (℘ (C1 + C2)) t gamma.
Proof.
  apply Gamma.Make.
   intros [[t1|t1]|] [[t2|t2]|] I [c|c] G; inverts G as G; (repeat inverts I as I);
    constructors~; try solve [ first [
        applys~ @Gamma.monotone G1; applys~ gamma_eq I
      | applys~ @Gamma.monotone G2; applys~ gamma_eq I
      | applys~ @Gamma.monotone G1 G
      | applys~ @Gamma.monotone G2 G ] ].
   intros [[t1|t1]|] [[t2|t2]|] [c|c] [Hg1 Hg2]; inverts Hg1 as Hg1; inverts Hg2 as Hg2;
    constructors~; (
        (applys~ @Gamma.meet_morph G1; simpl; splits~)
     || (applys~ @Gamma.meet_morph G2; simpl; splits~) || idtac);
     (applys~ @Gamma.monotone G1 || applys~ @Gamma.monotone G2).
Qed.

End SumLattice.

End SumLattice.


Section ProductSumGamma.

Variables T1 T2 : Type.
Variable L1 : LatticeDec.t T1.
Variable L2 : LatticeDec.t T2.
Variables C1 C2 : Type.
Variable gamma1 : T1 → C1 → Prop.
Variable gamma2 : T2 → C2 → Prop.
Hypothesis G1 : Gamma.t (℘ C1) T1 gamma1.
Hypothesis G2 : Gamma.t (℘ C2) T2 gamma2.

Inductive productSumGamma : (T1 * T2) → (C1 + C2) → Prop :=
  | productSumGamma_left : forall t1 t2 c,
    gamma1 t1 c →
    productSumGamma (t1, t2) (inl c)
  | productSumGamma_right : forall t1 t2 c,
    gamma2 t2 c →
    productSumGamma (t1, t2) (inr c)
  .

Global Instance ProductSumGamma : Gamma.t (℘ (C1 + C2)) (T1 * T2) productSumGamma.
Proof.
  apply Gamma.Make.
   intros [t11 t12] [t21 t22] [I1 I2] [c|c] G; inverts G as G; simpls; constructors.
    applys~ @Gamma.monotone G1 I1.
    applys~ @Gamma.monotone G2 I2.
   intros [t11 t12] [t21 t22] [c|c] [Hg1 Hg2]; inverts Hg1 as Hg1; inverts Hg2 as Hg2;
     constructors; simpls.
    applys @Gamma.meet_morph G1; simpl; splits~.
    applys @Gamma.meet_morph G2; simpl; splits~.
Qed.

End ProductSumGamma.


Section ProductProdGamma.

Variables T1 T2 : Type.
Variable L1 : LatticeDec.t T1.
Variable L2 : LatticeDec.t T2.
Variables C1 C2 : Type.
Variable gamma1 : T1 → C1 → Prop.
Variable gamma2 : T2 → C2 → Prop.
Hypothesis G1 : Gamma.t (℘ C1) T1 gamma1.
Hypothesis G2 : Gamma.t (℘ C2) T2 gamma2.

Definition productProdGamma (t : T1 * T2) (c : C1 * C2) : Prop :=
  gamma1 (fst t) (fst c) ∧ gamma2 (snd t) (snd c).

Global Instance ProductProdGamma : Gamma.t (℘ (C1 * C2)) (T1 * T2) productProdGamma.
Proof.
  apply Gamma.Make.
   intros [t11 t12] [t21 t22] [I1 I2] [c1 c2] G; inverts G as G; simpls; constructors.
    applys~ @Gamma.monotone G1 I1.
    applys~ @Gamma.monotone G2 I2.
   intros [t11 t12] [t21 t22] [c1 c2] [Hg1 Hg2]; inverts Hg1 as Hg1; inverts Hg2 as Hg2;
     constructors; simpls.
    applys @Gamma.meet_morph G1; simpl; splits~.
    applys @Gamma.meet_morph G2; simpl; splits~.
Qed.

End ProductProdGamma.


Module Extend.

(** A lattice for container objects ([option]-like), whose abstracted
  element is in the middle of it, alongside some additionnal
  information. **)

Section Extend.

Variable T : Type.
Hypothesis L : LatticeDec.t T.
Variable e : Type → Type.
Variable param : Type.
Hypothesis paramC : Comparable param.
Variable ext : forall A, param → A → e A.
Variable extr : forall A, e A → option (A * param)%type.
Hypothesis extr_ext : forall A a p, @extr A (@ext A p a) = Some (a, p).
(*Hypothesis ext_extr : forall A et a p, @extr A et = Some (a, p) → ext p a = et.*)
Hypothesis extr_none_comp : forall A et1 et2,
  @extr A et1 = None →
  @extr A et2 = None →
  Decidable (et1 = et2).

Definition t := LiftTop.t (LiftBot.t (e T)).

Global Instance Equiv : EquivDec.t t.
  clear extr_ext.
  do 2 apply LiftEquivDec.
  apply EquivDec.Make with
    (eq := fun et1 et2 =>
      match extr et1, extr et2 with
      | Some (a1, p1), Some (a2, p2) => a1 =♯ a2 ∧ p1 = p2
      | _, _ => et1 = et2
      end);
    repeat (let x := fresh "x" in let a := fresh "a" in
            let p := fresh "p" in let E := fresh "E" in
            intro x; destruct (extr x) as [[a p]|] eqn:E);
    repeat intros (?&?); intros; substs; autos~; tryfalse.
   splits~. eapply EquivDec.trans; eassumption.
   destruct (decide (p = p0)) eqn:D; fold_bool; rew_refl in D.
    substs. destruct (EquivDec.dec a a0); [left*|right*].
    right*.
   right. intro_subst. false.
   right. intro_subst. false.
   apply decidable_sumbool. apply~ extr_none_comp.
Defined.

Global Instance Poset : PosetDec.t t.
  clear extr_ext.
  apply PosetDec.Make with (eq := Equiv)
    (order := fun et1 et2 =>
      match et1, et2 with
      | _, None => True
      | None, Some _ => False
      | Some eb1, Some eb2 =>
        match eb1, eb2 with
        | None, _ => True
        | Some _, None => False
        | Some t1, Some t2 =>
          match extr t1, extr t2 with
          | Some (a1, p1), Some (a2, p2) => a1 ⊑♯ a2 ∧ p1 = p2
          | None, None => t1 = t2
          | _, _ => False
          end
        end
      end);
    repeat (let x := fresh "x" in let a := fresh "a" in
            let p := fresh "p" in let E := fresh "E" in
            intros [[x|]|]; [destruct (extr x) as [[a p]|] eqn:E| |]);
    intros; substs; autos~; tryfalse;
    try (simpls; try rewrite E in *; try rewrite E0 in *; tryfalse; autos~);
    intuition; substs~; try splits~.
   applys~ @PosetDec.trans a0.
   destruct (decide (p = p0)) eqn:D; fold_bool; rew_refl in D.
    substs. destruct (PosetDec.dec a a0); [left*|right*].
    right*.
   apply decidable_sumbool. apply~ extr_none_comp.
Defined.

Definition extr_destr A B t1 t2 C1 (C2 C3 C4 : B) :=
  match @extr A t1 as sa1, @extr A t2 as sa2
        return extr t1 = sa1 → extr t2 = sa2 → _ with
  | Some a1, Some a2 => fun _ _ => C1 a1 a2
  | None, None => fun E1 E2 =>
    if @decide _ (extr_none_comp E1 E2) then C2 else C3
  | _, _ => fun _ _ => C4
  end eq_refl eq_refl.

Lemma extr_destr_some_some : forall A B t1 t2 C1 (C2 C3 C4 : B) a1 a2,
  @extr A t1 = Some a1 →
  @extr A t2 = Some a2 →
  extr_destr t1 t2 C1 C2 C3 C4 = C1 a1 a2.
Proof.
  introv E1 E2. unfolds extr_destr.
  set (pi := fun E1 E2 => ifb t1 = t2 then C2 else C3). clearbody pi.
  destruct (extr t1); tryfalse. destruct (extr t2); tryfalse.
  inverts E1. inverts~ E2.
Qed.

Lemma extr_destr_none_none_eq : forall A B t1 C1 (C2 C3 C4 : B),
  @extr A t1 = None →
  extr_destr t1 t1 C1 C2 C3 C4 = C2.
Proof.
  introv E. unfolds extr_destr.
  set (pi := fun E1 E2 => ifb t1 = t1 then C2 else C3).
  asserts E': (pi = fun _ _ => C2).
    extens. intros E1 E2. unfold pi. cases_if~. fold_bool. rew_refl* in H.
  clearbody pi. rewrite E'. destruct~ (extr t1); tryfalse.
Qed.

Lemma extr_destr_none_none_neq : forall A B t1 t2 C1 (C2 C3 C4 : B),
  @extr A t1 = None →
  @extr A t2 = None →
  t1 ≠ t2 →
  extr_destr t1 t2 C1 C2 C3 C4 = C3.
Proof.
  introv E1 E2 D. unfolds extr_destr.
  set (pi := fun E1 E2 => ifb t1 = t2 then C2 else C3).
  asserts E': (pi = fun _ _ => C3).
    extens. intros E1' E2'. unfold pi. cases_if~. fold_bool. rew_refl* in H.
  clearbody pi. rewrite E'.
  destruct (extr t1); tryfalse. destruct~ (extr t2); tryfalse.
Qed.

Lemma extr_destr_some_none : forall A B t1 t2 C1 (C2 C3 C4 : B) a1,
  @extr A t1 = Some a1 →
  @extr A t2 = None →
  extr_destr t1 t2 C1 C2 C3 C4 = C4.
Proof.
  introv E1 E2. unfolds extr_destr.
  set (pi := fun E1 E2 => ifb t1 = t2 then C2 else C3). clearbody pi.
  destruct (extr t1); tryfalse. destruct~ (extr t2); tryfalse.
Qed.

Lemma extr_destr_none_some : forall A B t1 t2 C1 (C2 C3 C4 : B) a2,
  @extr A t1 = None →
  @extr A t2 = Some a2 →
  extr_destr t1 t2 C1 C2 C3 C4 = C4.
Proof.
  introv E1 E2. unfolds extr_destr.
  set (pi := fun E1 E2 => ifb t1 = t2 then C2 else C3). clearbody pi.
  destruct (extr t1); tryfalse. destruct~ (extr t2); tryfalse.
Qed.

Definition join et1 et2 : t :=
  match et1, et2 with
  | Some eb1, Some eb2 =>
    match eb1, eb2 with
    | Some t1, Some t2 =>
      extr_destr t1 t2
        (fun ap1 ap2 =>
           let (a1, p1) := ap1 in let (a2, p2) := ap2 in
           ifb p1 = p2 then
             Some (Some (ext p1 (a1 ⊔♯ a2)))
           else None)
        et1 None None
    | None, _ => et2
    | _, None => et1
    end
  | _, _ => None
  end.

Definition meet et1 et2 : t :=
  match et1, et2 with
  | Some eb1, Some eb2 =>
    match eb1, eb2 with
    | Some t1, Some t2 =>
      extr_destr t1 t2
        (fun ap1 ap2 =>
           let (a1, p1) := ap1 in let (a2, p2) := ap2 in
           ifb p1 = p2 then
             Some (Some (ext p1 (a1 ⊓♯ a2)))
           else Some None)
        et1 (Some None) (Some None)
    | _, _ => Some None
    end
  | None, _ => et2
  | _, None => et1
  end.


Global Instance Lat : LatticeDec.t t.
  apply LatticeDec.Make with (porder := Poset).
   apply JoinDec.Make with (op := join); unfolds join;
    repeat (let x := fresh "x" in let a := fresh "a" in
            let p := fresh "p" in let E := fresh "E" in
            intros [[x|]|]; [destruct (extr x) as [[a p]|] eqn:E| |]);
    repeat (erewrite extr_destr_some_some; [|eassumption|eassumption]);
    repeat (erewrite extr_destr_some_none; [|eassumption|eassumption]);
    repeat (erewrite extr_destr_none_some; [|eassumption|eassumption]);
    repeat (destruct (decidable_sumbool (extr_none_comp E E0)) as [Cmp|Cmp];
             [ substs; rewrite~ extr_destr_none_none_eq
             | rewrite~ extr_destr_none_none_neq ]);
    intros; substs; autos~; tryfalse; simpls; try cases_if~;
    try rewrite E in *; try rewrite E0 in *; try rewrite E1 in *;
    try rewrite extr_ext; intuition; substs; autos~; tryfalse; fold_bool; rew_refl~ in *.
   apply MeetDec.Make with (op := meet); unfolds meet;
    repeat (let x := fresh "x" in let a := fresh "a" in
            let p := fresh "p" in let E := fresh "E" in
            intros [[x|]|]; [destruct (extr x) as [[a p]|] eqn:E| |]);
    repeat (erewrite extr_destr_some_some; [|eassumption|eassumption]);
    repeat (erewrite extr_destr_some_none; [|eassumption|eassumption]);
    repeat (erewrite extr_destr_none_some; [|eassumption|eassumption]);
    repeat (destruct (decidable_sumbool (extr_none_comp E E0)) as [Cmp|Cmp];
             [ substs; rewrite~ extr_destr_none_none_eq
             | rewrite~ extr_destr_none_none_neq ]);
    intros; substs; autos~; tryfalse; simpls; try cases_if~;
    try rewrite E in *; try rewrite E0 in *; try rewrite E1 in *;
    try rewrite extr_ext; intuition; substs; autos~; tryfalse; fold_bool; rew_refl~ in *.
   apply BotDec.Make with (elem := Some None). intros [?|]; simpls*.
   exact (fun x y => join x y).
Defined.


Global Instance Top : TopDec.t t.
  apply TopDec.Make with (elem := None). intros [?|]; simpls*.
Defined.

Variable check : forall A1 A2, e A1 → e A2 → bool.
Hypothesis check_trans : forall A1 A2 A3 et1 et2 et3,
    @extr A1 et1 = None → @extr A2 et2 = None → @extr A3 et3 = None →
  check et1 et3 → check et2 et3 → check et1 et2.
Hypothesis check_correct : forall A et1 et2,
  @extr A et1 = None → @extr A et2 = None →
  check et1 et2 → et1 = et2.
Variable C : Type.
Variable gamma0 : T → C → Prop.
Hypothesis G : Gamma.t (℘ C) T gamma0.

Definition gamma t c :=
  match t with
  | None => True
  | Some b =>
    match b with
    | None => False
    | Some e =>
      match extr e, extr c with
      | None, None => check e c
      | Some (a, ap), Some (c, cp) => ap = cp ∧ gamma0 a c
      | _, _ => False
      end
    end
  end.

Global Instance Gamma : Gamma.t (℘ (e C)) t gamma.
Proof.
  apply Gamma.Make.
   intros [[t1|]|] [[t2|]|] E c Ec; simpls~; tryfalse.
    destruct (extr t1) as [[a1 p1]|]; destruct (extr t2) as [[a2 p2]|];
      destruct (extr c) as [[ec ep]|]; intuition; substs~; tryfalse.
    applys~ @Gamma.monotone G H H2.
   intros [[t1|]|] [[t2|]|] c [Hg1 Hg2]; simpls~; tryfalse.
    destruct (extr t1) as [[a1 p1]|] eqn:E1; destruct (extr t2) as [[a2 p2]|] eqn:E2;
      destruct (extr c) as [[ec ep]|] eqn:E; intuition; substs~; tryfalse.
     erewrite extr_destr_some_some; try eassumption. simpl.
      cases_if~; fold_bool; rew_refl in *; tryfalse.
      simpl. rewrite E. rewrite extr_ext. splits~.
      applys~ @Gamma.meet_morph G. split; autos~.
     forwards~ Hg: check_trans Hg1 Hg2. forwards~: check_correct Hg. substs.
      rewrite extr_destr_none_none_eq; try eassumption. simpl. rewrite E2. rewrite~ E.
Qed.

End Extend.

End Extend.


Section ImmerseGamma.

(** Adapting the concretisation functions if we change the concrete domain. **)

Variable T : Type.
Variable L : LatticeDec.t T.
Variable C : Type.
Variable gamma : T → C → Prop.
Hypothesis G : Gamma.t (℘ C) T gamma.

Variable C' : Type.
Variable f : C' → C.

Definition immerse_gamma t c := gamma t (f c).

Global Instance immerseGamma : Gamma.t (℘ C') T immerse_gamma.
Proof.
  apply Gamma.Make.
   intros t1 t2 I c g. applys @Gamma.monotone G I g.
   intros t1 t2 c [Hg1 Hg2]. applys @Gamma.meet_morph G. simpl. splits~.
Qed.

Variable gammaC : C → C' → Prop.

Definition immerse_gamma_prop t c' :=
  exists c, gamma t c ∧ gammaC c c'.

Section WithLat.

Hypothesis LC : LatticeDec.t C.
Hypothesis GammaC : Gamma.t _ _ gammaC.
Hypothesis gamma_monotone : forall t c1 c2, gamma t c1 → c2 ⊑♯ c1 → gamma t c2.

Global Instance immerseGammaLat : Gamma.t (℘ C') T immerse_gamma_prop.
Proof.
  apply Gamma.Make.
   intros t1 t2 I c' (c&g&fp). exists c. splits~. applys @Gamma.monotone G I g.
   intros t1 t2 c [(c1&g1&fp1) (c2&g2&fp2)].
    exists (c1 ⊓♯ c2). splits~.
     applys @Gamma.meet_morph G. simpl. splits~.
      applys~ gamma_monotone g1.
      applys~ gamma_monotone g2.
     applys @Gamma.meet_morph GammaC. simpl. splits~.
Qed.

End WithLat.

Section BeingUnique.

Hypothesis gammaC_unique : forall c1 c2 c, gammaC c1 c → gammaC c2 c → c1 = c2.

Global Instance immerseGammaUnique : Gamma.t (℘ C') T immerse_gamma_prop.
Proof.
  apply Gamma.Make.
   intros t1 t2 I c' (c&g&fp). exists c. splits~. applys @Gamma.monotone G I g.
   intros t1 t2 c [(c1&g1&fp1) (c2&g2&fp2)].
    forwards: gammaC_unique fp1 fp2. substs.
    exists c2. splits~. applys @Gamma.meet_morph G. simpl. splits~.
Qed.

End BeingUnique.

End ImmerseGamma.


Module Immerse.

Section Immerse.

(** Adapting the domain properties if we change the abstract domain. **)

Variable T : Type.
Variable L : LatticeDec.t T.

Variable T' : Type.
Variable f : T' → T.

Definition Equiv : EquivDec.t T'.
  apply EquivDec.Make with (eq := fun t1 t2 => f t1 =♯ f t2); introv.
   apply EquivDec.refl.
   apply EquivDec.sym.
   apply EquivDec.trans.
   apply EquivDec.dec.
Defined.

Definition Poset : PosetDec.t T'.
  apply PosetDec.Make with (eq := Equiv) (order :=  fun t1 t2 => f t1 ⊑♯ f t2); introv.
   apply PosetDec.refl.
   apply PosetDec.antisym.
   apply PosetDec.trans.
   apply PosetDec.dec.
Defined.

Variable fi : T → T'.
Hypothesis fi_spec : forall t, f (fi t) =♯ t.

(* Easier to use lemmae. *)
Lemma fi_spec_1 : forall x t, x ⊑♯ t → x ⊑♯ f (fi t).
Proof. autos*. Qed.

Lemma fi_spec_2 : forall x t, t ⊑♯ x → f (fi t) ⊑♯ x.
Proof. autos*. Qed.

Definition Join : @JoinDec.t T' Poset.
  apply JoinDec.Make with (op := fun t1 t2 => fi (f t1 ⊔♯ f t2)).
   introv. simpl. apply fi_spec_1. apply JoinDec.bound1.
   introv. simpl. apply fi_spec_1. apply JoinDec.bound2.
   introv ? ?. simpl. apply fi_spec_2. apply JoinDec.least_upper_bound; simpls~.
Defined.

Definition Meet : @MeetDec.t T' Poset.
  apply MeetDec.Make with (op := fun t1 t2 => fi (f t1 ⊓♯ f t2)).
   introv. simpl. apply fi_spec_2. apply MeetDec.bound1.
   introv. simpl. apply fi_spec_2. apply MeetDec.bound2.
   introv ? ?. simpl. apply fi_spec_1. apply MeetDec.greatest_lower_bound; simpls~.
Defined.

Definition Lat : LatticeDec.t T'.
  apply LatticeDec.Make with (porder := Poset).
   apply Join.
   apply Meet.
   applys BotDec.Make (fi (⊥♯)).
    introv. simpl. apply fi_spec_2. apply BotDec.prop.
   exact (fun t1 t2 => fi ((f t1) ∇♯(f t2))).
Defined.

Hypothesis Top0 : TopDec.t T.

Definition Top : @TopDec.t T' Poset.
  apply TopDec.Make with (elem := fi (⊤♯)).
   introv. simpl. apply fi_spec_1. apply TopDec.prop.
Defined.

Variable C : Type.
Variable gamma0 : T → C → Prop.
Hypothesis G : Gamma.t (℘ C) T gamma0.

Definition gamma t c := gamma0 (f t) c.

Global Instance Gamma : @Gamma.t (℘ C) T' _ Lat gamma.
Proof.
  apply Gamma.Make.
   intros t1 t2 I c g. applys @Gamma.monotone G I g.
   intros t1 t2 c [Hg1 Hg2]. unfolds. simpl.
    applys @Gamma.monotone G. apply fi_spec_1. apply PosetDec.refl. apply @EquivDec.refl.
    applys @Gamma.meet_morph G. simpl. splits~.
Qed.

End Immerse.

End Immerse.


Module FinitePowerSet.

Section FinitePowerSet.

Variable T : Type.
Hypothesis CT : Comparable T.
Hypothesis E : EquivDec.t T.

Definition t0 := LiftJoin.t (@FlatLatticeDec.TB T).
Definition t := LiftTop.t (list T).

Instance Lat0 : LatticeDec.t t0.
  eapply LiftJoin.Lat.
   typeclass.
   typeclass.
   typeclass.
   exact (fun x y => x ⊔♯ y).
Defined.

Instance Top0 : TopDec.t t0.
  apply LiftJoin.Top. typeclass.
Defined.

Definition const (e : T) : t := Some (cons e nil).

Definition f : t → t0.
  refine (option_case (⊤♯) (fun l =>
    nosimpl match LibList.map FlatLatticeDec.TB_Elem l with
    | nil => (⊥♯) (* No element here. *)
    | x :: l =>
      exist _ (x :: l) _
    end)).
  discriminate.
Defined.

Definition l : t0 → t :=
  fun l =>
    LibList.fold_right (fun c l =>
      match c, l with
      | _, None => None
      | FlatLatticeDec.TB_Top, _ => None
      | FlatLatticeDec.TB_Bot, Some l => Some l
      | FlatLatticeDec.TB_Elem e, Some l => Some (e :: l)
      end) (Some nil) (proj1_sig l).

Lemma f_cons : forall l,
  l ≠ nil →
  proj1_sig (f (Some l)) = LibList.map FlatLatticeDec.TB_Elem l.
Proof. intros l0 D. destruct~ l0; tryfalse. Qed.

Lemma f_const : forall e,
  proj1_sig (f (const e)) = [FlatLatticeDec.TB_Elem e].
Proof. reflexivity. Qed.

Lemma l_top : forall t1, l t1 = None → t1 =♯ (⊤♯).
Proof.
  introv Eq. destruct t1 as [l1 D]. unfolds l. induction l1; tryfalse.
  simpl in Eq.
  match goal with H : context [ fold_right ?f ?i ?l ] |- _ =>
    sets_eq F: (fold_right f i l) end. destruct a.
   apply LiftJoin.has_top. simpl. rew_refl~.
   asserts N: (F = None); [ destruct~ F; tryfalse |].
    rewrite N in EQF. clear - EQF IHl1. symmetry in EQF.
    tests N: (l1 = nil). lets Tl: (IHl1 N EQF). clear - Tl.
    applys @EquivDec.trans Tl. eapply EquivDec.trans; [| apply~ join_bot ].
    eapply EquivDec.trans; [| apply EquivDec.sym; apply LiftJoin.join_append ].
    apply~ LiftJoin.Order_base_Order_eq.
   asserts N: (F = None); [ destruct~ F; tryfalse |].
    rewrite N in EQF. clear - EQF IHl1. symmetry in EQF.
    tests N: (l1 = nil). lets Tl: (IHl1 N EQF). clear - Tl.
    applys @EquivDec.trans; [| applys join_top (LiftJoin.const (FlatLatticeDec.TB_Elem a)) Tl ].
    eapply EquivDec.trans; [| apply EquivDec.sym; apply LiftJoin.join_append ].
    apply~ LiftJoin.Order_base_Order_eq.
Qed.

Lemma fl : forall t1, f (l t1) =♯ t1.
Proof.
  introv. sets_eq r: (f (l t1)). asserts EQ': (r =♯ f (l t1)); [ substs~ |].
  clear EQr. destruct t1 as [l1 D1].
  destruct r as [l2 D2]. gen l2. induction l1; introv EQ; tryfalse.
  unfold l, proj1_sig in EQ. rew_list in EQ.
  match goal with H : context [ fold_right ?f ?i ?l ] |- _ =>
    sets_eq F: (fold_right f i l) end.
  tests D1': (l1 = nil).
   simpl in EQF. substs. applys @EquivDec.trans EQ.
    destruct a; apply~ LiftJoin.Order_base_Order_eq.
   change (F = l (exist _ l1 D1')) in EQF. destruct a.
    asserts Eq: (exist _ l2 D2 =♯ (⊤♯ : t0)).
      applys @EquivDec.trans EQ. clear. destruct~ F.
     applys @EquivDec.trans Eq. apply EquivDec.sym. apply LiftJoin.has_top.
     simpl. rew_refl~.
    asserts Eq: (exist _ l2 D2 =♯ f F).
      applys @EquivDec.trans EQ. clear. destruct~ F.
     substs. lets Eq': IHl1 Eq. applys @EquivDec.trans Eq'.
     eapply EquivDec.trans; [ apply join_bottom1 |].
     eapply EquivDec.trans; [ apply LiftJoin.join_append |].
     apply~ LiftJoin.Order_base_Order_eq.
    symmetry in EQF. applys @EquivDec.trans EQ. destruct F as [lf|].
     set_eq l3: (f (Some lf)). rewrite <- EQF in EQl3. destruct l3 as [l3 D3].
      forwards IH: IHl1 (eq_EquivDec _ EQl3). rewrite EQl3 in IH. rewrite EQF in IH.
      clear IHl1 EQF EQl3 D3 l3. asserts Ef: (EquivDec.eq (t := LiftJoin.Equiv_base _ _)
        (proj1_sig (f (Some (a :: lf))))
        (FlatLatticeDec.TB_Elem a :: proj1_sig (f (Some lf)))).
       clear. destruct~ lf. unfold f at 1, option_case. rew_list. unfold proj1_sig.
        apply EquivDec.sym. apply EquivDec.trans with (proj1_sig (f (Some nil)) ++ [FlatLatticeDec.TB_Elem a]).
         apply LiftJoin.eq_mix. extens. rew_refl*.
         eapply LiftJoin.cons_equiv.
          apply BotDec.prop.
          simpl. rew_refl*.
      apply LiftJoin.Order_base_Order_eq with (l1 := proj1_sig (f (Some (a :: lf)))).
      applys @EquivDec.trans Ef. rewrite <- app_nil_l with (l := l1). rewrite <- app_nil_l with (l := proj1_sig _).
      do 2 rewrite <- app_cons. applys @EquivDec.trans; [| apply LiftJoin.join_base_append ].
      applys @EquivDec.trans; [ apply EquivDec.sym; apply LiftJoin.join_base_append |].
      apply~ (@join_eq2 _ _ (LiftJoin.Join_base _ _) (cons (FlatLatticeDec.TB_Elem a) nil)).
     apply EquivDec.sym. applys @EquivDec.trans;
        [| applys join_top (LiftJoin.const (FlatLatticeDec.TB_Elem a)); applys l_top EQF ].
      eapply EquivDec.trans; [| apply EquivDec.sym; apply LiftJoin.join_append ].
      apply~ LiftJoin.Order_base_Order_eq.
Qed.

Global Instance Lat : LatticeDec.t t.
  apply Immerse.Lat with (L := Lat0) (f := f) (fi := l).
  apply fl.
Defined.

Global Instance Top : TopDec.t t.
  apply Immerse.Top with (L := Lat0) (f := f) (fi := l).
  apply fl.
  typeclass.
Defined.

Lemma const_bot : forall t1,
  ~ const t1 ⊑♯ (⊥♯).
Proof.
  introv A. simpls. rew_refl in A. do 2 unfolds in A. rewrite Forall_iff_forall_mem with (CA := _) in A.
  forwards Ex: (rm A).
   rew_refl. left~.
   rewrite Exists_iff_exists_mem with (CA := _) in Ex. lets (a&Ma&Oa): (rm Ex). simpls.
    rew_refl in Ma. inverts~ Ma.
Qed.

Lemma const_same : forall t1 t2,
  t1 =♯ t2 ↔ const t1 ⊑♯ const t2.
Proof.
  introv. simpls. unfolds LiftJoin.Order, LiftJoin.Order_base. rew_refl.
  rewrite Forall_iff_forall_mem with (CA := _). simpl. iff I.
   introv A. rew_refl in A. repeat inverts A as A. apply~ Exists_here.
   forwards Ex: I.
    rew_refl*.
    repeat inverts Ex as Ex. apply Ex.
Qed.

Lemma const_join : forall e t1 t2,
  const e ⊑♯ (t1 ⊔♯ t2) →
  const e ⊑♯ t1 ∨ const e ⊑♯ t2.
Proof.
  introv O. simpl in O. rew_refl in O. do 2 unfolds in O. simpl in O.
  rewrite Forall_iff_forall_mem with (CA := _) in O. forwards Ex: (rm O).
   rew_refl*.
   rewrite Exists_iff_exists_mem with (CA := _) in Ex. simpl in Ex. lets (a&M&Eq): (rm Ex).
    forwards (O1&O2): fl. do 2 unfolds in O1. rewrite Forall_iff_forall_mem with (CA := _) in O1.
    forwards Ex: O1 M. rewrite Exists_iff_exists_mem with (CA := _) in Ex. lets (a'&Ma'&Oa'): (rm Ex).
    destruct (f t1) as [l1 D1] eqn: E1. destruct (f t2) as [l2 D2] eqn: E2.
    simpl in Ma'. unfolds LiftJoin.join_base. rewrite mem_app in Ma'. rew_refl in Ma'.
    inverts Ma' as Ma'.
     left. simpl. rew_refl. do 2 unfolds. rewrite Forall_iff_forall_mem with (CA := _).
      introv I. simpl in I. rew_refl in I. repeat inverts I as I.
      rewrite Exists_iff_exists_mem. exists a'. rewrite E1. splits.
       autos*.
       applys~ @PosetDec.trans Oa'.
     rewrite filter_mem_eq in Ma'. rew_refl in Ma'. lets (Ma2&N): (rm Ma').
      right. simpl. rew_refl. do 2 unfolds. rewrite Forall_iff_forall_mem with (CA := _).
      introv I. simpl in I. rew_refl in I. repeat inverts I as I.
      rewrite Exists_iff_exists_mem. exists a'. rewrite E2. splits.
       autos*.
       applys~ @PosetDec.trans Oa'.
Qed.

Definition from_list : list T → t :=
  fold_right (fun e a => a ⊔♯ const e) (⊥♯).

Lemma from_list_spec : forall e es,
  const e ⊑♯ from_list es ↔ exists e', mem e' es ∧ e =♯ e'.
Proof.
  introv. gen e. induction es; introv; iff I.
   false* const_bot.
   lets (e'&M&Eq): (rm I). false*.
   unfolds from_list. rewrite fold_right_cons in I. apply const_join in I. inverts I as O.
    rewrite IHes in O. lets (e'&Me'&Ee'): (rm O). exists e'. splits~. simpl. rew_refl*.
    rewrite <- const_same in O. exists a. splits~. simpl. rew_refl*.
   lets (e'&M&Eq): (rm I). simpl in M. rew_refl in M.
    unfold from_list. rewrite fold_right_cons. inverts M as M.
     eapply PosetDec.trans.
      rewrite <- const_same. eassumption.
      apply JoinDec.bound2.
     eapply PosetDec.trans.
      rewrite~ IHes. eexists. splits*.
      apply JoinDec.bound1.
Qed.


Definition gamma (t : t) c :=
  option_case True (exists_st (fun c' => decide (c =♯ c'))) t.

Definition gamma' : t → T → Prop :=
  Immerse.gamma f (LiftJoin.gamma (flatGamma _)).

Lemma gamma_gamma' : gamma = gamma'.
Proof.
  extens. intros t c. iff I; unfolds; unfolds in I.
   destruct t as [l|]; do 2 unfolds; rewrite Exists_iff_exists_mem with (CA := _).
    apply Exists_exists_st in I. apply Exists_iff_exists_mem with (CA := _) in I.
     lets (e&Me&Ee): (rm I). rew_refl in Ee. tests: (l = nil).
      rewrite~ f_cons. eexists. splits.
       eapply map_mem. eexists. splits*.
       autos*.
    eexists. splits.
     simpl. rew_refl. left~.
     autos*.
   destruct t as [l|]; simpls~.
    do 2 unfolds in I. apply Exists_exists_st. rewrite Exists_iff_exists_mem in *.
    lets (e&Me&Ee): (rm I). tests: (l = nil).
     simpl in Me. rew_refl in Me. inverts Me; false.
     rewrite~ f_cons in Me. eapply map_mem in Me. lets (y&My&Ey): (rm Me).
      substs. simpls. exists y. splits*. cases_if~.
Qed.

Global Instance Gamma : @Gamma.t (℘ T) t _ Lat gamma.
Proof.
  rewrite gamma_gamma'. apply Immerse.Gamma.
  forwards G: FlatGamma T E. apply~ LiftJoin.Gamma; apply G.
Qed.

Lemma gamma_bot : forall e,
  ~ gamma (⊥♯) e.
Proof. introv G. rewrite gamma_gamma' in G. apply LiftJoin.gamma_bot in G; autos~. typeclass. Qed.

Lemma gamma_top : forall e,
  gamma (⊤♯) e.
Proof. introv. rewrite gamma_gamma'. apply~ LiftJoin.gamma_top. simpls~. Qed.


(** Return [None] when the input it the top element. **)
Definition to_list : t → option (list T) :=
  id.

Lemma in_inf : forall t1 t2 : list T,
  (forall e, mem e t1 → gamma (Some t2) e) →
  Some t1 ⊑♯ Some t2.
Proof.
  introv I. simpl. rew_refl. do 2 unfolds. rewrite Forall_iff_forall_mem with (CA := _).
  introv M. rewrite Exists_iff_exists_mem with (CA := FlatLatComparable CT). destruct (map FlatLatticeDec.TB_Elem t1) eqn: E1.
   simpl in M. rew_refl in M. repeat inverts M as M.
    destruct (map FlatLatticeDec.TB_Elem t2); eexists; splits; simpl; rew_refl~.
   simpl in M. change (mem x (t3 :: l0) : Prop) in M. rewrite <- E1 in M.
    rewrite map_mem in M. lets (y&My&Ey): (rm M). forwards Ex: I My. simpl in Ex.
    rewrite <- Exists_exists_st in Ex. rewrite Exists_iff_exists_mem with (CA := CT) in Ex.
    lets (y'&My'&Ey'): (rm Ex). destruct t2; tryfalse. simpl.
    exists (FlatLatticeDec.TB_Elem y'). splits~.
     rewrite <- mem_cons with (CA := FlatLatComparable CT).
      rewrite <- fold_right_cons with (f := fun x acc => FlatLatticeDec.TB_Elem x :: acc).
      fold (map FlatLatticeDec.TB_Elem (t2 :: t4)). rewrite map_mem. exists* y'.
     rew_refl in Ey'. cases_if; tryfalse. substs~.
Qed.

Lemma l_monotone : forall t1 t2,
  t1 ⊑♯ t2 → l t1 ⊑♯ l t2.
Proof.
  introv O. simpl. rew_refl. repeat unfolds. simpl in O. rew_refl in O.
  forwards (O1&O2): fl t1. forwards (O1'&O2'): fl t2. repeat unfolds in O O1 O2 O1' O2'.
  rewrite Forall_iff_forall_mem with (CA := _) in *. introv M. forwards Ex: O1 M.
  rewrite Exists_iff_exists_mem with (CA := _) in Ex. lets (a&Ma&Oa): (rm Ex).
  forwards Ex: O Ma. rewrite Exists_iff_exists_mem with (CA := _) in Ex. lets (a'&Ma'&Oa'): (rm Ex).
  forwards Ex: O2' Ma'. rewrite Exists_iff_exists_mem in Ex. lets (x'&Mx'&Ox'): (rm Ex).
  rewrite Exists_iff_exists_mem. exists* x'.
Qed.

Lemma l_func : forall t1 t2,
  t1 =♯ t2 → l t1 =♯ l t2.
Proof. introv Eq. apply PosetDec.antisym; apply l_monotone; apply~ @PosetDec.refl. Qed.

Lemma join_app : forall t1 t2 : list T,
  Some t1 ⊔♯ Some t2 =♯ Some (t1 ++ t2).
Proof.
  introv. unfold JoinDec.op, LatticeDec.join, Lat, Immerse.Lat, Lat0, Immerse.Join.
  destruct t1 as [|t1h t1t].
   change (l (⊥♯ ⊔♯ f (Some t2)) =♯ Some t2).
    applys* @EquivDec.trans (l (f (Some t2))) l_func fl.
  destruct t2 as [|t2h t2t].
   rewrite app_nil_r. change (l (f (Some (t1h :: t1t)) ⊔♯ ⊥♯) =♯ Some (t1h :: t1t)).
    applys* @EquivDec.trans (l (f (Some (t1h :: t1t)))) l_func fl.
  destruct (f (Some (t1h :: t1t))) eqn: E1. destruct (f (Some (t2h :: t2t))) eqn: E2.
  asserts N: (x ++ x0 ≠ []).
    introv Eq. destruct x; false*.
  eapply EquivDec.trans.
   apply l_func. apply LiftJoin.Order_base_Order_sig.
    unfold JoinDec.op, LatticeDec.join, LiftJoin.Lat, LiftJoin.Join, LiftJoin.join.
    eapply EquivDec.trans with (y := proj1_sig (exist (fun l => l ≠ []) _ N)).
     apply LiftJoin.join_base_append.
     apply EquivDec.refl.
  unfolds l, proj1_sig. match goal with |- context [ fold_right ?f ?i ] =>
    asserts Ex: (exists l', fold_right f i (x ++ x0) = Some l'
      ∧ (forall e, mem e l' → gamma (Some ((t1h :: t1t) ++ (t2h :: t2t))) e)
      ∧ (forall e, mem e ((t1h :: t1t) ++ (t2h :: t2t)) → gamma (Some l') e)) end.
     sets_eq l0: (x ++ x0). sets_eq l1: ((t1h :: t1t) ++ t2h :: t2t).
      forwards E1': f_equal E1. rewrite f_cons in E1'; try discriminate.
      forwards E2': f_equal E2. rewrite f_cons in E2'; try discriminate. simpl in E1', E2'.
      asserts Inv: (forall x, mem (FlatLatticeDec.TB_Elem x) l0 = mem x l1).
       introv. extens. substs. repeat rewrite mem_app. rew_refl. iff M; inverts M as M.
        left. rewrite map_mem in M. lets (?&M'&Eq): (rm M). inverts* Eq.
        right. rewrite map_mem in M. lets (?&M'&Eq): (rm M). inverts* Eq.
        left. rewrite* map_mem.
        right. rewrite* map_mem.
      asserts Inv': (forall x, mem x l0 → exists y, x = FlatLatticeDec.TB_Elem y).
       introv M. substs. rewrite mem_app in M. rew_refl in M.
        inverts M as M; rewrite map_mem with (CA := _) in M; lets* (?&?&?): M.
      clear - Inv Inv'. gen l1. induction l0; introv Inv.
       eexists. splits*.
        introv A. false*.
        introv A. rewrite <- Inv in A. false*.
       asserts (l'&I&P): (exists l1',
         incl l1' l1 ∧ forall x, mem (FlatLatticeDec.TB_Elem x) l0 = mem x l1').
        tests I: (mem a l0).
         exists l1. splits.
          apply incl_refl.
          introv. rewrite <- Inv. simpl. extens. rew_refl. iff* M. inverts~ M.
         forwards (a'&Ea'): Inv'.
          simpl. rew_refl*.
          exists (Filter (fun a => a <> a') l1). substs. splits.
           eapply mem_incl. introv M. rewrite <- Mem_mem in M.
            lets~ (M'&D): Filter_Mem_inv M. erewrite Mem_mem in M'. autos*.
           introv. extens. iff M.
            rewrite <- Mem_mem. apply Filter_Mem.
             erewrite Mem_mem. rewrite <- Inv. simpl. rew_refl*.
             introv A. substs. false*.
            rewrite <- Mem_mem in M. lets~ (M'&D): Filter_Mem_inv M. erewrite Mem_mem in M'.
             rewrite <- Inv in M'. simpl in M'. rew_refl in M'. inverts* M'.
       forwards (l2&E2&Ga&Gb): (rm IHl0) P.
        introv M. apply Inv'. simpl. rew_refl*.
        forwards (a'&Ea'): Inv'.
         simpl. rew_refl*.
         substs. rewrite fold_right_cons. eexists. splits.
          rewrite* E2.
          introv M. simpls. rewrite <- Exists_exists_st. rewrite Exists_iff_exists_mem.
           rew_refl in M. inverts M as M.
            eexists. splits.
             rewrite <- Inv. rew_refl*.
             cases_if*.
           forwards Ex: Ga M. rewrite <- Exists_exists_st in *.
            rewrite Exists_iff_exists_mem in *. lets (a&Ma&Ea): (rm Ex). exists a. splits~.
            rewrite <- Inv. rew_refl. right. rewrite* P.
          introv M. simpl. rewrite <- Exists_exists_st. rewrite Exists_iff_exists_mem with (CA := _).
           rewrite <- Inv in M. simpl in M. rew_refl in M. inverts M as M.
            inverts M. eexists. splits; simpl; rew_refl*. cases_if~.
            rewrite P in M. apply Gb in M. rewrite <- Exists_exists_st in M.
             rewrite Exists_iff_exists_mem in M. lets (a&Ma&Ea): (rm M).
             exists a. splits~. simpl. rew_refl*.
   lets (l'&El'&I1&I2): (rm Ex). rewrite El'. apply PosetDec.antisym; apply~ in_inf.
Qed.

Lemma from_list_to_list : forall t es,
  to_list t = Some es →
  from_list es =♯ t.
Proof.
  introv Eq. destruct t1; tryfalse. inverts~ Eq. induction es.
   autos*.
   unfolds from_list. rewrite fold_right_cons. eapply EquivDec.trans.
    applys join_eq1 IHes.
    eapply EquivDec.trans; [ apply join_sym |]. eapply EquivDec.trans.
     apply join_app.
     apply EquivDec.refl.
Qed.

Lemma to_list_none : forall t,
  to_list t = None → t = (⊤♯).
Proof. introv Eq. apply Eq. Qed.

Lemma to_list_some : forall t e es,
  to_list t = Some es →
  const e ⊑♯ t ↔ exists e', mem e' es ∧ e =♯ e'.
Proof.
  introv TL. forwards Eq: from_list_to_list TL.
  rewrite <- from_list_spec. iff I; applys~ @PosetDec.trans I.
Qed.

Lemma some_not_top : forall e es,
  ~ (exists e', mem e' es ∧ e =♯ e') →
  ~ Some es =♯ (⊤♯).
Proof.
  introv N Eq. false N. forwards (I1&I2): to_list_some; [ reflexivity |].
  apply I1. eapply PosetDec.trans.
   apply TopDec.prop.
   apply PosetDec.refl. apply* @EquivDec.sym.
Qed.

End FinitePowerSet.

Section Substitutions.

Variable T : Type.
Hypothesis CT : Comparable T.
Hypothesis E : EquivDec.t T.

Definition substs (s : t T) (e : T) (s' : t T) : t T :=
  match s with
  | None => None
  | Some l =>
    ifb Exists (fun e' => e =♯ e') l then
      Some (filter (fun e' => decide (~ e =♯ e')) l) ⊔♯ s'
    else Some l
  end.

Lemma substs_basic : forall e s,
  substs (const e) e s =♯ s.
Proof.
  introv. unfolds substs, const. cases_if as I; fold_bool; rew_refl in *.
  clear. apply join_bot with (BV := LatticeDec.bot).
   unfold filter, fold_right. cases_if~; fold_bool; rew_refl in *. false*.
  false I. apply Exists_here. rew_refl*.
Qed.

Lemma substs_nosubsts : forall s e1 e2,
  ~ const e1 ⊑♯ s →
  substs s e1 e2 = s.
Proof.
  introv NI. destruct s as [l|].
   unfolds. cases_if~. false NI. simpl. rew_refl. do 2 unfolds.
    apply Forall_iff_forall_mem with (CA := _). introv M. simpl in M. rew_refl in M. inverts M; tryfalse.
    tests N: (l = nil).
     lets Eq: f_cons N. simpl in Eq. rewrite Eq. clear Eq.
      fold_bool. rew_refl in *.
      rewrite Exists_iff_exists_mem in *. lets (a&Ma&Ea): (rm H).
      eexists. splits.
       eapply map_mem. exists a. splits*.
       rew_refl~ in Ea.
   simpls~.
Qed.

Lemma substs_top : forall st e s,
  st =♯ (⊤♯) →
  substs st e s =♯ ⊤♯.
Proof.
  introv Eq. destruct st as [l|].
   false. lets (_&E2): (rm Eq). do 2 unfolds in E2.
    rewrite Forall_iff_forall_mem with (CA := _) in E2. forwards~ Eq: E2.
     simpl. rew_refl. left~.
     rewrite Exists_iff_exists_mem with (CA := _) in Eq. lets (y&My&Oy): (rm Eq).
     tests N: (l = nil).
      simpl in My. rew_refl in My. inverts My; tryfalse.
      rewrite~ f_cons in My. apply map_mem with (CA := _) in My. lets (?&_&?): (rm My). substs*.
   apply EquivDec.refl.
Qed.

Lemma exist_proj12 : forall x : t0 T,
  x = exist _ (proj1_sig x) (proj2_sig x).
Proof. intro x. destruct~ x. Qed.

Lemma Exists_order : forall l1 l2 e,
  Some l1 ⊑♯ Some l2 →
  Exists (λ e' : T, decide (e =♯ e')) l1 →
  Exists (λ e' : T, decide (e =♯ e')) l2.
Proof.
  introv O Ex. induction Ex.
   rew_refl in *. simpl in O. rew_refl in O. do 2 unfolds in O. simpl in O.
    rewrite Forall_iff_forall_mem in O. forwards Ex: (rm O).
     simpl. rew_refl. left~.
     rewrite Exists_iff_exists_mem in Ex. lets (a&Ma&Oa): (rm Ex). tests N: (l2 = nil); simpls.
      rew_refl in Ma. inverts Ma; false.
     lets Ef: f_cons N. simpl in Ef. rewrite Ef in Ma. clear Ef.
     apply map_mem with (CA := _) in Ma. lets (y&My&Eq): (rm Ma). substs.
     rewrite Exists_iff_exists_mem. exists y. splits*. cases_if as I; autos~.
      false I. applys~ @EquivDec.trans Oa.
   apply IHEx. applys @PosetDec.trans O. apply LiftTop.lift_order_dec.
    rewrite exist_proj12. rewrite exist_proj12 at 1. apply LiftJoin.Order_base_Order_poset.
    rewrite f_cons with (l := x :: l0); try discriminate.
    tests N: (l0 = nil).
     apply~ LiftJoin.lift_order. typeclass.
     rewrite~ f_cons. rew_list. eapply PosetDec.trans; [|
       apply PosetDec.refl; apply EquivDec.sym; apply LiftJoin.cons_join ].
      lets S: (JoinDec.bound2 (t := LiftJoin.Join_base _ (FlatLatticeDec.tPoset E))).
      simpl in S. apply S.
Qed.

Lemma order_list : forall l1 l2,
  Some l1 ⊑♯ Some l2 →
  forall x, mem x l1 → exists x', x =♯ x' ∧ mem x' l2.
Proof.
  introv O M. simpl in O. rew_refl in O. do 2 unfolds in O. rewrite Forall_iff_forall_mem in O.
  tests N1: (l1 = nil). lets Eq: f_cons N1. simpl in Eq. rewrite Eq in O. clear Eq.
  tests N2: (l2 = nil).
   forwards Ex: O.
    eapply map_mem. exists~ x.
    simpl in Ex. repeat inverts Ex as Ex.
   lets Eq: f_cons N2. simpl in Eq. rewrite Eq in O. clear Eq. forwards Ex: O.
    eapply map_mem. exists~ x.
    rewrite Exists_iff_exists_mem with (CA := _) in Ex. lets (a&Ma&Oa): (rm Ex).
     eapply map_mem in Ma. lets~ (?&?&Oa'): (rm Ma). substs. simpls*.
Qed.

Lemma list_order : forall l1 l2,
  (forall x, mem x l1 → exists x', x =♯ x' ∧ mem x' l2) →
  Some l1 ⊑♯ Some l2.
Proof.
  introv O. simpl. rew_refl. do 2 unfolds. rewrite Forall_iff_forall_mem with (CA := _). introv M.
  tests N1: (l1 = nil).
   simpl in M. rew_refl in M. inverts M; tryfalse. destruct l2; apply Exists_here; simpls~.
   lets Eq: f_cons N1. simpl in Eq. rewrite Eq in M. clear Eq.
    eapply map_mem in M. lets (?&M'&?): (rm M). substs. forwards (x'&Ex'&Mx'): O M'.
    tests N2: (l2 = nil). lets Eq: f_cons N2. simpl in Eq. rewrite Eq. clear Eq.
    rewrite Exists_iff_exists_mem with (CA := _). exists (FlatLatticeDec.TB_Elem x'). splits~. apply* map_mem.
Qed.

Lemma filter_congruence : forall l1 l2 e,
  Some l1 ⊑♯ Some l2 →
  Some (filter (fun e' => decide (~ e =♯ e')) l1)
    ⊑♯ Some (filter (fun e' => decide (~ e =♯ e')) l2).
Proof.
  introv O. lets O': order_list (rm O). apply list_order.
  introv M. rewrite filter_mem_eq in M. rew_refl in M. inverts M as M N.
  forwards (x'&Ex'&Mx'): O' M. exists x'. splits~. rewrite filter_mem_eq. rew_refl. splits~.
  introv Eq. apply N. applys~ EquivDec.trans x'.
Qed.

Lemma filter_order : forall l1 l2 e,
  Some l1 ⊑♯ Some l2 →
  ~ Exists (λ e' : T, decide (e =♯ e')) l1 →
  Some l1 ⊑♯ Some (filter (fun e' => decide (~ e =♯ e')) l2).
Proof.
  introv O NM. asserts I: (l1 = filter (λ e' : T, decide (¬e =♯ e')) l1).
    clear - NM. induction~ l1. rewrite filter_cons. cases_if as I.
     fequals. apply IHl1. introv Ex. false NM. apply~ Exists_next.
     false NM. apply Exists_here. fold_bool. rew_refl~ in *. apply~ not_not_elim.
   rewrite I. apply~ filter_congruence.
Qed.

Lemma from_list_simpl : forall es,
  from_list _ E es =♯ Some es.
Proof.
  introv. induction es.
   autos*.
   unfolds from_list. rewrite fold_right_cons. eapply EquivDec.trans.
    applys join_eq1 IHes.
    eapply EquivDec.trans; [ apply join_sym |]. eapply EquivDec.trans.
     apply join_app.
     apply EquivDec.refl.
Qed.

Lemma substs_congruence : forall e s s1 s2,
  s1 ⊑♯ s2 →
  substs s1 e s ⊑♯ substs s2 e s.
Proof.
  introv O. unfolds substs. destruct s1 as [l1|].
   destruct s2 as [l2|].
    repeat cases_if~; fold_bool; rew_refl in *.
     apply~ join_monotone. apply~ filter_congruence.
     forwards F: Exists_order O.
      rewrite Exists_iff_exists_mem with (CA := _) in *. lets (a&Ma&Ea): (rm H).
       exists a. splits*. rew_refl~.
      autos*.
      false H0. rewrite Exists_iff_exists_mem with (CA := _) in *. lets (a&Ma&Ea): (rm F).
       exists a. splits*. rew_refl~ in Ea.
     applys @PosetDec.trans.
      applys filter_order O. introv A. false H. rewrite Exists_iff_exists_mem with (CA := _) in *.
       lets (a&Ma&Ea): (rm A). exists a. splits*. rew_refl* in Ea.
      autos*.
    apply TopDec.prop.
   forwards~ T2: sup_top O. applys @PosetDec.trans;
     [|apply PosetDec.refl; apply EquivDec.sym; apply~ substs_top].
    autos*.
Qed.

Lemma substs_congruence_eq : forall e s s1 s2,
  s1 =♯ s2 →
  substs s1 e s =♯ substs s2 e s.
Proof. introv O. apply PosetDec.antisym; apply~ substs_congruence. Qed.

Lemma substs_congruence_replacement : forall e s s1 s2,
  s1 ⊑♯ s2 →
  substs s e s1 ⊑♯ substs s e s2.
Proof. introv O. unfolds substs. destruct~ s. cases_if~. apply~ join_monotone. Qed.

Lemma substs_congruence_eq_replacement : forall e s s1 s2,
  s1 =♯ s2 →
  substs s e s1 =♯ substs s e s2.
Proof. introv O. apply PosetDec.antisym; apply~ substs_congruence_replacement. Qed.

Lemma substs_mem_out : forall e s1 s2 s3 s4,
  substs s1 e s2 =♯ s3 →
  s4 ⊑♯ s1 →
  ~ const e ⊑♯ s4 →
  s4 ⊑♯ s3.
Proof.
  introv S O NC. unfolds substs. destruct s1.
   cases_if as I; autos~.
    eapply PosetDec.trans.
     apply JoinDec.bound1.
     eapply PosetDec.trans; [| applys @PosetDec.refl S ].
      apply join_monotone; [| apply~ @PosetDec.refl ]. destruct s4.
       apply~ filter_order. introv Ex. false NC. apply list_order.
        introv M. simpl in M. rew_refl in M. repeat inverts M as M.
        rewrite Exists_iff_exists_mem in Ex. lets (a&Ma&Ea): (rm Ex).
        rew_refl in Ea. exists* a.
       false NC. apply TopDec.prop.
    applys~ @PosetDec.trans O.
   eapply PosetDec.trans.
    apply TopDec.prop.
    apply~ @PosetDec.refl.
Qed.

Lemma substs_mem_in : forall e s1 s2 s3 s4,
  substs s1 e s2 =♯ s3 →
  const e ⊑♯ s1 →
  s4 ⊑♯ s2 →
  s4 ⊑♯ s3.
Proof.
  introv S O O'. unfolds substs. destruct s1.
   cases_if as I; autos~.
    applys @PosetDec.trans O'. eapply PosetDec.trans; [| applys @PosetDec.refl S ]. autos*.
    fold_bool. rew_refl in I. false I. rewrite Exists_iff_exists_mem.
     forwards (a&Ea&Ma): order_list O.
      simpl. rew_refl*.
      exists* a.
   eapply PosetDec.trans.
    apply TopDec.prop.
    apply~ @PosetDec.refl.
Qed.

Lemma substs_inv : forall e1 e2 s1 s2 s3,
  substs s1 e1 s2 =♯ s3 →
  const e2 ⊑♯ s3 →
  const e2 ⊑♯ s1 ∧ (~ e1 =♯ e2 ∨ s1 =♯ (⊤♯)) ∨ const e2 ⊑♯ s2 ∧ const e1 ⊑♯ s1.
Proof.
  introv Eq O. destruct s1.
   forwards O': @PosetDec.trans O.
    apply PosetDec.refl. applys~ @EquivDec.sym Eq.
    unfolds substs. cases_if as I.
     apply const_join in O'. inverts O' as Oc.
      left. forwards Eq': from_list_simpl. apply EquivDec.sym in Eq'.
       apply PosetDec.refl in Eq'. forwards Ol: @PosetDec.trans Oc Eq'.
       apply from_list_spec in Ol. lets (e'&Me'&Ee'): (rm Ol).
       rewrite filter_mem_eq in Me'. rew_refl in Me'. lets (M&NE): (rm Me'). splits.
        apply list_order. introv M'. simpl in M'. rew_refl in M'. repeat inverts M' as M'.
         exists* e'.
        left*.
      right. splits~. apply list_order. introv M'. simpl in M'. rew_refl in M'.
       repeat inverts M' as M'. fold_bool. rew_refl in I. rewrite Exists_iff_exists_mem in I.
       lets (e'&Me'&Ee'): (rm I). exists*  e'.
     left. splits~. left. introv A. fold_bool. rew_refl in I. false I.
      forwards Eq': from_list_simpl. apply EquivDec.sym in Eq'. apply PosetDec.refl in Eq'.
      forwards Ol: @PosetDec.trans O' Eq'. apply from_list_spec in Ol.
      rewrite Exists_iff_exists_mem. lets (e'&Me'&Ee'): (rm Ol). exists* e'.
   left. splits~. apply TopDec.prop.
Qed.

End Substitutions.

End FinitePowerSet.

