
(** Some very general lattices. **)

Set Implicit Arguments.
Require Export SharedLat Shared.
Require Export TLC.LibTactics TLC.LibReflect.


Module UnitLattice.

Section UnitLattice.

Definition t := unit.

Global Instance EquivDec : EquivDec.t t.
  apply EquivDec.Make with (eq := fun _ _ => True); intros; autos*.
Defined.

Global Instance PosetDec : PosetDec.t t.
  apply PosetDec.Make with (eq := _) (order := fun _ _ => True); intros; autos*.
Defined.

Global Instance LatDec : LatticeDec.t t.
  apply LatticeDec.Make with (porder := PosetDec).
   apply JoinDec.Make with (op := fun _ _ => tt); simpls*.
   apply MeetDec.Make with (op := fun _ _ => tt); simpls*.
   apply BotDec.Make with (elem := tt); simpls*.
   exact (fun _ _ => tt).
Defined.

Global Instance TopDec : TopDec.t t.
  apply TopDec.Make with (elem := tt); simpls*.
Defined.

Global Instance Lat : Lattice.t t.
  typeclass. Defined.

Definition gamma : t → unit → Prop := fun _ _ => True.

Global Instance Monotone : @Monotone.t (℘ unit) t _ _ gamma.
Proof. apply Monotone.Make; unfolds gamma; simpl; autos*. Qed.

Global Instance Gamma : Gamma.t (℘ unit) t gamma.
Proof. apply Gamma.Make; unfolds gamma; simpl; autos*. Qed.

End UnitLattice.

End UnitLattice.


Section Flat.

Variable T : Type.
Hypothesis ET : EquivDec.t T.

Global Instance FlatLatTop : TopDec.t (@FlatLatticeDec.TB T).
  apply TopDec.Make with (elem := FlatLatticeDec.TB_Top).
  intro x. destruct x; simpls*.
Defined.

Global Instance FlatLatBot : BotDec.t (@FlatLatticeDec.TB T).
  apply BotDec.Make with (elem := FlatLatticeDec.TB_Bot).
  intro x. destruct x; simpls*.
Defined.

Hypothesis CT : Comparable T.

Global Instance FlatLatComparable : Comparable (@FlatLatticeDec.TB T).
  prove_comparable_simple_inductive. Defined.

Definition flatLatMap A B (f : A → B) t :=
  match t with
  | FlatLatticeDec.TB_Top => FlatLatticeDec.TB_Top
  | FlatLatticeDec.TB_Bot => FlatLatticeDec.TB_Bot
  | FlatLatticeDec.TB_Elem e => FlatLatticeDec.TB_Elem (f e)
  end.

Lemma flatLatMap_monotone : forall A B (EA : EquivDec.t A) (EB : EquivDec.t B) (f1 f2 : A → B) t1 t2,
  t1 ⊑♯ t2 →
  (forall a1 a2, a1 =♯ a2 → f1 a1 =♯ f2 a2) →
  flatLatMap f1 t1 ⊑♯ flatLatMap f2 t2.
Proof.
  introv O E. destruct t1; destruct t2; tryfalse; autos~.
  apply~ @PosetDec.refl. applys E O.
Qed.

End Flat.


Module CoLattice.

Section CoLattice.

Variable T : Type.

Section Indep.

Hypothesis P : Poset.t T.

Definition Poset : Poset.t T.
  apply Poset.Make with (eq := _) (order := fun x y => y ⊑ x); autos*.
Defined.

Hypothesis V : Bot.t P.

Definition Top : @Top.t T Poset.
  apply Top.Make with (elem := ⊥). simpl. autos*.
Defined.

Hypothesis TD : Top.t P.

Definition Bot : @Bot.t T Poset.
   apply Bot.Make with (⊤). simpl. apply TD.
Defined.

Hypothesis M : Meet.t P.

Definition Join : @Join.t T Poset.
   apply Join.Make with (op := fun x y => x ⊓ y); simpl; autos*.
Defined.

Hypothesis J : Join.t P.

Definition Meet : @Meet.t T Poset.
   apply Meet.Make with (fun x y => x ⊔ y); simpl; autos*.
Defined.

End Indep.

Section IndepLat.

Hypothesis L : Lattice.t T.

Definition Lat : Lattice.t T.
  apply Lattice.Make with (Poset _).
   apply Join. apply L.
   apply Meet. apply L.
   apply Bot. apply L.
   apply Top. apply L.
Defined.

End IndepLat.

Section IndepDec.

Hypothesis P : PosetDec.t T.

Definition PosetDec : PosetDec.t T.
  apply PosetDec.Make with (eq := _) (order := fun x y => y ⊑♯ x); autos~.
   autos*.
   introv. apply (PosetDec.dec y x).
Defined.

Hypothesis V : BotDec.t T.

Definition TopDec : @TopDec.t T PosetDec.
  apply TopDec.Make with (elem := ⊥♯). simpl. autos*.
Defined.

Hypothesis TD : TopDec.t T.

Definition BotDec : @BotDec.t T PosetDec.
   apply BotDec.Make with (⊤♯). simpl. apply TD.
Defined.

Hypothesis M : MeetDec.t T.

Definition JoinDec : @JoinDec.t T PosetDec.
   apply JoinDec.Make with (op := fun x y => x ⊓♯ y); simpl; autos*.
Defined.

Hypothesis J : JoinDec.t T.

Definition MeetDec : @MeetDec.t T PosetDec.
   apply MeetDec.Make with (fun x y => x ⊔♯ y); simpl; autos*.
Defined.

End IndepDec.

Section IndepLatDec.

Hypothesis L : LatticeDec.t T.
Hypothesis TL : TopDec.t T.

Definition LatDec_widen (widen : T → T → T) : LatticeDec.t T.
  apply LatticeDec.Make with (PosetDec _).
   apply JoinDec. apply L.
   apply MeetDec. apply L.
   apply BotDec. apply TL.
   exact widen.
Defined.

Definition LatDec := LatDec_widen (fun x y => x ⊔♯ y).

End IndepLatDec.

End CoLattice.

End CoLattice.


Section FlatLat.

Variable C : Type.
Hypothesis E : EquivDec.t C.

Definition flatGamma (a : @FlatLatticeDec.TB C) (c : C) : Prop :=
  match a with
  | FlatLatticeDec.TB_Top => True
  | FlatLatticeDec.TB_Bot => False
  | FlatLatticeDec.TB_Elem c' => c =♯ c'
  end.

Global Instance FlatMonotone : @Monotone.t (℘ C) (@FlatLatticeDec.TB C) _ _ flatGamma.
  apply Monotone.Make.
   intros [| |c1] [| |c2] I c G; simpls; tryfalse; autos*.
Qed.

Global Instance FlatGamma : Gamma.t (℘ C) (@FlatLatticeDec.TB C) flatGamma.
  apply Gamma.Make.
   apply Monotone.monotone.
   intros [| |c1] [| |c2] c (H1&H2); simpls; tryfalse; autos*.
    repeat cases_if~. false* H0.
Qed.

End FlatLat.


Module PowerSet.

Section PowerSet.

Variable T : Type.
Variable all : (T → bool) → bool.
Hypothesis all_spec : forall f, all f ↔ forall t, f t.

Definition t := T → bool.

Hypothesis CT : Comparable T.

Definition const : T → t := fun t1 t2 => decide (t1 = t2).

Global Instance Equiv : EquivDec.t t.
  apply EquivDec.Make with (fun f1 f2 => forall t, f1 t = f2 t); autos~.
   introv I1 I2; introv. rewrite~ I1.
   intros f1 f2. sets_eq <- ra: (all (fun t => decide (f1 t = f2 t))).
    destruct ra; [left|right].
     fold_bool; rewrite all_spec in EQra. intro t. lets: (EQra t). rew_refl~ in *.
     fold_bool. rew_refl in *. rewrite all_spec in EQra. intro P; apply EQra. introv. rew_refl~.
Defined.

Global Instance Poset : PosetDec.t t.
  apply PosetDec.Make with
    (eq := Equiv) (order := fun f1 f2 : _ → bool => forall t, f1 t → f2 t).
  intros f1 f2 E t. rewrite~ E.
  intros f1 f2 I1 I2 t. lets I1': (rm I1) t. lets I2': (rm I2) t.
   destruct (f1 t); fold_bool; rew_refl in *; autos*.
  firstorder.
  intros f1 f2. set_eq <- ra: (all (fun t => decide (f1 t → f2 t))).
   destruct ra; [left|right].
    fold_bool; rewrite all_spec in EQra. intro t. lets: (EQra t). rew_refl~ in *.
    fold_bool. rew_refl in *. rewrite all_spec in EQra. intro P; apply EQra. introv. rew_refl~.
Defined.

Global Instance Lattice : LatticeDec.t t.
  apply LatticeDec.Make with Poset.
   apply JoinDec.Make with (fun (f1 f2 : T → bool) t => decide (f1 t \/ f2 t));
     try solve [ repeat intro; rew_refl* in * ].
   apply MeetDec.Make with (fun (f1 f2 : T → bool) t => decide (f1 t /\ f2 t));
     try solve [ repeat intro; rew_refl* in * ].
   apply BotDec.Make with (fun _ => false);
     try solve [ repeat intro; rew_refl* in * ].
   (* Widen *)
   exact (fun f1 f2 t => decide (f1 t ∨ f2 t)).
Defined.

Global Instance Top : TopDec.t t.
   apply TopDec.Make with (fun _ => true);
     try solve [ repeat intro; rew_refl* in * ].
Defined.

Definition gamma (f : t) t : Prop := f t.

Global Instance Monotone : @Monotone.t (℘ T) t _ _ gamma.
Proof.
  apply Monotone.Make; unfolds~ gamma.
Qed.

Global Instance Gamma : Gamma.t (℘ T) t gamma.
Proof.
  apply Gamma.Make; unfolds~ gamma.
  intros f1 f2 t M. simpls. rew_refl~.
Qed.

Lemma gamma_const : forall t t', gamma (const t) t' ↔  t = t'.
Proof. introv. unfolds gamma, const. rew_refl*. Qed.

Lemma gamma_bot : forall e,
  ~ gamma (⊥♯) e.
Proof. introv A. false*. Qed.

Lemma gamma_top : forall e,
  gamma (⊤♯) e.
Proof. introv. compute. autos*. Qed.

End PowerSet.

End PowerSet.

Ltac constructor_list t_rec :=
  match type of t_rec with
  | forall (P : ?t -> _), _ =>
    let f := constr:(fun (x : t) => true) in
    let t_rec' := constr:(t_rec f) in
    let H := type of t_rec' in
    let rec aux L H :=
      match H with
      | _ ?x -> ?H' =>
        let L' := constr:(x :: L) in
        aux L' H'
      | _ => L
      end
    in aux (@nil t) H
  end.

Ltac build_comparable_enumerable :=
  try typeclass;
  apply make_comparable; intros () ();
  try (apply decidable_make with false; fold_bool; rew_refl~; discriminate);
  apply decidable_make with true; fold_bool; rew_refl~.

Ltac build_PowerSetPoset_enumerable t_rec :=
  let L := constructor_list t_rec in
  eapply PowerSet.Poset with (fun f : _ -> bool =>
    decide (LibList.Forall f L));
  [ introv; let I := fresh in iff I; rew_refl in *;
     [ intro t; destruct t;
       repeat (let I' := fresh in lets~ (?&I'): Forall_inv (rm I); rename I' into I)
     | apply Forall_forall; intros ? _; apply* I ]
  | build_comparable_enumerable ].

Ltac build_PowerSetLat_enumerable t_rec :=
  let L := constructor_list t_rec in
  eapply PowerSet.Lattice with (fun f : _ -> bool =>
    decide (LibList.Forall f L));
  [ introv; let I := fresh in iff I; rew_refl in *;
     [ intro t; destruct t;
       repeat (let I' := fresh in lets~ (?&I'): Forall_inv (rm I); rename I' into I)
     | apply Forall_forall; intros ? _; apply* I ]
  | build_comparable_enumerable ].


