
(** Adding some elements or operations in a lattice. **)

Set Implicit Arguments.
Require Export Generic.
Require Export Shared.


(** * Adding a Top Element **)

Module LiftTop.

Section LiftTop.

Variable A : Type.

Definition t := option A.

Definition const a : t := Some a.

Hypothesis CA : Comparable A.

Global Instance t_comparable : Comparable t.
  typeclass.
Defined.

Section Equiv.

Hypothesis E : Equiv.t A.

Definition Eq (x y : t) :=
  match x, y with
  | Some a1, Some a2 => Equiv.eq a1 a2
  | None, None => True
  | _, _ => False
  end.

Lemma Eq_LiftEquivDec : forall (E' : EquivDec.t A),
  (forall a1 a2, a1 =♯ a2 ↔ Equiv.eq a1 a2) →
  Eq = EquivDec.eq (t := LiftEquivDec E').
Proof.
  introv EE'. extens. intros [a1|] [a2|]; simpl; iff~ I.
   rewrite~ EE'.
   rewrite~ <- EE'.
Qed.

Instance Equiv : Equiv.t t.
  apply Equiv.Make with (eq := Eq).
   intros [a|]; simpl; autos~.
   intros [a1|] [a2|]; simpl; autos~.
   intros [a1|] [a2|] [a3|] E1 E2; simpls*.
Defined.

Hypothesis ED : forall a1 a2 : A, Decidable (Equiv.eq a1 a2).

Global Instance Eq_decidable : forall x y, Decidable (Eq x y).
  set (EA := Equiv_EquivDec _ ED).
  rewrite Eq_LiftEquivDec with EA; typeclass.
Defined.

End Equiv.

Section Indep.

Hypothesis P : Poset.t A.

Definition order (x y : t) :=
  match x, y with
    | Some x, Some y => x ⊑ y
    | _, None => True
    | None, Some _ => False
  end.

Instance Poset : Poset.t t.
  apply Poset.Make with (eq := Equiv _) (order := order);
    repeat intros [?|]; intros; simpl; autos~; tryfalse.
   apply @Poset.trans with a0; autos*.
Defined.

Instance Top : @Top.t t Poset.
  apply Top.Make with None.
  intros [?|]; simpls~.
Defined.

Lemma lift_order : forall a b : A,
  a ⊑ b →
  const a ⊑ const b.
Proof. introv O. simpls~. Qed.

Lemma lift_order_rev : forall a b : A,
  const a ⊑ const b →
  a ⊑ b.
Proof. introv O. simpls~. Qed.

Hypothesis B : @Bot.t A _.

Instance Bot : @Bot.t t _.
  apply Bot.Make with (elem := Some (⊥)); intros [?|]; simpls*.
Defined.

Hypothesis J : @Join.t A _.

Instance Join : @Join.t t _.
  apply Join.Make with (op := fun x y =>
                                match x, y with
                                | Some x, Some y => Some (x ⊔ y)
                                | _, _ => None
                                end);
   repeat intros [?|]; intros; simpl; autos~.
Defined.

Hypothesis M : @Meet.t A _.

Instance Meet : @Meet.t t _.
  apply Meet.Make with (op := fun x y =>
                                match x, y with
                                | Some x, Some y => Some (x ⊓ y)
                                | None, _ => y
                                | _, None => x
                                end);
   repeat intros [?|]; intros; simpl; autos~.
Defined.

Hypothesis ED : forall a1 a2 : A, Decidable (Equiv.eq a1 a2).
Hypothesis OD : forall a1 a2 : A, Decidable (a1 ⊑ a2).

Global Instance Poset_decidable : forall t1 t2 : t,
    Decidable (t1 ⊑ t2).
  intros [a1|] [a2|]; simpl; typeclass.
Defined.

End Indep.

Section IndepDec.

Hypothesis P : PosetDec.t A.

Instance PosetDec : PosetDec.t t.
  apply Poset_PosetDec with (P := Poset _).
   apply Eq_decidable. simpl. typeclass.
   apply Poset_decidable. simpl. typeclass.
Defined.

Lemma lift_order_dec : forall a b : A,
  a ⊑♯ b →
  const a ⊑♯ const b.
Proof. introv O. simpls~. Qed.

Lemma lift_order_rev_dec : forall a b : A,
  const a ⊑♯ const b →
  a ⊑♯ b.
Proof. introv O. simpls~. Qed.

End IndepDec.

Hypothesis L : Lattice.t A.

Definition widen (widen0 : A → A → A) : t → t → t.
  intros [x|] [y|]; exact (Some (widen0 x y)) || exact None.
Defined.

Definition Lat : Lattice.t t.
  apply Lattice.Make with (porder := Poset _).
   applys Join L.
   applys Meet L.
   applys Bot L.
   apply Top.
Defined.

Hypothesis ED : forall a1 a2 : A, Decidable (Equiv.eq a1 a2).
Hypothesis OD : forall a1 a2 : A, Decidable (a1 ⊑ a2).

Definition LatDec : LatticeDec.t t.
  applys update_widen (widen (fun x y => x ∇♯ y)).
  applys~ Lattice_LatticeDec Lat.
   apply~ Eq_decidable.
   apply~ Poset_decidable.
Defined.

Definition TopDec : TopDec.t (H := _ LatDec) t.
  apply~ Top_TopDec. typeclass.
Defined.

Variable C : Type.

Section Gamma.

Variable gamma0 : A → C → Prop.
Hypothesis G : Gamma.t (℘ C) A gamma0.

Definition gamma t c :=
  match t with
  | None => True
  | Some t => gamma0 t c
  end.

Instance Gamma : @Gamma.t (℘ C) t _ LatDec gamma.
Proof.
  apply Gamma.Make.
   intros [?|] [?|] I c Hg; simpl; autos~.
    applys* @Gamma.monotone G Hg.
    inverts I.
   intros [?|] [?|] c [Hg1 Hg2]; simpls; autos~.
    applys~ @Gamma.meet_morph G. simpls*.
Qed.

End Gamma.

Variables gamma0 gamma1 : A → C → Prop.
Hypothesis gamma_bigger : ∀ a1 a2 : A, a1 ⊑♯ a2 → gamma0 a1 ⊑ gamma1 a2.

Lemma gamma_param_monotone : forall t,
  gamma gamma0 t ⊑ gamma gamma1 t.
Proof. intros [t|] c G; simpls~. applys* gamma_bigger G. Qed.

End LiftTop.

Section Map.

Variables A B : Type.
Hypothesis PA : Poset.t A.
Hypothesis PB : Poset.t B.

Definition map (f : A → B) : t A → t B :=
  option_map f.

Lemma map_monotone : forall f t1 t2,
  (forall a1 a2, a1 ⊑ a2 → f a1 ⊑ f a2) →
  @Poset.order _ (Poset _) t1 t2 →
  @Poset.order _ (Poset _) (map f t1) (map f t2).
Proof. introv Mf O. destruct t1, t2; simpls~. Qed.

Definition map_flat (f : A → t B) : t A → t B :=
  option_case None f.

Lemma map_flat_monotone : forall f t1 t2,
  (forall a1 a2, a1 ⊑ a2 → @Poset.order _ (Poset _) (f a1) (f a2)) →
  @Poset.order _ (Poset _) t1 t2 →
  @Poset.order _ (Poset _) (map_flat f t1) (map_flat f t2).
Proof.
  introv Mf O. destruct t1 as [a1|], t2 as [a2|]; simpls~.
   destruct~ (f a1).
   false.
Qed.

End Map.

Section Flat.

Variable A : Type.
Hypothesis PA : Poset.t A.
Hypothesis TA : Top.t PA.

Definition flat : t A → A :=
  option_case (⊤) id.

Lemma flat_monotone : forall a1 a2 : t A,
  @Poset.order _ (Poset _) a1 a2 →
  flat a1 ⊑ flat a2.
Proof. introv O. destruct a1 as [a1|], a2 as [a2|]; simpls~. false. Qed.

End Flat.

Lemma flat_map_map_flat : forall A B PB (a : t A) f,
  @map_flat A B f a = flat (Top PB) (map f a).
Proof. introv. destruct~ a. Qed.

End LiftTop.


(** * Adding a Bottom Element **)

Module LiftBot.

Section LiftBot.

Variable A : Type.

Definition t := nosimpl (LiftTop.t A).

Definition const a : t := nosimpl (LiftTop.const a).

Hypothesis CA : Comparable A.

Global Instance t_comparable : Comparable t.
  typeclass.
Defined.

Instance Equiv : Equiv.t A → Equiv.t t := nosimpl (@LiftTop.Equiv _).

Global Instance Eq_decidable : forall (x y : t) (E : Equiv.t A),
    (forall a1 a2 : A, Decidable (Equiv.eq a1 a2)) →
    Decidable (Equiv.eq x y).
  introv D. apply~ LiftTop.Eq_decidable.
Defined.

Section Indep.

Hypothesis P : Poset.t A.

Instance Poset : Poset.t t.
  apply CoLattice.Poset. apply LiftTop.Poset. applys CoLattice.Poset P.
Defined.

Instance Bot : @Bot.t t _.
   apply CoLattice.Bot. apply LiftTop.Top.
Defined.

Lemma lift_order : forall a b : A,
  a ⊑ b →
  const a ⊑ const b.
Proof. introv O. simpls~. Qed.

Lemma lift_order_rev : forall a b : A,
  const a ⊑ const b →
  a ⊑ b.
Proof. introv O. simpls~. Qed.

Hypothesis J : @Join.t A _.

Instance Join : @Join.t t _.
  apply CoLattice.Join. apply LiftTop.Meet. applys CoLattice.Meet J.
Defined.

Hypothesis M : @Meet.t A _.

Instance Meet : @Meet.t t _.
   apply CoLattice.Meet. apply LiftTop.Join. applys CoLattice.Join M.
Defined.

Hypothesis T : @Top.t A _.

Definition Top : @Top.t t _.
  apply CoLattice.Top. apply LiftTop.Bot. applys CoLattice.Bot T.
Defined.

Hypothesis ED : forall a1 a2 : A, Decidable (Equiv.eq a1 a2).
Hypothesis OD : forall a1 a2 : A, Decidable (a1 ⊑ a2).

Global Instance Poset_decidable : forall t1 t2 : t,
    Decidable (t1 ⊑ t2).
  intros [a1|] [a2|]; simpl; typeclass.
Defined.

End Indep.

Section IndepDec.

Hypothesis P : PosetDec.t A.

Instance PosetDec : PosetDec.t t.
  apply Poset_PosetDec with (P := Poset _).
   apply LiftTop.Eq_decidable. simpl. typeclass.
   apply Poset_decidable. simpl. typeclass.
Defined.

Lemma lift_order_dec : forall a b : A,
  a ⊑♯ b →
  const a ⊑♯ const b.
Proof. introv O. simpls~. Qed.

Lemma lift_order_rev_dec : forall a b : A,
  const a ⊑♯ const b →
  a ⊑♯ b.
Proof. introv O. simpls~. Qed.

End IndepDec.

Hypothesis L : Lattice.t A.
Hypothesis T : @Top.t A _.

Definition widen (widen0 : A → A → A) : t → t → t.
  intros [x|] [y|].
   exact (Some (widen0 x y)).
   exact (Some x).
   exact (Some y).
   exact None.
Defined.

Definition Lat : Lattice.t t.
  apply Lattice.Make with (porder := Poset _).
   applys Join L.
   applys Meet L.
   applys Bot.
   applys Top T.
Defined.

Hypothesis ED : forall a1 a2 : A, Decidable (Equiv.eq a1 a2).
Hypothesis OD : forall a1 a2 : A, Decidable (a1 ⊑ a2).

Definition LatDec : LatticeDec.t t.
  applys update_widen (widen (fun x y => x ∇♯ y)).
  applys~ Lattice_LatticeDec Lat.
   introv. apply~ Eq_decidable.
   apply~ Poset_decidable.
Defined.

Definition TopDec : TopDec.t (H := _ LatDec) t.
  apply~ Top_TopDec. typeclass.
Defined.

Variable C : Type.

Section Gamma.

Variable gamma0 : A → C → Prop.
Hypothesis G : Gamma.t (℘ C) A gamma0.

Definition gamma t c :=
  match t with
  | None => False
  | Some t => gamma0 t c
  end.

Global Instance Gamma : @Gamma.t (℘ C) t _ LatDec gamma.
Proof.
  apply Gamma.Make.
   intros [?|] [?|] I c Hg; simpl; autos~.
    applys* @Gamma.monotone G Hg.
    inverts Hg.
   intros [?|] [?|] c [Hg1 Hg2]; simpls; autos~.
    applys~ @Gamma.meet_morph G. simpls*.
Qed.

End Gamma.

Variables gamma0 gamma1 : A → C → Prop.
Hypothesis gamma_bigger : ∀ a1 a2 : A, a1 ⊑♯ a2 → gamma0 a1 ⊑ gamma1 a2.

Lemma gamma_param_monotone : forall t,
  gamma gamma0 t ⊑ gamma gamma1 t.
Proof. intros [t|] c G; simpls~. applys* gamma_bigger G. Qed.

End LiftBot.

Section Map.

Variables A B : Type.
Hypothesis PA : Poset.t A.
Hypothesis PB : Poset.t B.

Definition map (f : A → B) : t A → t B :=
  option_map f.

Lemma map_monotone : forall f t1 t2,
  (forall a1 a2, a1 ⊑ a2 → f a1 ⊑ f a2) →
  @Poset.order _ (Poset _) t1 t2 →
  @Poset.order _ (Poset _) (map f t1) (map f t2).
Proof. introv Mf O. apply LiftTop.map_monotone with (PA := CoLattice.Poset PA); simpls~. Qed.

Definition map_flat (f : A → t B) : t A → t B :=
  option_case None f.

Lemma map_flat_monotone : forall f t1 t2,
  (forall a1 a2, a1 ⊑ a2 → @Poset.order _ (Poset _) (f a1) (f a2)) →
  @Poset.order _ (Poset _) t1 t2 →
  @Poset.order _ (Poset _) (map_flat f t1) (map_flat f t2).
Proof.
  introv Mf O. apply LiftTop.map_flat_monotone with (PA := CoLattice.Poset PA); autos~.
   clear - Mf. introv O. forwards~: Mf O.
Qed.

End Map.

Section Flat.

Variable A : Type.
Hypothesis PA : Poset.t A.
Hypothesis BA : @Bot.t A _.

Definition flat : t A → A :=
  option_case (⊥) id.

Lemma flat_monotone : forall a1 a2 : t A,
  @Poset.order _ (Poset _) a1 a2 →
  flat a1 ⊑ flat a2.
Proof. introv O. apply LiftTop.flat_monotone with (TA := CoLattice.Top BA). autos~. Qed.

End Flat.

Lemma flat_map_map_flat : forall A B PB (a : t A) f,
  @map_flat A B f a = flat (Bot PB) (map f a).
Proof. introv. destruct~ a. Qed.

End LiftBot.


(** * Adding both a Top and a Bottom Element **)

Module LiftTopBot.

Section LiftTopBot.

Variable T : Type.
Hypothesis ET : EquivDec.t T.

Definition t := @FlatLatticeDec.TB T.

Variable C : Type.
Variable gamma0 : T → C → Prop.
Hypothesis gamma0_fun : forall a1 a2 c,
  a1 =♯ a2 →
  gamma0 a1 c →
  gamma0 a2 c.
Hypothesis gamma0_inj : forall a1 a2 c,
  gamma0 a1 c →
  gamma0 a2 c →
  a1 =♯ a2.

(** In contrary to [flatGamma], this one is parametrised by another
  concretisation function. **)
Definition gamma : t → C → Prop :=
  fun e =>
    match e with
    | FlatLatticeDec.TB_Top => fun _ => True
    | FlatLatticeDec.TB_Elem e => gamma0 e
    | FlatLatticeDec.TB_Bot => fun _ => False
    end.

Global Instance Gamma : Gamma.t _ _ gamma.
Proof.
  apply Gamma.Make.
   introv O G. destruct N1; destruct N2; simpls~; tryfalse. applys gamma0_fun O G.
   introv (G1&G2). destruct N1; destruct N2; simpls~; tryfalse.
    forwards E: gamma0_inj G1 G2. repeat cases_if~.
Qed.

End LiftTopBot.

End LiftTopBot.


(** * Adding both a Join Operation **)

Module LiftJoin.

Section LiftJoin.

Variable T : Type.

Hypothesis CT : Comparable T.
Hypothesis P : PosetDec.t T.

Definition t : Type := { l : list T | l ≠ nil }.

Definition const (a : T) : t.
  refine (exist _ (cons a nil) _).
  discriminate.
Defined.

Global Instance t_comparable : Comparable t.
  constructors. introv. applys Decidable_equiv (proj1_sig x = proj1_sig y).
   destruct x as [x Ox], y as [y Oy]. iff E.
    simpl in E. substs. fequals.
    inverts~ E.
   typeclass.
Defined.

Definition Order_base (l1 l2 : list T) :=
  Forall (fun a1 => Exists (fun a2 => a1 ⊑♯ a2) l2) l1.

Definition Order (l1 l2 : t) :=
  Order_base (proj1_sig l1) (proj1_sig l2).

Local Instance Order_base_dec : forall l1 l2, Decidable (Order_base l1 l2).
  typeclass. Defined. (* Without this, Coq is not able to deal with later [typeclass] tactics... *)

Lemma Order_base_Order : forall l1 l2 D1 D2,
  Order_base l1 l2 →
  Order (exist _ l1 D1) (exist _ l2 D2).
Proof. introv O. unfolds~. Qed.

Lemma Order_base_refl : forall l, Order_base l l.
Proof.
  introv. unfolds. induction l; constructors~.
   apply~ Exists_here.
   applys~ Forall_weaken IHl. introv E. simpls. apply~ Exists_next.
Qed.

Lemma Order_refl : forall l, Order l l.
Proof. introv. destruct l as [l D]. apply Order_base_refl. Qed.

Lemma Order_base_trans : forall l1 l2 l3,
  Order_base l1 l2 → Order_base l2 l3 → Order_base l1 l3.
Proof.
  introv O1 O2. unfolds Order_base. rewrite Forall_iff_forall_mem with (CA := _) in *. introv I.
  forwards O1': O1 I. rewrite Exists_iff_exists_mem in *. lets (a2&I2&Oa2): (rm O1').
  forwards O2': O2 I2. rewrite Exists_iff_exists_mem in *. lets* (a3&I3&Oa3): (rm O2').
Qed.

Lemma Order_trans : forall l1 l2 l3,
  Order l1 l2 → Order l2 l3 → Order l1 l3.
Proof.
  introv. destruct l1 as [l1 D1], l2 as [l2 D2], l3 as [l3 D3].
  unfold Order. simpl. apply Order_base_trans.
Qed.

Definition Equiv_base : EquivDec.t (list T).
  apply EquivDec.Make with (eq := fun l1 l2 =>
    Order_base l1 l2 ∧ Order_base l2 l1).
   introv. splits; apply Order_base_refl.
   autos*.
   introv (O1&O2) (O3&O4). splits; eapply Order_base_trans; eassumption.
   introv. apply decidable_sumbool. typeclass.
Defined.

Lemma eq_mix : forall (l1 l2 : list T),
  (forall x, mem x l1 = mem x l2) →
  EquivDec.eq (t := Equiv_base) l1 l2.
Proof.
  introv Eq. simpl. unfold Order_base. do 2 rewrite Forall_iff_forall_mem.
  splits; introv M; rewrite Exists_iff_exists_mem.
   rewrite Eq in M. exists* x.
   rewrite <- Eq in M. exists* x.
Qed.

Instance Equiv : EquivDec.t t.
  apply EquivDec.Make with (eq := fun l1 l2 =>
    Order l1 l2 ∧ Order l2 l1); autos~.
   introv. apply @EquivDec.refl with (t := Equiv_base).
   introv. apply @EquivDec.sym with (t := Equiv_base).
   introv. apply @EquivDec.trans with (t := Equiv_base).
   introv. apply @EquivDec.dec with (t := Equiv_base).
Defined.

Lemma Order_base_Order_eq : forall l1 l2 D1 D2,
  EquivDec.eq (t := Equiv_base) l1 l2 ↔
  exist _ l1 D1 =♯ exist _ l2 D2.
Proof. iff O; unfolds~. Qed.

Lemma Order_base_Order_sig : forall l1 l2,
  EquivDec.eq (t := Equiv_base) (proj1_sig l1) (proj1_sig l2) →
  l1 =♯ l2.
Proof. introv E. destruct l1. destruct l2. rewrite~ <- Order_base_Order_eq. Qed.

Definition Poset_base : PosetDec.t (list T).
  apply PosetDec.Make with (eq := Equiv_base)
    (order := fun l1 l2 => decide (Order_base l1 l2)).
   introv (O1&O2). rew_refl~.
   introv O1 O2. rew_refl in *. split~.
   introv O1 O2. rew_refl in *. applys~ Order_base_trans O1 O2.
   introv. apply decidable_sumbool. typeclass.
Defined.

Instance Poset : PosetDec.t t.
  apply PosetDec.Make with (eq := Equiv)
    (order := fun l1 l2 => decide (Order l1 l2)).
   introv. apply @PosetDec.refl with (t := Poset_base).
   introv. apply @PosetDec.antisym with (t := Poset_base).
   introv. apply @PosetDec.trans with (t := Poset_base).
   introv. apply @PosetDec.dec with (t := Poset_base).
Defined.

Lemma Order_base_Order_poset : forall l1 l2 D1 D2,
  PosetDec.order (t := Poset_base) l1 l2 ↔
  exist _ l1 D1 ⊑♯ exist _ l2 D2.
Proof. iff O; unfolds~. Qed.

Lemma lift_order_base : forall a b : T,
  a ⊑♯ b →
  PosetDec.order (t := Poset_base) (a :: nil) (b :: nil).
Proof.
  introv O. simpls. rew_refl. unfolds. rewrite Forall_iff_forall_mem with (CA := _).
  introv Mx. simpl in Mx. rew_refl in Mx. inverts Mx; tryfalse.
  apply~ Exists_here.
Qed.

Lemma lift_order_base_rev : forall a b : T,
  PosetDec.order (t := Poset_base) (a :: nil) (b :: nil) →
  a ⊑♯ b.
Proof.
  introv O. simpls. rew_refl in O. unfolds in O. rewrite Forall_iff_forall_mem with (CA := _) in O.
  forwards E: O a; [ rew_refl* |]. repeat inverts E as E. autos~.
Qed.

Lemma lift_order : forall a b : T,
  a ⊑♯ b →
  const a ⊑♯ const b.
Proof. apply lift_order_base. Qed.

Lemma lift_order_rev : forall a b : T,
  const a ⊑♯ const b →
  a ⊑♯ b.
Proof. apply lift_order_base_rev. Qed.

Lemma const_order_base : forall a b l,
  a ⊑♯ b →
  mem b l →
  PosetDec.order (t := Poset_base) (a :: nil) l.
Proof.
  introv O M. eapply PosetDec.trans; [ applys lift_order_base O |].
  clear a O. simpl. rew_refl.
  unfolds. repeat constructors. rewrite* Exists_iff_exists_mem.
Qed.

Lemma const_order : forall a b l D,
  a ⊑♯ b →
  mem b l →
  const a ⊑♯ exist _ l D.
Proof. introv O M. applys const_order_base O M. Qed.

Lemma lift_equiv_base : forall a b : T,
  a =♯ b →
  EquivDec.eq (t := Equiv_base) (a :: nil) (b :: nil).
Proof. introv O. apply (PosetDec.antisym (t := Poset_base)); apply~ lift_order_base. Qed.

Lemma lift_equiv : forall a b : T,
  a =♯ b →
  const a =♯ const b.
Proof. apply lift_equiv_base. Qed.


Definition join_base (l1 l2 : list T) : list T :=
  l1 ++ filter (fun a2 =>
    decide (~ Exists (fun a1 => a2 ⊑♯ a1) l1)) l2.

Definition join (l1 l2 : t) : t.
  destruct l1 as [l1 D1], l2 as [l2 D2].
  refine (exist _ (join_base l1 l2) _).
  destruct l1; tryfalse. discriminate.
Defined.

Lemma join_base_mem_left : forall x l1 l2,
  mem x l1 →
  mem x (join_base l1 l2).
Proof. introv Mx. unfolds join_base. rewrite mem_app. rew_refl. left~. Qed.

Lemma join_mem_left : forall x l1 l2,
  mem x (proj1_sig l1) →
  mem x (proj1_sig (join l1 l2)).
Proof.
  introv Mx. destruct l1 as [l1 D1], l2 as [l2 D2]. simpls.
  apply~ join_base_mem_left.
Qed.

Lemma join_base_mem_right : forall x l1 l2,
  mem x l2 →
  exists a, mem a (join_base l1 l2) ∧ x ⊑♯ a.
Proof.
  introv Mx. tests E: (exists y, LibList.mem y l1 ∧ x ⊑♯ y).
   lets (y&Iy&Oy): (rm E). exists y. splits~.
    unfolds join_base. rewrite mem_app. rew_refl. left~.
   exists x. splits~.
    unfolds join_base. rewrite mem_app. rew_refl. right~.
    rewrite filter_mem_eq. rew_refl. splits~.
     rewrite Exists_iff_exists_mem. rew_logic~.
Qed.

Lemma join_mem_right : forall x l1 l2,
  mem x (proj1_sig l2) →
  exists a, mem a (proj1_sig (join l1 l2)) ∧ x ⊑♯ a.
Proof.
  introv Mx. destruct l1 as [l1 D1], l2 as [l2 D2]. simpls. clear D1 D2.
  apply~ join_base_mem_right.
Qed.

Definition Join_base : JoinDec.t (H := Poset_base) (list T).
  apply JoinDec.Make with (op := join_base); simpl.
   intros l1 l2. rew_refl. apply Forall_iff_forall_mem with (CA := _).
    introv I. apply Exists_iff_exists_mem with (CA := _). exists x. splits~.
    unfolds join, join_base. rewrite mem_app. rew_refl. left~.
   intros l1 l2. rew_refl. apply Forall_iff_forall_mem with (CA := _).
    introv I. apply Exists_iff_exists_mem with (CA := _). apply~ join_base_mem_right.
   intros l1 l2 l3 O1 O2. rew_refl in *. apply Forall_iff_forall_mem with (CA := _).
    unfolds Order, Order_base. rewrite Forall_iff_forall_mem in O1, O2.
    introv I. unfold join_base in I. rewrite mem_app in I. rew_refl in I.
    inverts I as I; autos*. rewrite filter_mem_eq in I. rew_refl in I.
    lets (I'&_): (rm I). apply* O2.
Defined.

Instance Join : JoinDec.t t.
  apply JoinDec.Make with (op := join); simpl.
   intros l1 l2. destruct l1 as [l1 D1], l2 as [l2 D2].
    apply @JoinDec.bound1 with (t := Join_base).
   intros l1 l2. destruct l1 as [l1 D1], l2 as [l2 D2].
    apply @JoinDec.bound2 with (t := Join_base).
   intros l1 l2 l3. destruct l1 as [l1 D1], l2 as [l2 D2].
    apply @JoinDec.least_upper_bound with (t := Join_base).
Defined.

Definition join_simpl (l1 l2 : t) : t.
  destruct l1 as [l1 D1], l2 as [l2 D2].
  refine (exist _ (l1 ++ l2) _).
  destruct l1; tryfalse. discriminate.
Defined.

Lemma join_simpl_proj1_sig : forall l1 l2,
  proj1_sig (join_simpl l1 l2) = proj1_sig l1 ++ proj1_sig l2.
Proof. destruct l1 as [l1 D1], l2 as [l2 D2]. reflexivity. Qed.

Lemma join_base_append : forall l1 l2,
  EquivDec.eq (t := Equiv_base) (join_base l1 l2) (l1 ++ l2).
Proof.
  introv. split; unfolds; rewrite Forall_iff_forall_mem; introv Mj; rewrite Exists_iff_exists_mem;
    unfolds join_base; rewrite mem_app in Mj; rew_refl in Mj; inverts Mj as Mj.
   exists x. splits~. rewrite mem_app. rew_refl*.
   rewrite filter_mem_eq in Mj. exists x. splits~. rewrite mem_app. rew_refl* in *.
   exists x. splits~. rewrite mem_app. rew_refl*.
   tests E: (Exists (λ a1 : T, x ⊑♯ a1) l1).
    rewrite Exists_iff_exists_mem in E. lets (a&Ma&Oa): (rm E).
     exists a. splits~. rewrite mem_app. rew_refl*.
    exists x. splits~. rewrite mem_app. rew_refl. right.
     rewrite filter_mem_eq. rew_refl in *. splits~.
Qed.

Lemma join_append : forall l1 l2,
  join l1 l2 =♯ join_simpl l1 l2.
Proof. introv. destruct l1 as [l1 D1], l2 as [l2 D2]. apply~ join_base_append. Qed.

Lemma cons_equiv : forall a b l,
  a ⊑♯ b →
  mem b l →
  EquivDec.eq (t := Equiv_base) (a :: l) l.
Proof.
  introv O M. apply EquivDec.trans with (y := join_base (a :: nil) l).
   apply EquivDec.sym. apply join_base_append.
   apply join_sup with (JV := Join_base).
   applys~ const_order_base M.
Qed.

Lemma cons_join : forall a l,
  EquivDec.eq (t := Equiv_base) (a :: l) (join_base (a :: nil) l).
Proof.
  introv. apply EquivDec.sym. eapply EquivDec.trans.
   apply join_base_append.
   rew_list~.
Qed.


Hypothesis M : MeetDec.t T.

Definition meet (l1 l2 : t) : t.
  destruct l1 as [l1 D1], l2 as [l2 D2].
  refine (exist _ (concat (LibList.map (fun a1 =>
    LibList.map (fun a2 =>
      a1 ⊓♯ a2) l2) l1)) _).
  destruct l1; tryfalse. destruct l2; tryfalse. discriminate.
Defined.

Instance Meet : MeetDec.t t.
  apply MeetDec.Make with (op := meet); simpl.
   intros l1 l2. destruct l1 as [l1 D1], l2 as [l2 D2].
    rew_refl. apply Forall_iff_forall_mem with (CA := _). simpl. clear D1 D2.
    introv I. apply Exists_iff_exists_mem with (CA := _). unfolds meet.
    apply concat_mem in I. lets (l&Il&Ix): (rm I).
    eapply map_mem with (CA := _) in Il. lets (y&Iy&E): (rm Il).
    substs. eapply map_mem with (CA := _) in Ix. lets (z&Iz&E): (rm Ix).
    exists y. splits*. substs*.
   intros l1 l2. destruct l1 as [l1 D1], l2 as [l2 D2].
    rew_refl. apply Forall_iff_forall_mem with (CA := _). simpl. clear D1 D2.
    introv I. apply Exists_iff_exists_mem with (CA := _). unfolds meet.
    apply concat_mem in I. lets (l&Il&Ix): (rm I).
    apply map_mem with (CA := _) in Il. lets (y&Iy&E): (rm Il).
    substs. eapply map_mem in Ix. lets (z&Iz&E): (rm Ix).
    exists z. splits*. substs*.
   intros l1 l2 l3. destruct l1 as [l1 D1], l2 as [l2 D2], l3 as [l3 D3].
    rew_refl. unfolds Order, Order_base. simpl. clear D1 D2 D3.
    introv O31 O32. rewrite Forall_iff_forall_mem in *.
    introv I. forwards I1: (rm O31) I. forwards I2: (rm O32) (rm I).
    rewrite Exists_iff_exists_mem in *.
    lets (a1&M1&O1): (rm I1). lets (a2&M2&O2): (rm I2).
    unfolds meet. exists (a1 ⊓♯ a2). splits~.
    apply concat_mem. eexists. splits.
     eapply map_mem. exists a1. splits*.
     eapply map_mem. exists a2. splits*.
Defined.

Hypothesis B : BotDec.t T.

Instance Bot_base : BotDec.t (H := Poset_base) (list T).
  apply BotDec.Make with (elem := nil).
   introv. simpl. rew_refl. repeat constructors~.
Defined.

Instance Bot : BotDec.t t.
  apply BotDec.Make with (elem := const (⊥♯)).
   introv. destruct x as [[|x l] D]; tryfalse. simpl. rew_refl. repeat constructors~.
Defined.

Lemma bot : const (⊥♯) =♯ (⊥♯).
Proof. apply EquivDec.refl. Qed.

Hypothesis widen0 : T → T → T.

Definition widen : t → t → t. (* This widenning operator can increase fast… *)
  refine (fun l1 l2 =>
    exist _ (concat (LibList.map (fun a1 =>
      LibList.map (fun a2 =>
        widen0 a1 a2) (proj1_sig l2)) (proj1_sig l1))) _).
  introv A. destruct l1 as [[|a1 l1] D1], l2 as [[|a2 l2] D2]; false.
Defined.

Instance Lat : LatticeDec.t t.
  apply LatticeDec.Make with (porder := Poset).
   apply Join.
   apply Meet.
   apply Bot.
   exact widen.
Defined.

Hypothesis Top0 : TopDec.t T.

Instance Top : TopDec.t t.
  apply TopDec.Make with (elem := const (⊤♯)).
   introv. destruct x as [l D]. simpl. rew_refl. do 2 unfolds.
   rewrite Forall_iff_forall_mem with (CA := _). simpl. clear D. introv Ml.
   rewrite Exists_iff_exists_mem with (CA := _). exists (⊤♯ : T). splits~.
    simpl. rew_refl*.
    apply TopDec.prop.
Defined.

Lemma top : const (⊤♯) =♯ (⊤♯).
Proof. apply EquivDec.refl. Qed.

Lemma has_top : forall l D,
  mem (⊤♯) l →
  exist _ l D =♯ (⊤♯).
Proof.
  introv Ml. applys~ @PosetDec.antisym (TopDec.prop).
  eapply PosetDec.trans.
   apply PosetDec.refl. apply EquivDec.sym. apply top.
   apply~ const_order.
Qed.


Variable C : Type.

Section Gamma.

Variable gamma0 : T → C → Prop.
Hypothesis monotone0 : ∀ N1 N2 : T, N1 ⊑♯ N2 → gamma0 N1 ⊑ gamma0 N2.
Hypothesis meet_morph0 : ∀ N1 N2 : T, (gamma0 N1 ⊓ gamma0 N2) ⊑ gamma0 (N1 ⊓♯ N2).

Definition gamma (l : t) c :=
  Exists (fun a => gamma0 a c) (proj1_sig l).

Lemma monotone : forall N1 N2 : t,
  N1 ⊑♯ N2 → gamma N1 ⊑ gamma N2.
Proof.
  introv O G. simpls. rew_refl in O. unfolds gamma. rewrite Exists_iff_exists_mem in *.
  lets (a&I&G0): (rm G). do 2 unfolds in O. rewrite Forall_iff_forall_mem with (CA := _) in O.
  apply O in I. rewrite Exists_iff_exists_mem in *. lets (a'&M'&O'): (rm I).
  exists a'. splits*.
Qed.

Lemma meet_morph : forall N1 N2 : t,
  (gamma N1 ⊓ gamma N2) ⊑ gamma (N1 ⊓♯ N2).
Proof.
  introv (G1&G2). unfolds gamma. rewrite Exists_iff_exists_mem in *.
  destruct N1 as [l1 D1], N2 as [l2 D2].
  lets (a1&I1&G01): (rm G1). lets (a2&I2&G02): (rm G2).
  exists (a1 ⊓♯ a2). splits.
   simpls. clear D1 D2. unfolds meet. apply concat_mem. eexists. splits.
    eapply map_mem. exists a1. splits*.
    eapply map_mem. exists a2. splits*.
   apply meet_morph0. simpls~.
Qed.

Instance Gamma : @Gamma.t (℘ C) t _ Lat gamma.
Proof.
  apply Gamma.Make.
   exact monotone.
   exact meet_morph.
Qed.

Hypothesis gamma0_bot : ∀ c, ~ gamma0 (⊥♯) c.

Lemma gamma_bot : forall c,
  ~ gamma (⊥♯) c.
Proof. introv G. repeat inverts G as G. applys gamma0_bot G. Qed.

Hypothesis gamma0_top : ∀ c, gamma0 (⊤♯) c.

Lemma gamma_top : forall c,
  gamma (⊤♯) c.
Proof.
  introv. unfolds. rewrite Exists_iff_exists_mem with (CA := _). exists (⊤♯ : T). splits~.
  simpl. rew_refl*.
Qed.

End Gamma.

Variables gamma0 gamma1 : T → C → Prop.
Hypothesis gamma_bigger : ∀ a1 a2 : T, a1 ⊑♯ a2 → gamma0 a1 ⊑ gamma1 a2.

Lemma gamma_param_monotone : forall l,
  gamma gamma0 l ⊑ gamma gamma1 l.
Proof.
  introv G. unfolds. unfolds in G. rewrite Exists_iff_exists_mem in *.
  lets (a&Ma&Ga): (rm G). exists a. splits*.
  applys~ gamma_bigger Ga.
Qed.

End LiftJoin.

Section Map.

Variables A B : Type.
Hypothesis CA : Comparable A.
Hypothesis CB : Comparable B.
Hypothesis PA : PosetDec.t A.
Hypothesis MA : MeetDec.t A.
Hypothesis BA : BotDec.t A.
Hypothesis PB : PosetDec.t B.
Hypothesis MB : MeetDec.t B.
Hypothesis BB : BotDec.t B.

Definition map (f : A → B) : t A → t B.
  introv [l D].
  refine (exist _ (map f l) _).
  destruct l; tryfalse. discriminate.
Defined.

Lemma map_monotone : forall f g t1 t2,
  (forall a1 a2, a1 ⊑♯ a2 → f a1 ⊑♯ g a2) →
  (forall a, a =♯ (⊥♯) → f a =♯ (⊥♯)) →
  @PosetDec.order _ (Poset _ _) t1 t2 →
  @PosetDec.order _ (Poset _ _) (map f t1) (map g t2).
Proof.
  introv Mf Bf O. simpls. rew_refl in *. unfolds Order, Order_base. rewrite Forall_iff_forall_mem in *.
  destruct t1 as [l1 D1], t2 as [l2 D2]. simpls. clear D1 D2.
  introv Mx. unfolds map. rewrite map_mem in Mx. lets (y&My&?): (rm Mx). substs.
  forwards~ E: O My. rewrite Exists_iff_exists_mem in *. lets (a&Ma&Oa): (rm E).
  exists (g a). splits~. apply* map_mem.
Qed.

Definition map_flat (f : A → t B) : t A → t B :=
  fun l =>
    fold_right (fun t1 l =>
      @JoinDec.op _ _ (Join _ _) l (f t1)) (@BotDec.elem _ _ (Bot _ _)) (proj1_sig l).

Definition map_flat_simpl (f : A → t B) : t A → t B :=
  fun l =>
    fold_right (fun t1 l =>
      join_simpl l (f t1)) (@BotDec.elem _ _ (Bot _ _)) (proj1_sig l).

Lemma map_flat_eq_simpl : forall f t,
  @EquivDec.eq _ (Equiv _ _) (map_flat f t) (map_flat_simpl f t).
Proof.
  introv. destruct t0 as [l D].
  unfolds map_flat, map_flat_simpl, proj1_sig. clear D. induction l; rew_list.
   apply EquivDec.refl.
   eapply EquivDec.trans; [| apply join_append ].
    eapply EquivDec.trans.
     apply (join_eq1 (t B) (Poset _ _) _ _ _ _ IHl).
     apply EquivDec.refl.
Qed.

Lemma map_flat_monotone : forall f g t1 t2,
  (forall a1 a2, a1 ⊑♯ a2 → @PosetDec.order _ (Poset _ _) (f a1) (g a2)) →
  (forall a, a =♯ (⊥♯) → @EquivDec.eq _ (Equiv _ _) (f a) (@BotDec.elem _ _ (Bot _ _))) →
  @PosetDec.order _ (Poset _ _) t1 t2 →
  @PosetDec.order _ (Poset _ _) (map_flat f t1) (map_flat g t2).
Proof.
  introv Mf Bf O. eapply PosetDec.trans; [ apply PosetDec.refl; apply map_flat_eq_simpl |].
  eapply PosetDec.trans; [| apply PosetDec.refl; apply EquivDec.sym; apply map_flat_eq_simpl ].
  simpl in O. simpl. rew_refl in *. unfolds Order, Order_base.
  destruct t1 as [l1 D1], t2 as [l2 D2]. repeat rewrite Forall_iff_forall_mem with (CA := _) in *. introv Mx.
  unfolds map_flat_simpl. simpls. clear D1 D2. induction l1.
   simpls. rew_refl in Mx. inverts Mx; tryfalse. clear O. induction l2.
    apply~ Exists_here.
    rewrite Exists_iff_exists_mem in *. lets (x&Mx&I): (rm IHl2). exists x. splits~.
     simpl. rewrite join_simpl_proj1_sig. rewrite mem_app. rew_refl*.
   simpl in Mx. rewrite join_simpl_proj1_sig in Mx.
    rewrite mem_app in Mx. rew_refl in Mx. inverts Mx as Mx.
     forwards~: IHl1 Mx. clear - O. introv M. apply O. simpl. rew_refl. right*.
     clear IHl1. forwards E: (rm O) a; [rew_refl~|]. induction E as [l a' E|l a' E IHE].
      forwards Of: Mf E. rew_refl in Of. do 2 unfolds in Of. rewrite Forall_iff_forall_mem in Of.
       forwards Ef: (rm Of) Mx. rewrite Exists_iff_exists_mem with (CA := _) in Ef. lets (a2&Ma2&Ia2): (rm Ef).
       rewrite Exists_iff_exists_mem. simpl. exists a2. splits~.
       rewrite join_simpl_proj1_sig. rewrite mem_app. rew_refl. right*.
      rewrite Exists_iff_exists_mem with (CA := _) in *. lets (a2&Ma2&Ia2): (rm IHE). exists a2. splits~.
       simpl. rewrite join_simpl_proj1_sig. rewrite mem_app. rew_refl. left*.
Qed.

End Map.

Section Flat.

Variable A : Type.

Hypothesis CA : Comparable A.
Hypothesis PA : PosetDec.t A.
Hypothesis JA : JoinDec.t A.
Hypothesis BA : BotDec.t A.

Definition flat : t A → A :=
  fun l =>
    fold_right (fun a1 a2 : A => a1 ⊔♯ a2) (⊥♯) (proj1_sig l).

Lemma flat_monotone : forall a1 a2 : t A,
  @PosetDec.order _ (Poset _ _) a1 a2 →
  flat a1 ⊑♯ flat a2.
Proof.
  introv O. destruct a1 as [l1 D1], a2 as [l2 D2]. unfolds flat. simpls.
  rew_refl in O. unfolds in O. simpls. clear D1 D2. induction O as [|a1 e O1 O2].
   apply BotDec.prop.
   rewrite fold_right_cons. apply~ JoinDec.least_upper_bound. gen a1.
    induction O1 as [a' e' O1|a' e' O1]; introv O2 IHO2; rewrite fold_right_cons.
     applys~ @PosetDec.trans O1.
     apply join_trans_2. apply~ IHO1. clear - CA.
      rewrite Forall_iff_forall_mem with (CA := _). introv M. rewrite Exists_iff_exists_mem. exists* x.
Qed.

Hypothesis TA : TopDec.t A.

Lemma flat_top : forall a,
   @EquivDec.eq _ (Equiv _ _) a (@TopDec.elem _ _ (Top _ _)) →
   flat a =♯ (⊤♯).
Proof.
  introv E. apply PosetDec.antisym.
   apply TopDec.prop.
   apply PosetDec.trans with (y := flat (@TopDec.elem _ _ (Top _ _))).
    clear. unfold flat. simpl. autos*.
    apply* flat_monotone.
Qed.

End Flat.

Lemma flat_map_map_flat : forall A B CB PB BB (a : t A) f,
  @EquivDec.eq _ (Equiv CB PB)
    (@map_flat A B CB PB BB f a)
    (flat (Join _ PB) (Bot CB BB) (map f a)).
Proof.
  introv. destruct a as [a D]. unfolds map_flat, map, flat, proj1_sig. clear D. induction~ a.
   rewrite map_cons. repeat rewrite fold_right_cons.
   eapply EquivDec.trans; [ apply join_sym with (P := Poset _ _) |].
   apply join_eq2 with (P := Poset _ _). autos*.
Qed.

Lemma map_flat_const : forall A CA PA BA e,
  @EquivDec.eq _ (Equiv _ _) (@map_flat A A CA PA BA (const (T := _)) e) e.
Proof.
  introv. destruct e as [e D]. unfolds map_flat, const, proj1_sig. induction e; tryfalse.
  tests N: (e = nil); rew_list.
   eapply EquivDec.trans.
    apply EquivDec.sym. apply join_bottom1 with (P := Poset _ _).
    clear. forwards O: Order_refl (exist (<> nil) (a :: nil) D).
     simpls. unfolds Order. splits~.
   eapply EquivDec.trans.
    apply join_eq1 with (P := Poset _ _). refine (IHe N).
    eapply EquivDec.trans.
     apply join_sym with (P := Poset _ _).
     apply @PosetDec.antisym with (t := Poset _ _); simpl; rew_refl;
       do 2 unfolds; rewrite Forall_iff_forall_mem; introv M; unfolds proj1_sig, join_base.
      rewrite mem_app in M. rew_refl in M. inverts M as M.
       simpl in M. rew_refl in M. inverts M; tryfalse.
        apply~ Exists_here.
       apply Exists_next. rewrite filter_mem_eq in M. rew_refl in M. apply* Exists_iff_exists_mem.
      simpl in M. rew_refl in M. destruct M as [M|M].
       substs. apply~ Exists_here.
       tests C: (x ⊑♯ a).
        apply~ Exists_here.
        apply Exists_next. rewrite Exists_iff_exists_mem with (CA := _). exists x. splits~.
         rewrite filter_mem_eq. rew_refl. splits*. intro I. repeat inverts I as I; false.
Qed.

End LiftJoin.


(** * Adding a Meet Operation **)

Module LiftMeet.

Section LiftMeet.

Variable T : Type.

Hypothesis CT : Comparable T.
Hypothesis P : PosetDec.t T.

Definition t : Type := nosimpl (LiftJoin.t T).

Definition const a : t := nosimpl (LiftJoin.const a).

Global Instance t_comparable : Comparable t.
  apply~ LiftJoin.t_comparable.
Defined.

Instance Equiv : EquivDec.t t.
  applys LiftJoin.Equiv CT. applys CoLattice.PosetDec P.
Defined.

Instance Poset : PosetDec.t t.
  apply CoLattice.PosetDec. applys LiftJoin.Poset CT. applys CoLattice.PosetDec P.
Defined.

Lemma lift_order : forall a b : T,
  a ⊑♯ b →
  const a ⊑♯ const b.
Proof. introv O. apply~ LiftJoin.lift_order. Qed.

Lemma lift_order_rev : forall a b : T,
  const a ⊑♯ const b →
  a ⊑♯ b.
Proof. introv O. applys~ LiftJoin.lift_order_rev O. Qed.

Lemma const_order : forall a b l D,
  b ⊑♯ a →
  mem b l →
  exist _ l D ⊑♯ const a.
Proof. apply LiftJoin.const_order with (P := CoLattice.PosetDec P). Qed.

Instance Meet : MeetDec.t t.
  apply CoLattice.MeetDec. apply LiftJoin.Join.
Defined.

Definition meet_simpl (l1 l2 : t) : t :=
  nosimpl (LiftJoin.join_simpl l1 l2).

Lemma meet_simpl_proj1_sig : forall l1 l2,
  proj1_sig (meet_simpl l1 l2) = proj1_sig l1 ++ proj1_sig l2.
Proof. apply LiftJoin.join_simpl_proj1_sig. Qed.

Lemma meet_append : forall l1 l2,
  l1 ⊓♯ l2 =♯ meet_simpl l1 l2.
Proof. apply LiftJoin.join_append. Qed.

Hypothesis J : JoinDec.t T.

Instance Join : JoinDec.t t.
  apply CoLattice.JoinDec. apply LiftJoin.Meet. applys CoLattice.MeetDec J.
Defined.

Hypothesis B : BotDec.t T.

Instance Bot : BotDec.t t.
  apply CoLattice.BotDec. apply LiftJoin.Top. applys CoLattice.TopDec B.
Defined.

Lemma bot : const (⊥♯) =♯ (⊥♯).
Proof. apply @LiftJoin.top with (P := CoLattice.PosetDec P). Qed.

Lemma has_bot : forall l D,
  mem (⊥♯) l →
  exist _ l D =♯ (⊥♯).
Proof.
  introv Ml. applys~ @PosetDec.antisym (BotDec.prop).
  eapply PosetDec.trans.
   apply~ const_order.
   apply PosetDec.refl. apply bot.
Qed.


Hypothesis widen0 : T → T → T.

Definition widen : t → t → t := (* This widenning operator can increase fast… *)
  fun l1 l2 =>
    let l :=
      concat (LibList.map (fun a1 =>
        LibList.map (fun a2 =>
          const (widen0 a1 a2)) (proj1_sig l2)) (proj1_sig l1)) in
  fold_left (fun a1 a2 =>
    a1 ⊔♯ a2) (⊥♯) l.

Instance Lat : LatticeDec.t t.
  apply LatticeDec.Make with (porder := Poset).
   apply Join.
   apply Meet.
   apply Bot.
   apply widen.
Defined.

Hypothesis TT : TopDec.t T.

Instance Top : TopDec.t t.
  apply CoLattice.TopDec. apply LiftJoin.Bot. applys CoLattice.BotDec TT.
Defined.

Lemma top : const (⊤♯) =♯ (⊤♯).
Proof. apply @LiftJoin.bot with (P := CoLattice.PosetDec P). Qed.

Variable C : Type.

Section Gamma.

Variable gamma0 : T → C → Prop.
Hypothesis monotone0 : ∀ N1 N2 : T, N1 ⊑♯ N2 → gamma0 N1 ⊑ gamma0 N2.
Hypothesis gamma0_top : ∀ c, gamma0 (⊤♯) c.

Definition gamma (l : t) c :=
  Forall (fun a => gamma0 a c) (proj1_sig l).

Lemma monotone : forall N1 N2 : t,
  N1 ⊑♯ N2 → gamma N1 ⊑ gamma N2.
Proof.
  introv O G. simpls. unfolds in G. unfolds. destruct N1 as [l1 D1], N2 as [l2 D2].
  rew_refl in O. do 2 unfolds in O. rewrite Forall_iff_forall_mem with (CA := _) in *.
  introv I. apply O in I. rewrite Exists_iff_exists_mem in I. simpls.
  lets (a&Ia&Oa): (rm I). apply G in Ia. applys~ monotone0 Ia.
Qed.

Lemma meet_morph : forall N1 N2 : t,
  (gamma N1 ⊓ gamma N2) ⊑ gamma (N1 ⊓♯ N2).
Proof.
  introv (G1&G2). unfolds in G1 G2. unfolds.
  destruct N1 as [l1 D1], N2 as [l2 D2].
  simpls. rewrite Forall_iff_forall_mem in *. introv I. unfolds LiftJoin.join_base.
  rewrite mem_app in I. rew_refl in I. inverts I as I.
   autos*.
   rewrite filter_mem_eq in I. rew_refl in I. inverts I as I ?. autos*.
Qed.

Instance Gamma : @Gamma.t (℘ C) t _ Lat gamma.
Proof.
  apply Gamma.Make.
   exact monotone.
   exact meet_morph.
Qed.

Hypothesis gamma0_bot : ∀ c, ~ gamma0 (⊥♯) c.

Lemma gamma_bot : forall c : C,
  ~ gamma (⊥♯) c.
Proof. introv G. inverts G as G _. apply* gamma0_bot. Qed.

Lemma gamma_top : forall c,
  gamma (⊤♯) c.
Proof.
  introv. unfolds. rewrite Forall_iff_forall_mem with (CA := _). introv M.
  simpl in M. rew_refl in M. inverts~ M; tryfalse.
Qed.

End Gamma.

Variables gamma0 gamma1 : T → C → Prop.
Hypothesis gamma_bigger : ∀ a1 a2 : T, a1 ⊑♯ a2 → gamma0 a1 ⊑ gamma1 a2.

Lemma gamma_param_monotone : forall l,
  gamma gamma0 l ⊑ gamma gamma1 l.
Proof.
  introv G. unfolds. unfolds in G. rewrite Forall_iff_forall_mem in *.
  introv M. forwards G': G M. applys~ gamma_bigger G'.
Qed.

End LiftMeet.

Section Map.

Variables A B : Type.
Hypothesis CA : Comparable A.
Hypothesis CB : Comparable B.
Hypothesis PA : PosetDec.t A.
Hypothesis JA : JoinDec.t A.
Hypothesis TA : TopDec.t A.
Hypothesis PB : PosetDec.t B.
Hypothesis JB : JoinDec.t B.
Hypothesis TB : TopDec.t B.

Definition map (f : A → B) : t A → t B :=
  nosimpl (LiftJoin.map f).

Lemma map_monotone : forall f g t1 t2,
  (forall a1 a2, a1 ⊑♯ a2 → f a1 ⊑♯ g a2) →
  (forall a, a =♯ (⊤♯) → g a =♯ (⊤♯)) →
  @PosetDec.order _ (Poset _ _) t1 t2 →
  @PosetDec.order _ (Poset _ _) (map f t1) (map g t2).
Proof.
  introv Mf Tf O. apply LiftJoin.map_monotone with
    (CA := CA) (BB := CoLattice.BotDec TB) (BA := CoLattice.BotDec TA); simpls~.
Qed.

Definition map_flat : (A → t B) → t A → t B :=
  nosimpl (LiftJoin.map_flat _ (CoLattice.BotDec _)).

Lemma map_flat_monotone : forall f g t1 t2,
  (forall a1 a2, a1 ⊑♯ a2 → @PosetDec.order _ (Poset _ _) (f a1) (g a2)) →
  (forall a, a =♯ (⊤♯) → @EquivDec.eq _ (Equiv _ _) (g a) (@TopDec.elem _ _ (Top _ _))) →
  @PosetDec.order _ (Poset _ _) t1 t2 →
  @PosetDec.order _ (Poset _ _) (map_flat f t1) (map_flat g t2).
Proof.
  introv Mf Tf O. apply LiftJoin.map_flat_monotone
    with (CA := CA) (BA := CoLattice.BotDec TA); simpls~.
Qed.

End Map.

Section Flat.

Variable A : Type.
Hypothesis CA : Comparable A.
Hypothesis PA : PosetDec.t A.
Hypothesis MA : MeetDec.t A.
Hypothesis TA : TopDec.t A.

Definition flat : t A → A :=
  nosimpl (LiftJoin.flat (CoLattice.JoinDec _) (CoLattice.BotDec _)).

Lemma flat_monotone : forall a1 a2 : t A,
  @PosetDec.order _ (Poset _ _) a1 a2 →
  flat a1 ⊑♯ flat a2.
Proof.
  introv O. apply LiftJoin.flat_monotone with
    (CA := CA) (JA := CoLattice.JoinDec MA) (BA := CoLattice.BotDec TA).
  exact O.
Qed.

Hypothesis BA : BotDec.t A.

Lemma flat_bot : forall a,
   @EquivDec.eq _ (Equiv _ _) a (@BotDec.elem _ _ (Bot _ _)) →
   flat a =♯ (⊥♯).
Proof.
  introv E. apply LiftJoin.flat_top with (JA := CoLattice.JoinDec MA)
    (CA := CA) (BA := CoLattice.BotDec TA) (TA := CoLattice.TopDec BA).
  exact E.
Qed.

End Flat.

Lemma flat_map_map_flat : forall A B CB PB TB (a : t A) f,
  @EquivDec.eq _ (Equiv _ _)
    (@map_flat A B CB PB TB f a)
    (flat (Meet _ _) (Top _ _) (map f a)).
Proof.
  introv. apply LiftJoin.flat_map_map_flat with (PB := CoLattice.PosetDec PB).
Qed.

Lemma map_flat_const : forall A C P T e,
  @EquivDec.eq _ (Equiv _ _) (@map_flat A A C P T (const (T := _)) e) e.
Proof. introv. apply LiftJoin.map_flat_const. Qed.

End LiftMeet.


(** * Adding both a Join and a Meet Operations **)

Module LiftJoinMeet.

Section LiftJoinMeet.

Variable T : Type.
Hypothesis CT : Comparable T.
Hypothesis P : PosetDec.t T.

Definition t :=
  nosimpl (LiftJoin.t (LiftMeet.t T)).

Definition const a : t :=
  nosimpl (LiftJoin.const (LiftMeet.const a)).

Instance Equiv : EquivDec.t t.
  apply~ LiftJoin.Equiv.
   apply~ LiftMeet.t_comparable.
   apply~ LiftMeet.Poset.
Defined.

Instance Poset : PosetDec.t t.
  apply~ LiftJoin.Poset.
   apply~ LiftMeet.t_comparable.
   apply~ LiftMeet.Poset.
Defined.

Lemma lift_order : forall a b : T,
  a ⊑♯ b →
  const a ⊑♯ const b.
Proof. introv O. apply LiftJoin.lift_order. apply~ LiftMeet.lift_order. Qed.

Lemma lift_order_rev : forall a b : T,
  const a ⊑♯ const b →
  a ⊑♯ b.
Proof. introv O. apply~ LiftMeet.lift_order_rev. applys LiftJoin.lift_order_rev O. Qed.

Instance Join : JoinDec.t t.
  apply LiftJoin.Join.
Defined.

Hypothesis B : BotDec.t T.

Instance Bot : BotDec.t t.
  apply LiftJoin.Bot. apply~ LiftMeet.Bot.
Defined.

Hypothesis widen0 : T → T → T.

Definition widen : t → t → t := (* This widenning operator can be slow in a really fast way… *)
  fun l1 l2 =>
    let flatten l :=
      concat (LibList.map (proj1_sig (P := _)) (proj1_sig l)) in
    let l :=
      concat (LibList.map (fun a1 =>
        LibList.map (fun a2 =>
          const (widen0 a1 a2))
      (flatten l2)) (flatten l1)) in
  fold_left (fun a1 a2 =>
    a1 ⊔♯ a2) (⊥♯) l.

Instance Lat : LatticeDec.t t.
  applys update_widen widen.
  eapply LiftJoin.Lat.
   apply~ LiftMeet.t_comparable.
   apply~ LiftMeet.Meet.
   apply~ LiftMeet.Bot.
   exact arbitrary. (* Is shadowed by [update_widen] above. *)
Defined.

Hypothesis TT : TopDec.t T.

Instance Top : TopDec.t t.
  apply LiftJoin.Top. apply~ LiftMeet.Top.
Defined.

Lemma bot : const (⊥♯) =♯ (⊥♯).
Proof. apply LiftJoin.bot. Qed.

Lemma top : const (⊤♯) =♯ (⊤♯).
Proof.
  eapply EquivDec.trans; [| apply LiftJoin.top ].
  apply @PosetDec.antisym with (t := Poset); apply LiftJoin.lift_order.
   apply TopDec.prop.
   apply PosetDec.refl. apply EquivDec.sym. apply LiftMeet.top.
Qed.


Variable C : Type.

Section Gamma.

Variable gamma0 : T → C → Prop.
Hypothesis monotone0 : ∀ N1 N2 : T, N1 ⊑♯ N2 → gamma0 N1 ⊑ gamma0 N2.
Hypothesis gamma0_bot : ∀ c, ~ gamma0 (⊥♯) c.
Hypothesis gamma0_top : ∀ c, gamma0 (⊤♯) c.

Definition gamma : t → C → Prop :=
  nosimpl (LiftJoin.gamma (LiftMeet.gamma gamma0)).

Lemma monotone : forall N1 N2 : t,
  N1 ⊑♯ N2 → gamma N1 ⊑ gamma N2.
Proof. apply LiftJoin.monotone. apply~ LiftMeet.monotone. Qed.

Lemma meet_morph : forall N1 N2 : t,
  (gamma N1 ⊓ gamma N2) ⊑ gamma (N1 ⊓♯ N2).
Proof. apply LiftJoin.meet_morph. apply~ LiftMeet.meet_morph. Qed.

Instance Gamma : Gamma.t _ _ gamma.
Proof.
  apply Gamma.Make.
   apply monotone.
   apply meet_morph.
Qed.

Lemma gamma_bot : forall c : C,
  ~ gamma (⊥♯) c.
Proof. introv G. applys LiftJoin.gamma_bot G. apply~ LiftMeet.gamma_bot. Qed.

Lemma gamma_top : forall c,
  gamma (⊤♯) c.
Proof.
  introv. apply~ LiftJoin.gamma_top.
  introv. apply~ LiftMeet.gamma_top.
Qed.

End Gamma.

End LiftJoinMeet.

Section Map.

Variables A B : Type.
Hypothesis CA : Comparable A.
Hypothesis PA : PosetDec.t A.
Hypothesis BA : BotDec.t A.
Hypothesis TA : TopDec.t A.
Hypothesis CB : Comparable B.
Hypothesis PB : PosetDec.t B.
Hypothesis BB : BotDec.t B.
Hypothesis TB : TopDec.t B.
Hypothesis widenB : B → B → B.

Definition map (f : A → B) : t A → t B :=
  nosimpl (LiftJoin.map (LiftMeet.map f)).

Lemma map_monotone : forall f g t1 t2,
  (forall a1 a2, a1 ⊑♯ a2 → f a1 ⊑♯ g a2) →
  (forall a, a =♯ (⊥♯) → g a =♯ (⊥♯)) →
  (forall a, a =♯ (⊤♯) → g a =♯ (⊤♯)) →
  @PosetDec.order _ (Poset _ _) t1 t2 →
  @PosetDec.order _ (Poset _ _) (map f t1) (map g t2).
Proof.
  introv Mfg Bf Tf O. apply LiftJoin.map_monotone with (CA := LiftMeet.t_comparable _) (BA := LiftMeet.Bot _ _) (BB := LiftMeet.Bot _ _); autos~.
   introv O'. apply* LiftMeet.map_monotone.
   simpl. unfolds LiftJoin.Order, LiftJoin.Order_base. introv (E1&E2). repeat rewrite Forall_iff_forall_mem with (CA := _) in *.
    unfolds LiftMeet.map. split~; introv M.
     constructors. simpl. apply BotDec.prop.
     simpl. forwards~ E: E2 (⊥♯ : A).
      rew_refl*.
      repeat rewrite Exists_iff_exists_mem with (CA := _) in *. lets (x'&Mx'&Ox'): (rm E). exists (f x'). splits.
       destruct a as [a D]. simpl. rewrite map_mem. exists* x'.
       simpl in M. rew_refl in M. inverts M; tryfalse. applys~ PosetDec.trans (g x').
Qed.

Definition map_flat (f : A → t B) : t A → t B :=
  nosimpl (LiftJoin.map_flat _ (LiftMeet.Bot _ _)
    (fun a =>
      LiftMeet.flat (@LatticeDec.meet _ (Lat _ _ widenB)) (Top _ _)
        (@LiftMeet.map A (t B) f a))).

Lemma map_flat_monotone : forall f g t1 t2,
  (forall a1 a2, a1 ⊑♯ a2 → @PosetDec.order _ (Poset _ _) (f a1) (g a2)) →
  (forall a,
    a =♯ (⊥♯) →
    @EquivDec.eq _ (Equiv _ _) (g a) (@BotDec.elem _ _ (@LatticeDec.bot _ (Lat _ _ widenB)))) →
  (forall a, a =♯ (⊤♯) → @EquivDec.eq _ (Equiv _ _) (g a) (@TopDec.elem _ _ (Top _ _))) →
  @PosetDec.order _ (Poset _ _) t1 t2 →
  @PosetDec.order _ (Poset _ _) (map_flat f t1) (map_flat g t2).
Proof.
  introv Mfg Bg Tg O. apply LiftJoin.map_flat_monotone with (CA := _) (BA := LiftMeet.Bot _ _); autos~.
   clear O. introv O. apply LiftMeet.flat_monotone with (CA := _). apply~ LiftMeet.map_monotone.
   clear O. introv Ba. apply LiftMeet.flat_bot with (CA := _) (BA := @LatticeDec.bot _ (Lat _ _ widenB)).
    eapply @PosetDec.antisym with (t := LiftMeet.Poset _ _); autos~.
    eapply PosetDec.trans.
     eapply LiftMeet.map_monotone; try (applys @PosetDec.refl; exact Ba); autos~; assumption.
     simpl. rew_refl. do 2 unfolds. rewrite Forall_iff_forall_mem with (CA := _).
      introv M. simpl in M. rew_refl in M. inverts M; tryfalse.
      destruct a as [[|a l] D]; tryfalse. apply Exists_here. eapply PosetDec.trans;
         [| apply PosetDec.refl; applys~ @EquivDec.sym Bg ]; autos~.
Qed.

End Map.

Section Flat.

Variable A : Type.
Hypothesis CA : Comparable A.
Hypothesis PA : PosetDec.t A.
Hypothesis JA : JoinDec.t A.
Hypothesis MA : MeetDec.t A.
Hypothesis TA : TopDec.t A.
Hypothesis BA : BotDec.t A.

Definition flat : t A → A.
  introv e. applys LiftJoin.flat; try typeclass.
  apply LiftJoin.map with (A := LiftMeet.t A).
  applys LiftMeet.flat. exact e.
Defined.

Lemma flat_monotone : forall a1 a2 : t A,
  @PosetDec.order _ (Poset _ _) a1 a2 →
  flat a1 ⊑♯ flat a2.
Proof.
  introv O. apply LiftJoin.flat_monotone with (CA := _). applys~ LiftJoin.map_monotone; try exact O.
   apply~ LiftMeet.flat_monotone.
   apply~ LiftMeet.flat_bot.
Qed.

End Flat.

Lemma flat_map_map_flat : forall A `{Comparable A} B CB PB BB TB widenB (a : t A) f,
  @EquivDec.eq _ (Equiv _ _)
    (@map_flat A B CB PB BB TB widenB f a)
    (flat
      (@LatticeDec.join _ (Lat _ _ widenB))
      (@LatticeDec.meet _ (Lat _ _ widenB))
      (Top _ _)
      (@LatticeDec.bot _ (Lat _ _ widenB))
      (map f a)).
Proof.
  introv CA. introv. unfolds map_flat, flat, map.
  eapply EquivDec.trans; [ apply LiftJoin.flat_map_map_flat |].
  asserts M: (forall A CA PA JA BA (a1 a2 : LiftJoin.t A),
      @EquivDec.eq _ (LiftJoin.Equiv CA _) a1 a2 →
      @LiftJoin.flat A PA JA BA a1 =♯ @LiftJoin.flat A PA JA BA a2).
    introv E. apply PosetDec.antisym; apply~ LiftJoin.flat_monotone.
  apply M with (CA := _) (PA := Poset _ _). clear. destruct a as [a D].
  apply LiftJoin.Order_base_Order_eq. clear D. induction a as [|a al]; autos~.
   asserts L: (forall (a1 a2 : t B) (al1 al2 : list (t B)),
     EquivDec.eq (t := Equiv _ _) a1 a2 →
     EquivDec.eq (t := LiftJoin.Equiv_base _ (Poset _ _)) al1 al2 →
     EquivDec.eq (t := LiftJoin.Equiv_base _ (Poset _ _)) (a1 :: al1) (a2 :: al2)).
    clear. introv E (O1&O2). split~; unfolds LiftJoin.Order, LiftJoin.Order_base;
     repeat rewrite Forall_iff_forall_mem with (CA := _) in *; introv M; simpl in M;
     rew_refl in M; inverts M as M;
     try solve [ apply~ Exists_here ]; try solve [ apply* Exists_next ].
   apply~ L.
Qed.

(* TODO: Monad structure.
Lemma map_flat_const : forall A P B T e,
  @EquivDec.eq _ (Equiv _) (@map_flat A A P B T (const (T := _)) e) e.
Proof.
  introv. eapply EquivDec.trans; [| apply LiftJoin.map_flat_const with (B := LiftMeet.Bot _) ].
  apply @PosetDec.antisym with (t := Poset _); apply LiftJoin.map_flat_monotone
    with (PA := LiftMeet.Poset _) (BA := LiftMeet.Bot _); autos~.
   Focus 4. introv O. eapply EquivDec.trans; [| apply LiftJoin.bot ].
    apply @PosetDec.antisym with (t := Poset _); apply~ LiftJoin.lift_order.
   introv O. destruct a1 as [a1 D1], a2 as [a2 D2].
    apply LiftJoin.Order_base_Order_poset with (D1 :=
      proj2_sig (LiftMeet.flat _ _ (LiftMeet.map (const (T:=A)) (exist (<> nil) a1 D1)))).
    simpls. rew_refl in *. unfolds LiftJoin.Order, LiftJoin.Order_base.
    rewrite Forall_iff_forall_mem in *. introv M. rewrite Exists_iff_exists_mem.
    unfolds LiftMeet.flat, LiftJoin.flat. simpl in O, M.
    simpls. eexists. rew_refl. splits*. clear D1. induction a1.
     simpl in M. rew_refl in M. inverts M; false. destruct a2 as [|x ?]; tryfalse.
      forwards~ Ab: O x; [ rew_refl~ | inverts~ Ab ].
     rewrite map_cons in M. rewrite fold_right_cons in M.
     match goal with M : istrue (mem x (proj1_sig (LiftJoin.meet _ ?c ?f))) |- _ => sets_eq F: f end.
     destruct F as [F DF]. simpl in M. apply concat_mem in M. lets (L&ML&Mx): (rm M).
     apply map_mem in ML. lets (y&M&E): (rm ML). substs. apply map_mem in Mx.
     simpl in M. rew_refl in M. inverts M; tryfalse. lets (y&M&E): (rm Mx). substs.

    unfolds LiftMeet.flat, LiftMeet.map, const, LiftMeet.const. rew_list.
     simpl in O. rew_refl in O. unfolds in O. rewrite Forall_iff_forall_mem in O. simpl in O.
     simpl. rew_refl. unfolds. rewrite Forall_iff_forall_mem.

    induction a1; tryfalse.
    apply @PosetDec.trans with (t := Poset _) (y := @TopDec.elem _ _ (Top _)).
     apply TopDec.prop.
     apply PosetDec.refl. apply EquivDec.sym. eapply EquivDec.trans; [| apply LiftJoin.top ].
      apply PosetDec.antisym; apply LiftJoin.lift_order.
       apply TopDec.prop.
       exact O.
    unfolds LiftMeet.flat, LiftMeet.map, const, LiftMeet.const. rew_list.
     simpl in O. rew_refl in O. unfolds in O. rewrite Forall_iff_forall_mem in O. simpl in O.
     simpl. rew_refl. unfolds. rewrite Forall_iff_forall_mem.
Qed.


(** ** Monad Structure **)

Definition ret : forall T, T → t T := const.
Definition bind : forall T1 T2 (P1 : PosetDec.t T1) (P2 : PosetDec.t T2),
    BotDec.t T1 → TopDec.t T1 → BotDec.t T2 → TopDec.t T2 →
    t T1 → (T1 → t T2) → t T2 :=
  fun T1 T2 P1 P2 BT1 TT1 BT2 TT2 e f =>
    ifb EquivDec.eq (t := Equiv BT1 TT1) e (TopDec.elem (t := Top BT1 TT1)) then f (⊤♯)
    else map_flat _ _ f e.


(** *** Monads Axioms **)

Lemma ret_bind : forall T1 T2 (P1 : PosetDec.t T1) (P2 : PosetDec.t T2)
    (BT1 : BotDec.t T1) (TT1 : TopDec.t T1) (BT2 : BotDec.t T2) (TT2 : TopDec.t T2)
    (a : T1) (f : T1 → t T2),
  (forall b1 b2, b1 ⊑♯ b2 → PosetDec.order (t := Poset _ _) (f b1) (f b2)) →
  EquivDec.eq (t := Equiv BT2 TT2) (bind _ _ _ _ (ret a) f) (f a).
Proof.
  introv Mon. unfolds bind. cases_if as I.
   rename a0 into I. (* Grmbl, tactic *)
    asserts At: ((⊤♯) =♯ a).
     apply PosetDec.antisym.
      lets (_&O): (rm I). unfolds in O. rewrite Forall_iff_forall_mem in O. forwards [B|E]: (rm O).
       rew_refl*.
       lets (_&O): (rm B). unfolds in O. rewrite Forall_iff_forall_mem in O. forwards [B|E]: (rm O).
        rew_refl*.
        apply PosetDec.trans with (⊥♯); autos*.
        inverts E.
       rewrite Exists_iff_exists_mem in E. lets (a'&Ma&Oa): (rm E). simpl in Ma. rew_refl in *.
        inverts Ma; tryfalse. unfolds in Oa. rewrite Forall_iff_forall_mem in Oa. forwards [B|E]: (rm Oa).
         rew_refl*.
         apply PosetDec.refl. applys @EquivDec.sym B.
         inverts E.
      apply TopDec.prop.
     apply @PosetDec.antisym with (t := Poset _ _); apply~ Mon.
   unfolds map_flat, LiftJoin.map_flat, ret, const, LiftMeet.const,
     LiftJoin.const, LiftMeet.flat.
    rew_list. eapply EquivDec.trans.
     eapply join_bot with (PV := Poset _ _); autos~.
     eapply EquivDec.trans.
      eapply meet_com with (H := Poset _ _).
      eapply meet_top with (P := Poset _ _).
Qed.

Lemma bind_ret : forall T (P : PosetDec.t T) (BT : BotDec.t T) (TT : TopDec.t T) (e : t T),
  EquivDec.eq (t := Equiv _ _) (bind _ _ _ _ e (ret (T := _))) e.
Proof.
  introv. unfolds bind. cases_if as I.
   applys @EquivDec.trans top. autos~.
   unfolds map_flat, LiftJoin.map_flat, ret, const, LiftMeet.const, LiftJoin.const, LiftMeet.flat, LiftMeet.map.
    apply @PosetDec.antisym with (t := Poset _ _).
     simpl. rew_refl. unfolds. rewrite Forall_iff_forall_mem. introv M.
      induction e; rew_list in *; tryfalse.

     eapply EquivDec.trans.
     eapply join_bot with (PV := Poset _ _); autos~.
     eapply EquivDec.trans.
      eapply meet_com with (H := Poset _ _).
      eapply meet_top with (P := Poset _ _).
Qed.

Lemma bind_bind : forall T1 T2 T3 (P2 : PosetDec.t T2) (P3 : PosetDec.t T3) (e : t T1) (f : T1 → t T2) (g : T2 → t T3) mf mg mb,
  max _ f mf →
  max _ g mg →
  max _ (fun x => bind (f x) g mg) mb →
  bind (bind e f mf) g mg =♯ bind e (λ x : T1, bind (f x) g mg) mb.

(** *** Compatibility with the order **)

Lemma ret_monotone : forall T (P : PosetDec.t T) (a1 a2 : T),
  a1 ⊑♯ a2 →
  ret a1 ⊑♯ ret a2.
Lemma monotone_bind : forall T1 T2 (CL1 : PosetDec.t T1) (CL2 : PosetDec.t T2)
    (e1 e2 : t T1) (f : T1 → t T2) mf,
  max _ f mf →
  (forall b1 b2, b1 ⊑♯ b2 → f b1 ⊑♯ f b2) →
  e1 ⊑♯ e2 →
  bind e1 f mf ⊑♯ bind e2 f mf.
Lemma bind_monotone : forall T1 T2 (CL2 : PosetDec.t T2)
    (e : t T1) (f g : T1 → t T2) mf mg,
  max _ f mf →
  max _ g mg →
  (forall b, f b ⊑♯ g b) →
  bind e f mf ⊑♯ bind e g mg.

(** *** Compatibility with the operations **)

Lemma meet_bind : forall T1 T2 (CL1 : PosetDec.t T1) (CL2 : PosetDec.t T2)
    (e1 e2 : t T1) (f : T1 → t T2) mf,
  max _ f mf →
  (forall b1 b2, b1 ⊑♯ b2 → f b1 ⊑♯ f b2) →
  (bind e1 f mf ⊓♯ bind e2 f mf) ⊑♯ bind (e1 ⊓♯ e2) f mf.
Lemma bind_meet : forall T1 T2 (CL2 : PosetDec.t T2)
    (e : t T1) (f g : T1 → t T2) mf mg mm,
  max _ f mf →
  max _ g mg →
  max _ (fun a => f a ⊓♯ g a) mm →
  (bind e f mf ⊓♯ bind e g mg) ⊑♯ bind e (fun a => f a ⊓♯ g a) mm.


(** *** Evaluate **)

Variable evaluate : forall T (P : PosetDec.t T),
  JoinDec.t T → MeetDec.t T → t T → T.

Lemma evaluate_ret : forall T (a : T) (P : PosetDec.t T) Jt Mt,
  evaluate Jt Mt (ret a) =♯ a.
Lemma evaluate_monotone : forall T (P : PosetDec.t T) Jt Mt (A B : t T),
  A ⊑♯ B →
  evaluate Jt Mt A ⊑♯ evaluate Jt Mt B.
Lemma evaluate_meet : forall T (P : PosetDec.t T) Jt Mt (A B : t T),
  evaluate Jt Mt A ⊓♯ evaluate Jt Mt B ⊑♯ evaluate Jt Mt (A ⊓♯ B).
Lemma evaluate_monotone_inv : forall A B : t Prop,
  (evaluate A → evaluate B) →
  A ⊑♯ B.

(** *** Miscellaneous Properties **)

Lemma ret_meet_prop : forall (P1 P2 : Prop),
  ret (P1 ∧ P2) =♯ (ret P1 ⊓♯ ret P2).

Lemma bind_generic : forall (T1 T2 : Type) (e1 : t T1) (e2 : t T2) (P : PosetDec.t T2),
  bind e1 (fun _ => e2) ⊑♯ e2.

*)

End LiftJoinMeet.


(* TODO: Being able to add some edges to a poset. *)

