
(** Lattices abstracting maps. **)

Require Import SharedLat.
Require Import Containers.Maps.
Require Export TLC.LibTactics TLC.LibReflect.


Section Lemmae.

Variables K V : Type.
Hypothesis OK : OrderedType K.

Lemma empty_find : forall (m : Map [K, V]) k,
  Empty m → m [k] = None.
Proof.
  introv E.
  apply MapFacts.not_find_in_iff.
  intro I. apply MF.elements_in_iff in I. lets (e&He): (rm I).
  apply MapFacts.elements_Empty in E. rewrite E in He. inverts He.
Qed.

Lemma fold_empty : forall A (f : K → V → A → A) i,
  fold f [] i = i.
Proof. introv. apply~ MapFacts.fold_Empty. apply empty_1. Qed.

End Lemmae.



Module MapDefault.

Section MapDefault.

Variables K V : Type.

Hypothesis OK : OrderedType K.
Hypothesis LV : LatticeDec.t V.
Hypothesis TV : TopDec.t V.

Definition t := (V * (Map [K, {v : V | ¬v =♯ (⊥♯)}]))%type.

Hypothesis CK : Comparable K.
Hypothesis CV : Comparable V.

Global Instance t_comparable : Comparable t.
  apply~ LibProd.prod_comparable. unfold MapAVLInstance.MapAVL_FMap.
  asserts R: (forall a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 a15 a16 a17 a18 a19 a20,
    @dict _ _ (Build_FMap a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 a15 a16 a17 a18 a19 a20) = a3).
   reflexivity.
  rewrite R. clear R. constructors. intros (t1&OK1) (t2&OK2).
  applys Decidable_equiv (t1 = t2).
   iff I.
    substs. fequals.
    inverts~ I.
   clear OK1 OK2. gen t2. induction t1; intro t2; induction t2.
    applys* Decidable_equiv True. typeclass.
    applys* Decidable_equiv False.
     iff I; inverts I.
     typeclass.
    applys* Decidable_equiv False.
     iff I; inverts I.
     typeclass.
    applys Decidable_equiv (t1_1 = t2_1 ∧ k = k0 ∧ e = e0 ∧ t1_2 = t2_2 ∧ z = z0).
     iff I.
      repeat inverts I as I. autos*.
      inverts~ I.
     repeat apply and_decidable; try typeclass.
      destruct e as [v ?], e0 as [v' ?]. applys Decidable_equiv (v = v').
       iff I; inverts~ I. fequals.
       typeclass.
Defined.

Definition read (h : t) k :=
  let (d, m) := h in
  match m [k] with
  | Some v => proj1_sig v
  | None => d
  end.

Definition write (h : t) k v : t :=
  let (d, m) := h in
  match EquivDec.dec v (⊥♯) with
  | right p => (d, m [k <- exist _ v p])
  | left _ => (⊥♯, []) (* Propagating bottom elements *)
  end.

Lemma read_write_eq : forall (h : t) k k' v,
  k === k' →
  read (write h k v) k' =♯ v.
Proof.
  introv E. unfolds read. unfolds write. destruct h as [d m].
  destruct (EquivDec.dec v (⊥♯)).
   rewrite~ empty_find. apply empty_1.
   rewrite MF.add_eq_o; simpls~.
Qed.

Lemma read_write_neq : forall (h : t) k1 k2 v,
  ~ (k1 === k2) →
  ~ v =♯ (⊥♯) →
  read (write h k1 v) k2 =♯ read h k2.
Proof.
  introv D NB. unfolds read. unfolds write. destruct h as [d m].
  destruct (EquivDec.dec v (⊥♯)).
   false.
   rewrite MF.add_neq_o; simpls~.
Qed.

Definition mapDefEq f g : Prop :=
  forall k, read f k =♯ read g k.

Definition mapDefLe f g : Prop :=
  forall k, read f k ⊑♯ read g k.

Definition mapDefLeb (f g : t) : bool :=
  let (df, mf) := f in let (dg, mg) := g in
  let check k _ b :=
    b && decide (read f k ⊑♯ read g k) in
  decide (df ⊑♯ dg) &&
  fold check mf true &&
  fold check mg true.

Lemma mapDefLeb_correct : forall f g,
  istrue (mapDefLeb f g) ↔ mapDefLe f g.
Proof.
  intros (df&mf) (dg&mg). iff L.
   unfolds. unfolds in L. skip. (* TODO: Doable *)
   unfolds. rewrite <- Shared.and_andb.
    asserts Od: (df ⊑♯ dg).
      skip. (* ≃ K is infinite. *)
    rew_refl. repeat splits~.
     asserts~ Hmf: (forall k v, mf [k] = Some v → mf [k] = Some v).
      gen L Hmf. generalize mf at - 3. introv.
      apply~ MapFacts.fold_rec. intros. rew_refl. splits.
       skip.
       unfolds read. lets Hmfk: Hmf k.
      clear H2.
Admitted. (* TODO *)

Definition join : t → t → t :=
  fun f g =>
  let (df, mf) := f in let (dg, mg) := g in
  let update f k y h := write h k (read f k ⊔♯ proj1_sig y) in
  fold (update f) mg
    (fold (update g) mf (df ⊔♯ dg, [])).

Definition meet : t → t → t :=
  fun f g =>
  let (df, mf) := f in let (dg, mg) := g in
  let update f k y h := write h k (read f k ⊓♯ proj1_sig y) in
  fold (update f) mg
    (fold (update g) mf (df ⊓♯ dg, [])).

Global Instance Equiv : EquivDec.t t.
  apply EquivDec.Make with mapDefEq.
  firstorder.
  firstorder.
  introv E1 E2. intro k. apply (EquivDec.trans _ _ _ (E1 k) (E2 k)).
  intros f g. set_eq <- b: (mapDefLeb f g && mapDefLeb g f).
   destruct b; fold_bool; rew_refl in EQb.
    lets (Le1&Le2): (rm EQb). apply mapDefLeb_correct in Le1. apply mapDefLeb_correct in Le2.
     left. intro k. apply PosetDec.antisym; [ apply Le1 | apply Le2 ].
    right. rew_logic in *. destruct EQb as [Le|Le]; intro M; apply Le;
      apply mapDefLeb_correct; intro k.
     apply PosetDec.refl. apply M.
     forwards M': (rm M) k. apply EquivDec.sym in M'.
      apply PosetDec.refl. apply M'.
Defined.

Global Instance Poset : PosetDec.t t.
  apply PosetDec.Make with Equiv mapDefLe.
  firstorder.
  firstorder.
  introv E1 E2. intro k. apply (PosetDec.trans _ _ _ (E1 k) (E2 k)).
  intros f g. set_eq <- b: (mapDefLeb f g). destruct b; fold_bool; rew_refl in EQb.
   left. apply~ mapDefLeb_correct.
   right. intro Le. apply EQb. apply~ mapDefLeb_correct.
Defined.

Global Instance Join : JoinDec.t t.
  apply JoinDec.Make with join; unfold join.
  (* bound1 *)
  skip. (* intros (df&mf) (dg&mg) k.
   sets_eq m: (fold (λ (k : K) (y : V), add k (dg ⊔♯ y)) mf []).
   asserts I: ((df, mf) ⊑♯ (df, m)).
     substs. clear. intro k. simpl.
     forwards E: MapFacts.fold_identity mf. sets_eq mf': (fold add mf []).
     forwards~ R: MF.fold_Equal_ord MapFacts.Equal_ST (λ k y, add k (dg ⊔♯ y)) E.
       clear. introv Ek Ev Eh. intro k. substs. apply~ MapFacts.add_m.
     rewrite <- R. rewrite <- E. rewrite EQmf'. pattern (fold add mf []).
     clear. apply MF.fold_rec. (* The invariant chosen by Coq is terribly wrong. *)
      introv E. rewrite~ fold_empty.
      introv M NI A I.
       forwards~ R: MapFacts.fold_commutes MapFacts.Equal_ST (λ k1 y, add k1 (dg ⊔♯ y)).
         clear. introv Ek Ev Eh. intro k. substs. apply~ MapFacts.add_m.
         clear. introv D. intro k''.
          tests C: (k'' === k); tests C': (k'' === k');
            repeat ((rewrite MapFacts.add_eq_o; [| solve [ autos* ] ])
                   || (rewrite MapFacts.add_neq_o; [| try unfolds; autos* ])); try reflexivity.
           false D. eapply equiv_transitive with k''; autos*.
         skip.
       rewrite R.
   clear EQm. gen I. apply MF.fold_rec. *)
  (* bound2 *)
  skip.
  (* least upper bound *)
  apply monotone_refl_least_upper_bound.
   skip.
   skip.
   skip.
Defined.

Global Instance Meet : MeetDec.t t.
  apply MeetDec.Make with meet; unfold meet.
  (* bound1 *)
  skip.
  (* bound2 *)
  skip.
  (* greatest upper bound *)
  apply monotone_refl_greatest_lower_bound.
   skip.
   skip.
   skip.
Defined.

Definition widen : t → t → t :=
  fun f g =>
  let (df, mf) := f in let (dg, mg) := g in
  let update f k y h := write h k (read f k ∇♯ proj1_sig y) in
  fold (update f) mg
    (fold (update g) mf (df ∇♯ dg, [])).


Global Instance Lattice : LatticeDec.t t.
  eapply LatticeDec.Make.
   apply Join.
   apply Meet.
   apply BotDec.Make with (⊥♯, []).
    intros (d&m) k. simpl. rewrite~ empty_find. apply empty_1.
   exact widen.
Defined.

Global Instance Top : TopDec.t t.
   apply TopDec.Make with (⊤♯, []).
   intros h k. simpl. rewrite MapFacts.empty_o. apply TopDec.prop.
Defined.

Lemma read_top : forall x, read (⊤♯) x = ⊤♯.
Proof. introv. simpls. rewrite MapFacts.empty_o. autos~. Qed.

Lemma write_monotone : forall h1 h2 k v1 v2,
  h1 ⊑♯ h2 → v1 ⊑♯ v2 →
  write h1 k v1 ⊑♯ write h2 k v2.
Proof.
  intros (d1&m1) (d2&m2). introv O1 O2. unfolds write.
  destruct (EquivDec.dec v1 (⊥♯)) eqn: D1; destruct (EquivDec.dec v2 (⊥♯)) eqn:D2.
   apply PosetDec.refl; autos*.
   change (⊥♯, []) with (⊥♯ : t). apply BotDec.prop.
   false n. apply~ @PosetDec.antisym. applys @PosetDec.trans O2. apply PosetDec.refl; autos*.
   intro k'. tests C: (k === k').
    do 2 eapply PosetDec.trans; try apply O2; apply PosetDec.refl; try apply EquivDec.refl.
     forwards R: read_write_eq (d1, m1) v1 C. unfolds write. rewrite D1 in R. exact R.
     forwards R: read_write_eq (d2, m2) v2 C. unfolds write. rewrite D2 in R.
      applys @EquivDec.sym R.
    do 2 eapply PosetDec.trans; try apply O1; apply PosetDec.refl; try apply EquivDec.refl.
     forwards R: read_write_neq (d1, m1) C n. unfolds write. rewrite D1 in R. exact R.
     forwards R: read_write_neq (d2, m2) C n0. unfolds write. rewrite D2 in R.
      applys @EquivDec.sym R.
Qed.

Lemma read_monotone : forall h1 h2 k,
  h1 ⊑♯ h2 →
  read h1 k ⊑♯ read h2 k.
Proof. intros (d1&m1) (d2&m2). introv O. apply O. Qed.


Require Export LibHeap.

Variable VC : Type.
Variable gamma0 : V → VC → Prop.
Hypothesis G : Gamma.t (℘ VC) V gamma0.
Hypothesis IVC : Inhab VC.

Definition gamma (h : t) (H : heap K VC) : Prop :=
  forall k : K,
    indom H k →
    gamma0 (MapDefault.read h k) (read H k).

Global Instance Gamma : Gamma.t (℘ _) t gamma.
Proof.
  apply Gamma.Make; unfolds~ gamma.
   intros h1 h2 I H g k ID. applys~ Gamma.monotone I.
   intros h1 h2 H (M1&M2) k ID.
    forwards GC: Gamma.meet_morph (conj (M1 k ID) (M2 k ID)).
    skip. (* This proof is complex as it deals with the propagation of ⊥♯:
            M1 and M2 enforce the existence of a concrete heap H that
            respects the condition of both abstract heaps h1 and h2, and
            thus those two abstract heap were not propagated to ⊥♯ while
            merging them. *) (* TODO *)
Qed.

Lemma gamma_write : forall h H k v V,
  ~ v =♯ (⊥♯) →
  gamma0 v V →
  gamma h H →
  gamma (MapDefault.write h k v) (write H k V).
Proof.
  introv NB Gv Gh. intros k' ID. tests C: (k' === k).
   eapply gamma_eq; [autos*| |].
    apply EquivDec.sym. apply* MapDefault.read_write_eq.
    skip. (* Actually, [heap]s lack this quotient ability to pass through. *)
   eapply gamma_eq; [autos*| |].
    apply EquivDec.sym. apply MapDefault.read_write_neq; autos*.
    asserts: (k' ≠ k).
      intro_subst. apply C. apply Equivalence_Reflexive.
     forwards~: @indom_write_inv ID. erewrite read_write_neq; autos*.
Qed.

Definition gammaf (h : t) (H : K → VC) : Prop :=
  forall k : K, gamma0 (MapDefault.read h k) (H k).

Global Instance Gammaf : Gamma.t (℘ _) t gammaf.
Proof.
  apply Gamma.Make; unfolds~ gammaf.
  intros h1 h2 I H g k. applys~ @Gamma.monotone G.
  intros h1 h2 H (M1&M2) k.
   forwards: @Gamma.meet_morph G (conj (M1 k) (M2 k)).
    skip. (* This is the same… *)
Qed.

Lemma gammaf_write : forall h H H' k v,
  ~ v =♯ (⊥♯) →
  gammaf h H →
  (forall k', ~ k === k' → H k' = H' k') →
  (forall k', k === k' → gamma0 v (H' k')) →
  gammaf (MapDefault.write h k v) H'.
Proof.
  introv NB Gh HD Gv. intros k'. tests C: (k' === k).
   eapply gamma_eq; [autos*| |].
    apply EquivDec.sym. apply* MapDefault.read_write_eq.
    apply Gv. apply~ @Equivalence_Symmetric.
   eapply gamma_eq; [autos*| |].
    apply EquivDec.sym. apply MapDefault.read_write_neq; autos*.
    rewrite~ <- HD.
Qed.

End MapDefault.

End MapDefault.


