
(** Some shared properties about lattices. **)

Set Implicit Arguments.
Require Export lattice_def.
Require Export lat_prop.
Require Export CompleteLattice.
Require Import Shared.


Definition max A B (P : Poset.t B) (f : A → B) y :=
  (forall x, f x ⊑ y) ∧
  forall y', (forall x, f x ⊑ y') → y ⊑ y'.

Definition max_dec A B (P : PosetDec.t B) (f : A → B) y :=
  (forall x, f x ⊑♯ y) ∧
  forall y', (forall x, f x ⊑♯ y') → y ⊑♯ y'.

Lemma max_const : forall A B (P : Poset.t B) b,
  A → max P (fun _ : A => b) b.
Proof. introv a. split~. Qed.

Lemma max_dec_const : forall A B (P : PosetDec.t B) b,
  A → max_dec P (fun _ : A => b) b.
Proof. introv a. split~. Qed.

Lemma max_empty : forall A B (PB : Poset.t B) (BB : @Bot.t B _) (f : A → B),
  (A → False) → max _ f (⊥).
Proof. introv na. split~. intro a. false~. Qed.

Lemma max_dec_empty : forall A B (PB : PosetDec.t B) (BB : BotDec.t B) (f : A → B),
  (A → False) → max_dec _ f (⊥♯).
Proof. introv na. split~. intro a. false~. Qed.


Global Instance Bot_Inhab : forall A (P : PosetDec.t A) (B : BotDec.t A),
    Inhab A.
  introv B. apply Build_Inhab. exists~ (⊥♯).
Defined.

Global Instance Decidable_EquivDec : forall A (E : EquivDec.t A) x y,
    Decidable (x =♯ y).
  introv. apply sumbool_decidable. apply EquivDec.dec.
Defined.

Global Instance Decidable_PosetDec : forall A (P : PosetDec.t A) x y,
    Decidable (x ⊑♯ y).
  introv. apply sumbool_decidable. apply PosetDec.dec.
Defined.

Lemma Gamma_eq : forall (a A : Type) (La : Lattice.t a) (LA : LatticeDec.t A) (op : A → a) N1 N2,
  Gamma.t _ _ op →
  N1 =♯ N2 → Equiv.eq (op N1) (op N2).
Proof. introv G E. apply Poset.antisym; apply~ Gamma.monotone. Qed.

Lemma Gamma_equiv : forall (a A : Type) (LA : LatticeDec.t A) (op1 op2 : A → a → Prop),
  Gamma.t _ _ op1 →
  (forall X x, op1 X x ↔ op2 X x) →
  Gamma.t _ _ op2.
Proof.
  introv G E. asserts: (op1 = op2).
    extens~.
  substs~.
Qed.


Definition update_widen : forall T, LatticeDec.t T → (T → T → T) → LatticeDec.t T :=
  fun T L widen =>
    LatticeDec.Make _ (LatticeDec.porder (t := L)) (LatticeDec.join (t := L))
      (LatticeDec.meet (t := L)) (LatticeDec.bot (t := L)) widen.


Section Lemmae.

Variables K V : Type.

Section EquivDec_eq.

Lemma eq_EquivDec : forall (E : EquivDec.t V) (x y : V),
  x = y → x =♯ y.
Proof. introv ?. substs~. Qed.

Hypothesis dec : ∀ x y : V, {x = y} + {x ≠ y}.

Definition EquivDec_eq : EquivDec.t V.
  apply EquivDec.Make with (eq := fun x y => x = y); intros; substs~.
Defined.

Lemma join_trans_1 : forall A (P : PosetDec.t A) (t : JoinDec.t A) (x y z : A),
  z ⊑♯ x → z ⊑♯ (x ⊔♯ y).
Proof. introv S. applys PosetDec.trans S. apply JoinDec.bound1. Qed.

Lemma join_trans_2 : forall A (P : PosetDec.t A) (t : JoinDec.t A) (x y z : A),
  z ⊑♯ y → z ⊑♯ (x ⊔♯ y).
Proof. introv S. applys PosetDec.trans S. apply JoinDec.bound2. Qed.

Lemma meet_same : forall A (P : PosetDec.t A) (t : MeetDec.t A) (x : A),
  x =♯ (x ⊓♯ x).
Proof. introv. apply PosetDec.antisym; autos*. Qed.

Lemma meet_com : forall A (P : PosetDec.t A) (t : MeetDec.t A) (x y : A),
  (x ⊓♯ y) =♯ (y ⊓♯ x).
Proof. introv. apply* PosetDec.antisym. Qed.

Lemma meet_same_NDec : forall A (P : Poset.t A) (t : Meet.t P) (x : A),
  Equiv.eq x (x ⊓ x).
Proof. introv. apply Poset.antisym; autos*. Qed.

End EquivDec_eq.

Lemma Comparable_EquivDec : Comparable V → EquivDec.t V.
  introv C. apply EquivDec_eq. introv. apply decidable_sumbool. apply C.
Defined.

Lemma Comparable_EquivDec_eq : forall H (v1 v2 : V),
  @EquivDec.eq V (Comparable_EquivDec H) v1 v2 → v1 = v2.
Proof. simpls~. Qed.

Section Indep.

Hypothesis PV : PosetDec.t V.
Hypothesis JV : JoinDec.t V.
Hypothesis MV : MeetDec.t V.
Hypothesis BV : BotDec.t V.
Hypothesis TV : TopDec.t V.

Lemma inf_bot : forall x y : V,
  x =♯ (⊥♯) →
  y ⊑♯ x →
  y =♯ (⊥♯).
Proof.
  introv E O. apply PosetDec.antisym.
   refine (PosetDec.trans _ _ _ O _). apply PosetDec.refl; autos*.
   apply BotDec.prop.
Qed.

Lemma sup_top : forall x y : V,
  x =♯ (⊤♯) →
  x ⊑♯ y →
  y =♯ (⊤♯).
Proof.
  introv E O. apply PosetDec.antisym.
   apply TopDec.prop.
   refine (PosetDec.trans _ _ _ _ O). apply PosetDec.refl; autos*.
Qed.

Lemma join_sup : forall x y : V,
  x ⊑♯ y →
  x ⊔♯ y =♯ y.
Proof.
  introv Ex. apply PosetDec.antisym.
   apply JoinDec.least_upper_bound; autos*.
   autos*.
Qed.

Lemma join_bot : forall x y : V,
  x =♯ (⊥♯) →
  x ⊔♯ y =♯ y.
Proof. introv Ex. apply join_sup. apply @PosetDec.trans with (y := (⊥♯)); autos~. Qed.

Lemma join_top : forall x y : V,
  y =♯ (⊤♯) →
  x ⊔♯ y =♯ (⊤♯).
Proof.
  introv Ey. apply @EquivDec.trans with (y := y); autos~.
  apply~ join_sup. apply @PosetDec.trans with (y := (⊤♯)); autos~.
  apply TopDec.prop.
Qed.

Lemma join_not_bot : forall x y : V,
  ¬x =♯ (⊥♯) →
  ¬y =♯ (⊥♯) →
  ¬x ⊔♯ y =♯ (⊥♯).
Proof.
  introv Nx Ny M. asserts F: (x ⊑♯ ⊥♯).
    eapply PosetDec.trans. exact (JoinDec.bound1 x y).
    autos*.
  autos*.
Qed.

End Indep.

Hypothesis LV : LatticeDec.t V.

Variable C : Type.
Variable gamma : V → C → Prop.
Hypothesis G : Gamma.t (℘ C) V gamma.

Lemma gamma_eq : forall x y c,
  x =♯ y →
  gamma x c →
  gamma y c.
Proof. introv E Hg. applys* @Gamma.monotone G Hg. Qed.


Variable A : Type.
Hypothesis P : PosetDec.t A.

Lemma monotone_refl_least_upper_bound :
  forall op : A -> A -> A,
    (∀ x y z, y ⊑♯ z → op y x ⊑♯ op z x) ->
    (∀ x y z, y ⊑♯ z → op x y ⊑♯ op x z) ->
    (∀ x, op x x ⊑♯ x) ->
  ∀ x y z : A, x ⊑♯ z → y ⊑♯ z → op x y ⊑♯ z.
Proof. introv M1 M2 R I1 I2. applys* PosetDec.trans (op z z). Qed.

Lemma monotone_refl_greatest_lower_bound :
  forall op : A -> A -> A,
    (∀ x y z, y ⊑♯ z → op y x ⊑♯ op z x) ->
    (∀ x y z, y ⊑♯ z → op x y ⊑♯ op x z) ->
    (∀ x, x ⊑♯ op x x) ->
  ∀ x y z : A, z ⊑♯ x → z ⊑♯ y → z ⊑♯ op x y.
Proof. introv M1 M2 R I1 I2. applys* PosetDec.trans (op z z). Qed.

End Lemmae.


(** The equality is also an order relation.  It is usually not the one
  we want, but it is very useful to turn a lemma about some
  conservation of the order by some function into a lemma about the
  same conservation of equality. **)
Definition EquivDec_PosetDec : forall T, EquivDec.t T → PosetDec.t T.
  introv E. apply PosetDec.Make with (eq := E) (order := fun a b => a =♯ b); autos*.
  apply EquivDec.dec.
Defined.


Module Monotone.

Class t (C A : Type) (CP : Poset.t C) (AP : PosetDec.t A) (op : A → C) : Prop :=
  Make {
    monotone : ∀ N1 N2 : A, N1 ⊑♯ N2 → op N1 ⊑ op N2
  }.

End Monotone.

Global Instance GammaMonotone : forall C A LC LA op,
  @Gamma.t C A LC LA op → Monotone.t _ _ op.
Proof.
  introv G. constructors. apply Gamma.monotone.
Defined.


(** Going from a structure to a decidable structure and vice versa. **)

Global Instance EquivDec_Equiv : forall T,
    EquivDec.t T →
    Equiv.t T.
  introv E. applys Equiv.Make (fun t1 t2 => t1 =♯ t2); apply E.
Defined.

Global Instance PosetDec_Poset : forall T,
    PosetDec.t T →
    Poset.t T.
  introv P. eapply Poset.Make with (order := fun t1 t2 => t1 ⊑♯ t2); apply P.
Defined.

Global Instance JoinDec_Join : forall T (P : PosetDec.t T),
    JoinDec.t T →
    @Join.t T _.
  introv J. eapply Join.Make with (op := fun t1 t2 => t1 ⊔♯ t2); apply J.
Defined.

Global Instance MeetDec_Meet : forall T (P : PosetDec.t T),
    MeetDec.t T →
    @Meet.t T _.
  introv M. eapply Meet.Make with (op := fun t1 t2 => t1 ⊓♯ t2); apply M.
Defined.

Global Instance TopDec_Top : forall T (P : PosetDec.t T),
    TopDec.t T →
    @Top.t T _.
  introv TT. eapply Top.Make with (⊤♯). apply TT.
Defined.

Global Instance BotDec_Bot : forall T (P : PosetDec.t T),
    BotDec.t T →
    @Bot.t T _.
  introv B. eapply Bot.Make with (⊥♯). apply B.
Defined.

Global Instance LatticeDec_Lattice : forall T (L : LatticeDec.t T),
    TopDec.t T →
    Lattice.t T.
  introv TT. eapply Lattice.Make with (porder := _ LatticeDec.porder).
   apply JoinDec_Join. apply L.
   apply MeetDec_Meet. apply L.
   apply BotDec_Bot. apply L.
   apply~ TopDec_Top.
Defined.

Global Instance Equiv_EquivDec : forall T (E : Equiv.t T),
    (forall t1 t2, Decidable (Equiv.eq t1 t2)) →
    EquivDec.t T.
  introv D. apply EquivDec.Make with Equiv.eq; try apply E.
  introv. apply decidable_sumbool. typeclass.
Defined.

Global Instance Poset_PosetDec : forall T (P : Poset.t T),
    (forall t1 t2 : T, Decidable (Equiv.eq t1 t2)) →
    (forall t1 t2 : T, Decidable (Poset.order t1 t2)) →
    PosetDec.t T.
  introv DE DP. apply PosetDec.Make with (eq := _ DE) (order := Poset.order); try apply P.
  introv. apply decidable_sumbool. typeclass.
Defined.

Global Instance Join_JoinDec : forall T (P : Poset.t T)
    (DE : forall t1 t2 : T, Decidable (Equiv.eq t1 t2))
    (DP : forall t1 t2 : T, Decidable (Poset.order t1 t2)),
    Join.t P →
    JoinDec.t T.
  introv J. apply JoinDec.Make with (op := fun t1 t2 => t1 ⊔ t2); apply J.
Defined.

Global Instance Meet_MeetDec : forall T (P : Poset.t T)
    (DE : forall t1 t2 : T, Decidable (Equiv.eq t1 t2))
    (DP : forall t1 t2 : T, Decidable (Poset.order t1 t2)),
    Meet.t P →
    MeetDec.t T.
  introv M. apply MeetDec.Make with (op := fun t1 t2 => t1 ⊓ t2); apply M.
Defined.

Global Instance Top_TopDec : forall T (P : Poset.t T)
    (DE : forall t1 t2 : T, Decidable (Equiv.eq t1 t2))
    (DP : forall t1 t2 : T, Decidable (Poset.order t1 t2)),
    @Top.t T _ →
    TopDec.t T.
  introv TT. apply TopDec.Make with (elem := Top.elem). apply TT.
Defined.

Global Instance Bot_BotDec : forall T (P : Poset.t T)
    (DE : forall t1 t2 : T, Decidable (Equiv.eq t1 t2))
    (DP : forall t1 t2 : T, Decidable (Poset.order t1 t2)),
    @Bot.t T _ →
    BotDec.t T.
  introv B. apply BotDec.Make with (elem := Bot.elem). apply B.
Defined.

Global Instance Lattice_LatticeDec : forall T (L : Lattice.t T)
    (DE : forall t1 t2 : T, Decidable (Equiv.eq t1 t2))
    (DP : forall t1 t2 : T, Decidable (Poset.order t1 t2)),
    LatticeDec.t T.
  introv DE DP. apply LatticeDec.Make with (porder := _ Lattice.porder).
   apply Join_JoinDec. apply L.
   apply Meet_MeetDec. apply L.
   apply Bot_BotDec. apply L.
   exact (fun t1 t2 => t1 ⊔ t2).
Defined.

Lemma Poset_PosetDec_propagates : forall T (P : Poset.t T) (t1 t2 : T)
    (DE : forall t1 t2 : T, Decidable (Equiv.eq t1 t2))
    (DP : forall t1 t2 : T, Decidable (Poset.order t1 t2)),
  t1 ⊑ t2 →
  t1 ⊑♯ t2.
Proof. introv E. simpls*. Qed.

Lemma Lattice_LatticeDec_propagates : forall T (L : Lattice.t T) (t1 t2 : T)
    (DE : forall t1 t2 : T, Decidable (Equiv.eq t1 t2))
    (DP : forall t1 t2 : T, Decidable (Poset.order t1 t2)),
  t1 ⊑ t2 →
  t1 ⊑♯ t2.
Proof. introv E. simpls*. Qed.

Lemma PosetDec_Poset_propagates : forall T (P : PosetDec.t T) (t1 t2 : T),
  t1 ⊑♯ t2 →
  t1 ⊑ t2.
Proof. introv E. simpls*. Qed.

Lemma LatticeDec_Lattice_propagates : forall T (L : LatticeDec.t T) (t1 t2 : T),
  t1 ⊑♯ t2 →
  t1 ⊑ t2.
Proof. introv E. simpls*. Qed.


(** There may be more than one way to generate these type classes.
  Here follow some lemmae to help dealing with this. **)

Lemma Poset_PosetDec_Lattice_LatticeDec : forall T (L : Lattice.t T) DE DP,
  Poset_PosetDec (Lattice.porder (t := L)) DE DP
  = LatticeDec.porder (t := Lattice_LatticeDec _ DE DP).
Proof. reflexivity. Qed.

Lemma PosetDec_Poset_LatticeDec_Lattice : forall T (L : LatticeDec.t T) (TT : TopDec.t T),
  PosetDec_Poset (LatticeDec.porder (t := L))
  = Lattice.porder (t := LatticeDec_Lattice _ _).
Proof. reflexivity. Qed.

Lemma Join_JoinDec_Lattice_LatticeDec : forall T (L : Lattice.t T) DE DP,
  Join_JoinDec DE DP (Lattice.join (t := L))
  = LatticeDec.join (t := Lattice_LatticeDec _ DE DP).
Proof. reflexivity. Qed.

Lemma JoinDec_Join_LatticeDec_Lattice : forall T (L : LatticeDec.t T) (TT : TopDec.t T),
  JoinDec_Join (LatticeDec.join (t := L))
  = Lattice.join (t := LatticeDec_Lattice _ _).
Proof. reflexivity. Qed.

Lemma Meet_MeetDec_Lattice_LatticeDec : forall T (L : Lattice.t T) DE DP,
  Meet_MeetDec DE DP (Lattice.meet (t := L))
  = LatticeDec.meet (t := Lattice_LatticeDec _ DE DP).
Proof. reflexivity. Qed.

Lemma MeetDec_Meet_LatticeDec_Lattice : forall T (L : LatticeDec.t T) (TT : TopDec.t T),
  MeetDec_Meet (LatticeDec.meet (t := L))
  = Lattice.meet (t := LatticeDec_Lattice _ _).
Proof. reflexivity. Qed.

Lemma Bot_BotDec_Lattice_LatticeDec : forall T (L : Lattice.t T) DE DP,
  Bot_BotDec DE DP (Lattice.bot (t := L))
  = LatticeDec.bot (t := Lattice_LatticeDec _ DE DP).
Proof. reflexivity. Qed.

Lemma BotDec_Bot_LatticeDec_Lattice : forall T (L : LatticeDec.t T) (TT : TopDec.t T),
  BotDec_Bot (LatticeDec.bot (t := L))
  = Lattice.bot (t := LatticeDec_Lattice _ _).
Proof. reflexivity. Qed.

Lemma TopDec_Top_LatticeDec_Lattice : forall T (L : LatticeDec.t T) (TT : TopDec.t T),
  TopDec_Top TT
  = Lattice.top (t := LatticeDec_Lattice _ _).
Proof. reflexivity. Qed.

