
(** This file contains some way to build new semantics from already
  defined semantics, and to lift all lemmae proven for the initial
  semantics to the generated ones. **)

Set Implicit Arguments.
Require Export PrettyBigStep.


(** * Immerse Into a Monad **)

Section ImmerseSem.

(** A generic way to build an extended semantics from an semantics
  defined over a poset instead of a lattice.  The domains of this
  semantics are supposed to be embedded into a monad, which is an
  extension of the originals whith an added lattice structure (which
  could be more complex than the boolean one). **)

Variable str : structure.
Let term := term str.
Let name := name str.
Let Rule_struct := Rule_struct term.
Let left (n : name) : term := left str n.
Let rule_struct (n : name) : Rule_struct := rule_struct str n.
Let semantics := semantics str.
Variable adom : domains.
Let ast := st adom.
Let ares := res adom.
Variable asem : semantics adom.
Let acond n (sigma : ast) := cond asem n sigma.
Let arule n : Rule ast ares := rule asem n.
Let aRule := Rule ast ares.
Variable dom : domains.
Let st := st dom.
Let res := res dom.
Variable sem : semantics dom.
Let cond n (sigma : st) := cond sem n sigma.
Let rule n : Rule st res := rule sem n.

Hypothesis Past : Poset.t ast.
Hypothesis Pares : Poset.t ares.
Hypothesis Tast : @Top.t ast _.
Hypothesis Tares : @Top.t ares _.

(** As we don’t have any lattice structure in the initial domains,
  only the monotonicity of the concretisation functions makes sense. **)
Variable gst : ast → st → Prop.
Hypothesis Gst_monotone : ∀ N1 N2 : ast, N1 ⊑ N2 → gst N1 ⊑ gst N2.
Variable gres : ares → res → Prop.
Hypothesis Gres_monotone : ∀ N1 N2 : ares, N1 ⊑ N2 → gres N1 ⊑ gres N2.

Hypothesis acond_monotone : forall n (sigma sigma' : ast),
   sigma' ⊑ sigma →
   acond n sigma' →
   acond n sigma.

Hypothesis acond_correct : forall n asigma sigma,
  gst asigma sigma →
  cond n sigma →
  acond n asigma.

Hypothesis rule_format_correct_all_sem : rule_format_correct_all sem.
Hypothesis rule_format_correct_all_asem : rule_format_correct_all asem.

Hypothesis arules_incr_asem : forall n,
  arules_incr _ _ _ (arule n).

Hypothesis propagates_asem : forall n,
  propagates adom dom gst gres
    (acond n) (cond n)
    (arule n) (rule n).

Hypothesis asem_semantics_full : semantics_full asem.

Section Lift.

(** Definition of the monad **)
Variable M : Type → Type.
Variable ret : forall t, t → M t.
Variable bind : forall t1 t2, M t1 → (t1 → M t2) → M t2 → M t2.

Hypothesis LM : forall t, Poset.t t → Lattice.t (M t).
Hypothesis TM : forall t (P : Poset.t t), @Top.t (M t) _.

Local Instance LMast : Lattice.t (M ast).
  typeclass. Defined.

Local Instance LMares : Lattice.t (M ares).
  typeclass. Defined.

Require Export SharedLat.


(** Monads axioms **)
Hypothesis ret_bind : forall t1 t2 (P2 : Poset.t t2) (a : t1) (f : t1 → M t2) mf,
  max _ f mf →
  Equiv.eq (bind (ret a) f mf) (f a).
Hypothesis bind_ret : forall t (P : Poset.t t) (e : M t) m,
  max _ (ret (t := _)) m →
  Equiv.eq (bind e (ret (t := _)) m) e.
Hypothesis bind_bind : forall t1 t2 t3 (P2 : Poset.t t2) (P3 : Poset.t t3) (e : M t1) (f : t1 → M t2) (g : t2 → M t3) mf mg mb,
  max _ f mf →
  max _ g mg →
  max _ (fun x => bind (f x) g mg) mb →
  Equiv.eq (bind (bind e f mf) g mg) (bind e (λ x : t1, bind (f x) g mg) mb).

(** Compatibility with the order **)
Hypothesis ret_monotone : forall t (P : Poset.t t) (a1 a2 : t),
  a1 ⊑ a2 →
  ret a1 ⊑ ret a2.
Hypothesis monotone_bind : forall t1 t2 (CL1 : Poset.t t1) (CL2 : Poset.t t2)
    (e1 e2 : M t1) (f : t1 → M t2) mf,
  max _ f mf →
  (forall b1 b2, b1 ⊑ b2 → f b1 ⊑ f b2) →
  e1 ⊑ e2 →
  bind e1 f mf ⊑ bind e2 f mf.
Hypothesis bind_monotone : forall t1 t2 (CL2 : Poset.t t2)
    (e : M t1) (f g : t1 → M t2) mf mg,
  max _ f mf →
  max _ g mg →
  (forall b, f b ⊑ g b) →
  bind e f mf ⊑ bind e g mg.

(** Compatibility with the operations **)
Hypothesis meet_bind : forall t1 t2 (CL1 : Poset.t t1) (CL2 : Poset.t t2)
    (e1 e2 : M t1) (f : t1 → M t2) mf,
  max _ f mf →
  (forall b1 b2, b1 ⊑ b2 → f b1 ⊑ f b2) →
  (bind e1 f mf ⊓ bind e2 f mf) ⊑ bind (e1 ⊓ e2) f mf.
Hypothesis bind_meet : forall t1 t2 (CL2 : Poset.t t2)
    (e : M t1) (f g : t1 → M t2) mf mg mm,
  max _ f mf →
  max _ g mg →
  max _ (fun a => f a ⊓ g a) mm →
  (bind e f mf ⊓ bind e g mg) ⊑ bind e (fun a => f a ⊓ g a) mm.

Lemma meet_bind_eq : forall t1 t2 (CL1 : Poset.t t1) (CL2 : Poset.t t2)
    (e1 e2 : M t1) (f : t1 → M t2) mf,
  max _ f mf →
  (forall b1 b2, b1 ⊑ b2 → f b1 ⊑ f b2) →
  Equiv.eq (bind (e1 ⊓ e2) f mf) (bind e1 f mf ⊓ bind e2 f mf).
Proof.
  introv Mf O. apply Poset.antisym.
   apply Meet.meet_greatest_lower_bound; apply~ monotone_bind.
   applys~ meet_bind O.
Qed.

Lemma bind_meet_eq : forall t1 t2 (CL2 : Poset.t t2)
    (e : M t1) (f g : t1 → M t2) mf mg mm,
  max _ f mf →
  max _ g mg →
  max _ (fun a => f a ⊓ g a) mm →
  Equiv.eq (bind e f mf ⊓ bind e g mg) (bind e (fun a => f a ⊓ g a) mm).
Proof.
  introv Mf Mg Mm. apply Poset.antisym.
   applys~ bind_meet.
   apply Meet.meet_greatest_lower_bound; apply~ bind_monotone.
Qed.


(** We will want to use this monad and t extract properties from it.
  To this end, we will map the elements into properties.  We would
  thus like a way to flatten properties stuck in the monad into
  tangible properties. **)
Require Import PropLat.

Local Instance MPropLat : Lattice.t (M Prop).
  exact (LM LProp.Poset). Defined.

Variable evaluate : M Prop → Prop.

Hypothesis evaluate_ret : forall P : Prop,
  evaluate (ret P) ↔ P.
Hypothesis evaluate_monotone : forall A B : M Prop,
  A ⊑ B →
  evaluate A → evaluate B.
Hypothesis evaluate_meet : forall A B : M Prop,
  evaluate A ∧ evaluate B → evaluate (A ⊓ B).
Hypothesis ret_meet_prop : forall (P1 P2 : Prop),
  Equiv.eq (ret (P1 ∧ P2)) (ret P1 ⊓ ret P2).

Hypothesis bind_generic : forall (t1 t2 : Type) (e1 : M t1) (e2 : M t2) (P : Poset.t t2),
  bind e1 (fun _ => e2) e2 ⊑ e2.

Definition bind_prop : forall t, M t → (t → M Prop) → M Prop :=
  fun t e P =>
    bind e P (ret (exists a, evaluate (P a))).

Hypothesis evaluate_monotone_inv : forall A B : M Prop,
  (evaluate A → evaluate B) →
  A ⊑ B.

Lemma ret_evaluate : forall P : M Prop,
  Equiv.eq (ret (evaluate P)) P.
Proof. introv. apply Poset.antisym; apply evaluate_monotone_inv; introv E; apply~ evaluate_ret. Qed.

Lemma max_prop : forall t (P : t → M Prop),
  max _ P (ret (exists a, evaluate (P a))).
Proof.
  introv. splits.
   introv. eapply Poset.trans.
    apply Poset.refl. apply Equiv.sym. apply ret_evaluate.
    apply ret_monotone. introv E. exists~ x.
   introv A. apply evaluate_monotone_inv.
    introv E. rewrite evaluate_ret in E. lets (a&Ea): (rm E).
    applys evaluate_monotone Ea. apply~ A.
Qed.


(** ** Abstract semantics **)

(** Because of the monad, we want t use a slightly different
  definition for the semantics. **)

(** We would like to have this, but it can’t be accepted by Coq as-is.

[[
CoInductive aevalm : term → ast → M ares → Prop :=
  | aevalm_cons : forall t asigma er,
      (forall n,
        t = left n →
        acond n asigma →
        aapplym n asigma er) →
      aevalm t asigma er
with aapplym : name → ast → M ares → Prop :=
  | aapplym_cons : forall n asigma asigma' er er',
      asigma ⊑ asigma' →
      er' ⊑ er →
      aapplym_step n asigma' er' →
      aapplym n asigma er
with aapplym_step : name → ast → M ares → Prop :=
  | aapplym_step_Ax : forall n ax asigma ar,
      rule_struct n = Rule_struct_Ax _ →
      arule n = Rule_Ax ax →
      ax asigma = Some ar →
      aapplym_step n asigma (ret ar)
  | aapplym_step_R1 : forall n t up asigma asigma' er,
      rule_struct n = Rule_struct_R1 t →
      arule n = Rule_R1 _ up →
      up asigma = Some asigma' →
      aevalm t asigma' er →
      aapplym_step n asigma er
  | aapplym_step_R2 : forall n t1 t2 up next asigma asigma1 er er',
      rule_struct n = Rule_struct_R2 t1 t2 →
      arule n = Rule_R2 up next →
      up asigma = Some asigma1 →
      aevalm t1 asigma1 er →
      evaluate (bind er (fun ar =>
        ret (exists asigma2,
          next asigma ar = Some asigma2
          ∧ aevalm t2 asigma2 er'))) →
      aapplym_step n asigma er'
  .
]]

**)

Section CheckStep.

Variable aevalm : term → ast → M ares → Prop.

Inductive aapplym_check_step : name → ast → M ares → Prop :=
  | aapplym_check_step_Ax : forall n ax asigma ar,
      rule_struct n = Rule_struct_Ax _ →
      arule n = Rule_Ax ax →
      ax asigma = Some ar →
      aapplym_check_step n asigma (ret ar)
  | aapplym_check_step_R1 : forall n t up asigma asigma' er,
      rule_struct n = Rule_struct_R1 t →
      arule n = Rule_R1 _ up →
      up asigma = Some asigma' →
      aevalm t asigma' er →
      aapplym_check_step n asigma er
  | aapplym_check_step_R2 : forall n t1 t2 up next asigma asigma1 er er',
      rule_struct n = Rule_struct_R2 t1 t2 →
      arule n = Rule_R2 up next →
      up asigma = Some asigma1 →
      aevalm t1 asigma1 er →
      evaluate (bind_prop er (fun ar =>
        ret (exists asigma2,
          next asigma ar = Some asigma2
          ∧ aevalm t2 asigma2 er'))) →
      aapplym_check_step n asigma er'
  .

End CheckStep.


Definition asemm_dom :=
  make_domains ast (M ares).

(** A version of [hyp_in] independant from the semantics [asem].
  We need it as we can not yet introduce the abstract rules (for
  typing reasons). **)
Inductive hypm_in (aapplym_check : name → ast → M ares → Prop) : term → ast → M ares → Prop :=
  | hypm_in_cons : forall t sigma r,
      (forall n,
        t = left n →
        acond n sigma →
        aapplym_check n sigma r) →
      hypm_in aapplym_check t sigma r
  .

(** A first semantics, taking only a basic semantic context. **)
Definition aevalm : term → ast → M ares → Prop :=
  aeval_f str asemm_dom hypm_in aapplym_check_step (aapply_check (glue_weaken _ _ _ _)).

Lemma Park_aevalm : forall R,
  Park str asemm_dom hypm_in aapplym_check_step (aapply_check (glue_weaken _ _ _ _)) R →
  forall t sigma r, R t sigma r → aevalm t sigma r.
Proof. introv P D. apply* aeval_f_cons. Qed.

(** The conservation property **)

Lemma aeval_aevalm : forall t asigma ar,
  aeval asem (glue_weaken _ _ _ _) t asigma ar →
  aevalm t asigma (ret ar).
Proof.
  introv A. apply aeval_f_cons with (fun t asigma er =>
    exists ar, er = ret ar ∧ aeval asem (glue_weaken _ _ _ _) t asigma ar); [| autos* ].
  clear t asigma ar A. introv (ar&E&D). substs. constructor.
  introv E C. inverts D as F. forwards A: (rm F) (rm E) (rm C).
  inverts A as G A. inverts G as G. inverts G as O1 O2. constructors.
   constructors. constructors.
    exact O1.
    applys ret_monotone O2.
   introv (E1&E2). substs. forwards~ A': A. inverts A' as S R E1 A1 E2 A2.
    apply* aapplym_check_step_Ax.
    apply* aapplym_check_step_R1.
    applys~ aapplym_check_step_R2 S R E1.
     eexists. splits*.
     eapply evaluate_monotone.
      apply Poset.refl. apply Equiv.sym. apply ret_bind. apply max_prop.
      apply evaluate_ret. eexists. splits*.
Qed.

(** The final semantics **)

Definition aevalM : term → M ast → M ares → Prop :=
  fun t esigma er =>
    evaluate (bind_prop esigma (fun asigma => ret (aevalm t asigma er))).

(** The conservation property **)

Lemma aeval_aevalM : forall t asigma ar,
  aeval asem (glue_weaken _ _ _ _) t asigma ar →
  aevalM t (ret asigma) (ret ar).
Proof.
  introv D. eapply evaluate_monotone.
   apply Poset.refl. apply Equiv.sym. apply ret_bind. apply max_prop.
   apply evaluate_ret. applys~ aeval_aevalm D.
Qed.


(** ** Conservation Properties **)

(** *** Concretisation Functions **)

Definition gammaM : forall t c, (t → c → Prop) → M t → c → Prop :=
  fun t c gamma e c =>
    evaluate (bind_prop e (fun a => ret (gamma a c))).

(** Compatibility with the concretisation function **)
Lemma gammaM_ret : forall t c (g : t → c → Prop) a b,
  g a b →
  gammaM g (ret a) b.
Proof.
  introv G. eapply evaluate_monotone.
   apply Poset.refl. apply Equiv.sym. apply ret_bind. apply max_prop.
   apply~ evaluate_ret.
Qed.

Lemma gammaM_monotone : forall t c (g : t → c → Prop) CT,
  (∀ N1 N2 : t, N1 ⊑ N2 → g N1 ⊑ g N2) →
  ∀ N1 N2 : M t, Poset.order (t := Lattice.porder (t := LM CT)) N1 N2 →
    gammaM g N1 ⊑ gammaM g N2.
Proof.
  introv monotone O G. applys evaluate_monotone G. apply~ monotone_bind.
   apply max_prop.
   introv B. eapply ret_monotone.
   lets~ I: monotone B x.
Qed.

Lemma gammaM_meet_morph : forall t c (g : t → c → Prop) CT,
  (∀ N1 N2 : t, N1 ⊑ N2 → g N1 ⊑ g N2) →
  ∀ N1 N2 : M t, ((gammaM g N1) ⊓ (gammaM g N2))
    ⊑ gammaM g (Meet.op (t := Lattice.meet (t := LM CT)) N1 N2).
Proof.
  introv monotone (G1&G2). unfolds. eapply evaluate_monotone.
   apply Poset.refl. apply Equiv.sym. apply meet_bind_eq.
    apply max_prop.
    introv B. eapply ret_monotone.
    lets~ I: monotone B x.
   apply evaluate_meet. splits~.
Qed.

Definition gstM := gammaM gst.

Definition gresM := gammaM gres.


(** *** Correctness **)

Lemma correctness_aevalm : forall t asigma er,
  aevalm t asigma er → forall sigma r,
  gst asigma sigma →
  eval sem t sigma r →
  gresM er r.
Proof.
  introv Da G D. inverts Da as P PR.
  cuts (IHE&IHA): ((forall t sigma r,
      eval sem t sigma r →
      forall asigma ar,
      R t asigma ar →
      gst asigma sigma →
      gresM ar r)
    /\ forall n sigma r,
      apply sem n sigma r →
      cond n sigma →
      forall asigma er,
      aapplym_check_step R n asigma er →
      gst asigma sigma →
      gresM er r).
    forwards*: IHE.
  clear PR G D. apply eval_mutind.
  (** eval **)
  introv Et C A IHA aE G. forwards All: P aE. inverts All as All. forwards aA: All Et.
    apply* acond_correct.
  inverts aA as Gl T. inverts Gl as Gl. inverts Gl as O1 O2. forwards~ G': IHA T.
   applys~ Gst_monotone O1 G.
   applys~ gammaM_monotone G'.
  (** apply_Ax **)
  introv Estr ER Eax C aA G. inverts aA; rewrite_term (rule_struct n); inverts Estr.
  eapply evaluate_monotone.
    apply Poset.refl. apply Equiv.sym. apply ret_bind. apply max_prop.
  apply evaluate_ret. forwards Pr: propagates_asem n.
  rewrite_term (rule n). rewrite_term (arule n). inverts Pr as Hyp. applys~ Hyp Eax H1.
  apply* acond_correct.
  (** apply_R1 **)
  introv Estr ER Eup E IH C aA G.
  inverts aA as Estr' ER' Eup' aE'; rewrite_term (rule_struct n); inverts Estr.
  forwards Pr: propagates_asem n. rewrite_term (rule n). rewrite_term (arule n).
  inverts Pr as Hyp. forwards~: Hyp Eup Eup'.
   apply* acond_correct.
   apply* IH.
  (** apply_R2 **)
  introv Estr ER Eup E1 IH1 Enext Ae2 IH2 C aA. introv G.
  inverts aA as Estr1 ER1 Eup1 aE1 Enext1 aE2; rewrite_term (rule_struct n); inverts Estr.
  forwards Pr: propagates_asem n. rewrite_term (rule n). rewrite_term (arule n).
  inverts Pr as Hyp1 Hyp2. forwards~ G1: Hyp1 Eup Eup1.
    apply* acond_correct.
  forwards G2: IH1 aE1 G1. lets B2: (conj G2 Enext1). apply evaluate_meet in B2.
  eapply evaluate_monotone in B2.
   apply evaluate_ret. exact B2.
   eapply Poset.trans.
    apply bind_meet; apply max_prop.
    eapply Poset.trans.
     apply bind_monotone.
      apply max_prop.
      apply max_prop.
      introv. eapply Poset.trans.
       apply Poset.refl. apply Equiv.sym. apply ret_meet_prop.
       apply ret_monotone. introv (G3&asigma1&Easigma1&D2).
        apply* IH2.
     eapply Poset.trans; [| apply bind_generic ].
      apply bind_monotone.
       apply max_prop.
       apply max_const. exact (⊤).
       introv _. autos*.
Qed.

Lemma correctness_aevalM : forall t esigma er,
  aevalM t esigma er → forall sigma r,
  gstM esigma sigma →
  eval sem t sigma r →
  gresM er r.
Proof.
  introv DA G D. lets B: (conj DA G). apply evaluate_meet in B.
  eapply evaluate_monotone in B.
   apply evaluate_ret. exact B.
   eapply Poset.trans.
    apply bind_meet; apply max_prop.
    eapply Poset.trans.
     apply bind_monotone.
      apply max_prop.
      apply max_prop.
      introv. eapply Poset.trans.
       apply Poset.refl. apply Equiv.sym. apply ret_meet_prop.
       apply ret_monotone. introv (Da&G'). applys* correctness_aevalm Da.
     eapply Poset.trans; [| apply bind_generic ].
      apply bind_monotone.
       apply max_prop.
       apply max_const. exact (⊤).
       introv _. autos*.
Qed.

End Lift.

End ImmerseSem.

