
(** This file contains the definition of pretty-big-step derivations,
  concrete and abstract, as well te predicates stating that these are
  related.  The correctness theorem is stated and proven.  An example
  of a concrete semantics is given in the file “ConcreteWhile.v”, as
  well as some derivations example.  File “AbstractSign.v” gives an
  example of an abstract semantics. **)

Set Implicit Arguments.
Require Import Utf8.
Require Export Shared TLC.LibFunc.

(* Congruence doesn’t work as expected, so I’m removing it from this tactic. *)
Ltac false_post ::=
  solve [ assumption | discriminate ].

(* [logics] doesn’t work as expected, so I’m removing it from this tactic. *)
Ltac case_if_post ::=
  tryfalse.

Ltac rewrite_term t :=
  fold t in *;
  repeat try match goal with
  | H : t = _ |- _ => rewrite H in *
  | H : _ = t |- _ => rewrite <- H in *
  end.


Section RuleStructure.

(** * Rule Structure **)

(** Defines all things common to the concrete and abstract semantics. **)

Inductive Rule_struct term :=
  | Rule_struct_Ax : Rule_struct term
  | Rule_struct_R1 : term →  Rule_struct term
  | Rule_struct_R2 : term → term → Rule_struct term
  .

(** The following structure represents rule schemes **)
(** A rule scheme maps a name **)
(** - to the term that the rule scheme can evaluate **)
(** - to a rule_struct containing the other terms necessary to define **)
(**   an instance of the rule scheme **)

Record structure := {

    term : Type;
    name : Type;

    left : name → term;
    rule_struct : name → Rule_struct term

  }.

Variable str : structure.
Let term := term str.
Let name := name str.
Let Rule_struct := Rule_struct term.
Let left (n : name) : term := left str n.
Let rule_struct (n : name) : Rule_struct := rule_struct str n.

Inductive Rule st res :=
  | Rule_Ax : (st → option res) → Rule st res
  | Rule_R1 : (st → option st) → Rule st res
  | Rule_R2 : (st → option st) → (st → res → option st) → Rule st res
  .

Record domains := make_domains {
    st : Type;
    res : Type
  }.

Variable dom : domains.
Let st := st dom.
Let res := res dom.

Record semantics := make_semantics {
    cond : name → st → Prop;
    rule : name → Rule st res
  }.

Variable sem : semantics.
Let cond n (sigma : st) := cond sem n sigma.
Let rule n : Rule st res := rule sem n.
Let Rule := Rule st res.

Inductive rule_format_correct : Rule_struct → Rule → Prop :=
  | rule_format_correct_Ax : forall ax,
    rule_format_correct (Rule_struct_Ax _) (Rule_Ax ax)
  | rule_format_correct_R1 : forall t up,
    rule_format_correct (Rule_struct_R1 t) (Rule_R1 _ up)
  | rule_format_correct_R2 : forall t1 t2 up next,
    rule_format_correct (Rule_struct_R2 t1 t2) (Rule_R2 up next)
  .

Definition rule_format_correct_all :=
  forall n, rule_format_correct (rule_struct n) (rule n).

Inductive applies : Rule → st → Prop :=
  | applies_Ax : forall sigma r ax,
    ax sigma = Some r →
    applies (Rule_Ax ax) sigma
  | applies_R1 : forall sigma sigma' up,
    up sigma = Some sigma' →
    applies (Rule_R1 _ up) sigma
  | applies_R2 : forall sigma sigma' up next,
    up sigma = Some sigma' →
    (forall r, exists sigma', next sigma r = Some sigma') →
    applies (Rule_R2 up next) sigma
  .

Definition semantics_full :=
  forall n sigma,
   cond n sigma →
   applies (rule n) sigma.

(** Viewing the previous definition, one might ask why we do not directly define
  [cond] as being [applies] with the correct arguments.  The reason is that most
  axioms, up and next functions may be more general than one might expect: most
  of them are structural and nothing prevent them from simply being constant or
  simply wrapping their argument into another structure.  Furthermore, we want
  [cond] to be easily usable in proofs, which [applies] does not.  For these
  reasons, we want to separate [cond] from [applies]. **)

End RuleStructure.


Section ConcreteWorld.

(** * Concrete Semantics **)

Variable str : structure.
Let term := term str.
Let name := name str.
Let Rule_struct := Rule_struct term.
Let left (n : name) : term := left str n.
Let rule_struct (n : name) : Rule_struct := rule_struct str n.
Variable dom : domains.
Let st := st dom.
Let res := res dom.
Let semantics := semantics str dom.
Variable sem : semantics.
Let cond n (sigma : st) := cond sem n sigma.
Let rule n : Rule st res := rule sem n.
Let Rule := Rule st res.


(** I’ve put it in [Type] instead of [Prop] as it’s meaningful to have
  a concrete interpreter returning such an object… but apart from
  that, one can think of it as in [Prop].  Furthermore, this
  development is based on TLC, which contains proof irrelevance:
  different derivations should be considered as different objects, and
  thus [Prop] does not fit here. **)

Inductive eval : term → st → res → Type :=
  | eval_cons : forall t sigma r n,
      t = left n →
      cond n sigma →
      apply n sigma r →
      eval t sigma r
with apply : name → st → res → Type :=
  | apply_Ax : forall n ax sigma r,
      rule_struct n = Rule_struct_Ax _ →
      rule n = Rule_Ax ax →
      ax sigma = Some r →
      apply n sigma r
  | apply_R1 : forall n t up sigma sigma' r,
      rule_struct n = Rule_struct_R1 t →
      rule n = Rule_R1 _ up →
      up sigma = Some sigma' →
      eval t sigma' r →
      apply n sigma r
  | apply_R2 : forall n t1 t2 up next sigma sigma1 sigma2 r r',
      rule_struct n = Rule_struct_R2 t1 t2 →
      rule n = Rule_R2 up next →
      up sigma = Some sigma1 →
      eval t1 sigma1 r →
      next sigma r = Some sigma2 →
      eval t2 sigma2 r' →
      apply n sigma r'
  .

Scheme eval_ind_2 := Induction for eval Sort Prop
  with apply_ind_2 := Induction for apply Sort Prop.
Combined Scheme eval_mutind from eval_ind_2, apply_ind_2.

(** The depth of the derivation. **)
Fixpoint eval_depth t sigma r (D : eval t sigma r) :=
  match D with
  | eval_cons E C A =>
    apply_depth A
  end
with apply_depth n sigma r (A : apply n sigma r) :=
  match A with
  | apply_Ax _ _ E1 E2 E3 => 0
  | apply_R1 _ _ E1 E2 E3 D => S (eval_depth D)
  | apply_R2 _ _ E1 E2 E3 D1 E4 D2 =>
    S (max (eval_depth D1) (eval_depth D2))
  end.

End ConcreteWorld.


Section AbstractWorld.

(** * Abstract Semantics **)

(** The abstract semantics uses similar structural data than the
  concrete semantics, with three exceptions (see papers for details):
  first, every rule which can apply are applied at each step, second,
  there is a “glue” step between two rules to allows generalisation,
  third, the semantics is coinductive. **)

Require Export CompleteLattice.

Variable str : structure.
Let term := term str.
Let name := name str.
Let Rule_struct := Rule_struct term.
Let left (n : name) : term := left str n.
Let rule_struct (n : name) : Rule_struct := rule_struct str n.
Variable adom : domains.
Let ast := st adom.
Let ares := res adom.
Let semantics := semantics str adom.
Variable asem : semantics.
Let acond n (sigma : ast) := cond asem n sigma.
Let arule n : Rule ast ares := rule asem n.
Let aRule := Rule ast ares.

(** The glue provides some ways of extending existing triples: if we
  proved all the triple [asigma', t, ar'] and that [glue [asigma',
  ar'] asigma ar] holds, then the triple [asigma, t, ar] should
  hold. See Section Glue to get some instances. **)
Definition glue_type : Type :=
  name → (ast → ares → Prop) → ast → ares → Prop.

Variable glue : glue_type.


(** ** Definitions **)

(** We give two definitions of the abstract semantics.  [aeval] is
  based on Coq’s coinduction, while [aeval_f] is defined to be the
  greatest fixed point of a functionnal.  Both definitions are proven
  equivalent.**)

Section CheckStep.

Variable aeval : term → ast → ares → Prop.

Inductive aapply_check_step : name → ast → ares → Prop :=
  | aapply_check_step_Ax : forall n ax sigma r,
      rule_struct n = Rule_struct_Ax _ →
      arule n = Rule_Ax ax →
      ax sigma = Some r →
      aapply_check_step n sigma r
  | aapply_check_step_R1 : forall n t up sigma sigma' r,
      rule_struct n = Rule_struct_R1 t →
      arule n = Rule_R1 _ up →
      up sigma = Some sigma' →
      aeval t sigma' r →
      aapply_check_step n sigma r
  | aapply_check_step_R2 : forall n t1 t2 up next sigma sigma1 sigma2 r r',
      rule_struct n = Rule_struct_R2 t1 t2 →
      arule n = Rule_R2 up next →
      up sigma = Some sigma1 →
      aeval t1 sigma1 r →
      next sigma r = Some sigma2 →
      aeval t2 sigma2 r' →
      aapply_check_step n sigma r'
  .

End CheckStep.

Section Check.

Variable aapply_check_step : name → ast → ares → Prop.

Inductive aapply_check : name → ast → ares → Prop :=
  | aapply_check_cons : forall n sigma r P,
      glue n P sigma r →
      (forall sigma' r',
        P sigma' r' →
        aapply_check_step n sigma' r') →
      aapply_check n sigma r
  .

End Check.

Section HypIn.

Variable aapply_check : name → ast → ares → Prop.

Inductive hyp_in : term → ast → ares → Prop :=
  | hyp_in_cons : forall t sigma r,
      (forall n,
        t = left n →
        acond n sigma →
        aapply_check n sigma r) →
      hyp_in t sigma r
  .

End HypIn.

Section Park.

Variable hyp_in : (name → ast → ares → Prop) → term → ast → ares → Prop.
Variable aapply_check_step : (term → ast → ares → Prop) → name → ast → ares → Prop.
Variable aapply_check : (name → ast → ares → Prop) → name → ast → ares → Prop.

Definition Park (R : term → ast → ares → Prop) :=
  forall t sigma r,
    R t sigma r →
    hyp_in (aapply_check (aapply_check_step R)) t sigma r.

Inductive aeval_f : term → ast → ares → Prop :=
  | aeval_f_cons : forall R t sigma r,
    Park R →
    R t sigma r →
    aeval_f t sigma r
  .

End Park.


CoInductive aeval : term → ast → ares → Prop :=
  | aeval_cons : forall t sigma r,
      (forall n,
        t = left n →
        acond n sigma →
        aapply n sigma r) →
      aeval t sigma r
with aapply : name → ast → ares → Prop :=
  | aapply_cons : forall n sigma r P,
      glue n P sigma r →
      (forall sigma' r',
        P sigma' r' →
        aapply_step n sigma' r') →
      aapply n sigma r
with aapply_step : name → ast → ares → Prop :=
  | aapply_step_Ax : forall n ax sigma r,
      rule_struct n = Rule_struct_Ax _ →
      arule n = Rule_Ax ax →
      ax sigma = Some r →
      aapply_step n sigma r
  | aapply_step_R1 : forall n t up sigma sigma' r,
      rule_struct n = Rule_struct_R1 t →
      arule n = Rule_R1 _ up →
      up sigma = Some sigma' →
      aeval t sigma' r →
      aapply_step n sigma r
  | aapply_step_R2 : forall n t1 t2 up next sigma sigma1 sigma2 r r',
      rule_struct n = Rule_struct_R2 t1 t2 →
      arule n = Rule_R2 up next →
      up sigma = Some sigma1 →
      aeval t1 sigma1 r →
      next sigma r = Some sigma2 →
      aeval t2 sigma2 r' →
      aapply_step n sigma r'
  .


(** *** Correction Lemmae Between [aeval] and [aeval_f]. **)

Lemma aapply_check_step_impl : forall (R1 R2 : term → ast → ares → Prop) t sigma r,
  (forall t sigma r, R1 t sigma r → R2 t sigma r) →
  aapply_check_step R1 t sigma r →
  aapply_check_step R2 t sigma r.
Proof.
  introv I A. inverts A.
   apply* aapply_check_step_Ax.
   apply* aapply_check_step_R1.
   apply* aapply_check_step_R2.
Qed.

Section AapplyCheck.

Hypothesis aapply_check_step : (term → ast → ares → Prop) → name → ast → ares → Prop.

Hypothesis aapply_check_step_impl : forall (R1 R2 : term → ast → ares → Prop) t sigma r,
  (forall t sigma r, R1 t sigma r → R2 t sigma r) →
  aapply_check_step R1 t sigma r →
  aapply_check_step R2 t sigma r.

Lemma aapply_check_impl : forall (R1 R2 : term → ast → ares → Prop) n sigma r,
  (forall t sigma r, R1 t sigma r → R2 t sigma r) →
  aapply_check (aapply_check_step R1) n sigma r →
  aapply_check (aapply_check_step R2) n sigma r.
Proof.
  introv I A. inverts A as G D. constructors; try eassumption.
  introv OK. applys~ aapply_check_step_impl I D.
Qed.

End AapplyCheck.

Section HypInPark.

Variable aapply_check_step : (term → ast → ares → Prop) → name → ast → ares → Prop.
Variable aapply_check : (name → ast → ares → Prop) → name → ast → ares → Prop.

Hypothesis aapply_check_impl : forall (R1 R2 : term → ast → ares → Prop) n sigma r,
  (forall t sigma r, R1 t sigma r → R2 t sigma r) →
  aapply_check (aapply_check_step R1) n sigma r →
  aapply_check (aapply_check_step R2) n sigma r.

Lemma hyp_in_impl : forall (R1 R2 : term → ast → ares → Prop) t sigma r,
  (forall t sigma r, R1 t sigma r → R2 t sigma r) →
  hyp_in (aapply_check (aapply_check_step R1)) t sigma r →
  hyp_in (aapply_check (aapply_check_step R2)) t sigma r.
Proof.
  introv I A. inverts A as Da. constructors. introv E C.
  applys* aapply_check_impl I.
Qed.

Definition Park_modulo_set (R1 R2 : term → ast → ares → Prop) :=
  forall t sigma r,
    R2 t sigma r →
    hyp_in (aapply_check (aapply_check_step (fun t sigma r => R1 t sigma r ∨ R2 t sigma r)))
      t sigma r.

Lemma Park_modulo_complete : forall R1 R2,
  Park hyp_in aapply_check_step aapply_check R1 →
  Park_modulo_set R1 R2 →
  Park hyp_in aapply_check_step aapply_check (fun t sigma r => R1 t sigma r ∨ R2 t sigma r).
Proof.
  introv P1 P2 [D|D].
   applys* hyp_in_impl P1.
   apply~ P2.
Qed.

End HypInPark.

Lemma aapply_step_check : aapply_step = aapply_check_step aeval.
Proof.
  extens. intros t sigma r. iff A.
    inverts A.
     apply* aapply_check_step_Ax.
     apply* aapply_check_step_R1.
     apply* aapply_check_step_R2.
    inverts A.
     apply* aapply_step_Ax.
     apply* aapply_step_R1.
     apply* aapply_step_R2.
Qed.

Theorem aevals_equiv : forall t sigma r,
  aeval t sigma r ↔ aeval_f hyp_in aapply_check_step aapply_check t sigma r.
Proof.
  introv. iff D.
   constructors*. clear D t sigma r. introv D. inverts D as Ac.
    constructors. introv E C. forwards Aac: Ac E C. inverts Aac as O1 O2 D.
    constructors; try eassumption. rewrite~ <- aapply_step_check.
   inverts D as R_check Rc. gen t sigma r. cofix Dv. introv Rc.
    forwards Dc: R_check Rc. inverts Dc as Ac.
    constructors. introv E C. forwards Aac: Ac E C. inverts Aac as G D.
    constructors; try eassumption. introv OK. forwards D': D OK. inverts D'.
     apply* aapply_step_Ax.
     apply* aapply_step_R1.
     apply* aapply_step_R2.
Qed.

Corollary aevals_same : aeval = aeval_f hyp_in aapply_check_step aapply_check.
Proof. extens. exact aevals_equiv. Qed.

Corollary Park_aeval : forall R,
  Park hyp_in aapply_check_step aapply_check R →
  forall t sigma r, R t sigma r → aeval t sigma r.
Proof. introv P D. rewrite aevals_same. apply* aeval_f_cons. Qed.

Lemma aeval_hyp_in_impl : forall t sigma r,
  aeval t sigma r → hyp_in (aapply_check (aapply_check_step aeval)) t sigma r.
Proof.
  introv D. rewrite aevals_same in *. inverts D as I D.
  forwards D': I D. applys~ hyp_in_impl aapply_check_impl D'.
    apply~ aapply_check_step_impl.
  clear - I. introv D. constructors*.
Qed.

Lemma check_fix_aeval : forall (R : term → ast → ares → Prop),
  (forall t sigma r,
    R t sigma r →
    hyp_in (aapply_check (aapply_check_step R)) t sigma r) →
  forall t sigma r, R t sigma r → aeval t sigma r.
Proof. introv ER. introv D. rewrite aevals_same. constructors*. Qed.

Lemma aeval_hyp_in : aeval = hyp_in (aapply_check (aapply_check_step aeval)).
Proof.
  extens. intros t sigma r. iff D.
   inverts D as Dec. constructors. introv E C. forwards A: Dec E C.
    inverts A as O1 O2 A. constructors; try eassumption.
    rewrite~ <- aapply_step_check.
   inverts D as Dec. constructors. introv E C. forwards A: Dec E C.
    inverts A as O1 O2 A. constructors; try eassumption.
    rewrite~ aapply_step_check.
Qed.

Section HypInIter.

Variable aapply_check_step : (term → ast → ares → Prop) → name → ast → ares → Prop.
Variable aapply_check : (name → ast → ares → Prop) → name → ast → ares → Prop.

Hypothesis aapply_check_impl : forall (R1 R2 : term → ast → ares → Prop) n sigma r,
  (forall t sigma r, R1 t sigma r → R2 t sigma r) →
  aapply_check (aapply_check_step R1) n sigma r →
  aapply_check (aapply_check_step R2) n sigma r.

Inductive hyp_in_iter : (term → ast → ares → Prop) → term → ast → ares → Prop :=
  | hyp_in_iter_exact : forall (R : term → ast → ares → Prop) t sigma r,
    R t sigma r →
    hyp_in_iter R t sigma r
  | hyp_in_iter_cons : forall (R R' : term → ast → ares → Prop) t sigma r,
    (forall t sigma r, R' t sigma r → hyp_in_iter R t sigma r) →
    hyp_in (aapply_check (aapply_check_step R')) t sigma r →
    hyp_in_iter R t sigma r
  .

Lemma hyp_in_iter_applyn : forall n R t sigma r,
  applyn n (fun R => hyp_in (aapply_check (aapply_check_step R))) R t sigma r →
  hyp_in_iter R t sigma r.
Proof.
  induction n; introv D.
   apply* hyp_in_iter_exact.
   apply hyp_in_iter_cons with
     (R' := applyn n (fun R => hyp_in (aapply_check (aapply_check_step R))) R); autos*.
Qed.

Lemma hyp_in_iter_impl : forall (R1 R2 : term → ast → ares → Prop),
  (forall t sigma r, R1 t sigma r → R2 t sigma r) →
  (forall t sigma r, hyp_in_iter R1 t sigma r → hyp_in_iter R2 t sigma r).
Proof.
  introv I D. gen I. induction D; introv I.
   apply* hyp_in_iter_exact.
   apply hyp_in_iter_cons with (R' := hyp_in_iter R2); trivial.
    applys hyp_in_impl aapply_check_impl H1.
     introv D. apply* H0.
Qed.

Lemma hyp_in_iter_iter : forall (R : term → ast → ares → Prop),
   hyp_in_iter (hyp_in_iter R) = hyp_in_iter R.
Proof.
  extens. intros t sigma r. iff D; [| apply~ hyp_in_iter_exact ].
  sets_eq R': (hyp_in_iter R). rewrite EQR'.
  asserts I: (forall t sigma r, R' t sigma r → hyp_in_iter R t sigma r). substs*.
  clear EQR'. gen R. induction~ D; introv I.
   apply hyp_in_iter_cons with (R' := hyp_in_iter R0); trivial.
    applys hyp_in_impl aapply_check_impl H1.
     introv D. apply* H0.
Qed.

Lemma hyp_in_iter_check : forall (R : term → ast → ares → Prop) t sigma r,
  hyp_in_iter (hyp_in (aapply_check (aapply_check_step R))) t sigma r →
  hyp_in (aapply_check (aapply_check_step (hyp_in_iter R))) t sigma r.
Proof.
  introv D. sets_eq R': (hyp_in (aapply_check (aapply_check_step R))).
  asserts I: (forall t sigma r,
               R' t sigma r →
               hyp_in (aapply_check (aapply_check_step (hyp_in_iter R))) t sigma r).
   rewrite EQR'. clear D. introv D. applys hyp_in_impl aapply_check_impl D.
    apply hyp_in_iter_exact.
  clear EQR'. induction* D.
  applys hyp_in_impl aapply_check_impl H1.
   introv D. apply hyp_in_iter_cons with (R' := hyp_in_iter R); trivial.
    applys H0 D I.
Qed.

End HypInIter.

Lemma check_fix_aeval_iter_cons : forall (R : term → ast → ares → Prop),
  (forall t sigma r,
    R t sigma r →
    hyp_in (aapply_check (aapply_check_step (hyp_in_iter aapply_check_step aapply_check R)))
      t sigma r) →
  forall t sigma r, R t sigma r → aeval t sigma r.
Proof.
  introv ER D. rewrite aevals_same.
  apply aeval_f_cons with (R := hyp_in_iter aapply_check_step aapply_check R);
    try solve [ constructors* ].
  clear t sigma r D. introv D. inverts* D. applys* hyp_in_impl aapply_check_impl.
  apply~ aapply_check_step_impl.
Qed.

Lemma hyp_in_aeval : hyp_in (aapply_check (aapply_check_step aeval)) = aeval.
Proof.
  extens. intros t sigma r. iff D; [| apply~ aeval_hyp_in_impl ].
  rewrite aevals_same in *. applys aeval_f_cons D.
  clear t sigma r D. introv D. applys hyp_in_impl aapply_check_impl D.
    apply aapply_check_step_impl.
  clear t sigma r D. introv D. rewrite <- aevals_same in *. apply* aeval_hyp_in_impl.
Qed.

Lemma hyp_in_iter_aeval : hyp_in_iter aapply_check_step aapply_check aeval = aeval.
Proof.
  extens. intros t sigma r. iff D.
   sets_eq R: aeval. induction~ D; substs~.
    rewrite <- hyp_in_aeval. applys* hyp_in_impl aapply_check_impl H1.
    apply~ aapply_check_step_impl.
   applys hyp_in_iter_exact D.
Qed.

Section ParkWeak.

Variable aapply_check_step : (term → ast → ares → Prop) → name → ast → ares → Prop.
Variable aapply_check : (name → ast → ares → Prop) → name → ast → ares → Prop.

Hypothesis aapply_check_impl : forall (R1 R2 : term → ast → ares → Prop) n sigma r,
  (forall t sigma r, R1 t sigma r → R2 t sigma r) →
  aapply_check (aapply_check_step R1) n sigma r →
  aapply_check (aapply_check_step R2) n sigma r.

Definition Park_weak (R : term → ast → ares → Prop) :=
  forall t sigma r,
    R t sigma r →
    hyp_in (aapply_check (aapply_check_step (hyp_in_iter aapply_check_step aapply_check R)))
      t sigma r.

Lemma Park_weak_hyp_in_iter : forall R,
  Park_weak R →
  Park_weak (hyp_in_iter aapply_check_step aapply_check R).
Proof.
  introv Pw D. induction D.
   forwards* D: Pw. rewrite~ <- hyp_in_iter_iter in D.
   applys hyp_in_impl aapply_check_impl H1. introv D. forwards~ D': H0 D.
    applys~ hyp_in_iter_cons D'.
Qed.

End ParkWeak.

Lemma Park_weak_aeval : forall R,
  Park_weak aapply_check_step aapply_check R →
  forall t sigma r, R t sigma r → aeval t sigma r.
Proof.
  introv Pw D. forwards D': hyp_in_iter_exact (rm D).
  forwards P: Park_weak_hyp_in_iter aapply_check_impl Pw.
    apply~ aapply_check_step_impl.
  rewrite aevals_same. apply* aeval_f_cons.
  introv D. rewrite <- hyp_in_iter_iter.
   apply~ P.
   applys~ aapply_check_impl aapply_check_step_impl.
Qed.

Definition triple_union (R1 R2 : term → ast → ares → Prop) t sigma r :=
  R1 t sigma r ∨ R2 t sigma r.

Definition Park_weak_union (R0 R : term → ast → ares → Prop) :=
  forall t sigma r,
    R t sigma r →
    hyp_in (aapply_check (aapply_check_step (hyp_in_iter aapply_check_step aapply_check (triple_union R R0))))
      t sigma r.

Lemma Park_weak_union_aeval : forall (R0 R : term → ast → ares → Prop),
  (forall t sigma r, R0 t sigma r → aeval t sigma r) →
  Park_weak_union R0 R →
  forall t sigma r, R t sigma r → aeval t sigma r.
Proof.
  introv C P D. asserts D': (triple_union R aeval t sigma r). left~.
  applys Park_weak_aeval D'. clear t sigma r D D'.
  introv [D|D].
   forwards~ D': P D. applys hyp_in_impl aapply_check_impl D'.
    apply~ aapply_check_step_impl.
    clear - C. introv D. applys hyp_in_iter_impl aapply_check_impl D.
     apply~ aapply_check_step_impl.
     clear - C. introv [D|D]; [left~|right~].
   rewrite <- hyp_in_aeval in D. applys hyp_in_impl aapply_check_impl D.
    apply~ aapply_check_step_impl.
    clear - C. introv D. apply hyp_in_iter_exact. right~.
Qed.

End AbstractWorld.


Section Correctness.

(** * Correctness **)

Require Export lat_prop.
Require Export CompleteLattice.

Variable str : structure.
Let term := term str.
Let name := name str.
Let Rule_struct := Rule_struct term.
Let left (n : name) : term := left str n.
Let rule_struct (n : name) : Rule_struct := rule_struct str n.
Let semantics := semantics str.
Variable adom : domains.
Let ast := st adom.
Let ares := res adom.
Variable asem : semantics adom.
Let acond n (sigma : ast) := cond asem n sigma.
Let arule n : Rule ast ares := rule asem n.
Let aRule := Rule ast ares.
Variable dom : domains.
Let st := st dom.
Let res := res dom.
Variable sem : semantics dom.
Let cond n (sigma : st) := cond sem n sigma.
Let rule n : Rule st res := rule sem n.
Let Rule := Rule st res.

Hypothesis TC : rule_format_correct_all sem.

Variable gst : ast → st → Prop.
Variable gres : ares → res → Prop.


(** We could add as hypothesis for every condition underneath that
  [cond] is respected, but I don’t think that it would really be
  necessary in practise: abstract transfert functions are usually more
  or less close to the concrete ones.  Thus would further more
  complexify the definition of this predicate. **)

Inductive propagates : (℘ ast) → (℘ st) → aRule → Rule → Prop :=
  | propagates_Ax : forall (cond acond : ℘ _)  ax aax,
      (forall sigma asigma r ar,
        cond sigma →
        acond asigma →
        gst asigma sigma →
        ax sigma = Some r →
        aax asigma = Some ar →
        gres ar r) →
      propagates acond cond (Rule_Ax aax) (Rule_Ax ax)
  | propagates_R1 : forall (cond acond : ℘ _) up aup,
      (forall sigma asigma sigma' asigma',
        cond sigma →
        acond asigma →
        gst asigma sigma →
        up sigma = Some sigma' →
        aup asigma = Some asigma' →
        gst asigma' sigma') →
      propagates acond cond (Rule_R1 _ aup) (Rule_R1 _ up)
  | propagates_R2 : forall (cond acond : ℘ _) up aup next anext,
      (forall sigma asigma sigma' asigma',
        cond sigma →
        acond asigma →
        gst asigma sigma →
        up sigma = Some sigma' →
        aup asigma = Some asigma' →
        gst asigma' sigma') →
      (forall sigma asigma r ar sigma' asigma',
        cond sigma →
        acond asigma →
        gst asigma sigma →
        gres ar r →
        next sigma r = Some sigma' →
        anext asigma ar = Some asigma' →
        gst asigma' sigma') →
      propagates acond cond (Rule_R2 aup anext) (Rule_R2 up next)
  .

Hypothesis Pr : forall n, propagates (acond n) (cond n) (arule n) (rule n).

Lemma format_correct_conserved : rule_format_correct_all asem.
Proof.
  intro n. lets P: Pr n. lets T: TC n.
  fold (rule_struct n) in *. fold (arule n). fold (rule n) in T.
  inverts T; rewrite_term (rule n); inverts P; constructors.
Qed.


Hypothesis acond_correct : forall n asigma sigma,
  gst asigma sigma →
  cond n sigma →
  acond n asigma.


(** Correctness of the glue. See in Section Glue for some instances. **)

Variable glue : glue_type str adom.

Definition correct_up_to_depth k n asigma ar :=
  forall sigma r (A : apply sem n sigma r),
    gst asigma sigma →
    cond n sigma →
    apply_depth A < k →
    gres ar r.

Definition glue_correct := forall n P asigma ar k,
  glue n P asigma ar →
  (forall asigma' ar',
    P asigma' ar' →
    correct_up_to_depth k n asigma' ar') →
  correct_up_to_depth k n asigma ar.

Hypothesis Cglue : glue_correct.


Theorem correctness : forall t asigma ar,
  aeval asem glue t asigma ar →
  forall sigma r,
    gst asigma sigma →
    eval sem t sigma r →
    gres ar r.
Proof.
  introv aD G D. sets_eq k': (eval_depth D). asserts I: (k' < S k'); [ math |].
  rewrite EQk' in I at 1. clear EQk'. gen t asigma sigma r ar. induction (S k') as [|k].
   math.
   introv G I aD. destruct D as [? ? ? ? E C A] eqn: ED. inverts aD as allBranches.
    forwards~ aA: (rm allBranches) E.
      apply* acond_correct.
    inverts aA as Gl aA. forwards~: Cglue Gl A I. introv OK G0 C0 I0. forwards aA': aA OK.
    forwards Prn: Pr n. destruct A0 as
        [ n ax sigma0 r0 E1 E2 E3
        | n t0 up sigma0 sigma1 r1 E1 E2 E3 D0
        | n t1 t2 up next sigma0 sigma1 sigma2 r1 r2 E1 E2 E3 D1 E4 D2] eqn: E0;
      destruct aA' as
        [ n aax asigma0 ar0 aE1 aE2 aE3
        | n at0 aup asigma0 asigma1 ar1 aE1 aE2 aE3 aD0
        | n at1 at2 aup anext asigma0 asigma1 asigma2 ar1 ar2 aE1 aE2 aE3 aD1 aE4 aD2]
        eqn: aE0; rewrite_term (rule n); rewrite_term (arule n); inverts Prn as P1 P2;
        lets E1': E1; rewrite aE1 in E1'; inverts E1'.
    (** Axiom **)
    apply* P1.
    (** Type 1 Rule **)
    applys~ IHk D0 aD0.
     apply* P1.
     unfolds apply_depth. unfolds eval_depth. math.
    (** Type 2 Rule **)
    asserts Lem: ((max (eval_depth D1) (eval_depth D2) < k)%nat).
      unfolds apply_depth. unfolds eval_depth. math.
    rewrite Nat.max_lub_lt_iff in Lem. lets (I1&I2): (rm Lem).
    applys~ IHk D2 aD2.
     apply* P2.
      applys~ IHk D1 aD1.
       apply* P1.
       math.
      math.
Qed.

End Correctness.


(** * Some Additional Generic Definitions **)

Section Abstract.

(** ** Best Abstraction **)

Section BestAcond.

Variable str : structure.
Let term := term str.
Let name := name str.
Let Rule_struct := Rule_struct term.
Let left (n : name) : term := left str n.
Let rule_struct (n : name) : Rule_struct := rule_struct str n.
Variable dom : domains.
Let st := st dom.
Let res := res dom.
Let semantics := semantics str dom.
Variables ast ares : Type.
Let aRule := Rule ast ares.
Variable sem : semantics.
Let cond n (sigma : st) := cond sem n sigma.
Let rule n : Rule st res := rule sem n.
Let Rule := Rule st res.

Hypothesis Lst : Poset.t ast.

Variable gst : ast → st → Prop.
Hypothesis Gst_monotone : ∀ N1 N2 : ast, N1 ⊑ N2 → gst N1 ⊑ gst N2.

(** ** A generic definition of [acond] as being the best abstraction of [cond]. **)

Definition best_acond n asigma :=
  exists sigma,
    gst asigma sigma
    ∧ cond n sigma.

Lemma best_acond_correct : forall n asigma sigma,
  gst asigma sigma →
  cond n sigma →
  best_acond n asigma.
Proof. introv G C. unfolds. autos*. Qed.

Lemma best_acond_monotone : forall n (sigma sigma' : ast),
   sigma' ⊑ sigma →
   best_acond n sigma' → best_acond n sigma.
Proof.
  introv I (csigma&G&C). exists csigma. splits~.
  applys~ Gst_monotone I.
Qed.

End BestAcond.


Section Glue.

Variable str : structure.
Let term := term str.
Let name := name str.
Let Rule_struct := Rule_struct term.
Let left (n : name) : term := left str n.
Let rule_struct (n : name) : Rule_struct := rule_struct str n.
Let semantics := semantics str.
Variable adom : domains.
Let ast := st adom.
Let ares := res adom.
Variable asem : semantics adom.
Let acond n (sigma : ast) := cond asem n sigma.
Let arule n : Rule ast ares := rule asem n.
Let aRule := Rule ast ares.
Variable dom : domains.
Let st := st dom.
Let res := res dom.
Variable sem : semantics dom.
Let cond n (sigma : st) := cond sem n sigma.
Let rule n : Rule st res := rule sem n.
Let Rule := Rule st res.

Variable gst : ast → st → Prop.
Variable gres : ares → res → Prop.


(** ** About Glues **)

(** This predicates states that [glue1] accepts more operations than [glue2]. **)

Definition glue_contains (glue1 glue2 : glue_type str adom) :=
  forall n P asigma ar,
    glue2 n P asigma ar →
    glue1 n P asigma ar.

Lemma glue_contains_refl : forall glue,
  glue_contains glue glue.
Proof. introv; intro; intros. autos*. Qed.

Lemma glue_contains_trans : forall glue1 glue2 glue3,
  glue_contains glue1 glue2 →
  glue_contains glue2 glue3 →
  glue_contains glue1 glue3.
Proof. introv C1 C2; intro; intros. autos*. Qed.

Lemma glue_contains_aeval : forall glue1 glue2,
  glue_contains glue1 glue2 → forall asigma t ar,
    aeval asem glue2 asigma t ar →
    aeval asem glue1 asigma t ar.
Proof.
  introv GC. cofix F. introv D. inverts D as A. constructors.
  introv E C. forwards aA: A E C. inverts aA as G AS. constructors.
   applys GC G.
   introv OK. forwards AS': AS OK. inverts AS' as E1 E2 E3 D1 E4 D2.
    applys* aapply_step_Ax.
    forwards F1: F D1. applys* aapply_step_R1 F1.
    forwards F1: F D1. forwards F2: F D2. applys* aapply_step_R2 F1 F2.
Qed.

Lemma glue_contains_aapply_step : forall glue1 glue2,
  glue_contains glue1 glue2 → forall n asigma ar,
    aapply_step asem glue2 n asigma ar →
    aapply_step asem glue1 n asigma ar.
Proof.
  introv GC AS. inverts AS as E1 E2 E3 D1 E4 D2.
   applys* aapply_step_Ax.
   forwards* F1: glue_contains_aeval D1. applys* aapply_step_R1 F1.
   forwards* F1: glue_contains_aeval D1. forwards* F2: glue_contains_aeval D2.
    applys* aapply_step_R2 F1 F2.
Qed.

Lemma glue_contains_aapply : forall glue1 glue2,
  glue_contains glue1 glue2 → forall n asigma ar,
    aapply asem glue2 n asigma ar →
    aapply asem glue1 n asigma ar.
Proof.
  introv GC A. inverts A as G AS. constructors.
   applys GC G.
   introv OK. forwards AS': AS OK. apply* glue_contains_aapply_step.
Qed.

Lemma glue_contains_aapply_check : forall glue1 glue2,
  glue_contains glue1 glue2 → forall n asigma ar aapply_check_step,
    aapply_check glue2 aapply_check_step n asigma ar →
    aapply_check glue1 aapply_check_step n asigma ar.
Proof. introv GC A. inverts A as G AS. applys aapply_check_cons AS. applys GC G. Qed.


(** This predicates states that [glue] is iterable, i.e., that we can
  apply the given glue several times at a row and staying in the
  glue. **)

Definition glue_iterable (glue : glue_type str adom) :=
  forall n (P1 P2 : ast → ares → Prop) asigma3 ar3,
    (forall asigma2 ar2, P2 asigma2 ar2 → glue n P1 asigma2 ar2) →
    glue n P2 asigma3 ar3 →
    glue n P1 asigma3 ar3.

Definition glue_refl (glue : glue_type str adom) :=
  forall n (P : ast → ares → Prop) asigma ar,
    P asigma ar →
    glue n P asigma ar.


(** If a glue requires more conditions to accept a conclusion than a
  correct glue, then this bigger glue is of course still correct.  The
  iteration requires to the glue to be invariant to such a
  transformation.  This is expressed by the predicate
  [glue_more_premisses glue1 glue2]: [glue1] always requires more
  premisses than [glue1]. **)

Definition glue_more_premisses (glue1 glue2 : glue_type str adom) :=
  forall n P asigma ar,
    glue1 n P asigma ar →
    exists (P' : ast → ares → Prop),
      (forall asigma ar,
        P' asigma ar → P asigma ar)
      ∧ glue2 n P' asigma ar.

Lemma glue_more_premisses_refl : forall glue,
  glue_more_premisses glue glue.
Proof. introv G. exists~ P. Qed.

Lemma glue_more_premisses_trans : forall glue1 glue2 glue3,
  glue_more_premisses glue1 glue2 →
  glue_more_premisses glue2 glue3 →
  glue_more_premisses glue1 glue3.
Proof. introv GS12 GS23 G. forwards (P1&I1&G1): GS12 G. forwards* (P2&I2&G2): GS23 G1. Qed.


(** A glue which requires more than another correct glue is itself correct. **)

Lemma glue_more_premisses_correct : forall glue1 glue2,
  glue_more_premisses glue1 glue2 →
  glue_correct sem gst gres glue2 →
  glue_correct sem gst gres glue1.
Proof. introv O C G OK. forwards (P'&F&G'): O G. applys* C G'. Qed.


(** Such a glue accept to be given more premisses. **)

Definition glue_accepts_more_premisses (glue : glue_type str adom) :=
  forall n (P P' : ast → ares → Prop) asigma ar,
    glue n P asigma ar →
    (forall asigma ar,
      P asigma ar → P' asigma ar) →
    glue n P' asigma ar.


End Glue.

End Abstract.


Section Glue.

(** * Instantiations of the Glue **)

Variable str : structure.
Let term := term str.
Let name := name str.
Let Rule_struct := Rule_struct term.
Let left (n : name) : term := left str n.
Let rule_struct (n : name) : Rule_struct := rule_struct str n.
Let semantics := semantics str.
Variable adom : domains.
Let ast := st adom.
Let ares := res adom.
Variable asem : semantics adom.
Let acond n (sigma : ast) := cond asem n sigma.
Let arule n : Rule ast ares := rule asem n.
Let aRule := Rule ast ares.
Variable dom : domains.
Let st := st dom.
Let res := res dom.
Variable sem : semantics dom.
Let cond n (sigma : st) := cond sem n sigma.
Let rule n : Rule st res := rule sem n.
Let Rule := Rule st res.

Variable gst : ast → st → Prop.
Variable gres : ares → res → Prop.

Let glue_type := glue_type str adom.


(** The following function computes the glue in practise. It is the
  one required by the analysers in file [Analysers.v]. **)

Hypothesis Cast : Comparable ast.
Hypothesis Cares : Comparable ares.

Class glue_computable (glue : glue_type) := glue_computable_make {
    glue_compute : name → list (ast * ares) → ast → option (list (ast * ares) * ares) ;
    glue_compute_spec : forall srs0 n sigma srs r,
      glue_compute n srs0 sigma = Some (srs, r) →
      glue n (fun asigma' r' => mem (asigma', r') srs) sigma r
    (** Note that [srs] is not required to be included in [srs0]: to
      compute a glue, we are allowed to provide intermediary steps. **)
  }.

Arguments glue_compute glue {glue_computable} _ _ _.


(** ** Glue Closure **)

Section Closure.

(** A closure by the [glue_more_premisses] relation. This glue is the
  same as the given glue, but accepts more (useless) hypothesis. It is
  useful in combination with [glue_iter]. **)

Variables glue : glue_type.

Inductive glue_closure : glue_type :=
  | glue_closure_intro : forall n (P' P : ast → ares → Prop) asigma ar,
      glue n P asigma ar →
      (forall asigma ar, P asigma ar → P' asigma ar) →
      glue_closure n P' asigma ar
  .

Lemma glue_closure_more_premisses : glue_more_premisses glue glue_closure.
Proof. introv G. exists P. splits~. constructors*. Qed.

Hypothesis glue0_correct : glue_correct sem gst gres glue.

Lemma glue_closure_correct : glue_correct sem gst gres glue_closure.
Proof. introv Gl CA. inverts Gl as Gl I. applys~ glue0_correct Gl. Qed.

(** As there is no way to know which subset of premisses [glue] needs,
  the computability has to be proven in a case-base manner. **)

Lemma glue_contains_closure : glue_contains glue_closure glue.
Proof. introv Gl. constructors*. Qed.

Lemma glue_closure_accepts_more_premisses : glue_accepts_more_premisses glue_closure.
Proof. introv G I. inverts G as G I'. constructors*. Qed.

(** The coherence with [acond] is by definition not conserved. **)

End Closure.

Global Instance glue_closure_computable : forall glue,
    glue_computable (glue_closure glue) →
    glue_computable (glue_closure (glue_closure glue)).
  introv C. applys~ glue_computable_make (glue_compute (glue_closure glue)).
  introv Eq. apply glue_compute_spec in Eq. constructors*.
Defined.


(** ** Sum Glue **)

Section Sum.

(** Usually, we want more than just one glue: we would like to be able
  to both perform weakenings and filterings, for instance. This
  general glue is composed of two glues. It is then possible to
  compound a third one to the compound, and so on. Note that this glue
  only allows to apply one rule: it is usually a good idea to combine
  it with the iterating glue defined below. **)

Variables glue1 glue2 : glue_type.

Inductive glue_sum : glue_type :=
  | glue_sum_left : forall n P asigma ar,
      glue1 n P asigma ar →
      glue_sum n P asigma ar
  | glue_sum_right : forall n P asigma ar,
      glue2 n P asigma ar →
      glue_sum n P asigma ar
  .

Hypothesis glue1_correct : glue_correct sem gst gres glue1.
Hypothesis glue2_correct : glue_correct sem gst gres glue2.

Lemma glue_sum_correct : glue_correct sem gst gres glue_sum.
Proof.
  introv Gl CA G C I. inverts Gl as Gl.
   applys~ glue1_correct Gl G I.
   applys~ glue2_correct Gl G I.
Qed.

Hypothesis glue1_computable : glue_computable glue1.
Hypothesis glue2_computable : glue_computable glue2.

(** The following instance checks first whether [glue1] has a
  solution, then [glue2]: it is important to put the most precise
  instance left. **)

Global Instance glue_sum_computable : glue_computable glue_sum.
  applys glue_computable_make (fun n srs sigma =>
    match glue_compute glue1 n srs sigma with
    | Some srsr => Some srsr
    | None => glue_compute glue2 n srs sigma
    end).
  introv Eq. destruct glue_compute eqn: E.
   inverts Eq. apply glue_compute_spec in E. apply~ glue_sum_left.
   apply glue_compute_spec in Eq. apply~ glue_sum_right.
Defined.

Lemma glue_contains_sum_left : glue_contains glue_sum glue1.
Proof. introv Gl. apply~ glue_sum_left. Qed.

Lemma glue_contains_sum_right : glue_contains glue_sum glue2.
Proof. introv Gl. apply~ glue_sum_right. Qed.

Hypothesis glue1_acond : forall n P asigma ar asigma' ar',
  glue1 n P asigma ar →
  acond n asigma →
  P asigma' ar' →
  acond n asigma'.
Hypothesis glue2_acond : forall n P asigma ar asigma' ar',
  glue2 n P asigma ar →
  acond n asigma →
  P asigma' ar' →
  acond n asigma'.

Lemma glue_sum_acond : forall n P asigma ar asigma' ar',
  glue_sum n P asigma ar →
  acond n asigma →
  P asigma' ar' →
  acond n asigma'.
Proof.
  introv G C I. inverts G as G.
   applys~ glue1_acond G C I.
   applys~ glue2_acond G C I.
Qed.

End Sum.


(** ** Iterating Glue **)

Section Iter.

(** The above glues only allow to apply a gluing rule once. We usually
  want some more for convenience. This glue generalises this principle
  with any finite amount of glue. Note that it would not work in
  general for an infinite amount, although the abstract semantics is
  defined coinductively. **)

Variable glue : glue_type.

Inductive glue_iter : glue_type :=
  | glue_iter_refl : forall n (P : ast → ares → Prop) asigma ar,
      P asigma ar →
      glue_iter n P asigma ar
  | glue_iter_cons : forall n (P1 P2 : ast → ares → Prop) asigma3 ar3,
      (forall asigma2 ar2, P2 asigma2 ar2 → glue_iter n P1 asigma2 ar2) →
      glue n P2 asigma3 ar3 →
      glue_iter n P1 asigma3 ar3
  .

Hypothesis glue0_correct : glue_correct sem gst gres glue.

Lemma glue_iter_correct : glue_correct sem gst gres glue_iter.
Proof. introv Gl CA. induction* Gl. Qed.

Hypothesis glue0_computable : glue_computable glue.

Hypothesis glue0_closure_accepts_more_premisses : glue_accepts_more_premisses glue.

Lemma glue_iter_accepts_more_premisses : glue_accepts_more_premisses glue_iter.
Proof.
  introv Gl I. induction Gl.
   applys~ glue_iter_refl.
   applys* glue_iter_cons.
Qed.

(** Remember how the result of a glue computation may contain
  additional materials. These materials may have to be proven
  again. This glue iterates the search until all the materials are in
  the original set. It is parameterised by some fuel, as this may not
  terminate. If it runs out of fuel, it returns the currently computed
  required triples. They still follow the specification of
  [glue_computable], but may not be included in the original set. **)

Fixpoint glue_iter_compute fuel n srs asigma :=
  match fuel with
  | 0 => None
  | S fuel =>
    LibOption.map_on (glue_compute glue n srs asigma) (fun srsr' : _ * _ =>
      let (srs', ar) := srsr' in
      'let good := filter (fun sr => mem sr srs) srs' in
      'let bad := filter (fun sr => ! mem sr srs) srs' in
      'let deal_with_bad :=
        map (fun sr : _ * _ =>
          let (sigma, r) := sr in
          match glue_iter_compute fuel n srs sigma with
          | None => (sr :: nil)
          | Some (srs', r') =>
            ifb r = r' then srs'
            (** In case where [r ≠ r'], there is not much to do other
              than understand how the given glue works. To avoid this,
              have a glue which reuse the exact the result in the
              hints as much as possible. **)
            else (sr :: nil)
          end) bad in
      (Remove_duplicates (good ++ concat deal_with_bad), ar))
  end.

Global Instance glue_iter_computable (fuel : nat) : glue_computable glue_iter.
  applys glue_computable_make (glue_iter_compute fuel).
  induction fuel; introv E; inverts E as E.
  forwards ((srs'&ar)&G&E'): map_on_inv (rm E). repeat let_name. inverts E'.
  apply glue_compute_spec in G. asserts Lem: (forall sr, mem sr srs' = mem sr (good ++ bad)).
    introv. rewrite mem_app. substs. do 2 rewrite filter_mem_eq.
    destruct (mem sr srs0); rew_bool~. typeclass.
  applys glue_iter_cons G. introv I. rewrite Lem in I.
  rewrite mem_app in I. rew_refl in I. inverts I as I.
   apply glue_iter_refl. rewrite Remove_duplicates_mem. rewrite mem_app. rew_refl*.
   rewrite EQbad in I. rewrite filter_mem_eq in I. rew_refl in I. rewrite Lem in I.
    rewrite mem_app in I. rew_refl in I. lets ([I1|I1]&NI): (rm I).
     apply glue_iter_refl. rewrite Remove_duplicates_mem. rewrite mem_app. rew_refl*.
     destruct (glue_iter_compute fuel n srs0 asigma2) eqn: E.
      destruct p as [srs3 ar3]. forwards G': IHfuel E. tests E': (ar3 = ar2).
       applys~ glue_iter_accepts_more_premisses G'. introv I. rewrite Remove_duplicates_mem.
        rewrite mem_app. rew_refl. right. rewrite concat_mem. exists srs3. splits~.
        rewrite EQdeal_with_bad. rewrite map_mem. exists (asigma2, ar2). splits*.
        rewrite E. case_if*.
       apply glue_iter_refl. rewrite Remove_duplicates_mem. rewrite mem_app. rew_refl. right.
        rewrite concat_mem. exists ((asigma2, ar2) :: nil). splits.
         rewrite EQdeal_with_bad. rewrite map_mem. rew_refl. exists (asigma2, ar2). splits*.
          rewrite E. case_if~. substs*.
         simpl. rew_refl*.
      apply glue_iter_refl. rewrite Remove_duplicates_mem. rewrite mem_app. rew_refl. right.
       rewrite concat_mem. exists ((asigma2, ar2) :: nil). splits.
        rewrite EQdeal_with_bad. rewrite map_mem. rew_refl. exists (asigma2, ar2). splits*.
         rewrite~ E.
        simpl. rew_refl*.
Defined.

Lemma glue_contains_iter : glue_contains glue_iter glue.
Proof. introv Gl. applys~ glue_iter_cons Gl. introv I. applys~ glue_iter_refl I. Qed.

Lemma glue_iterable_iter : glue_iterable glue_iter.
Proof. introv G1 G2. induction~ G2. applys~ glue_iter_cons H1. Qed.

Lemma glue_refl_iter : glue_refl glue_iter.
Proof. introv G. applys~ glue_iter_refl G. Qed.

End Iter.

Lemma glue_closure_iter : forall glue n P asigma ar,
  glue_closure (glue_iter glue) n P asigma ar →
  glue_iter (glue_closure glue) n P asigma ar.
Proof.
  introv I. inverts I as Gl I. induction Gl.
    applys* glue_iter_refl.
    applys* glue_iter_cons. constructors*.
Qed.

Global Instance glue_closure_iter_computable : forall glue,
    glue_computable (glue_closure (glue_iter glue)) →
    glue_computable (glue_iter (glue_closure glue)).
  introv C. applys glue_computable_make (glue_compute (glue_closure (glue_iter glue))).
  introv E. apply glue_compute_spec in E. applys glue_closure_iter E.
Defined.


(** ** Axiomatic Glue **)

Section Axiomatic.

(** There are times where we know that some triples are correct, but
  can’t easily match a derivation. This glue allows to add such
  triples as-is in the semantics. **)

Variable axiom : name → ast → ares → Prop.
Hypothesis axiom_correct : forall n asigma ar sigma r,
  axiom n asigma ar →
  gst asigma sigma →
  cond n sigma →
  gres ar r.

Inductive axiomatic_glue : glue_type :=
  | axiomatic_glue_intro : forall n asigma ar,
      axiom n asigma ar →
      axiomatic_glue n (fun _ _ => False) asigma ar
  .

Lemma axiomatic_glue_correct : glue_correct sem gst gres axiomatic_glue.
Proof. introv Gl CA G C Aa. inverts Gl. apply* axiom_correct. Qed.

Hypothesis axiom_partially_pickable : forall n asigma, Partially_pickable (axiom n asigma).

Global Instance axiomatic_glue_computable : glue_computable axiomatic_glue.
  apply~ (glue_computable_make axiomatic_glue (fun n srs asigma =>
    LibOption.map_on (try_to_pick (axiom n asigma)) (fun ar => (nil, ar)))).
  introv _ E. forwards (ar&E1&E2): map_on_inv E. inverts E2. apply try_to_pick_correct in E1.
  erewrite prop_ext_2 with (P := fun asigma' r' => istrue (mem (asigma', r') nil)).
   applys~ axiomatic_glue_intro E1.
   introv. iff*.
Defined.

End Axiomatic.


(** ** Constant Glue **)

Section Constant.

(** Not all glue use the full power of the [P] set of premisses. Most
  are just accepting singletons. The construction [const_glue]
  converts such a glue into an expected glue. **)

Variable glue : name → ast → ares → ast → ares → Prop.

Inductive const_glue : glue_type :=
  | const_glue_intro : forall n asigma' ar' asigma ar,
      glue n asigma' ar' asigma ar →
      const_glue n (fun asigma'' ar'' => asigma'' = asigma' ∧ ar'' = ar') asigma ar
  .

Lemma const_glue_correct :
  (forall asigma' ar' asigma ar k n,
    glue n asigma' ar' asigma ar →
    correct_up_to_depth adom sem gst gres k n asigma' ar' →
    correct_up_to_depth adom sem gst gres k n asigma ar) →
  glue_correct sem gst gres const_glue.
Proof. introv OK G OK'. inverts G as G. applys OK G. apply~ OK'. Qed.

Hypothesis glue_partially_pickable : forall n asigma ar asigma',
  Partially_pickable (glue n asigma ar asigma').

Global Instance const_glue_computable : glue_computable const_glue.
  applys glue_computable_make (fun n srs sigma =>
    match srs with
    | (asigma, ar) :: nil =>
      LibOption.map_on (try_to_pick (glue n asigma ar sigma)) (fun ar => (srs, ar))
    | _ => None
    end).
  introv Eq. destruct srs0 as [|[asigma ar] ()]; inverts Eq as Eq.
  forwards (ar'&E1&E2): LibOption.map_on_inv Eq. inverts E2. apply try_to_pick_correct in E1.
  erewrite prop_ext_2 with (P := fun asigma' r' => istrue (mem (asigma', r') ((asigma, ar) :: nil))).
   applys~ const_glue_intro E1.
   introv. simpl. rew_refl. iff I; repeat inverts I as I; substs~.
Defined.

Global Instance const_glue_closure_computable : glue_computable (glue_closure const_glue).
  applys glue_computable_make (fun n srs sigma =>
    'let srs' :=
      LibList.map (fun sr : _ * _ =>
        let (asigma, ar) := sr in
        LibOption.map (fun r => (sr :: nil, r)) (try_to_pick (glue n asigma ar sigma))) srs in
    LibOption.unsome_default None (list_take_first (fun o => decide (o ≠ None)) srs')).
  introv Eq. let_name. destruct list_take_first eqn: E; inverts Eq as Eq.
  forwards (I&D): list_take_first_prop (rm E). rew_refl in D. destruct o; tryfalse.
  subst srs'. rewrite map_mem with (CA := _) in I. lets ((asigma&ar)&Isr&Esr): (rm I).
  symmetry in Esr. forwards (ar'&Gl&E): LibOption.map_inv (rm Esr). subst p. inverts Eq.
  apply try_to_pick_correct in Gl. constructors.
   applys~ const_glue_intro Gl.
   introv (E1&E2). simpl. rew_refl*.
Defined.

End Constant.


(** ** Trivial Glue **)

Section Trivial.

(** A glue which does nothing. **)

Inductive glue_trivial_constr : name → ast → ares → ast → ares → Prop :=
  | glue_trivial_intro : forall n asigma ar,
      glue_trivial_constr n asigma ar asigma ar
  .

Definition glue_trivial := const_glue glue_trivial_constr.

Lemma glue_trivial_iterable : glue_iterable glue_trivial.
Proof. introv F G. do 2 inverts G as G. forwards* G: F. Qed.

Lemma glue_trivial_correct : glue_correct sem gst gres glue_trivial.
Proof. apply const_glue_correct. introv G. inverts~ G. Qed.

Global Instance glue_trivial_pickable_option : forall n asigma ar asigma',
    Pickable_option (glue_trivial_constr n asigma ar asigma').
  introv. applys pickable_option_make (ifb asigma = asigma' then Some ar else None).
   introv Eq. case_if as I; inverts Eq. substs. constructors~.
   introv (ar'&G). inverts G. exists ar'. case_if*.
Defined.

Global Instance glue_trivial_computable : glue_computable glue_trivial.
  typeclass.
Defined.

Lemma glue_trivial_acond : forall n asigma1 ar1 asigma2 ar2,
  glue_trivial_constr n asigma1 ar1 asigma2 ar2 →
  acond n asigma2 →
  acond n asigma1.
Proof. introv G. inverts* G. Qed.

Lemma glue_contains_iter_trivial : forall glue,
  glue_contains (glue_iter glue) glue_trivial.
Proof. introv Gl. do 2 inverts Gl as Gl. apply~ glue_iter_refl. Qed.

End Trivial.


(** ** Weakening Glue **)

Section Weaken.

Hypothesis Lst : Poset.t ast.
Hypothesis Lr : Poset.t ares.

Hypothesis Gst_monotone : ∀ N1 N2 : ast, N1 ⊑ N2 → gst N1 ⊑ gst N2.
Hypothesis Gres_monotone : ∀ N1 N2 : ares, N1 ⊑ N2 → gres N1 ⊑ gres N2.

(** This glue allows to perform some approximations along the
  derivation. **)

Inductive glue_weaken_constr : name → ast → ares → ast → ares → Prop :=
  | glue_weaken_intro : forall n asigma ar asigma' ar',
      asigma ⊑ asigma' →
      ar' ⊑ ar →
      glue_weaken_constr n asigma' ar' asigma ar
  .

Definition glue_weaken := const_glue glue_weaken_constr.

Lemma glue_weaken_correct : glue_correct sem gst gres glue_weaken.
Proof.
  apply const_glue_correct. introv Gl CA G C I. inverts Gl as O1 O2.
  applys Gres_monotone O2. applys CA C I. applys~ Gst_monotone O1.
Qed.

Hypothesis ast_lat_partially_decidable : forall sigma sigma' : ast,
  PartiallyDecidable (sigma ⊑ sigma').

Global Instance glue_weaken_partially_pickable : forall n asigma' ar' asigma,
    Partially_pickable (glue_weaken_constr n asigma' ar' asigma).
  introv. applys partially_pickable_make
    (if try_to_decide (asigma ⊑ asigma') then Some ar' else None).
  introv Eq. case_if as I; inverts Eq. fold_bool. apply try_to_decide_spec in I.
  constructors~.
Defined.

Global Instance glue_weaken_computable : glue_computable glue_weaken.
  typeclass.
Defined.

Lemma glue_contains_weaken_trivial : glue_contains glue_weaken glue_trivial.
Proof. introv Gl. do 2 inverts Gl as Gl. do 2 constructors*. Qed.

Lemma glue_weaken_iterable : glue_iterable glue_weaken.
Proof.
  introv F G. inverts G as G. inverts G as O1 O2. forwards* G: F.
  inverts G as G. inverts G as O3 O4. do 2 constructors*.
Qed.

Hypothesis acond_monotone : forall n sigma sigma',
  sigma' ⊑ sigma →
  acond n sigma' →
  acond n sigma.

Lemma glue_weaken_acond : forall n asigma1 ar1 asigma2 ar2,
  glue_weaken_constr n asigma1 ar1 asigma2 ar2 →
  acond n asigma2 →
  acond n asigma1.
Proof. introv G C. inverts G as O1 O2. applys acond_monotone O1 C. Qed.

End Weaken.


(** ** Filtering Glue *)

Section Filter.

(** This glue allows to derives some informations from the applied
  rule. For instance, if the current term is [if x s1 s2] and the
  applied rule is [if-true] (with their natural definitions), then we
  can infer that [x] is [true] here. **)

Variable filter : name → ast → option ast.

Hypothesis filter_correct : forall n asigma asigma' sigma,
  filter n asigma = Some asigma' →
  gst asigma sigma →
  gst asigma' sigma.

Inductive glue_filter_constr : name → ast → ares → ast → ares → Prop :=
  | glue_filter_intro : forall n asigma ar asigma',
      filter n asigma = Some asigma' →
      glue_filter_constr n asigma' ar asigma ar
  .

Definition glue_filter := const_glue glue_filter_constr.

Lemma glue_filter_correct : glue_correct sem gst gres glue_filter.
Proof.
  apply const_glue_correct. introv Gl CA G C I. inverts Gl as E. applys CA C I.
  applys filter_correct E G.
Qed.

Global Instance glue_filter_pickable_option : forall n asigma' ar' asigma,
    Pickable_option (glue_filter_constr n asigma' ar' asigma).
  introv. destruct (filter n asigma) eqn: E.
   applys pickable_option_make (ifb a = asigma' then Some ar' else None).
    introv Eq. case_if. inverts Eq. substs. constructors~.
    introv (a'&G). inverts G as E'. rewrite E' in E. inverts E.
     exists a'. case_if*.
   applys pickable_option_make (@None ares).
    introv A. inverts A.
    introv (a&A). inverts A as E'. rewrite E' in E. inverts E.
Defined.

Global Instance glue_filter_computable : glue_computable glue_filter.
  typeclass.
Defined.

Hypothesis acond_filter : forall n asigma1 asigma2,
  acond n asigma1 →
  filter n asigma1 = Some asigma2 →
  acond n asigma2.

Lemma glue_filter_acond : forall n asigma1 ar1 asigma2 ar2,
  glue_filter_constr n asigma1 ar1 asigma2 ar2 →
  acond n asigma2 →
  acond n asigma1.
Proof. introv G C. inverts G as E. applys acond_filter C E. Qed.

Hypothesis acond_correct : forall n asigma sigma,
  gst asigma sigma →
  cond n sigma →
  acond n asigma.

Lemma acond_complete_acond_filter :
  (forall n asigma,
    acond n asigma →
    exists sigma, cond n sigma ∧ gst asigma sigma) → forall n asigma1 asigma2,
  acond n asigma1 →
  filter n asigma1 = Some asigma2 →
  acond n asigma2.
Proof.
  introv Complete C E. forwards (sigma&Csigma&G): Complete C.
  applys acond_correct Csigma. applys~ filter_correct E G.
Qed.

End Filter.


(** ** Framing Glue **)

Section Frame.

Variable context : Type.

(** The intuitive meaning of [frame_ast asigma' c asigma] is that
  [asigma'] taken in the context [c] is the same than
  [asigma]. [frame_ares] works in the same way. **)
Variable frame_ast : ast → context → ast → Prop.
Variable frame_ares : ares → context → ares → Prop.

Inductive glue_frame_constr : name → ast → ares → ast → ares → Prop :=
  | glue_frame_cons : forall n sigma sigma' r r' c,
      frame_ast sigma' c sigma →
      frame_ares r' c r →
      glue_frame_constr n sigma' r' sigma r
  .

Definition glue_frame := const_glue glue_frame_constr.

(** Correctness has to be proven on a case-based manner. **)

Hypothesis frame_ast_decidable : forall asigma c asigma',
  Decidable (frame_ast asigma' c asigma).
Hypothesis frame_ares_partially_pickable : forall ar' c,
  Partially_pickable (frame_ares ar' c).

(** To be partially pickable, this glue asks for an oracle able to
  provide a context. The goal would be that if [oracle asigma' asigma]
  returns [c], then [frame_ast asigma' c asigma]. The computation
  checks it anyway, so it has not to be proven. **)
Variable oracle : ast → ast → option context.

Global Instance glue_frame_partially_pickable : forall n asigma ar' asigma',
    Partially_pickable (glue_frame_constr n asigma' ar' asigma).
  introv. destruct (oracle asigma' asigma) eqn: E.
   destruct (decide (frame_ast asigma' c asigma)) eqn: B.
    (** The oracle was right. **)
    fold_bool. rew_refl in B. applys partially_pickable_make (try_to_pick (frame_ares ar' c)).
    introv Eq. apply try_to_pick_correct in Eq. constructors*.
    (** The oracle was wrong, we can fail. **)
    apply partially_pickable_false.
   (** No clues are given, we can fail. **)
   apply partially_pickable_false.
Defined.

Global Instance glue_frame_computable : glue_computable glue_frame.
  typeclass.
Defined.

Hypothesis frame_ast_acond : forall n asigma C asigma',
  frame_ast asigma C asigma' →
  acond n asigma' →
  acond n asigma.

Lemma glue_frame_acond : forall n asigma1 ar1 asigma2 ar2,
  glue_frame_constr n asigma1 ar1 asigma2 ar2 →
  acond n asigma2 →
  acond n asigma1.
Proof. introv G C. inverts G as F1 F2. applys frame_ast_acond F1 C. Qed.

End Frame.

End Glue.


Section Miscellanous.

Variable str : structure.
Let term := term str.
Let name := name str.
Let Rule_struct := Rule_struct term.
Let left (n : name) : term := left str n.
Let rule_struct (n : name) : Rule_struct := rule_struct str n.
Let semantics := semantics str.
Variable adom : domains.
Let ast := st adom.
Let ares := res adom.
Variable asem : semantics adom.
Let acond n (sigma : ast) := cond asem n sigma.
Let arule n : Rule ast ares := rule asem n.
Let aRule := Rule ast ares.
Variable dom : domains.
Let st := st dom.
Let res := res dom.
Variable sem : semantics dom.
Let cond n (sigma : st) := cond sem n sigma.
Let rule n : Rule st res := rule sem n.
Let Rule := Rule st res.

Hypothesis Lst : Poset.t ast.
Hypothesis Lr : Poset.t ares.

Variable gst : ast → st → Prop.
Hypothesis Gst_monotone : ∀ N1 N2 : ast, N1 ⊑ N2 → gst N1 ⊑ gst N2.
Variable gres : ares → res → Prop.
Hypothesis Gres_monotone : ∀ N1 N2 : ares, N1 ⊑ N2 → gres N1 ⊑ gres N2.

Let glue_type := glue_type str adom.
Let glue_weaken := glue_weaken str adom Lst Lr.

Hypothesis acond_monotone : forall n sigma sigma',
  sigma' ⊑ sigma →
  acond n sigma' →
  acond n sigma.


(** ** Miscellanous Properties About [aeval] for Some Specific Glues **)

Section Weaken.

Variables glue : glue_type.
Hypothesis glue_contains_weaken : glue_contains glue glue_weaken.
Hypothesis glue_iterable : glue_iterable glue.
Hypothesis glue_accepts_more_premisses : glue_accepts_more_premisses glue.

Lemma glue_contains_weaken_iterable : forall n P sigma sigma' r r',
  sigma ⊑ sigma' →
  r' ⊑ r →
  glue n P sigma' r' →
  glue n P sigma r.
Proof.
  introv O1 O2 Gl. applys~ glue_iterable (fun sigma r => sigma = sigma' ∧ r = r').
   introv (E1&E2). substs~.
   apply glue_contains_weaken. do 2 constructors; eassumption.
Qed.

Lemma aapply_st_pos : forall n sigma sigma' r,
  sigma' ⊑ sigma →
  aapply asem glue n sigma r →
  aapply asem glue n sigma' r.
Proof.
  introv O DA. inverts DA as G D. constructors*.
  applys~ glue_contains_weaken_iterable O.
Qed.

Lemma aapply_check_st_pos : forall R n sigma sigma' r,
  sigma' ⊑ sigma →
  aapply_check glue R n sigma r →
  aapply_check glue R n sigma' r.
Proof.
  introv O DA. inverts DA as G D. constructors*.
  applys~ glue_contains_weaken_iterable O.
Qed.

Lemma hyp_in_st_pos : forall R t sigma sigma' r,
  sigma' ⊑ sigma →
  hyp_in asem (aapply_check glue R) t sigma r →
  hyp_in asem (aapply_check glue R) t sigma' r.
Proof.
  introv O D. inverts D as app. constructors.
  introv E C. forwards C': acond_monotone O C. forwards DA: app E C'.
  applys~ aapply_check_st_pos DA.
Qed.

Lemma aeval_st_pos : forall t sigma sigma' r,
  sigma' ⊑ sigma →
  aeval asem glue t sigma r →
  aeval asem glue t sigma' r.
Proof. rewrite aeval_hyp_in. apply hyp_in_st_pos. Qed.

Lemma aapply_res_pos : forall n sigma r r',
  r ⊑ r' →
  aapply asem glue n sigma r →
  aapply asem glue n sigma r'.
Proof.
  introv O DA. inverts DA as G D. constructors*.
  applys~ glue_contains_weaken_iterable O.
Qed.

Lemma aapply_check_res_pos : forall R n sigma r r',
  r ⊑ r' →
  aapply_check glue R n sigma r →
  aapply_check glue R n sigma r'.
Proof.
  introv O DA. inverts DA as G D. constructors*.
  applys~ glue_contains_weaken_iterable O.
Qed.

Lemma hyp_in_res_pos : forall R t sigma r r',
  r ⊑ r' →
  hyp_in asem (aapply_check glue R) t sigma r →
  hyp_in asem (aapply_check glue R) t sigma r'.
Proof.
  introv O D. inverts D as app. constructors.
  introv E C. forwards DA: app E C.
  applys~ aapply_check_res_pos DA.
Qed.

Lemma aeval_res_pos : forall t sigma r r',
  r ⊑ r' →
  aeval asem glue t sigma r →
  aeval asem glue t sigma r'.
Proof. rewrite aeval_hyp_in. apply hyp_in_res_pos. Qed.

End Weaken.


Section Meet.

Hypothesis ares_meet : @Meet.t ares _.
Hypothesis ast_join : @Join.t ast _.
Hypothesis ast_meet : @Meet.t ast _.

Lemma acond_join : forall n sigma sigma',
  acond n sigma →
  acond n sigma' →
  acond n (sigma ⊔ sigma').
Proof. introv C1 C2. applys* acond_monotone C1. Qed.

Require Import Lift.

(** We use the [LiftBot.Poset] construction from the lattices of [liblat].
  It is a poset on an option type, where the [Some] are ordered as in the
  underlying order; and [None] is considered as bottom, that is as the least
  element of the lattice.
  In practise, this enforces the transfer functions to be monotone where
  they are defined, and their definition domains to be closed under the
  lattice order: if a transfer function is defined for an element, then
  it is defined for every element greater in the lattice.
  We need this precision as we only stated that the transfer functions
  should be defined if the side-condition applies. However, we did not
  ensure any further properties if the side-conditions are not defined,
  in particular, the definition domain could be chaotical. We thus ensure it
  not to be. **)

Inductive arules_incr : aRule → Prop :=
  | arules_incr_Ax : forall aax,
      (forall asigma asigma' ar ar',
        aax asigma = ar →
        aax asigma' = ar' →
        asigma' ⊑ asigma →
        @Poset.order _ (LiftBot.Poset _) ar' ar) →
      (** ∀ asigma asigma', asigma' ⊑♯ asigma → aax asigma' ⊑♯ aax asigma **)
      arules_incr (Rule_Ax aax)
  | arules_incr_R1 : forall aup,
      (forall asigma asigma' asigma0 asigma0',
        aup asigma = asigma0 →
        aup asigma' = asigma0' →
        asigma' ⊑ asigma →
        @Poset.order _ (LiftBot.Poset _) asigma0' asigma0) →
       (** ∀ asigma asigma', asigma' ⊑♯ asigma → aup asigma' ⊑♯ aup asigma **)
      arules_incr (Rule_R1 _ aup)
  | arules_incr_R2 : forall aup anext,
      (forall asigma asigma' asigma0 asigma0',
        aup asigma = asigma0 →
        aup asigma' = asigma0' →
        asigma' ⊑ asigma →
        @Poset.order _ (LiftBot.Poset _) asigma0' asigma0) →
       (** ∀ asigma asigma', asigma' ⊑♯ asigma → aup asigma' ⊑♯ aup asigma **)
      (forall asigma asigma' ar ar' asigma0 asigma0',
        anext asigma ar = asigma0 →
        anext asigma' ar' = asigma0' →
        asigma' ⊑ asigma →
        ar' ⊑ ar →
        @Poset.order _ (LiftBot.Poset _) asigma0' asigma0) →
      (** ∀ asigma asigma' ar ar', asigma' ⊑♯ asigma → ar' ⊑♯ ar → anext asigma' ar' ⊑♯ anext asigma ar **)
      arules_incr (Rule_R2 aup anext)
  .

Hypothesis Icr : forall n, arules_incr (arule n).
Hypothesis Fsem : semantics_full asem.

(** Stability for [⊓♯].  This allows to have more precise oracles. **)

Definition rel_stable_meet (R : term → ast → ares → Prop) :=
  forall t sigma1 sigma2 r r',
  R t sigma1 r → R t sigma2 r' →
  R t (sigma1 ⊓ sigma2) (r ⊓ r').

Lemma rel_stable_meet_equiv : forall R1 R2,
  (forall t sigma r, R1 t sigma r ↔ R2 t sigma r) →
  rel_stable_meet R1 →
  rel_stable_meet R2.
Proof. introv E M D1 D2. forwards: M; try apply E; [apply D1|apply D2|autos~]. Qed.

Definition rel_st_incr (R : term → ast → ares → Prop) :=
  forall t sigma sigma' r,
    sigma' ⊑ sigma → R t sigma r → R t sigma' r.

Lemma rel_stable_meet_altern : forall (R : term → ast → ares → Prop),
  rel_st_incr R →
  rel_stable_meet R ↔ forall t sigma sigma1 sigma2,
    sigma ⊑ (sigma1 ⊓ sigma2) → forall r r',
    R t sigma1 r →
    R t sigma2 r' →
    R t sigma (r ⊓ r').
Proof. introv pos. iff Re. autos*. introv D1 D2. applys~ Re D1 D2. Qed.

Lemma hyp_in_meet : forall (R : term → ast → ares → Prop),
  rel_st_incr R →
  rel_stable_meet R →
  rel_stable_meet (hyp_in asem (aapply_check glue_weaken (aapply_check_step asem R))).
Proof.
  introv pos M. apply rel_stable_meet_altern.
    unfolds. apply hyp_in_st_pos.
     apply glue_contains_refl.
     apply glue_weaken_iterable.
  set (M' := proj1 (rel_stable_meet_altern pos) M). clearbody M'. clear M.
  introv O D1 D2. inverts D1 as app1. inverts D2 as app2. constructors.
  introv E C. forwards Icn: Icr n.
  forwards C1: acond_monotone C; [ applys @Poset.trans O; apply Meet.meet_bound1 |].
  forwards C2: acond_monotone C; [ applys @Poset.trans O; apply Meet.meet_bound2 |].
  forwards DA1: app1 E C1. forwards DA2: app2 E C2.
  inverts DA1 as G1 D1. inverts G1 as G1. inverts G1 as Ost1 Ores1.
  inverts DA2 as G2 D2. inverts G2 as G2. inverts G2 as Ost2 Ores2.
  asserts Ir: ((ar' ⊓ ar'0) ⊑ (r ⊓ r')). clear - Ores1 Ores2. autos*.
  asserts Is: ((sigma1 ⊓ sigma2) ⊑ (asigma' ⊓ asigma'0)). clear - Ost1 Ost2. autos*.
  asserts CM: (acond n (asigma' ⊓ asigma'0)).
    applys~ acond_monotone C. applys~ @Poset.trans O.
  forwards App: Fsem CM. fold (arule n) in App. forwards* A1: D1. forwards* A2: D2.
  clear D1 D2 E C Ost1 Ores1 Ost2 Ores2 CM.
  destruct (rule_struct n) eqn: S; unfolds rule_struct; unfolds arule;
    inverts A1 as E11 E12 E13 I11 N1 I12; rewrite S in E11; inverts E11;
    inverts A2 as E21 E22 E23 I21 N2 I22; rewrite S in E21; inverts E21;
    rewrite E22 in *; inverts E12; inverts Icn as I1 I2 I3; inverts App as App1 App2 App3.
   asserts Ir': (r0 ⊑ (r ⊓ r')).
      forwards~ P1: I1 E13 App1. forwards~ P2: I1 E23 App1.
      applys* @Poset.trans Ir.
    constructors.
     constructors. constructors.
      applys~ @Poset.trans O Is.
      exact Ir'.
     introv (E1&E2). substs. eapply aapply_check_step_Ax; try eassumption.
   constructors.
    constructors. constructors.
     applys~ @Poset.trans O Is.
     exact Ir.
    introv (E1&E2). substs. eapply aapply_check_step_R1; try eassumption.
     applys M' I11 I21. forwards~: I1 E13 App1. forwards~: I1 E23 App1.
   constructors.
    constructors. constructors.
     applys~ @Poset.trans O Is.
     exact Ir.
    introv (E1&E2). substs. lets (sigma_next&Esigma_next): (App2 (r0 ⊓ r1)).
     eapply aapply_check_step_R2; try eassumption.
      applys M' I11 I21.
       forwards~: I1 E13 App1. forwards~: I1 E23 App1.
      applys M' I12 I22.
       forwards~: I2 N1 Esigma_next. forwards~: I2 N2 Esigma_next.
Qed.

Inductive transfert_fun_meet : aRule → Prop :=
  | transfert_fun_meet_ax : forall ax,
    transfert_fun_meet (Rule_Ax ax)
  | transfert_fun_meet_R1 : forall up,
    (forall sigma1 sigma1' sigma2 sigma2',
      up sigma1 = Some sigma1' →
      up sigma2 = Some sigma2' →
      up (sigma1 ⊓ sigma2) = Some (sigma1' ⊓ sigma2')) →
    transfert_fun_meet (Rule_R1 _ up)
  | transfert_fun_meet_R2 : forall up next,
    (forall sigma1 sigma1' sigma2 sigma2',
      up sigma1 = Some sigma1' →
      up sigma2 = Some sigma2' →
      up (sigma1 ⊓ sigma2) = Some (sigma1' ⊓ sigma2')) →
    (forall sigma1 r1 sigma1' sigma2 r2 sigma2',
      next sigma1 r1 = Some sigma1' →
      next sigma2 r2 = Some sigma2' →
      next (sigma1 ⊓ sigma2) (r1 ⊓ r2) = Some (sigma1' ⊓ sigma2')) →
    transfert_fun_meet (Rule_R2 up next)
  .

Lemma hyp_in_meet_transert_fun : forall (R : term → ast → ares → Prop),
  (forall n, transfert_fun_meet (arule n)) →
  rel_stable_meet R →
  rel_stable_meet (hyp_in asem (aapply_check glue_weaken (aapply_check_step asem R))).
Proof.
  introv T M. cuts P: (forall t sigma sigma1 sigma2 r r',
      hyp_in asem (aapply_check glue_weaken (aapply_check_step asem R)) t sigma1 r →
      hyp_in asem (aapply_check glue_weaken (aapply_check_step asem R)) t sigma2 r' →
      sigma = (sigma1 ⊓ sigma2) →
      hyp_in asem (aapply_check glue_weaken (aapply_check_step asem R)) t sigma (r ⊓ r')).
    introv D1 D2. applys~ P D1 D2.
  introv D1 D2 O. inverts D1 as app1. inverts D2 as app2. constructors.
  introv E C. forwards Icn: Icr n.
  forwards C1: acond_monotone C.
    rewrite O. apply Meet.meet_bound1.
  forwards C2: acond_monotone C.
    rewrite O. apply Meet.meet_bound2.
  forwards DA1: app1 E C1. forwards DA2: app2 E C2.
  inverts DA1 as G1 D1. inverts G1 as G1. inverts G1 as Ost1 Ores1.
  inverts DA2 as G2 D2. inverts G2 as G2. inverts G2 as Ost2 Ores2.
  asserts Ir: ((ar' ⊓ ar'0) ⊑ (r ⊓ r')). clear - Ores1 Ores2. autos*.
  asserts Is: ((sigma1 ⊓ sigma2) ⊑ (asigma' ⊓ asigma'0)). clear - Ost1 Ost2. autos*.
  asserts CM: (acond n (asigma' ⊓ asigma'0)).
    applys~ acond_monotone C. rewrite~ O.
  forwards App: Fsem CM. fold (arule n) in App. forwards Tr: T n.
  forwards* A1: (rm D1). forwards* A2: (rm D2).
  clear E C Ost1 Ores1 Ost2 Ores2 CM T.
  destruct (rule_struct n) eqn: S; unfolds rule_struct; unfolds arule;
    inverts A1 as E11 E12 E13 I11 N1 I12; rewrite S in *; inverts E11;
    inverts A2 as E21 E22 E23 I21 N2 I22; rewrite S in *; inverts E21;
    rewrite E22 in *; inverts E12; inverts Icn as I1 I2; inverts App as App1 App2.
   asserts Ir': (r0 ⊑ (r ⊓ r')).
      forwards~ P1: I1 E13 App1. forwards~ P2: I1 E23 App1.
      applys* @Poset.trans Ir.
    constructors.
     constructors. constructors.
      rewrite O. exact Is.
      exact Ir'.
     introv (E1&E2). substs. eapply aapply_check_step_Ax; try eassumption.
   constructors.
    constructors. constructors.
     rewrite O. exact Is.
     exact Ir.
    introv (E1&E2). substs. forwards D: M I11 I21.
     applys aapply_check_step_R1 D; try eassumption. inverts Tr as T. apply* T.
   constructors.
    constructors. constructors.
     rewrite O. exact Is.
     exact Ir.
    introv (E1&E2). substs. lets (sigma_next&Esigma_next): (App2 (r0 ⊓ r1)).
     inverts Tr as Tup Tnext. forwards~ T: Tup E13 E23. unfolds ast. rewrite App1 in T.
     inverts T. forwards~ T: Tnext N1 N2. unfolds ares. rewrite Esigma_next in T. inverts T.
     eapply aapply_check_step_R2; try eassumption.
      forwards~ D: M I11 I21.
      forwards~ D: M I12 I22.
Qed.

Inductive meet_closure (R : term → ast → ares → Prop) : term → ast → ares → Prop :=
  | meet_closure_exact : forall t sigma r,
      R t sigma r →
      meet_closure R t sigma r
  | meet_closure_meet : forall t sigma sigma1 sigma2 r r1 r2,
      sigma ⊑ (sigma1 ⊓ sigma2) →
      (r1 ⊓ r2) ⊑ r →
      meet_closure R t sigma1 r1 →
      meet_closure R t sigma2 r2 →
      meet_closure R t sigma r
  .

Lemma meet_closure_stable_meet : forall R,
  rel_stable_meet (meet_closure R).
Proof. introv D1 D2. applys~ meet_closure_meet D1 D2. Qed.

Lemma meet_closure_st_incr : forall R,
  rel_st_incr R →
  rel_st_incr (meet_closure R).
Proof.
  introv Re O D. induction D.
   apply meet_closure_exact. applys~ Re H.
   applys meet_closure_meet D1 D2; eapply Poset.trans; try eassumption; autos*.
Qed.

Lemma hyp_in_meet_rel : forall t sigma r,
  meet_closure (aeval asem glue_weaken) t sigma r →
  hyp_in asem (aapply_check glue_weaken (aapply_check_step asem (meet_closure (aeval asem glue_weaken)))) t sigma r.
Proof.
  introv D. induction D.
   applys~ hyp_in_impl aapply_check_impl.
    apply~ aapply_check_step_impl.
    apply meet_closure_exact.
    apply~ aeval_hyp_in_impl.
   applys hyp_in_st_pos H.
    apply~ glue_contains_refl.
    apply~ glue_weaken_iterable.
    applys hyp_in_res_pos H0.
     apply~ glue_contains_refl.
     apply~ glue_weaken_iterable.
     apply~ hyp_in_meet.
      clear - acond_monotone name glue_type. introv O D. inverts D as O1 O2 D1 D2.
       apply meet_closure_exact. applys~ aeval_st_pos O1.
        apply~ glue_contains_refl.
        apply~ glue_weaken_iterable.
       applys meet_closure_meet D1 D2; try eassumption.
        applys~ @Poset.trans O.
      clear. introv D1 D2. applys* meet_closure_meet D1 D2.
Qed.

Lemma aeval_meet : rel_stable_meet (aeval asem glue_weaken).
Proof.
  introv D1 D2. rewrite aevals_same.
  apply aeval_f_cons with (R := meet_closure (aeval asem glue_weaken)).
   clear t sigma1 sigma2 r r' D1 D2. introv D.
    forwards M: hyp_in_meet (meet_closure (aeval asem glue_weaken)).
     clear - acond_monotone name glue_type. introv O D. inverts D as O1 O2 D1 D2.
      apply meet_closure_exact. applys~ aeval_st_pos O.
       apply~ glue_contains_refl.
       apply~ glue_weaken_iterable.
      applys~ meet_closure_meet D1 D2. applys~ @Poset.trans O.
     clear. introv D1 D2. applys* meet_closure_meet D1 D2.
     inverts D as O1 O2 D1 D2.
      applys hyp_in_impl aapply_check_impl.
       apply~ aapply_check_step_impl.
       apply meet_closure_exact.
       apply~ aeval_hyp_in_impl.
     applys hyp_in_st_pos O1.
      apply~ glue_contains_refl.
      apply~ glue_weaken_iterable.
      applys hyp_in_res_pos O2.
       apply~ glue_contains_refl.
       apply~ glue_weaken_iterable.
       apply~ M; apply~ hyp_in_meet_rel.
   eapply meet_closure_meet; try apply* @Poset.refl; apply~ meet_closure_exact.
Qed.


Definition Park_meet (R : term → ast → ares → Prop) :=
  forall t sigma r,
    R t sigma r →
    hyp_in asem (aapply_check glue_weaken (aapply_check_step asem (meet_closure R))) t sigma r.

Lemma Park_meet_Park : forall R,
  (forall n, transfert_fun_meet (arule n)) →
  Park_meet R →
  Park str adom (hyp_in asem) (aapply_check_step asem) (aapply_check glue_weaken)
    (meet_closure R).
Proof.
  introv T P D. induction D.
   forwards~: R.
   eapply hyp_in_st_pos; try eassumption.
    apply~ glue_contains_refl.
    apply~ glue_weaken_iterable.
    eapply hyp_in_res_pos; try eassumption.
     apply~ glue_contains_refl.
     apply~ glue_weaken_iterable.
     applys~ hyp_in_meet_transert_fun T meet_closure_stable_meet.
Qed.

Lemma Park_meet_Park_st_incr : forall R,
  rel_st_incr R →
  Park_meet R →
  Park str adom (hyp_in asem) (aapply_check_step asem) (aapply_check glue_weaken)
    (meet_closure R).
Proof.
  introv T P D. induction D.
   forwards~: R.
   eapply hyp_in_st_pos; try eassumption.
    apply~ glue_contains_refl.
    apply~ glue_weaken_iterable.
    eapply hyp_in_res_pos; try eassumption.
     apply~ glue_contains_refl.
     apply~ glue_weaken_iterable.
     applys~ hyp_in_meet meet_closure_stable_meet. apply~ meet_closure_st_incr.
Qed.

End Meet.

End Miscellanous.


(** ** A Threaded Version **)

(** Note depreciated: This is not compatible with the current
  treatment of the glue.  Instead, please use a symbolically joined
  domain and use a filtering glue.  This is equivalent to the trace
  partitionning presented below, but much more powerful as it comes
  with the whole development (it is even possible to frame contexts
  and still have trace partitionning, for instance). **)

Section Threads.

(** There are cases where we can’t afford to define a proper lattice
  for a domain.  We are stuck to complete a poset to a lattice with
  symbolic joins.  We then want to do trace partitionning, that is to
  split the input into several smaller bits and to analyse each of
  them separately.  If we don’t do that, we indeed will mix different
  results as soon as an R2 rule shall appear. **)

Variable str : structure.
Let term := term str.
Let name := name str.
Let Rule_struct := Rule_struct term.
Let left (n : name) : term := left str n.
Let rule_struct (n : name) : Rule_struct := rule_struct str n.
Let semantics := semantics str.
Variable adom : domains.
Let ast := st adom.
Let ares := res adom.
Variable asem : semantics adom.
Let acond n (sigma : ast) := cond asem n sigma.
Let arule n : Rule ast ares := rule asem n.
Let aRule := Rule ast ares.
Variable dom : domains.
Let st := st dom.
Let res := res dom.
Variable sem : semantics dom.
Let cond n (sigma : st) := cond sem n sigma.
Let rule n : Rule st res := rule sem n.
Let Rule := Rule st res.

Require Export lat_prop.
Require Export CompleteLattice.

Hypothesis Pst : Poset.t ast.
Hypothesis Pr : Poset.t ares.

Variable gst : ast → st → Prop.
Hypothesis Gst_monotone : ∀ N1 N2 : ast, N1 ⊑ N2 → gst N1 ⊑ gst N2.
Variable gres : ares → res → Prop.
Hypothesis Gres_monotone : ∀ N1 N2 : ares, N1 ⊑ N2 → gres N1 ⊑ gres N2.


CoInductive aevalt : term → ast → ares → Prop :=
  | aevalt_cons : forall t sigma r,
      (forall n,
        t = left n →
        acond n sigma →
        aapplyt n sigma r) →
      aevalt t sigma r
with aapplyt : name → ast → ares → Prop :=
  | aapplyt_cons : forall n sigma sigma' (Psigma : _ → Prop) r,
      sigma ⊑ sigma' →
      (forall sigmap,
        (** Propagation **)
        Psigma sigmap →
        exists rp,
          rp ⊑ r ∧ aapplyt_step n sigmap rp) →
      (forall sigmac,
        (** We didn’t forget a trace **)
        gst sigma' sigmac →
        exists sigmap,
          Psigma sigmap ∧ gst sigmap sigmac) →
      aapplyt n sigma r
with aapplyt_step : name → ast → ares → Prop :=
  | aapplyt_step_Ax : forall n ax sigma r,
      rule_struct n = Rule_struct_Ax _ →
      arule n = Rule_Ax ax →
      ax sigma = Some r →
      aapplyt_step n sigma r
  | aapplyt_step_R1 : forall n t up sigma sigma' r,
      rule_struct n = Rule_struct_R1 t →
      arule n = Rule_R1 _ up →
      up sigma = Some sigma' →
      aevalt t sigma' r →
      aapplyt_step n sigma r
  | aapplyt_step_R2 : forall n t1 t2 up next sigma sigma1 sigma2 r r',
      rule_struct n = Rule_struct_R2 t1 t2 →
      arule n = Rule_R2 up next →
      up sigma = Some sigma1 →
      aevalt t1 sigma1 r →
      next sigma r = Some sigma2 →
      aevalt t2 sigma2 r' →
      aapplyt_step n sigma r'
  .

Section Check.

Variable aapply_check_step : name → ast → ares → Prop.

Inductive aapplyt_check : name → ast → ares → Prop :=
  | aapplyt_check_cons : forall n sigma sigma' (Psigma : _ → Prop) r,
      sigma ⊑ sigma' →
      (forall sigmap,
        (** Propagation **)
        Psigma sigmap →
        exists rp,
          rp ⊑ r ∧ aapply_check_step n sigmap rp) →
      (forall sigmac,
        (** We didn’t forget a trace **)
        gst sigma' sigmac →
        exists sigmap,
          Psigma sigmap ∧ gst sigmap sigmac) →
      aapplyt_check n sigma r.

End Check.


(** *** Correction Lemmae Between [aeval] and [aeval_f]. **)

Lemma aapplyt_step_check : aapplyt_step = aapply_check_step asem aevalt.
Proof.
  extens. intros t sigma r. iff A.
    inverts A.
     apply* aapply_check_step_Ax.
     apply* aapply_check_step_R1.
     apply* aapply_check_step_R2.
    inverts A.
     apply* aapplyt_step_Ax.
     apply* aapplyt_step_R1.
     apply* aapplyt_step_R2.
Qed.

Theorem aevalts_equiv : forall t sigma r,
  aevalt t sigma r ↔ aeval_f _ _ (hyp_in asem) (aapply_check_step asem) aapplyt_check t sigma r.
Proof.
  introv. iff D.
   constructors*. clear D t sigma r. introv D. inverts D as Ac.
    constructors. introv E C. forwards Aac: Ac E C. inverts Aac as O1 O2 D.
    constructors; try eassumption. rewrite~ <- aapplyt_step_check.
   inverts D as R_check Rc. gen t sigma r. cofix Dv. introv Rc.
    forwards Dc: R_check Rc. inverts Dc as Ac.
    constructors. introv E C. forwards Aac: Ac E C. inverts Aac as O1 O2 D.
    constructors; try eassumption. introv P. forwards (rp&Orp&A): O2 P.
    exists rp. splits~. inverts A.
     apply* aapplyt_step_Ax.
     apply* aapplyt_step_R1.
     apply* aapplyt_step_R2.
Qed.

Corollary aevalts_same : aevalt = aeval_f _ _ (hyp_in asem) (aapply_check_step asem) aapplyt_check.
Proof. extens. exact aevalts_equiv. Qed.


(** *** Correction Lemmae of [aeval] With Respect to [aevalt]. **)

Lemma aeval_aevalt : forall t sigma r,
  aeval asem (glue_weaken str adom _ _) t sigma r →
  aevalt t sigma r.
Proof.
  cofix F. introv R. inverts R as R. constructors.
  introv E C. forwards A: (rm R) (rm E) (rm C).
  inverts A as G A. inverts G as G. inverts G as O1 O2.
  applys aapplyt_cons (fun sigmap => sigmap = asigma') O1.
   introv E. substs. eexists. splits.
    exact O2.
    forwards* A': A. inverts A' as E1 E2 E3 E4 E5 E6.
     applys~ aapplyt_step_Ax E1 E2 E3.
     apply F in E4. applys~ aapplyt_step_R1 E1 E2 E3 E4.
     apply F in E4. apply F in E6. applys~ aapplyt_step_R2 E2 E4 E6.
   introv G. eexists. splits~.
Qed.



(** *** Miscellanous Properties About [aevalt]. **)

Hypothesis acond_monotone : forall n sigma sigma',
  sigma' ⊑ sigma →
  acond n sigma' →
  acond n sigma.

Lemma aapplyt_st_pos : forall n sigma sigma' r,
  sigma' ⊑ sigma →
  aapplyt n sigma r →
  aapplyt n sigma' r.
Proof.
  introv O DA. inverts DA as Ost D G. constructors.
   eapply @Poset.trans. exact O. exact Ost.
   apply D.
   apply G.
Qed.

Lemma aevalt_st_pos : forall t sigma sigma' r,
  sigma' ⊑ sigma →
  aevalt t sigma r →
  aevalt t sigma' r.
Proof.
  introv O D. inverts D as D. constructors.
  introv E C. forwards A: D E.
   applys~ acond_monotone C.
   applys~ aapplyt_st_pos A.
Qed.

Lemma aapplyt_res_pos : forall n sigma r r',
  r ⊑ r' →
  aapplyt n sigma r →
  aapplyt n sigma r'.
Proof.
  introv O DA. inverts DA as Ost D G. constructors.
   apply Ost.
   introv P. forwards (rp&Orp&Arp): D P.
    eexists. splits.
     eapply @Poset.trans.
      exact Orp.
      exact O.
     exact Arp.
   apply G.
Qed.

Lemma aevalt_res_pos : forall t sigma r r',
  r ⊑ r' →
  aevalt t sigma r →
  aevalt t sigma r'.
Proof.
  introv O D. inverts D as D. constructors.
  introv E C. forwards A: D E C.
  applys~ aapplyt_res_pos A.
Qed.


Hypothesis ares_join : @Join.t ares _.
Hypothesis ares_meet : @Meet.t ares _.
Hypothesis ast_join : @Join.t ast _.
Hypothesis ast_meet : @Meet.t ast _.

Hypothesis gst_join_morph : forall sigma1 sigma2 : ast,
  gst (sigma1 ⊔ sigma2) ⊑ (gst sigma1 ⊔ gst sigma2).

Hypothesis Icr : forall n, arules_incr _ _ _ (arule n).
Hypothesis Fsem : semantics_full asem.

Definition rel_stable_join (R : term → ast → ares → Prop) :=
  forall t sigma1 sigma2 r r',
  R t sigma1 r → R t sigma2 r' →
  R t (sigma1 ⊔ sigma2) (r ⊔ r').

Hypothesis acond_correct : forall n asigma sigma,
  gst asigma sigma →
  cond n sigma →
  acond n asigma.

Hypothesis Pro : forall n, propagates _ _ gst gres (acond n) (cond n) (arule n) (rule n).

Theorem correctness_thread : forall t asigma ar,
  aevalt t asigma ar →
  forall sigma r,
    gst asigma sigma →
    eval sem t sigma r →
    gres ar r.
Proof.
  cuts (IHE&IHA): ((forall t sigma r,
      eval sem t sigma r →
      forall asigma ar,
      aevalt t asigma ar →
      gst asigma sigma →
      gres ar r)
    /\ forall n sigma r,
      apply sem n sigma r →
      cond n sigma →
      forall asigma ar,
      aapplyt_step n asigma ar →
      gst asigma sigma →
      gres ar r).
    introv aE G E. forwards*: IHE.
  apply eval_mutind.
  (** eval **)
  introv Et C A IHA aE G. inverts aE as allBranches.
  forwards aA: allBranches Et.
    apply* acond_correct.
  inverts aA as Ssigma P T R. forwards (sigmap&Psigmap&Gsigmap): T.
    applys~ Gst_monotone Ssigma G.
  forwards (rp&Prp&aA): P Psigmap. forwards~ G': IHA aA.
  applys~ Gres_monotone G'.
  (** apply_Ax **)
  introv Estr ER Eax C aA G. inverts aA; rewrite_term (rule_struct n); inverts Estr.
  forwards P: Pro n. rewrite_term (rule n). rewrite_term (arule n). inverts P as Hyp.
  applys~ Hyp C G. applys~ acond_correct G.
  (** apply_R1 **)
  introv Estr ER Eup E IH C aA G.
  inverts aA as Estr' ER' Eup' aE'; rewrite_term (rule_struct n); inverts Estr.
  forwards P: Pro n. rewrite_term (rule n). rewrite_term (arule n). inverts P as Hyp.
  applys~ IH aE'. applys~ Hyp C G. applys~ acond_correct G.
  (** apply_R2 **)
  introv Estr ER Eup E1 IH1 Enext Ae2 IH2 C aA. introv G.
  inverts aA as Estr1 ER1 Eup1 aE1 Enext1 aE2; rewrite_term (rule_struct n); inverts Estr.
  forwards P: Pro n. rewrite_term (rule n). rewrite_term (arule n). inverts P as Hyp1 Hyp2.
  applys~ IH2 aE2. applys~ Hyp2 C G Enext Enext1.
   applys~ acond_correct G.
   applys~ IH1 aE1. applys~ Hyp1 C G. applys~ acond_correct G.
Qed.

End Threads.

