{-# LANGUAGE BangPatterns, ScopedTypeVariables #-}
{-# OPTIONS -fno-warn-name-shadowing #-}
module GHC.ExplicitCallStack.Stack
( throwStack
, pushStack
, emptyStack
, StackElement()
, Stack()
, mkKnownStackElem
, mkUnknownStackElem
) where

import Control.Exception
import Control.Monad
import Control.Concurrent.MVar
import Data.HashTable (HashTable)
import Data.Int
import Data.Unique
import GHC.Base
import GHC.Show
import Data.Maybe
import qualified Data.HashTable as HashTable
import System.IO.Unsafe

hashSE :: StackElement -> Int32
hashSE = fromIntegral . f . elementID
  where
    f (SEID x y) = x + 0xdeadbeef * y
{- Invariant, each StackElement occurs at most once in the Stack -}

data Stack = Empty         { stackDetails :: !StackDetails }
{- read these like: (stackDetails, _stackElement) `Then` _restOfStack -}
           | Then          { _stackElement :: !StackElement
                           , stackDetails :: !StackDetails
                           , _restOfStack  :: !Stack
                           }
           | RecursionThen { _stackElement :: !StackElement
                           , stackDetails :: !StackDetails
                           , _restOfStack :: !Stack
                           }

instance Eq Stack where
  Empty {} == Empty {}                       = True
  s1@Then {} == s2@Then {}                   = (stackDetails s1) == (stackDetails s2)
  s1@RecursionThen {} == s2@RecursionThen {} = (stackDetails s1) == (stackDetails s2)
  _ == _ = False

instance Eq StackDetails where
  s1 == s2 = (stackUnique s1) == (stackUnique s2)

data StackElems = Stop
                | !StackElement `SEThen` !StackElems
                | !StackElement `SERecursionThen` !StackElems

data StackDetails = StackDetails { stackUnique :: !Unique
                                 , stackTable :: !(MVar (HashTable StackElement Stack))
                                 } 

data StackElement 
  = KnownLocation { elementID         :: !StackElementID
                  , _lineNo           :: Int
                  , _colNo            :: Int
                  , _file             :: String
                  , enclosingVarName  :: String
                  }
  | UnknownLocation { elementID        :: !StackElementID
                    , enclosingVarName :: String
                    }

instance Eq StackElement where
  a == b = elementID a == elementID b

instance Ord StackElement where
  a `compare` b = elementID a `compare` elementID b

data StackElementID = SEID { moduleKey   :: !Int
                           , localUnique :: !Int
                           }
  deriving (Eq, Ord, Show)

instance Show Stack where
  show Empty {} = ""
  show (Then se _ Empty {}) = show se
  show (Then se _ rest) = show se ++ "\n" ++ show rest
  show (RecursionThen se _ Empty {}) = show se ++ "\n..."
  show (RecursionThen se _ rest) = show se ++ "\n...\n" ++ show rest

instance Show StackElement where
  show (UnknownLocation{ enclosingVarName = enclosingVarName }) 
    = concat ["in ", enclosingVarName, ", ", "<unknown location>"]
  show k@(KnownLocation { enclosingVarName = enclosingVarName
                        , _file = file})  
    = concat ["in ", enclosingVarName, ", ", file, ":", lineNo, ",", colNo]
    where
      lineNo = show $ _lineNo k
      colNo = show $ _colNo k

throwStack :: forall e a . Exception e => (Stack -> e) -> a
throwStack f = throw (f emptyStack)

{-# NOINLINE emptyStack #-}
emptyStack :: Stack
emptyStack = unsafePerformIO $ do
  stackDetails <- newStackDetails 
  return $ Empty { stackDetails = stackDetails }

mkKnownStackElem   :: Int -> Int -> String -> FilePath -> Int -> Int -> StackElement
mkKnownStackElem modKey unq evn fp line col 
  = KnownLocation { elementID = SEID { moduleKey = modKey
                                     , localUnique = unq
                                     }
                  , _lineNo = line
                  , _colNo = col
                  , _file = fp
                  , enclosingVarName = evn
                  }

mkUnknownStackElem :: Int -> Int -> String -> StackElement
mkUnknownStackElem modKey unq evn 
  = UnknownLocation { elementID = SEID { moduleKey = modKey
                                       , localUnique = unq
                                       }
                    , enclosingVarName = evn
                    }

pushStack :: Stack -> StackElement -> Stack
pushStack stack elem = unsafePerformIO $ do
  mstack <- memoPush stack elem
  case mstack of
    Just stack' -> return stack'
    Nothing -> do
      maybe (createPush elem stack) 
            (uncurry (assemble elem))
            (stack `breakAt` elem) 
  where  
    assemble :: StackElement -> Stack -> StackElems -> IO Stack
    assemble elem bottom Stop = createPush elem =<< createPush elem bottom
    assemble elem bottom (se `SEThen` elems) = createPush elem =<< assemble se bottom elems
    assemble elem bottom (se `SERecursionThen` elems) = createPush elem =<< 
                                                        createPush se =<< 
                                                        assemble se bottom elems

{- push that only looks at the current stack's memo table -}
memoPush :: Stack -> StackElement -> IO (Maybe Stack)
memoPush stack elem = withMVar mvht (flip HashTable.lookup elem)
  where
    mvht = stackTable . stackDetails $ stack

{- push that may create a new stack if not memoizied -}
{- if not memoizied it only checks the top of the stack to
   see if it is the same as the element being pushed,
   doesn't take the stack apart -}
createPush :: StackElement -> Stack -> IO Stack
createPush elem stack = do
  mstack <- memoPush stack elem
  case mstack of
    Just stack' -> return stack'
    Nothing -> do
      newDetails <- newStackDetails

      if (hasStackElement stack && _stackElement stack == elem)
        {- invariant, if we are here and the top element == elem, 
         - then the stack isn't of the form [elem, ..., ] as we would 
         - have found it in the memo table that we initialize below with this rule -}
        then do
          let newStack = RecursionThen  { stackDetails  = newDetails
                                        , _stackElement = elem
                                        , _restOfStack  = _restOfStack stack
                                        }

          stackTableInsert (stackDetails $ stack) elem newStack
          stackTableInsert newDetails elem newStack -- this maintains the above invariant
          return newStack
        else do
          let newStack = Then { stackDetails  = newDetails
                              , _stackElement = elem
                              , _restOfStack  = stack
                              }

          stackTableInsert (stackDetails $ stack) elem newStack
          return newStack

newStackDetails :: IO StackDetails
newStackDetails = do
          newStackUnique <- newUnique
          mvar <- newMVar =<< HashTable.new (==) hashSE

          return $ StackDetails { stackUnique = newStackUnique
                                , stackTable  = mvar
                                }

stackTableInsert :: StackDetails -> StackElement -> Stack -> IO ()
stackTableInsert details key val = do
  modifyMVar_ (stackTable details)
              (\ht -> HashTable.insert ht key val >> return ht) 

breakAt :: Stack -> StackElement -> Maybe (Stack, StackElems)
(Empty _) `breakAt` _ = Nothing
(Then elem _ stack) `breakAt` splitElem
  | elem == splitElem = return (stack, Stop)
  | otherwise = do
      (bottom, top) <- stack `breakAt` splitElem
      return (bottom, elem `SEThen` top)
(RecursionThen elem _ stack) `breakAt` splitElem
  | elem == splitElem = return (stack, Stop)
  | otherwise = do
      (bottom, top) <- stack `breakAt` splitElem
      return (bottom, elem `SERecursionThen` top)

hasStackElement :: Stack -> Bool
hasStackElement Empty {} = False
hasStackElement _ = True
