{-# OPTIONS_GHC -Wall -fno-warn-name-shadowing #-}
{-# LANGUAGE TemplateHaskell #-}
module SimpleDebugTranslate(stackPassTransform) where

#ifdef GHCI
import Annotations
import CoreMonad
import CoreSyn
import Data.Typeable (Typeable)
import FastString
#if __GLASGOW_HASKELL__ > 609
import Data.Data (Data)
#else
import Data.Generics (Data)
#endif
import FiniteMap
import HscTypes
import Id
import Maybes
import MkCore
import Name hiding (varName)
import PrelNames
import RdrName
import RnEnv
import Serialized
import SrcLoc
import TcRnMonad hiding (getModule)
import TyCon
import Type
import CoreUtils
import UniqSupply
import Var

import Control.Monad

import qualified Language.Haskell.TH.Syntax as TH
import GHC.ExplicitCallStack.Annotation
import OccName hiding (varName)
import Module
import DynFlags
import Unique
import Serialized

newtype DebugM a = DebugM { _runDebugM :: DebugContext -> Location -> CoreM a}

data Location = Loc { locSrcLoc  :: SrcLoc
                    , locVarName :: FastString
                    }

data DebugContext = DC { dcLocalDebuggedMap :: FiniteMap Var Var
                       , dcStackType :: Type
                       , dcThrowStackVar :: Var 
                       , dcStackPushVar :: Var
                       , dcEmptyStackVar :: Var
                       , dcMkKnownStackElem :: Var
                       , dcMkUnknownStackElem :: Var
                       , dcThrowVar :: Var
                       }

instance Monad DebugM where
  return a = DebugM $ \_ _ -> (return a)
  (DebugM lf) >>= rf = DebugM $ \dc sl -> (lf dc sl) >>= (\a -> _runDebugM (rf a) dc sl)

instance MonadUnique DebugM where
  getUniqueSupplyM = liftCoreM getUniqueSupplyM

instance Functor DebugM where
  fmap = liftM

liftCoreM :: CoreM a -> DebugM a
liftCoreM c = DebugM $ \_ _ -> c

stackType :: DebugM Type
stackType      = DebugM $ \dc _ -> (return . dcStackType $ dc)
throwStackVar :: DebugM Var
throwStackVar  = DebugM $ \dc _ -> (return . dcThrowStackVar $ dc)
stackPushVar :: DebugM Var
stackPushVar   = DebugM $ \dc _ -> (return . dcStackPushVar $ dc)
emptyStackVar :: DebugM Var
emptyStackVar = DebugM $ \dc _ -> (return . dcEmptyStackVar $ dc)
mkKnownStackElem :: DebugM Var
mkKnownStackElem = DebugM $ \dc _ -> (return . dcMkKnownStackElem $ dc)
mkUnknownStackElem :: DebugM Var
mkUnknownStackElem = DebugM $ \dc _ -> (return . dcMkUnknownStackElem $ dc)
throwVar :: DebugM Var
throwVar = DebugM $ \dc _ -> (return . dcThrowVar $ dc)

currentSrcLoc :: DebugM Location
currentSrcLoc = DebugM $ \_ sl -> return sl

withSrcLoc :: Location -> DebugM a -> DebugM a
withSrcLoc sl (DebugM f) = (DebugM $ \dc _ -> f dc sl)

refineSrcSpan :: SrcSpan -> DebugM a -> DebugM a
refineSrcSpan ss (DebugM f) = DebugM $ \dc l -> 
  let l' = l { locSrcLoc = (srcSpanStart ss) }
  in f dc l'
    

runDebugM :: FiniteMap Var Var -> Type -> DebugM a -> CoreM a
runDebugM localDebuggedMap stackTy code = do
  
  throwStackVar <- rdrNameToId throwStack_RDR
  throwVar <- rdrNameToId throw_RDR
  stackPushVar <- rdrNameToId pushStack_RDR 
  emptyStackVar <- rdrNameToId emptyStack_RDR 

  mkKnownStackEl <- rdrNameToId mkKnownStackElem_RDR
  mkUnknownStackEl <- rdrNameToId mkUnknownStackElem_RDR

  
  let dc = DC { dcLocalDebuggedMap = localDebuggedMap
              , dcStackType = stackTy
              , dcThrowStackVar = throwStackVar
              , dcStackPushVar = stackPushVar
              , dcMkKnownStackElem = mkKnownStackEl
              , dcMkUnknownStackElem = mkUnknownStackEl
              , dcEmptyStackVar = emptyStackVar
              , dcThrowVar = throwVar
              }

  let loc = Loc { locSrcLoc = noSrcLoc
                , locVarName = mkFastString "No Variable"
                }

  _runDebugM code dc loc


stackPassTransform :: ModGuts -> CoreM ModGuts
stackPassTransform guts = do
  let binds = mg_binds guts

  stackTy <- liftM mkTyConTy $ rdrNameToTyCon stackType_RDR
  localDebuggedMap <- buildLocalDebuggedMap stackTy binds 

  bindResults <- runDebugM localDebuggedMap stackTy $ mapM processBind binds

  let binds' = map buildBinds bindResults
  debuggedAnns' <- mkAndSetAnnotations localDebuggedMap

  return $ guts { mg_binds = binds',
                  mg_anns = debuggedAnns' ++ (mg_anns guts) }


buildBinds :: BindResult -> CoreBind
buildBinds (NonRecBR DRDebugged { dr_OriginalVar = x
                                , dr_OriginalExpr = e
                                , dr_DebbugedVar = x'
                                , dr_DebuggedExpr = e' })   = Rec [(x,e), (x',e')]
buildBinds (NonRecBR DRNotDebugged { dr_OriginalVar = x
                                   , dr_OriginalExpr = e }) = NonRec x e
buildBinds (RecBR drs) = Rec $ concatMap collapseBindings drs
  where
    collapseBindings :: DebugResult -> [(Var, CoreExpr)]
    collapseBindings DRDebugged { dr_OriginalVar = x
                                , dr_OriginalExpr = e
                                , dr_DebbugedVar = x'
                                , dr_DebuggedExpr = e' } = [(x,e), (x',e')]
    collapseBindings DRNotDebugged { dr_OriginalVar = x
                                   , dr_OriginalExpr = e } = [(x,e)]


mkAndSetAnnotations :: FiniteMap Var Var -> CoreM [Annotation]
mkAndSetAnnotations = mapM (uncurry mkAndSetAnnotation) . fmToList
  where
    mkAndSetAnnotation :: Var -> Var -> CoreM Annotation
    mkAndSetAnnotation origVar debuggedVar = do
      let target = NamedTarget (varName origVar)
      thValue <- reifyName (varName debuggedVar)
      let annValue = Debugged thValue
      let ann = Annotation { ann_target = target
                           , ann_value  = toSerialized serializeWithData annValue
                           }

      addAnnotation serializeWithData target annValue 
      return ann

    reifyName :: Name -> CoreM TH.Name
    reifyName name = do
      mod <- getModule

      let pkg_str = packageIdString (modulePackageId mod)
      let mod_str = moduleNameString (moduleName mod)
      let occ     = nameOccName name
      let occ_str = occNameString occ

      return $ TH.mkNameG_v pkg_str mod_str occ_str
      where

processBind :: CoreBind -> DebugM BindResult
processBind (NonRec b e) = liftM NonRecBR $ processDebug b e
processBind (Rec bs)     = liftM RecBR $ mapM (uncurry processDebug) bs

data BindResult = NonRecBR DebugResult
                | RecBR [DebugResult]

data DebugResult 
  = DRDebugged { dr_OriginalVar :: Var
               , dr_DebbugedVar :: Var
               , dr_OriginalExpr :: CoreExpr
               , dr_DebuggedExpr :: CoreExpr
               }
  | DRNotDebugged { dr_OriginalVar :: Var
                  , dr_OriginalExpr :: CoreExpr
                  }

processDebug :: Var -> CoreExpr -> DebugM DebugResult
processDebug x e = withSrcLoc currentLocation $ do
  toDebug <- liftM2 (||) (dopt Opt_ExplicitCallStackCoreAll `liftM` liftCoreM getDynFlags)
                         (hasDebugAnnotation x)

  case toDebug of
    False -> do
      emptyStackVar <- emptyStackVar
      new_e <- useStackVar e emptyStackVar
      return $ DRNotDebugged { dr_OriginalVar = x
                             , dr_OriginalExpr = new_e
                             }
    True  -> do
      emptyStackVar <- emptyStackVar
      e' <- wrapInStackLambdaAndUseIt e
      x' <- liftM (expectJust "looking up local debugged version") $ lookupLocalDebuggedVersion x

      let new_e = App (Var x') (Var emptyStackVar)

      return $ DRDebugged { dr_OriginalVar = x
                          , dr_DebbugedVar = x'
                          , dr_OriginalExpr = new_e
                          , dr_DebuggedExpr = e'
                          }
  where
    currentLocation = Loc { locSrcLoc  = nameSrcLoc . varName $ x
                          , locVarName = occNameFS . nameOccName . varName $ x
                          }
    

hasDebugAnnotation :: Var -> DebugM Bool
hasDebugAnnotation var = do
  (anns :: [Debug]) <- liftCoreM $ findAnnotations deserializeWithData (NamedTarget (varName var))
  return . not . null $ anns

lookupLocalDebuggedVersion :: Var -> DebugM (Maybe Var)
lookupLocalDebuggedVersion x = DebugM $ \dc _ -> do
  let dbgMap = dcLocalDebuggedMap dc
  return $ lookupFM dbgMap x

wrapInStackLambdaAndUseIt ::  CoreExpr ->  DebugM CoreExpr
wrapInStackLambdaAndUseIt exp = do
  stackVar <- mkStackVariable
  _stackVar <- mkStackVariable
  exp' <- useStackVar exp stackVar 
  return $ Lam stackVar (Case (Var stackVar) _stackVar (exprType exp') [(DEFAULT, [], exp')])
  where
    mkStackVariable :: DebugM Id
    mkStackVariable = do
      st <- stackType
      mkSysLocalM stackFString st
      where
        stackFString :: FastString
        stackFString = mkFastString "stack"
       
useStackVar ::  CoreExpr -> Id ->  DebugM CoreExpr
useStackVar e@(Var v) stackVar = do
  tsv <- throwStackVar
  if ((varName tsv) == (varName v))
    then rewriteThrowStackVar
    else checkForDebuggedVersion
  where
    rewriteThrowStackVar    = do
      currentLoc <- currentSrcLoc
      stack <- (currentLoc >: stackVar)
      tv <- throwVar

      -- Using: throwStack :: forall e a . Exception e -> (Stack -> e) -> a
      let throwStackType = varType v
      let ([evar, avar], arrBit) = splitForAllTys throwStackType
      let ([exETy, stToETy], _) = splitFunTys arrBit

      dv <- mkSysLocalM (mkFastString "dv") exETy
      f  <- mkSysLocalM (mkFastString "f") stToETy

      -- /\ e a . \ d :: (Excn e) . \f :: (Stack -> e) . throw a e d (f stack)
      let outExp = mkLams [evar, avar, dv, f] $ 
                      mkApps (mkVarApps (Var tv) [evar, avar, dv]) [mkApps (Var f) [stack]]

      return outExp


    checkForDebuggedVersion = do
      mdv <- debuggedVersion v
      case mdv of
        Just dv -> do
          currentLoc <- currentSrcLoc 
          liftM (callDebuggedVersion dv) (currentLoc >: stackVar)
        Nothing -> return e

useStackVar e@(Lit _)   _ = return e
useStackVar (App e1 e2) stackVar = liftM2 App (useStackVar e1 stackVar) 
                                              (useStackVar e2 stackVar)
useStackVar (Lam x e)   stackVar = liftM (Lam x) (useStackVar e stackVar)
useStackVar (Let b e)   stackVar = liftM2 Let (useStackVarBind b stackVar)
                                               (useStackVar e stackVar)
useStackVar (Case e x ty alts) stackVar 
  = liftM2 (\e' alts' -> Case e' x ty alts') (useStackVar e stackVar)
                                             (mapM (\alt -> useStackVarAlt alt stackVar) alts)
useStackVar (Cast e c) stackVar = liftM (\e' -> Cast e' c) (useStackVar e stackVar)
useStackVar (Note n@(CoreSrcLoc loc) e) stackVar = liftM (Note n) (refineSrcSpan loc $ useStackVar e stackVar)
useStackVar (Note n e) stackVar = liftM (Note n) (useStackVar e stackVar)
useStackVar e@(Type _) _ = return e


useStackVarBind ::  CoreBind -> Id ->  DebugM CoreBind
useStackVarBind (NonRec b e) stackVar = liftM (NonRec b) (useStackVar e stackVar)
useStackVarBind (Rec bs) stackVar     = liftM Rec (mapM (\(b,e) -> liftM ((,)b) (useStackVar e stackVar)) bs)

useStackVarAlt ::  CoreAlt -> Id ->  DebugM CoreAlt
useStackVarAlt (con, bs, e) stackVar = liftM (\e' -> (con, bs, e')) (useStackVar e stackVar)

debuggedVersion ::  Id ->  DebugM (Maybe Var)
debuggedVersion var = do
  anns <- liftCoreM $ findAnnotations deserializeWithData (NamedTarget (varName var))
  case anns of
    (Debugged n:_) -> liftCoreM $ do
                        mna <- thNameToGhcName n
                        maybe (return Nothing) (liftM Just . lookupId) mna
    _              -> lookupLocalDebuggedVersion var

thNameToId :: TH.Name -> CoreM (Maybe Id)
thNameToId thn = (thNameToGhcName thn) >>= maybe (return Nothing) (liftM Just . lookupId)

rdrNameToId :: RdrName -> CoreM Id
rdrNameToId rdrName = lookupName rdrName >>= lookupId

rdrNameToTyCon :: RdrName -> CoreM TyCon
rdrNameToTyCon rdrName = lookupName rdrName >>= lookupTyCon

lookupName :: RdrName -> CoreM Name
lookupName rdrName = do
  hscEnv <- getHscEnv
  liftIO $ initTcForLookup hscEnv (lookupGlobalOccRn rdrName)
  where
    initTcForLookup :: HscEnv -> TcM a -> IO a
    initTcForLookup hsc_env = liftM (expectJust ("T - initTcInteractive:" ++ showRdrName rdrName) . snd) 
                            . initTc hsc_env HsSrcFile False iNTERACTIVE


(>:) :: Location -> Var -> DebugM CoreExpr
srcLoc >: var = do
  pushVar <- stackPushVar
  stackElem <- mkStackElem srcLoc
  return $ mkCoreApps  (Var pushVar) [Var var, stackElem] 

mkStackElem :: Location -> DebugM CoreExpr
mkStackElem location = do
  modUnique <- liftM getUnique (liftCoreM getModule)
  unique    <- getUniqueM  

  let modUniqueInt = mkIntExprInt $ getKey modUnique
  let uniqueInt    = mkIntExprInt $ getKey unique

  encVarNameFS <- liftCoreM (mkStringExprFS $ locVarName location)

  f <- mkStackElemFn
  args <- sequence mkStackElemArgs

  let stackElemType = snd . splitFunTys . varType $ f
  tmpVar <- mkSysLocalM (mkFastString "sl") stackElemType
  return $ Let (NonRec tmpVar (mkCoreApps (Var f) ([modUniqueInt, uniqueInt, encVarNameFS] ++ args))) (Var tmpVar)
  
  where
    srcLoc = locSrcLoc location
    
    (mkStackElemFn, mkStackElemArgs)
      | isGoodSrcLoc srcLoc = (mkKnownStackElem, [ liftCoreM . mkStringExprFS . srcLocFile $ srcLoc
                                                 , return . mkIntExprInt . srcLocLine           $ srcLoc
                                                 , return . mkIntExprInt . succ . srcLocCol     $ srcLoc
                                                 ])
      | otherwise           = (mkUnknownStackElem, [])


callDebuggedVersion :: Id -> CoreExpr -> CoreExpr
callDebuggedVersion debugFn arg = App (Var debugFn) arg

buildLocalDebuggedMap :: Type -> [CoreBind] -> CoreM (FiniteMap Var Var)
buildLocalDebuggedMap stackType = liftM (listToFM . catMaybes) 
                                . mapM (buildLocalDebuggedMapVar stackType)
                                . bindersOfBinds

buildLocalDebuggedMapVar :: Type -> Var -> CoreM (Maybe (Var,Var))
buildLocalDebuggedMapVar stackType x = do
  (dgdans :: [Debugged]) <- getAnns
  case dgdans of
    [] -> do 
      (dans :: [Debug]) <- getAnns
      debugOverride <- dopt Opt_ExplicitCallStackCoreAll `liftM` getDynFlags
      case debugOverride || (not . null $ dans) of
        False -> return $ Nothing
        True ->  do 
          x' <- mkDebuggedVar x stackType
          return $ Just (x, x')
    (Debugged name:_) -> do
      mDebuggedName <- thNameToId name
      return $ fmap (\dn -> (x, dn)) mDebuggedName
  where
    getAnns :: (Typeable a, Data a) => CoreM [a]
    getAnns = findAnnotations deserializeWithData (NamedTarget $ varName x)

    mkDebuggedVar :: Var -> Type -> CoreM Var
    mkDebuggedVar x stackType = do
        let destType = mkFunTy stackType (varType x)

        let srcStr = occNameString . nameOccName . varName $ x
        let destStr = mkFastString $ srcStr ++ "_$_debugged"

        unique' <- getUniqueM
        let x'Name = mkSystemVarName unique' destStr

        return $ mkVar x'Name destType
      where
        mkVar name tipe
          | isGlobalId x   = mkExportedLocalVar (idDetails x) name tipe (idInfo x)
          | otherwise      = flip setVarType tipe
                           . flip setVarName name $ x
#else

import HscTypes
import CoreMonad

stackPassTransform :: ModGuts -> CoreM ModGuts
stackPassTransform = return
#endif

